/*!
 * \file       dia_Lock.cpp
 *
 * \brief      Locking class used for synchronized access in multiple thread environments
 *
 * \details    Locking class used for synchronized access in multiple thread environments
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_LOCK__
#include <common/framework/application/dia_Lock.h>
#endif

#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
#include <errno.h>      //needed by ETIMEDOUT
#endif

using namespace std;

namespace dia {

//------------------------------------------------------------------------------

Lock::Lock ( const char* name )
   : mName(name)
#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
     //mMutex(PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP) //extended initializer lists only available with -std=c++11 or -std=gnu++11
#else
     ,mHandle(OSAL_C_INVALID_HANDLE)
#endif
{
#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
   DIA_DBG_TR_INF("dia::Lock::Lock \"%s\" this=%p", mName.c_str(), this);
   int ret = pthread_mutexattr_init(&mAttr);
   if (0 != ret) DIA_DBG_TR_ERR("dia::Lock::Lock ERROR pthread_mutexattr_init returned %d. \"%s\"", ret, mName.c_str());
   DIA_ASSERT(0 == ret);
   ret = pthread_mutexattr_settype(&mAttr, PTHREAD_MUTEX_RECURSIVE);
   if (0 != ret) DIA_DBG_TR_ERR("dia::Lock::Lock ERROR pthread_mutexattr_settype returned %d. \"%s\"", ret, mName.c_str());
   DIA_ASSERT(0 == ret);
   ret = pthread_mutex_init(&mMutex, &mAttr);
   if (0 != ret) DIA_DBG_TR_ERR("dia::Lock::Lock ERROR pthread_mutex_init returned %d. \"%s\"", ret, mName.c_str());
   DIA_ASSERT(0 == ret);
#else
   tS32 s32Result = OSAL_s32SemaphoreCreate(mName.c_str(), &mHandle,1);
   DIA_ASSERT(OSAL_OK == s32Result);
#endif
}

//------------------------------------------------------------------------------

Lock::Lock ( const std::string& name )
   : mName(name)
#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
     //mMutex(PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP) //extended initializer lists only available with -std=c++11 or -std=gnu++11
#else
     ,mHandle(OSAL_C_INVALID_HANDLE)
#endif
{
#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
   DIA_DBG_TR_INF("dia::Lock::Lock \"%s\" this=0x%08X", mName.c_str(), this);
   int ret = pthread_mutexattr_init(&mAttr);
   if (0 != ret) DIA_DBG_TR_ERR("dia::Lock::Lock ERROR pthread_mutexattr_init returned %d. \"%s\"", ret, mName.c_str());
   DIA_ASSERT(0 == ret);
   ret = pthread_mutexattr_settype(&mAttr, PTHREAD_MUTEX_RECURSIVE);
   if (0 != ret) DIA_DBG_TR_ERR("dia::Lock::Lock ERROR pthread_mutexattr_settype returned %d. \"%s\"", ret, mName.c_str());
   DIA_ASSERT(0 == ret);
   ret = pthread_mutex_init(&mMutex, &mAttr);
   if (0 != ret) DIA_DBG_TR_ERR("dia::Lock::Lock ERROR pthread_mutex_init returned %d. \"%s\"", ret, mName.c_str());
   DIA_ASSERT(0 == ret);
#else
   tS32 s32Result = OSAL_s32SemaphoreCreate(mName.c_str(), &mHandle,1);
   DIA_ASSERT(OSAL_OK == s32Result);
#endif
}

//------------------------------------------------------------------------------

Lock::~Lock ( void )
{
#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
   DIA_DBG_TR_INF("dia::Lock::~Lock \"%s\" this=0x%08X", mName.c_str(), this);
   DIA_ASSERT(0 == pthread_mutex_destroy(&mMutex));
   DIA_ASSERT(0 == pthread_mutexattr_destroy(&mAttr));
#else
   // destructors should not throw exceptions (--> lint), but functions called in the
   // could possibly throw excepections. So we catch them here and raise an assert
   _BP_TRY_BEGIN
   {
      if( OSAL_C_INVALID_HANDLE != mHandle )
      {
         DIA_ASSERT( OSAL_OK == OSAL_s32SemaphoreClose(mHandle) );
         DIA_ASSERT( OSAL_OK == OSAL_s32SemaphoreDelete(mName.c_str()) );
      }
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia::Lock::~Lock !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
#endif
}

//------------------------------------------------------------------------------

tDiaResult
Lock::unlock ( void )
{
#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
   DIA_DBG_TR_INF("dia::Lock::unlock \"%s\" this=0x%08X", mName.c_str(), this);

   int retVal = pthread_mutex_unlock(&mMutex); //lint !e455: Warning: A thread mutex that had not been locked is being unlocked
//   DIA_DBG_TR_INF("dia::Lock \"%s\" unlocked by thread %d", mName.c_str(), (int) OSAL_ThreadWhoAmI());
   if (0 != retVal)
   {
      DIA_TR_ERR("### Lock::unlock pthread_mutex_unlock returned %d", retVal);
      return DIA_FAILED;
   }
   return DIA_SUCCESS;
#else
   tS32 result = OSAL_s32SemaphorePost( mHandle );

   DIA_TR_INF("dia::Lock::unlock OSAL_s32SemaphorePost returned 0x%08X", (tU32)result);
   return ( OSAL_OK == result ) ? DIA_SUCCESS : DIA_FAILED;
#endif
}

//------------------------------------------------------------------------------

tDiaResult
Lock::lock ( void )
{
   tDiaResult retCode = DIA_SUCCESS;

#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
   DIA_DBG_TR_INF("dia::Lock::lock \"%s\" this=0x%08X", mName.c_str(), this);
   DIA_ASSERT(0 == pthread_mutex_lock(&mMutex));
//   DIA_DBG_TR_INF("dia::Lock \"%s\" locked by thread %d", mName.c_str(), (int) OSAL_ThreadWhoAmI());
#else
   tS32 result = OSAL_s32SemaphoreWait( mHandle, OSAL_C_TIMEOUT_FOREVER );

   DIA_TR_INF("Locking Mutex %s. OSAL_s32SemaphoreWait returned 0x%08X", mName.c_str(), (tU32)result);

   if ( result != OSAL_OK )
   {
      retCode = (OSAL_u32ErrorCode() == OSAL_E_TIMEOUT) ? DIA_E_LOCK_TIMEOUT : DIA_E_LOCK_FAILED;
   }
#endif

   return retCode;  //lint !e454: Warning: A thread mutex has been locked but not unlocked
}

//------------------------------------------------------------------------------

tDiaResult
Lock::lock ( tU32 timeoutInMilliSec )
{
   tDiaResult retCode = DIA_SUCCESS;

#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
   DIA_DBG_TR_INF("dia::Lock::lock(timeoutMsec=%u) \"%s\" (ThreadID=0x%08X) this=0x%08X", timeoutInMilliSec, mName.c_str(), (int) pthread_self(), this);

   timespec timeSpec;

   timeSpec.tv_sec  = (long)  timeoutInMilliSec / 1000;
   timeSpec.tv_nsec = (long) (timeoutInMilliSec % 1000) * 1000000;

   if ( pthread_mutex_timedlock(&mMutex,&timeSpec) == ETIMEDOUT )
   {
      DIA_DBG_TR_INF("dia::Lock::lock(timeoutMsec=%u) \"%s\" returned DIA_E_LOCK_TIMEOUT", timeoutInMilliSec, mName.c_str() );
      retCode = DIA_E_LOCK_TIMEOUT;
   }
#else
   tS32 result = OSAL_s32SemaphoreWait( mHandle, /*OSAL_C_TIMEOUT_FOREVER*/ timeoutInMilliSec );

   if ( result != OSAL_OK )
   {
      retCode = (OSAL_u32ErrorCode() == OSAL_E_TIMEOUT) ? DIA_E_LOCK_TIMEOUT : DIA_E_LOCK_FAILED;
   }
#endif

   return retCode;
}

}

////------------------------------------------------------------------------------
//
//dia_Lock::dia_Lock ( tCString name /*, enLockState mode*/ )
//   : mName(name)
//#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
//     //mMutex(PTHREAD_RECURSIVE_MUTEX_INITIALIZER_NP) //extended initializer lists only available with -std=c++11 or -std=gnu++11
//#else
//     ,mHandle(OSAL_C_INVALID_HANDLE)
//#endif
//{
//#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
////#ifdef __DIA_UNIT_TESTING__
////   DIA_TR_INF("dia_Lock::dia_Lock \"%s\" this=0x%08X", mName.c_str(), this);
////#endif
//   int ret = pthread_mutexattr_init(&mAttr);
////#ifdef __DIA_UNIT_TESTING__
////   if (0!=ret) DIA_TR_ERR("dia_Lock::dia_Lock ERROR pthread_mutexattr_init returned %d. \"%s\"", ret, mName.c_str());
////#endif
//   DIA_ASSERT(0==ret);
//   ret=pthread_mutexattr_settype(&mAttr, PTHREAD_MUTEX_RECURSIVE);
////#ifdef __DIA_UNIT_TESTING__
////   if (0!=ret) DIA_TR_ERR("dia_Lock::dia_Lock ERROR pthread_mutexattr_settype returned %d. \"%s\"", ret, mName.c_str());
////#endif
//   DIA_ASSERT(0==ret);
//   ret=pthread_mutex_init(&mMutex, &mAttr);
////#ifdef __DIA_UNIT_TESTING__
////   if (0!=ret) DIA_TR_ERR("dia_Lock::dia_Lock ERROR pthread_mutex_init returned %d. \"%s\"", ret, mName.c_str());
////#endif
//   DIA_ASSERT(0==ret);
//#else
//   tS32 s32Result = OSAL_s32SemaphoreCreate( mName.c_str(), &mHandle, /*(mode == enUnlocked) ?*/ 1 /*: 0*/ );
//   DIA_ASSERT(OSAL_OK == s32Result);
//#endif
//}
//
////------------------------------------------------------------------------------
//
//dia_Lock::~dia_Lock ( void )
//{
//#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
//#ifdef __DIA_UNIT_TESTING__
//   DIA_TR_INF("dia_Lock::~dia_Lock \"%s\" this=0x%08X", mName.c_str(), this);
//#endif
//   DIA_ASSERT(0==pthread_mutex_destroy(&mMutex));
//   DIA_ASSERT(0==pthread_mutexattr_destroy(&mAttr));
//#else
//   // destructors should not throw exceptions (--> lint), but functions called in the
//   // could possibly throw excepections. So we catch them here and raise an assert
//   _BP_TRY_BEGIN
//   {
//      if( OSAL_C_INVALID_HANDLE != mHandle )
//      {
//         DIA_ASSERT( OSAL_OK == OSAL_s32SemaphoreClose(mHandle) );
//         DIA_ASSERT( OSAL_OK == OSAL_s32SemaphoreDelete(mName.c_str()) );
//      }
//   }
//   _BP_CATCH_ALL
//   {
//      DIA_TR_ERR("EXCEPTION CAUGHT: dia_Lock::~dia_Lock !!!");
//      DIA_ASSERT_ALWAYS();
//   }
//   _BP_CATCH_END
//#endif
//}
//
////------------------------------------------------------------------------------
//
//tDiaResult
//dia_Lock::unlock ( void )
//{
//#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
//#ifdef __DIA_UNIT_TESTING__
//   DIA_TR_INF("dia_Lock::unlock \"%s\" this=0x%08X", mName.c_str(), this);
//#endif
//
//#if 0  /* 2015.09.02: Ignore the problem this time (assert has been commented out), the issue will be resolved by gib2hi. */
//   DIA_ASSERT(0==pthread_mutex_unlock(&mMutex)); //lint !e455: Warning: A thread mutex that had not been locked is being unlocked
//#else
//   int retVal = pthread_mutex_unlock(&mMutex); //lint !e455: Warning: A thread mutex that had not been locked is being unlocked
//#ifndef __DIA_UNIT_TESTING__
//   DIA_TR_INF("dia_Lock \"%s\" unlocked by thread %d", mName.c_str(), (int) OSAL_ThreadWhoAmI());
//#endif
//   if (0!=retVal)
//   {
//      DIA_TR_ERR("### dia_Lock::unlock pthread_mutex_unlock returned %d", retVal);
//      return DIA_FAILED;
//   }
//#endif
//   return DIA_SUCCESS;
//#else
//   tS32 result = OSAL_s32SemaphorePost( mHandle );
//
//   DIA_TR_INF("dia_Lock::unlock OSAL_s32SemaphorePost returned 0x%08X", (tU32)result);
//   return ( OSAL_OK == result ) ? DIA_SUCCESS : DIA_FAILED;
//#endif
//}
//
////------------------------------------------------------------------------------
//
//tDiaResult
//dia_Lock::lock ( void )
//{
//   tDiaResult retCode = DIA_SUCCESS;
//
//#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
//#ifdef __DIA_UNIT_TESTING__
//   DIA_TR_INF("dia_Lock::lock \"%s\" this=0x%08X", mName.c_str(), this);
//#endif
//   DIA_ASSERT(0==pthread_mutex_lock(&mMutex));
//#ifndef __DIA_UNIT_TESTING__
//   DIA_TR_INF("dia_Lock \"%s\" locked by thread %d", mName.c_str(), (int) OSAL_ThreadWhoAmI());
//#endif
//#else
//   tS32 result = OSAL_s32SemaphoreWait( mHandle, OSAL_C_TIMEOUT_FOREVER );
//
//   DIA_TR_INF("Locking Mutex %s. OSAL_s32SemaphoreWait returned 0x%08X", mName.c_str(), (tU32)result);
//
//   if ( result != OSAL_OK )
//   {
//      retCode = (OSAL_u32ErrorCode() == OSAL_E_TIMEOUT) ? DIA_E_LOCK_TIMEOUT : DIA_E_LOCK_FAILED;
//   }
//#endif
//
//   return retCode;  //lint !e454: Warning: A thread mutex has been locked but not unlocked
//}
//
////------------------------------------------------------------------------------
//
//tDiaResult
//dia_Lock::lock ( tU32 timeoutInMilliSec )
//{
//   tDiaResult retCode = DIA_SUCCESS;
//
//#ifdef VARIANT_S_FTR_ENABLE_LOCK_AS_PTHREAD_MUTEX
//#ifdef __DIA_UNIT_TESTING__
//   DIA_TR_INF("dia_Lock::lock(timeoutMsec=%u) \"%s\" (ThreadID=0x%08X) this=0x%08X", timeoutInMilliSec, mName.c_str(), (int) pthread_self(), this);
//#endif
//
//   timespec timeSpec;
//
//   timeSpec.tv_sec  = (long)  timeoutInMilliSec / 1000;
//   timeSpec.tv_nsec = (long) (timeoutInMilliSec % 1000) * 1000000;
//
//   if ( pthread_mutex_timedlock(&mMutex,&timeSpec) == ETIMEDOUT )
//   {
//#ifdef __DIA_UNIT_TESTING__
//      DIA_TR_INF("dia_Lock::lock(timeoutMsec=%u) \"%s\" returned DIA_E_LOCK_TIMEOUT", timeoutInMilliSec, mName.c_str() );
//#endif
//      retCode = DIA_E_LOCK_TIMEOUT;
//   }
//#else
//   tS32 result = OSAL_s32SemaphoreWait( mHandle, /*OSAL_C_TIMEOUT_FOREVER*/ timeoutInMilliSec );
//
//   if ( result != OSAL_OK )
//   {
//      retCode = (OSAL_u32ErrorCode() == OSAL_E_TIMEOUT) ? DIA_E_LOCK_TIMEOUT : DIA_E_LOCK_FAILED;
//   }
//#endif
//
//   return retCode;
//}
