/**
 * \file      dia_File.cpp
 *
 * \brief     { File operations }
 *
 * \details   {File operations}
 *
 * \author    bhs1hi, kaa1hi
 * \date      Jan 25, 2014
 *
 * \copyright Robert Bosch Car Multimedia 2013
 */

#ifndef __INCLUDED_DIA_FILE__
#include "dia_File.h"
#endif

#include <sys/types.h> //lint !e451 !e537 repeatedly included header file without standard include guard
#include <sys/stat.h>  //lint !e451 !e537 repeatedly included header file without standard include guard
#include <grp.h>       //lint !e451 !e537 repeatedly included header file without standard include guard
#include <fcntl.h>     //lint !e451 !e537 repeatedly included header file without standard include guard
#include <unistd.h>    //lint !e451 !e537 repeatedly included header file without standard include guard
#include <errno.h>
#include <limits>
#include <fstream>

#define DIA_FILE_BUFF_SIZE    (5*1000)

using namespace dia;

//-----------------------------------------------------------------------------

dia_File::dia_File ( tCString fileWithPath )
   : mFileName(fileWithPath),
     mFd(NULL)
{
   ScopeTrace trc("dia_File::dia_File(tCString)");
   DIA_TR_INF("File Name: \"%s\"",mFileName.c_str());
}

//-----------------------------------------------------------------------------

dia_File::dia_File ( const std::string& fileWithPath )
   : mFileName(fileWithPath),
     mFd(NULL)
{
   ScopeTrace trc("dia_File::dia_File(const std::string&)");
   DIA_TR_INF("File Name: \"%s\"",mFileName.c_str());
}

//-----------------------------------------------------------------------------

dia_File::~dia_File ( void )
{
   mFd = NULL;
}

//-----------------------------------------------------------------------------

bool
dia_File::doesExist ( void ) const
{
   ScopeTrace trc("dia_File::doesExist");

   struct stat buffer;
   bool retVal = (stat(mFileName.c_str(), &buffer)==0) ? true : false;

   DIA_TR_INF("dia_File::doesExist returned %s", (retVal) ? "\"YES\"" : "\"NO\"" );

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::open ( tCString options )
{
   ScopeTrace trc("dia_File::open");

   mFd = fopen(mFileName.c_str(), options);
   tDiaResult retVal = (mFd) ? DIA_SUCCESS : DIA_E_FILE_OPEN_FAILED;

   DIA_TR_INF("dia_File::open returned 0x%08x (name=\"%s\", options %s)", retVal, mFileName.c_str(), options );

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::close ( void )
{
   ScopeTrace trc("dia_File::close");

   if ( !mFd ) return DIA_FAILED;

   tDiaResult retCode = (fclose(mFd) != EOF) ? DIA_SUCCESS : DIA_E_FILE_CLOSE_FAILED;
   DIA_TR_INF("dia_File::close returned 0x%08x", retCode );

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::remove ( void )
{
   ScopeTrace trc("dia_File::remove");

   tDiaResult retVal = ( ::remove(mFileName.c_str()) == 0 ) ? DIA_SUCCESS : DIA_FAILED;

   DIA_TR_INF("dia_File::remove returned 0x%08x", retVal );

   return retVal;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::getLine ( std::string& line )
{
   if ( !mFd ) return DIA_E_INVALID_FILE_HANDLE;

   char buff[DIA_FILE_BUFF_SIZE] = {0};
   if ( fgets(buff, sizeof(buff), mFd) != NULL )
   {
      line.assign(buff, strnlen(buff, DIA_FILE_BUFF_SIZE));
   }
   else return DIA_FAILED;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::getLine ( tChar *buff,  tS32 buffSize, tU32 *lenReadStr )
{
   if ( !buff ) return DIA_E_INVALID_POINTER;
   if ( !lenReadStr ) return DIA_E_INVALID_LENGTH;
   if ( !mFd ) return DIA_E_INVALID_FILE_HANDLE;

   if ( fgets(buff, buffSize, mFd) != NULL )
   {
      *lenReadStr = (tU32) (strlen (buff));
   }
   else return DIA_FAILED;

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

tDiaResult
   dia_File::rawRead ( tU8* data, size_t* dataLength )
{
   ScopeTrace trc("dia_File::rawRead");

   if ( !mFd ) return DIA_E_INVALID_FILE_HANDLE;
   if ( !dataLength ) return DIA_E_INVALID_LENGTH;
   if ( !data ) return DIA_E_INVALID_POINTER;

   DIA_TR_INF("dia_File::rawRead maxBytes = %zu", *dataLength );

   tDiaResult retCode = DIA_FAILED;

   size_t numBytesRead = fread(data, sizeof(tU8), *dataLength, mFd);
   if ( numBytesRead > 0 )
   {
      *dataLength = (tU32) numBytesRead;
      retCode = DIA_SUCCESS;
   }
   else
   {
      DIA_TR_ERR("dia_File::rawRead: fread failed (numBytesRead = %zu, errno=%s)", numBytesRead,strerror(errno) );
   }

   DIA_TR_INF("dia_File::rawRead returned 0x%08x", retCode );

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::writeLine ( tChar *line,  tU32 /*u32LineSize*/ )
{
   ScopeTrace trc("dia_File::writeLine");

   tDiaResult retCode = DIA_FAILED;

   if ( mFd && line )
   {
      if ( fprintf(mFd, "%s\n", line ) > 0 )
      {
          retCode = DIA_SUCCESS;
      }
   }

   DIA_TR_INF("dia_File::writeLine returned 0x%08x", retCode );

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::rawWrite ( tU8* data, size_t* dataLength )
{
   ScopeTrace trc("dia_File::rawWrite");

   if ( !mFd ) return DIA_E_INVALID_FILE_HANDLE;
   if ( !dataLength ) return DIA_E_INVALID_LENGTH;
   if ( !data ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   tU32 numBytesWritten = (tU32) fwrite(data, sizeof(tU8), *dataLength, mFd);
   if ( numBytesWritten > 0 )
   {
      DIA_TR_INF("dia_File::rawWrite numBytesWritten = %u", numBytesWritten );

      *dataLength = numBytesWritten;
      retCode = DIA_SUCCESS;
   }

   DIA_TR_INF("dia_File::rawWrite returned 0x%08x", retCode );

   return retCode;
}

//-----------------------------------------------------------------------------

bool
dia_File::endOfFile ( void )
{
   ScopeTrace trc("dia_File::endOfFile");

   if ( !mFd )
   {
      DIA_TR_ERR("dia_File::endOfFile returned true. File descriptor is null.");
      return true;
   }

   bool retCode = ( feof(mFd) ) ? true : false;

   DIA_TR_INF("dia_File::endOfFile returned %s", (retCode) ? "true" : "false" );

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::setGroupByName ( tCString groupName )
{
   ScopeTrace trc("dia_File::setGroupByName");

   int filedesc = ::open(mFileName.c_str(),O_RDWR);

   if ( filedesc == -1 )
   {
      DIA_TR_ERR("##### INVALID FILE DESCRIPTOR #####");
      return DIA_E_INVALID_FILE_HANDLE;
   }

   tDiaResult retCode = DIA_FAILED;

   struct group* pGroupInfo = ::getgrnam(groupName);

   if ( pGroupInfo )
   {
      if ( ::fchown(filedesc, (uid_t) -1, pGroupInfo->gr_gid) != -1 )
      {
         retCode = DIA_SUCCESS;
      }
      else
      {
         //TODO: Add trace with errno translation here.
         DIA_TR_ERR("##### FAILED TO MODIFY OWNWERSHIP #####");
      }
   }
   else
   {
      //TODO: Add trace with errno translation here.
      DIA_TR_ERR("##### INVALID GROUP NAME #####");
   }

   (void) ::close(filedesc);

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::setGroupByID ( tU32 groupID )
{
   ScopeTrace trc("dia_File::setGroupByID");

   int filedesc = ::open(mFileName.c_str(),O_RDWR);

   if ( filedesc == -1 )
   {
      DIA_TR_ERR("##### INVALID FILE DESCRIPTOR #####");
      return DIA_E_INVALID_FILE_HANDLE;
   }

   tDiaResult retCode = DIA_FAILED;

   if ( ::fchown(filedesc, (uid_t) -1, (gid_t) groupID) != -1 )
   {
      retCode = DIA_SUCCESS;
   }
   else
   {
      //TODO: Add trace with errno translation here.
      DIA_TR_ERR("##### FAILED TO MODIFY OWNWERSHIP #####");
   }

   (void) ::close(filedesc);

   return retCode;
}

//-----------------------------------------------------------------------------

tDiaResult
dia_File::changeMode ( dia_FileMode mode )
{
   ScopeTrace trc("dia_File::changeMode");

   int filedesc = ::open(mFileName.c_str(),O_RDWR);

   if ( filedesc == -1 )
   {
      DIA_TR_ERR("##### INVALID FILE DESCRIPTOR #####");
      return DIA_E_INVALID_FILE_HANDLE;
   }

   tDiaResult retCode = DIA_FAILED;

   // Following code might be moved to dia_File.chmod(mode_t mode)
   if ( ::fchmod(filedesc,mode) != -1 )
   {
      retCode = DIA_SUCCESS;
   }
   else
   {
      //TODO: Add trace with errno translation here.
      DIA_TR_ERR("##### FAILED TO CHANGE MODE #####");
   }

   (void) ::close(filedesc);

   return retCode;
}

//-----------------------------------------------------------------------------

tU32
dia_File::getSize(void)
{
   ScopeTrace trc("dia_File::getSize");

   tU32 retVal = std::numeric_limits<tU32>::max(); //error
   ifstream file(mFileName.c_str(),ios::binary);

   if (file)
   {
      file.seekg((long long) 0,ios::end);

      int size = (int)(file.tellg()); // This limits the filesize to ~2GB which should be acceptable for now
      if (-1==size)
      {
         DIA_TR_ERR("dia_File::getSize std::istream::tellg returned -1 for file %s.", mFileName.c_str());
      }
      else
      {
         DIA_TR_INF("dia_File::getSize File size is %d bytes", size);
         if (size>=0)
         {
            retVal = (tU32)size;
         }
         else
         {
            DIA_TR_ERR("dia_File::getSize Incorrect size %d for file %s.", size, mFileName.c_str());
         }
      }
   }
   else
   {
      DIA_TR_ERR("dia_File::getSize File '%s' cannot be opened.", mFileName.c_str());
   }

   return retVal;
}
