/*!
 * \file       dia_CommandController.h
 *
 * \brief      Manager class that manages different kinds of command plugins and commands
 *
 * \details    Manager class that manages and maintains different kinds of diagnostic command plugins and commands
 *
 * \component  Diagnostics
 *
 * \ingroup    diaCoreAppFrw
 *
 * \copyright  (c) 2012-2017 Robert Bosch GmbH
 *
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 */

#ifndef __INCLUDED_DIA_COMMAND_CONTROLLER__
#include <common/framework/application/dia_CommandController.h>
#endif

#ifndef __INCLUDED_DIA_COMMAND__
#include <common/framework/application/dia_Command.h>
#endif

#ifndef __INCLUDED_DIA_PREDICATE__
#include <common/framework/application/dia_Predicate.h>
#endif

#ifndef __INCLUDED_DIA_LOCK_SCOPE__
#include <common/framework/application/dia_LockScope.h>
#endif

using namespace dia;

DIA_IMPL_SINGLETON(dia_CommandController)

#ifndef __DIA_UNIT_TESTING__

dia_CommandController*
getInstanceOfCommandController ( void )
{
   return dia_CommandController::getInstance();
}

#endif

//-----------------------------------------------------------------------------

dia_CommandController::dia_CommandController ( void )
   : mSyncObj("dia_CommandController_LK")
{
   ScopeTrace trc("dia_CommandController::dia_CommandController()");
}

//-----------------------------------------------------------------------------

dia_CommandController::~dia_CommandController ( void )
{
   _BP_TRY_BEGIN
   {
      mPluginRep.clear();
   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_CommandController::~dia_CommandController !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommandController::addCommandControllerPlugin ( dia_CommandControllerPlugin* pPlugin )
{
   ScopeTrace oTrace("dia_CommandController::addCommandControllerPlugin()");

   LockScope lock(mSyncObj);

   if ( !pPlugin ) return DIA_E_INVALID_POINTER;

   tDiaResult retCode = DIA_FAILED;

   std::map<dia_enCmdCtrlPluginType,dia_CommandControllerPlugin*>::iterator iter = mPluginRep.find(pPlugin->getID());
   if ( iter == mPluginRep.end() )
   {
      DIA_TR_INF("#######################################################");
      DIA_TR_INF("#");
      DIA_TR_INF("# ADDING COMMAND CONTROLLER PLUGIN \"%d\"", pPlugin->getID());
      DIA_TR_INF("#");
      DIA_TR_INF("#######################################################");
      mPluginRep[pPlugin->getID()] = pPlugin;
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tDiaResult
dia_CommandController::removeCommandControllerPlugin ( dia_enCmdCtrlPluginType type )
{
   ScopeTrace oTrace("dia_CommandController::removeCommandControllerPlugin()");

   LockScope lock(mSyncObj);

   tDiaResult retCode = DIA_E_NOT_FOUND;

   std::map<dia_enCmdCtrlPluginType,dia_CommandControllerPlugin*>::iterator iter = mPluginRep.find(type);
   if ( iter != mPluginRep.end() )
   {
      mPluginRep.erase(type);
      retCode = DIA_SUCCESS;
   }

   return retCode;
}

//------------------------------------------------------------------------------

tU16
dia_CommandController::numberOfCommandControllerPlugins ( void ) const
{
   LockScope lock(mSyncObj);
   return (tU16) mPluginRep.size();
}

//------------------------------------------------------------------------------

tU16
dia_CommandController::numberOfCommands ( void ) const
{
   LockScope lock(mSyncObj);
   return (tU16) mCommandRep.size();
}

//------------------------------------------------------------------------------

std::list<dia_Command*>&
dia_CommandController::getCommands ( void )
{
   return mCommandRep;
}

//-----------------------------------------------------------------------------

void
dia_CommandController::pushFront ( dia_Command* pCmd )
{
   ScopeTrace trc("dia_CommandController::push_front()");
   LockScope lock(mSyncObj);
   mCommandRep.push_front(pCmd);
}

//-----------------------------------------------------------------------------

void
dia_CommandController::pushBack ( dia_Command* pCmd )
{
   ScopeTrace trc("dia_CommandController::push_back()");
   LockScope lock(mSyncObj);
   mCommandRep.push_back(pCmd);
}

//-----------------------------------------------------------------------------

tDiaResult
dia_CommandController::addCommand ( dia_Command* pCmd )
{
   ScopeTrace trc("dia_CommandController::addCommand()");

   if ( !pCmd ) return DIA_E_INVALID_POINTER;

   // now evaluate all the predicates attached to this command
   pCmd->update();

   if ( pCmd->isReady() )
   {
      // all predicates are matched so we can execute the command immediately
      DIA_TR_INF( "### Executing Command !!!");
      (void) pCmd->execute();

      // command was executed so we need to deallocate its memory
      delete pCmd;
   }
   else
   {
      pushBack(pCmd);
   }

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

void
dia_CommandController::update ( dia_UID predicateID, const std::vector<tU32>& args )
{
   ScopeTrace trc("dia_CommandController::update(predicateID,std::vector<tU32>&)");

   LockScope lock(mSyncObj);

   DIA_TR_INF("##### UPDATE RECEIVED FOR PREDICATE 0x%08X", predicateID);

   std::list<dia_Command*> obsoleteCmds;
   std::list<dia_Command*>::iterator cmdIter;

   // iterate over all pending commands and update them if they are having the given predicate attached
   for ( cmdIter=mCommandRep.begin(); cmdIter != mCommandRep.end(); ++cmdIter )
   {
	  if(!(*cmdIter)) continue;    //Check for NULL pointer exception
      dia_Command* pCmd = (*cmdIter);
      if ( !pCmd ) continue;

      // check if the given predicate is attached to the command
      dia_Predicate* pPredicate = pCmd->queryPredicate(predicateID);
      if ( pPredicate )
      {
         DIA_TR_INF( "### Predicate 0x%08X Found !!!", predicateID);

         // update the predicate of the command
         pPredicate->update(args);

         // now check if all predicates of the command are matched. if so we can execute the command
         if ( pCmd->isReady() )
         {
            // run the command (we assume synchronous execution)
            pCmd->execute();
            obsoleteCmds.push_back(pCmd);
         }
      }
      else
      {
         DIA_TR_INF( "### Predicate 0x%08X __NOT__ Found !!!", predicateID);
      }
   }

   for ( cmdIter=obsoleteCmds.begin(); cmdIter != obsoleteCmds.end(); ++cmdIter )
   {
      std::list<dia_Command*>::iterator iter = std::find(mCommandRep.begin(), mCommandRep.end(), (*cmdIter));
      if ( iter != mCommandRep.end() )
      {
         mCommandRep.erase(iter);
      }
      delete (*cmdIter);
    }
    obsoleteCmds.clear();
}

//-----------------------------------------------------------------------------

void
dia_CommandController::destroy ( void )
{
   LockScope lock(mSyncObj);

   std::list<dia_Command*>::iterator listIter = mCommandRep.begin();
   for ( ; listIter != mCommandRep.end(); ++listIter ) delete (*listIter);
   mCommandRep.clear();

   std::map<dia_enCmdCtrlPluginType,dia_CommandControllerPlugin*>::iterator mapIter = mPluginRep.begin();
   for ( ; mapIter != mPluginRep.end(); ++mapIter ) delete mapIter->second;
   mPluginRep.clear();
}

