/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief Routine Control Listener-Interface - Diagnosis Library - Introduced in MFD Nav
*
* Important:
* Listeners run in the applications body thread when a message arrives, so don't
* block for a long time and delegate longer running calculations to worker threads.
*
* (c) 2008 Blaupunkt Werke GmbH
*
* \Todo 2008-07-10 Add documentation, extend scenarios, component reaction
*//****************************************************************************/

#ifndef DIAGLIB_ROUTINE_CONTROL_LISTENER_H
#define DIAGLIB_ROUTINE_CONTROL_LISTENER_H

#ifndef __INCLUDED_DIAGLIB_COMMON__
#include "diaglib_common.h"
#endif

#ifndef __INCLUDED_DIAGLIB_ENUMERATIONS__
#include "Enumerations.h"
#endif

namespace diaglib {

/**************************************************************************//**
* \brief Listener interface for incoming Routine control requests
*
* Listener interface that is used to handle Routine control message events.
* Derive from this interface and implement the event handler according to your
* needs.
* See this sequence diagram for typical calling/message sequence in case of short
* tests:
* \image html seqRoutineCtrlShort.png
* See this sequence diagram for typical calling/message sequence in case of long
* tests:
* \image html seqRoutineCtrlLong.png
*
* \ingroup        diaglibAPI
*//****************************************************************************/
class tclRoutineControlListenerIF
{
public:
  /***********************************************************************//**
   * \brief Incoming Get Routine result event handler
   *
   * Is called on incoming "Get Routine Result" requests. Components reaction
   * should be one of the following:
   * \li You found the routine results: Send back a RoutineResult message
   *     containing poMsgContext, tenRoutineCtrlResult (PASSED or
   *     FAILED)and the result vector as specififed in the ID spec.
   * \li You can't find the routine results: Send back a RoutineResult
   *     message containing poMsgContext, tenRoutineCtrlResult
   *     (NO_RESULT_AVAILABLE) and an ResultVector containing exactly one
   *     parameter of type U8_ERROR and the value ROUTINE_NEVER_RAN or
   *     ROUTINE_STILL_RUNNING.
   *
   * \param[in]      u32RoutineId Routine control ID of the incoming request
   * \param[in]      poMsgContext Internal message context. Just return exactly
   *                 this context when you answer to this request.
   *
   * \return         tU32 ResultCode, currently not checked. Please return 0.
   * \todo           Insert scenario description and link sequence diagrams
   *//*************************************************************************/
   virtual tU32 vOnGetRoutineResult (
                                      tU32 u32RoutineId,
                                      tContext MsgContext
                                    ) = 0;

  /***********************************************************************//**
   * \brief Incoming Routine control abort event handler
   *
   * Is called on incoming "Abort routine control" requests. Components
   * reaction should be one of the following:
   * \li The routine is running: Inform the routine (Worker Thread) that is
   *     supposed to stop. After it stopped in fact or immediately before it
   *     does send a RoutineResult message containig the following:
   *     poMsgContext, tenRoutineCtrlResult (ABORTED) and an empty
   *     result vector.
   * \li The routine is NOT running: Send back a RoutineResult message
   *     containing poMsgContext, tenRoutineCtrlResult
   *     (COULD_NOT_BE_ABORTED) and a ResultVector containing exactly one
   *     parameter of type U8_ERROR and the value ROUTINE_NOT_RUNNING.
   *
   * \param[in]      u32RoutineId RoutineID diagnosis wants to be aborted
   * \param[in]      poMsgContext Internal message context. Just return exactly
   *                 this context when you answer to this request.
   *
   * \return         tU32 ResultCode, currently not checked. Please return 0.
   * \todo           Insert scenario description and link sequence diagrams
   *//*************************************************************************/
   virtual tU32 vOnRoutineControlAbort (
                                         tU32 u32RoutineId,
                                         tContext MsgContext
                                       ) = 0;

  /***********************************************************************//**
   * \brief Incoming Routine control start event handler
   *
   * Is called on incoming "Start routine control" requests. Components
   * reaction should be one of the following:
   * \li Start long running test and answer with ROUTINE_CONTROL_STARTED
   * \li Execute short running test (one second or less) and respond with
   *     ROUTINE_CONTROL_OK and result
   * \li In case of error answer with ROUTINE_CONTROL_NOT_OK and provide
   *     internal error code in vector.
   *
   * \param[in]      u32RoutineId Routine ID diagnosis wants to be started
   * \param[in]      u32Bitmask Bitmask containing flags that specify how the
   *                 routine should be runned. The ID dependent part of this
   *                 bitmask is described in the ID spec.
   * \param[in]      oParamVector Parameter vector for the routine that should
   *                 be started. Defined in the ID spec.
   * \param[in]      poMsgContext Internal message context. Just return exactly
   *                 this context when you answer to this request.
   *
   * \return         tU32 ResultCode, currently not checked. Please return 0.
   * \todo           Insert scenario description and link sequence diagrams
   *//*************************************************************************/
   virtual tU32 vOnRoutineControlStart (
                                         tU32 u32RoutineId,
                                         tU32 u32Bitmask,
                                         const tclParameterVector& oParamVector,
                                         tContext MsgContext
                                       ) = 0;

   virtual ~tclRoutineControlListenerIF() {}
};
}
#endif // DIAGLIB_ROUTINE_CONTROL_LISTENER_H



