/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief Message Context manager - Diagnosis Library - Introduced in MFD Nav
*
* \todo Maybe change timer to 5 second steps to improve performance
*
* 25.08.2008
* Adding timing functionality. When activated using the feature-switch this
* traces out the time a message took until it was responded to.
* Feature switch: VARIANT_S_FTR_ENABLE_FEAT_DIAGLIB_MSG_TIMING
*
* (c) 2008 Blaupunkt Werke GmbH
*//****************************************************************************/

#ifdef DIAGLIB_FILE_NUMBER
#undef DIAGLIB_FILE_NUMBER
#endif
#define DIAGLIB_FILE_NUMBER F_DIAGLIB_MESSAGE_CONTEXT_MANAGER

#include "tclMessageContextManager.h"

#define REG_S_IMPORT_INTERFACE_GENERIC
#include "reg_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#include "tclMessageContext.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ET_DIAGLIB_GENERAL
#include "trcGenProj/Header/tclMessageContextManager.cpp.trc.h"
#endif


#define DIAGLIB_THREAD_NAME                        "diaglib"
#define DIAGLIB_REG_THREAD_PRIO                    "DIAGLIB_PRIO"
#define DIAGLIB_REG_THREAD_STACK_SIZE              "DIAGLIB_STCK"
#define DIAGLIB_REG_THREAD_PATH                    "/dev/registry/LOCAL_MACHINE/SOFTWARE/BLAUPUNKT/PROCESS/FUNCTIONAL_COMPONENT/FC_DIAGNOSIS/APP_THREAD"
#define DIAGLIB_MESSAGE_CONTEXT_PERIOD_DEFAULT_SEC 5
#define DIAGLIB_STANDARD_SEM_TIMEOUT_MSEC          50

// Semaphore Name. ProcessId will be added at the end of name.
#ifdef OSAL_DYNAMIC_BASE
#define DIAGLIB_SEM_NAME                           "dlibdyn"
#else
#define DIAGLIB_SEM_NAME                           "dlib"
#endif

#define DIAGLIB_WORKER_THREAD_DEFAULT_PRIO         90
#define DIAGLIB_WORKER_THREAD_DEFAULT_STCK         10000


//forward declaration
extern "C" {tS32 OSAL_s32ThreadJoin(OSAL_tThreadID tid, OSAL_tMSecond msec);}

namespace diaglib {

tU32                 tclMessageContextManager::u32StaticCounter         = 0;                                            // Context counter
OSAL_tTimerHandle    tclMessageContextManager::hStaticTimeoutTimer      = OSAL_C_INVALID_HANDLE;                        // Timer handle
OSAL_tSemHandle      tclMessageContextManager::StaticSemHandle          = OSAL_C_INVALID_HANDLE;                        // Semaphore handle
tBool                tclMessageContextManager::bStaticTimerActive       = FALSE;                                        // Is timer active?
tU16                 tclMessageContextManager::u16StaticCtxtPeriod      = DIAGLIB_MESSAGE_CONTEXT_PERIOD_DEFAULT_SEC;   // Period contexts timeout in
OSAL_tThreadID       tclMessageContextManager::s32StaticThreadId        = 0;                                            // Thread ID of diaglib worker thread
OSAL_tEventHandle    tclMessageContextManager::s32StaticEventHdl        = OSAL_C_INVALID_HANDLE;                        // Event Handle of diaglib event
tclLock              tclMessageContextManager::mLock(DIAGLIB_SEM_NAME);
tU16                 tclMessageContextManager::u16UserCount             = 0;                                            // Number of users of MsgCtxtMgr

map<tU32, tclMessageContext*>   tclMessageContextManager::oContextMap = map<tU32, tclMessageContext*>();


/***********************************************************************//**
 * \brief Initializer
 *
 * Initializes all members and sets up the timer.
 *
 * \return n/a
 *//************************************************************************/
tVoid tclMessageContextManager::vIncreaseUsers ( )
{
   ETG_TRACE_USR3_THR(( "--> tclMessageContextManager::vIncreaseUsers"));
   //diaglib_FuncTrace oTrace(I_DIAGLIB_F_INCREASE);

   //DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_NEW_USER, (u16UserCount + 1));
   ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::vIncreaseUsers => Users now = %d",u16UserCount+1));
   mLock.s32Lock();

   ++u16UserCount;

   if(u16UserCount == 1)
   {
      // creat event
      char acEventName[20] = "";
      sprintf(acEventName,DIAGLIB_EVENT_NAME"_%lx", (long int)OSAL_ProcessWhoAmI());
      ETG_TRACE_USR1_THR(( "--- tclMessageContextManager::vIncreaseUsers => Create Event (%s)",acEventName));

      tS32 s32EventReturnCode = OSAL_s32EventCreate(acEventName, &s32StaticEventHdl);
      if(OSAL_ERROR == s32EventReturnCode)
      {
         s32StaticEventHdl = OSAL_C_INVALID_HANDLE;
         //DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNABLE_TO_INIT_TIMEOUT_TIMER);
         ETG_TRACE_ERR_THR(( "!!! tclMessageContextManager::vIncreaseUsers => ERROR: OSAL_s32EventCreate FAILED  Name:%s",acEventName));
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::vIncreaseUsers => EventCreated: Hdl=%d Name:%s",s32StaticEventHdl,acEventName));
      }

      // create thread
      ETG_TRACE_USR1_THR(( "--- tclMessageContextManager::vIncreaseUsers => Create Thread"));
      spawnTimeoutThread();

      // create timer
      ETG_TRACE_USR1_THR(( "--- tclMessageContextManager::vIncreaseUsers => Create Timer"));
      tS32 s32TimerReturnCode = OSAL_s32TimerCreate((OSAL_tpfCallback)(tclMessageContextManager::vOnSecondsPassed), OSAL_NULL, &hStaticTimeoutTimer);
      if(OSAL_ERROR == s32TimerReturnCode)
      {
         hStaticTimeoutTimer = OSAL_C_INVALID_HANDLE;
         //DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNABLE_TO_INIT_TIMEOUT_TIMER);
         ETG_TRACE_ERR_THR(( "!!! tclMessageContextManager::vIncreaseUsers => ERROR: OSAL_s32TimerCreate FAILED"));
         NORMAL_M_ASSERT_ALWAYS();
      }
      bStaticTimerActive   = FALSE;
   }
   mLock.bUnlock();
   ETG_TRACE_USR3_THR(( "<-- tclMessageContextManager::vIncreaseUsers"));
}

/***********************************************************************//**
 * \brief DeInitializer
 *
 * DeInitializes all members and sets up the timer.
 *
 * \return n/a
 *//************************************************************************/
tVoid tclMessageContextManager::vDecreaseUsers ( )
{
   ETG_TRACE_USR3_THR(( "--> tclMessageContextManager::vDecreaseUsers"));

   ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::vDecreaseUsers => Users now = %d",u16UserCount-1));

   mLock.s32Lock();

   --u16UserCount;

   if(u16UserCount == 0)
   {

      char acEventName[20] = "";
      sprintf(acEventName,DIAGLIB_EVENT_NAME"_%lx", (long int)OSAL_ProcessWhoAmI());

      // destory timer
      ETG_TRACE_USR1_THR(( "--- tclMessageContextManager::vIncreaseUsers => Deltete Timer"));
      bStaticTimerActive   = FALSE;
      OSAL_s32TimerSetTime ( hStaticTimeoutTimer, 0, 0);
      OSAL_s32TimerDelete(hStaticTimeoutTimer);

      // destory thread
      ETG_TRACE_USR1_THR(( "--- tclMessageContextManager::vIncreaseUsers => Deltete Thread"));
      killTimeoutThread();

      // destory event
      ETG_TRACE_USR1_THR(( "--- tclMessageContextManager::vIncreaseUsers => Deltete Event: Hdl=%d Name:%s",s32StaticEventHdl,acEventName));
      tS32 s32Result = OSAL_s32EventClose(s32StaticEventHdl);
      NORMAL_M_ASSERT(OSAL_ERROR != s32Result);

      s32Result = OSAL_s32EventDelete(acEventName);
      NORMAL_M_ASSERT(OSAL_ERROR != s32Result);
   }
   mLock.bUnlock();
   ETG_TRACE_USR3_THR(( "<-- tclMessageContextManager::vDecreaseUsers"));
}

/***********************************************************************//**
 * \brief Add Context into manages observation
 *
 * Adds the provided message context into the internal container and returns a
 * void pointer as a key. Key should be issued to components, in order to
 * provide it again on a response on the message.
 *
 * \param[in] poMsgContext Message context to add to the container
 *
 * \return tVoid* Handle of message context carmouflaged as a void pointer
 *//************************************************************************/
tU32 tclMessageContextManager::poAddContext(tclMessageContext* poMsgContext)
{
   ETG_TRACE_USR3_THR(( "--> tclMessageContextManager::poAddContext"));
   //diaglib_FuncTrace oTrace(I_DIAGLIB_F_ADD_CONTEXT);

   tS32 s32ReturnCode = OSAL_OK;
   tU32 u32ContextId = 0;


   if( OSAL_OK == mLock.s32Lock(DIAGLIB_STANDARD_SEM_TIMEOUT_MSEC) )
   {
      // ==================================================================================
      // CRITICAL SECTION
      // ==================================================================================
      // Activate when you need semaphore traces
      // DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_ENTER_CRITICAL_SECTION);
      ++u32StaticCounter;
      oContextMap[u32StaticCounter] = poMsgContext;
      u32ContextId = u32StaticCounter;

      /*
      Start timer if it is not already running
      */
      if(bStaticTimerActive == FALSE && hStaticTimeoutTimer != OSAL_C_INVALID_HANDLE)
      {
         ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::poAddContext => start Timer"));
         s32ReturnCode = OSAL_s32TimerSetTime (
                                                hStaticTimeoutTimer,
                                                DIAGLIB_MESSAGE_CONTEXT_PERIOD_DEFAULT_SEC * 1000,
                                                DIAGLIB_MESSAGE_CONTEXT_PERIOD_DEFAULT_SEC * 1000
                                              ); // 1 second = 1000 ms, so the timer calls back every DIAGLIB_MESSAGE_CONTEXT_PERIOD seconds

         bStaticTimerActive = TRUE;
         // DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_TIMER_ACTIVATED);
      }

      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CREATED_CTXT_WITH_HANDLE, u32StaticCounter );
      // Activate when you need semaphore traces
      // DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_LEAVE_CRITICAL_SECTION);
      // ==================================================================================
      mLock.bUnlock();
   }
   else
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNABLE_TAKE_SEM_UNTIL_TIMEOUT);
   }

   if(s32ReturnCode != OSAL_OK)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NEGATIVE_RESPONSE);
   }

   ETG_TRACE_USR3_THR(( "<-- tclMessageContextManager::poAddContext"));
   return u32ContextId;
}

/***********************************************************************//**
 * \brief Returns the context of the provided key
 *
 * Looks up the MessageContext that was stored using the provided key.
 * When found, it is removed from the list and the context is returned to
 * the caller. If it is not found a warning is traced and OSAL_NULL is
 * returned.
 *
 * \param[in] poHandle The key that was issued before when added the context
 *
 * \return tclMessageContext* Message context when found, else OSAL_NULL
 *//************************************************************************/
tclMessageContext* tclMessageContextManager::bGetContextBack(tU32 u32Handle)
{
   diaglib_FuncTrace oTrace(I_DIAGLIB_F_GET_CONTEXT_BACK);
   tS32 s32ReturnCode = OSAL_OK;
   tclMessageContext* poMsgContext = OSAL_NULL;


   if( OSAL_OK != mLock.s32Lock(DIAGLIB_STANDARD_SEM_TIMEOUT_MSEC) )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNABLE_TAKE_SEM_UNTIL_TIMEOUT);
      return poMsgContext;
   }

   // ==================================================================================
   // CRITICAL SECTION
   // ==================================================================================
   // Activate when you need semaphore traces
   // DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_ENTER_CRITICAL_SECTION);
   // Lint 1702 -> comparison of std iterators. We have no influence on implementation
   map<tU32, tclMessageContext*>::iterator iter = oContextMap.find(u32Handle);
   if(iter != oContextMap.end())//lint !e1702
   {
      poMsgContext = oContextMap[u32Handle];
      oContextMap.erase(u32Handle);
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_FOUND_CTXT_FOR_ID, u32Handle );

      /*
       * If the map is empty deactivate the timer.
       */
      if(oContextMap.size() == 0 && hStaticTimeoutTimer != OSAL_C_INVALID_HANDLE)
      {
         ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::bGetContextBack => stop Timer"));

         s32ReturnCode = OSAL_s32TimerSetTime(hStaticTimeoutTimer, 0, 0);

         bStaticTimerActive = FALSE;
         // DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_TIMER_DEACTIVATED);
      }
   }
   else
   {
      DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_NO_CONTEXT_FOUND_FOR_ID, u32Handle );
   }
   // Activate when you need semaphore traces
   // DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_LEAVE_CRITICAL_SECTION);
   // ==================================================================================
   mLock.bUnlock();

   if(s32ReturnCode != OSAL_OK)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NEGATIVE_RESPONSE);
   }

#ifdef VARIANT_S_FTR_ENABLE_FEAT_DIAGLIB_MSG_TIMING
   if(OSAL_NULL != poMsgContext)
   {
      tU32 u32CurrentTick = OSAL_ClockGetElapsedTime();
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_MESSAGE_TIME_UNTIL_RESPONSE, (u32CurrentTick - poMsgContext->u32GetCreationTick()));
   }
#endif

   return poMsgContext;
}

/***********************************************************************//**
 * \brief Handles timer callback on library level
 *
 * This is the central callback function for all timer instances that handle
 * MessageContext Timeout. On a callback the this pointer of the
 * MessageContextManager is provided. This function calls the component
 * specific timer handler.
 *
 * \param[in] poThis Pointer to component specific timer handler (MsgCtxMgr)
 *
 * \return OSAL_tpfCallback Some OSAL specific return code
 *//************************************************************************/
OSAL_tpfCallback tclMessageContextManager::vOnSecondsPassed( const tVoid* pDontCare)
{
   ETG_TRACE_USR3_THR(( "--> tclMessageContextManager::vOnSecondsPassed"));
   //diaglib_FuncTrace oTrace(I_DIAGLIB_F_ON_SECONDS_PASSED);

   if(OSAL_NULL == pDontCare)
   {
      // lint Info 715 unreferenced parameter
   }

   ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::vOnSecondsPassed => Post Event DIAGLIB_EVENT_TIMER"));

   tS32 s32Result = OSAL_s32EventPost(s32StaticEventHdl, DIAGLIB_EVENT_TIMER, OSAL_EN_EVENTMASK_OR);
   NORMAL_M_ASSERT(OSAL_ERROR != s32Result);

   ETG_TRACE_USR3_THR(( "<-- tclMessageContextManager::vOnSecondsPassed"));
   return OSAL_NULL;
}

tVoid tclMessageContextManager::vTimeoutWorkerThread( tVoid* /*pDontCare*/)
{
   ETG_TRACE_USR3_THR(( "--> tclMessageContextManager::vTimeoutWorkerThread"));
   //diaglib_FuncTrace oTrace(I_DIAGLIB_F_TIMEOUT_WORKER_THREAD);

   OSAL_tEventMask hEvRequest = 0;
   tS32 s32Result = 0;
   bool isDone = false;

   while ( !isDone )
   {
      NORMAL_M_ASSERT(OSAL_C_INVALID_HANDLE != s32StaticEventHdl);

      s32Result = OSAL_s32EventWait ( s32StaticEventHdl,
                                  DIAGLIB_WORKER_EVENT_MASK,
                                  OSAL_EN_EVENTMASK_OR,
                                  OSAL_C_TIMEOUT_FOREVER,
                                  &hEvRequest
                                );
      if( OSAL_OK != s32Result )
      {
         NORMAL_M_ASSERT_ALWAYS();
         isDone = true;
         continue;
      }

      if ( OSAL_s32EventPost(s32StaticEventHdl, ~hEvRequest, OSAL_EN_EVENTMASK_AND) != OSAL_OK)
      {
         NORMAL_M_ASSERT_ALWAYS();
         isDone = true;
         continue;
      }

      if ( hEvRequest & DIAGLIB_EVENT_KILL_THREAD )
      {
         ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::vTimeoutWorkerThread => DIAGLIB_EVENT_KILL_THREAD"));
         isDone = true;
         continue;
      }

      if ( hEvRequest & DIAGLIB_EVENT_TIMER )
      {
         ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::vTimeoutWorkerThread => DIAGLIB_EVENT_TIMER"));
         vHandleThreadOnSecPassed();
      }
   }// while (TRUE)
   ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::vTimeoutWorkerThread => KILLED"));

   //s32Result = OSAL_s32EventPost(s32StaticEventHdl, DIAGLIB_EVENT_THREAD_ENDED, OSAL_EN_EVENTMASK_OR);
   //NORMAL_M_ASSERT(OSAL_OK == s32Result);
   ETG_TRACE_USR3_THR(( "<-- tclMessageContextManager::vTimeoutWorkerThread"));
}

tVoid tclMessageContextManager::spawnTimeoutThread()
{
   ETG_TRACE_USR3_THR(( "--> tclMessageContextManager::spawnTimeoutThread"));
   //diaglib_FuncTrace oTrace(I_DIAGLIB_F_SPAWN_TIMEOUT_THREAD);
   OSAL_trThreadAttribute rThreadAttributes;

   vFillThreadAttributes(rThreadAttributes);

   s32StaticThreadId = OSAL_ThreadCreate(&rThreadAttributes);
   tS32 s32ThreadActivateResult = OSAL_s32ThreadActivate(s32StaticThreadId);

   if(OSAL_ERROR != s32StaticThreadId && s32ThreadActivateResult != OSAL_ERROR)
   {
      ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::spawnTimeoutThread => WORKER SPAWNED WITH ID Id:%x Result:%x",s32StaticThreadId, s32ThreadActivateResult));
      //DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_WORKER_SPAWNED_WITH_ID, s32StaticThreadId);
   }
   else
   {
      ETG_TRACE_ERR_THR(( "!!! tclMessageContextManager::spawnTimeoutThread => ERROR: WORKER SPAWNING FAILED Id:%x Result:%x",s32StaticThreadId, s32ThreadActivateResult));
      //DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_WORKER_SPAWNING_FAILED);
   }
   ETG_TRACE_USR3_THR(( "<-- tclMessageContextManager::spawnTimeoutThread"));
}

tVoid tclMessageContextManager::killTimeoutThread()
{
   ETG_TRACE_USR3_THR(( "--> tclMessageContextManager::killTimeoutThread"));

    tS32 s32Result = OSAL_s32EventPost(s32StaticEventHdl, DIAGLIB_EVENT_KILL_THREAD, OSAL_EN_EVENTMASK_OR);
    NORMAL_M_ASSERT(OSAL_ERROR != s32Result);

    /*
    s32Result = OSAL_s32EventWait ( s32StaticEventHdl,
                                    DIAGLIB_EVENT_THREAD_ENDED,
                                    OSAL_EN_EVENTMASK_OR,
                                    OSAL_C_TIMEOUT_FOREVER,
                                    &hEvRequest
                                  );
                                  */


    s32Result  = OSAL_s32ThreadJoin((OSAL_tThreadID) s32StaticEventHdl, (OSAL_tMSecond) OSAL_C_TIMEOUT_FOREVER);
    NORMAL_M_ASSERT(OSAL_ERROR != s32Result);

   ETG_TRACE_USR3_THR(( "<-- tclMessageContextManager::killTimeoutThread"));
}

tVoid tclMessageContextManager::vFillThreadAttributes(OSAL_trThreadAttribute& attrib)
{
   reg_tclRegKey oReg;

   tBool   bRegReadStatus  = FALSE;
   tU32    u32ThreadPrio   = DIAGLIB_WORKER_THREAD_DEFAULT_PRIO;
   tU32    u32StackSize    = DIAGLIB_WORKER_THREAD_DEFAULT_STCK;

   // Open Registry
   if(oReg.bOpen(DIAGLIB_REG_THREAD_PATH) == TRUE)
   {

      // Read the thread Priority from the Registry
      bRegReadStatus = oReg.bQueryU32(DIAGLIB_REG_THREAD_PRIO, &u32ThreadPrio);
      if ( bRegReadStatus == FALSE )
      {
         u32ThreadPrio = DIAGLIB_WORKER_THREAD_DEFAULT_PRIO;
         DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_USE_DEFAULT_PRIO, u32ThreadPrio);
      }
      else
      {
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_READ_PRIO_FROM_REG, u32ThreadPrio);
      }

      bRegReadStatus = oReg.bQueryU32(DIAGLIB_REG_THREAD_STACK_SIZE, &u32StackSize);
      if (bRegReadStatus == FALSE)
      {
         u32StackSize  = DIAGLIB_WORKER_THREAD_DEFAULT_STCK;
         DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_USE_DEFAULT_STCK, u32StackSize);
      }
      else
      {
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_READ_STCK_FROM_REG, u32StackSize);
      }
      oReg.vClose();
   }

   char acIdName[20] = "";
   sprintf(acIdName,DIAGLIB_THREAD_NAME"_%lx", (long int)OSAL_ProcessWhoAmI());
   //mName += acIdName;
   //mName += (static_cast<tChar>(OSAL_ProcessWhoAmI())+0x30);
   ETG_TRACE_COMP_THR(( "--- tclMessageContextManager::vFillThreadAttributes => ThreadName: %s",acIdName));

   attrib.szName = acIdName;
   attrib.u32Priority = u32ThreadPrio;
   attrib.s32StackSize = (tS32)u32StackSize;
   attrib.pfEntry = vTimeoutWorkerThread;
   attrib.pvArg = OSAL_NULL;
}

tVoid tclMessageContextManager::vHandleThreadOnSecPassed()
{
   tS32 s32ReturnCode = 0;

   if( OSAL_OK != mLock.s32Lock(DIAGLIB_STANDARD_SEM_TIMEOUT_MSEC) )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNABLE_TAKE_SEM_UNTIL_TIMEOUT);
      return;
   }

   // ==================================================================================
   // CRITICAL SECTION
   // ==================================================================================
   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_ENTER_CRITICAL_SECTION);

   vHandlerAllContextsOnSecPassed();

   /*
    * If the map is empty deactivate the timer.
    */
   if(oContextMap.size() == 0 && hStaticTimeoutTimer != OSAL_C_INVALID_HANDLE)
   {
      s32ReturnCode = OSAL_s32TimerSetTime(hStaticTimeoutTimer, 0, 0);

      if(s32ReturnCode != OSAL_OK)
      {
         DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NEGATIVE_RESPONSE);
      }
      bStaticTimerActive = FALSE;
      DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_TIMER_DEACTIVATED);
   }

   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_LEAVE_CRITICAL_SECTION);
   // ==================================================================================
   mLock.bUnlock();
}

tVoid tclMessageContextManager::vHandlerAllContextsOnSecPassed()
{
   map<tU32, tclMessageContext*>::iterator it = oContextMap.begin();
   // Lint 1702 -> comparison of std iterators. We have no influence on implementation
   while(it != oContextMap.end())//lint !e1702
   {
      it->second->vSecondsPassed( DIAGLIB_MESSAGE_CONTEXT_PERIOD_DEFAULT_SEC );

      if(TRUE == it->second->bDidContextTimeout())
      {
         DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_CONTEXT_TIMED_OUT, it->first );
         oContextMap.erase(it++);
      }
      else
      {
         it++;
      }
   }
}

}
