/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief Message Context - Diagnosis Library - Introduced in MFD Nav
*
* See header file
* 25.08.2008
* Adding timing functionality. When activated using the feature-switch this
* traces out the time a message took until it was responded to.
* Feature switch: VARIANT_S_FTR_ENABLE_FEAT_DIAGLIB_MSG_TIMING
*
* (c) 2008 Blaupunkt Werke GmbH
*//****************************************************************************/

#ifdef DIAGLIB_FILE_NUMBER
#undef DIAGLIB_FILE_NUMBER
#endif
#define DIAGLIB_FILE_NUMBER F_DIAGLIB_MESSAGE_CONTEXT_CCA

#include "tclMessageContext.h"

#include "Trace.h"

// No flow trace on purpose

namespace diaglib {

/***********************************************************************//**
 * \brief Constructor
 *
 * Initializes all members according to the provided values.
 *
 * \param[in] enGroup Describes to what message group the message belongs
 * \param[in] u32Id The unique ID of the received message
 * \param[in] u32AdditionalValue Some message groups have an additional
 *            value to be stored.
 *
 * \return n/a
 *//************************************************************************/
tclMessageContext::tclMessageContext (
                                        ten_MessageGroup enGroup,
                                        tU32 u32Id,
                                        tU32 u32AdditionalValue
                                     )
{
   enMessageGroup = enGroup;
   u32UniqueId = u32Id;
   tclMessageContext::vResestTimeout(enGroup);
   bValid = true;

#ifdef VARIANT_S_FTR_ENABLE_FEAT_DIAGLIB_MSG_TIMING
   u32CreationTick = OSAL_ClockGetElapsedTime();
#endif

   switch (enMessageGroup)
   {
      case EN_MESSAGE_ROUTINE_CONTROL:
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CREATED_CTXT_FOR_ID, u32Id);
      break;

      case EN_MESSAGE_IO_CONTROL:
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CREATED_CTXT_FOR_ID, u32Id);
         enIoControlAction = static_cast<tenIoControlAction> (u32AdditionalValue);
      break;

      case EN_MESSAGE_DIAG_DATA:
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CREATED_CTXT_FOR_ID, u32Id);
      break;

      case EN_MESSAGE_SYSTEM_SET:
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CREATED_CTXT_FOR_ID, u32Id);
         enSystemSetType = static_cast<tenSystemSetType> (u32AdditionalValue);
      break;

      default:
         bValid = false;
         DIAGLIB_TRACE_ERROR_U32(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNKNOWN_MESSAGE_GRP, static_cast<tU32>(enMessageGroup));
   }
}

/***********************************************************************//**
 * \brief Destructor
 *
 * DeInit all pointer members
 *
 * \return n/a
 *//************************************************************************/
tclMessageContext::~tclMessageContext()
{
}

/***********************************************************************//**
 * \brief Gets the message group, of the message, this is the context
 *
 * Returns the group of the message this context is associated to.
 *
 * \return ten_MessageGroup Group identifier
 *//************************************************************************/
tclMessageContext::ten_MessageGroup tclMessageContext::getMessageGroup() const
{
   if(false == bValid)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_REQUEST_INVALID_CTXT);
   }

   return enMessageGroup;
}

/***********************************************************************//**
 * \brief Gets the unique message ID of the context
 *
 * Returns the unique ID of the message this context is associated to.
 * The unique ID identifies the type of the message not the message itself.
 * See documentation for unique ID concept on details
 *
 * \return tU32 Unique ID of the associated message
 *//************************************************************************/
tU32 tclMessageContext::getId() const
{
   if(false == bValid)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_REQUEST_INVALID_CTXT);
   }

   return u32UniqueId;
}

/***********************************************************************//**
 * \brief Gets the type of a systemset message
 *
 * Returns the type of a systemset message. If the context is not associated
 * with a systemset message false is returned.
 *
 * \param[out] enType Contains the system set type, if return value is true
 *
 * \return tBool true if this is a systemset message and the context is valid
 *//************************************************************************/
tBool tclMessageContext::getSystemSetType(tenSystemSetType& enType) const
{
   if(false == bValid)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_REQUEST_INVALID_CTXT);
      return false;
   }

   if(enMessageGroup == EN_MESSAGE_SYSTEM_SET)
   {
      enType = enSystemSetType;
      return true;
   }
   else
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_ASSUMES_WRONG_CTX_TYPE);
      return false;
   }
}

/***********************************************************************//**
 * \brief Gets the type of a IOcontrol message
 *
 * Returns the type of a IOcontrol action. If the context is not associated
 * with a IOcontrol message false is returned.
 *
 * \param[out] enAction Contains the IOcontrol action, if return value is true
 *
 * \return tBool true if this is a IOcontrol message and the context is valid
 *//************************************************************************/
tBool tclMessageContext::getIoControlAction(tenIoControlAction& enAction) const
{
   if(false == bValid)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_REQUEST_INVALID_CTXT);
      return false;
   }

   if(enMessageGroup == EN_MESSAGE_IO_CONTROL)
   {
      enAction = enIoControlAction;
      return true;
   }
   else
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_ASSUMES_WRONG_CTX_TYPE);
      return false;
   }
}

/***********************************************************************//**
 * \brief Resets the timeout counter to start value
 *
 * Sets the internal timeout value (usually time in seconds) to start value.
 *
 * \return none
 *//************************************************************************/

tVoid
tclMessageContext::vResestTimeout ( ten_MessageGroup enGroup )
{
   static tU32 timeoutValue[4] = {
         DIAGLIB_MESSAGE_CONTEXT_TIMEOUT_SEC,            // EN_MESSAGE_ROUTINE_CONTROL
         DIAGLIB_MESSAGE_CONTEXT_TIMEOUT_SEC,            // EN_MESSAGE_IO_CONTROL
         DIAGLIB_MESSAGE_CONTEXT_TIMEOUT_SEC,            // EN_MESSAGE_DIAG_DATA
         DIAGLIB_MESSAGE_CONTEXT_TIMEOUT_SEC_SYSTEM_SET  //EN_MESSAGE_SYSTEM_SET
   };

   u32SecondsToTimeout = timeoutValue[enGroup];
}

/***********************************************************************//**
 * \brief Subtract seconds from timeout value
 *
 * Decrements the internal timeout value by the amount of seconds. If
 * provided value is bigger than remaining timeout, timeout value is set to
 * zero.
 *
 * \param[in] u16Seconds Ammount of seconds that passed.
 *
 * \return none
 *//************************************************************************/
tVoid tclMessageContext::vSecondsPassed(tU16 u16Seconds)
{
   if(u32SecondsToTimeout >= u16Seconds)
   {
      u32SecondsToTimeout -= u16Seconds;
   }
   else
   {
      u32SecondsToTimeout = 0;
   }
}

/***********************************************************************//**
 * \brief Returns whether or not context timed out
 *
 * Returns true if timeout value is zero and false if value is greater zero
 *
 * \return tBool Did context time out? (yes = true, no = false)
 *//************************************************************************/
tBool tclMessageContext::bDidContextTimeout() const
{
   if(false == bValid)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_REQUEST_INVALID_CTXT);
   }

   return (u32SecondsToTimeout == 0);
}

#ifdef VARIANT_S_FTR_ENABLE_FEAT_DIAGLIB_MSG_TIMING
tU32  tclMessageContext::u32GetCreationTick()
{
   return u32CreationTick;
}
#endif

}
