/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief SystemSet Handler - Diagnosis library - Introduced in MFD Nav
*
* (c) 2008 Blaupunkt Werke GmbH
*//****************************************************************************/

#ifdef DIAGLIB_FILE_NUMBER
#undef DIAGLIB_FILE_NUMBER
#endif
#define DIAGLIB_FILE_NUMBER F_DIAGLIB_SYSTEMSET_HANDLER

#include "tclSystemSetHandler.h"

#ifndef __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_TYPES
#define OSAL_S_IMPORT_INTERFACE_MEMORY
#include "osal_if.h"
#endif


#include "../tclSystemSetDefaultListener.h"
#include "../tclInterfaceManager.h"
#include "../IdSpec.h"
#include "../tclHelperFunctions.h"

#include "../Trace.h"

namespace diaglib {

/***********************************************************************//**
 * \brief Constructor
 *
 * Initializes all members according to the provided values. And registers
 * itself with the provided ServiceHandler.
 *
 * \param[in] poServiceHandler Describes to what message group the message belongs
 * \param[in] poResponsibleManager The unique ID of the received message
 *
 * \return n/a
 *//************************************************************************/
tclSystemSetHandler::tclSystemSetHandler
                              (
                                 tclServiceHandler* poServiceHandler,
                                 tclInterfaceManager* poResponsibleManager
                              ):
                                 poService(poServiceHandler),
                                 poManager(poResponsibleManager)
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_CONSTRUCTOR);

   if(poService == OSAL_NULL || poResponsibleManager == OSAL_NULL)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
   }

   for(tU32 i = 0; i<=DIAGLIB_SYSTEM_SET_TYPE_MAX; ++i)
   {
      au32MainId[i] = SYSSETID_ALL_COMP;
   }

   poDefaultListener = OSAL_NULL;
   poDefaultListener = OSAL_NEW tclSystemSetDefaultListener(this);

   if(poService != OSAL_NULL)
      poService->vRegisterSystemSetListener(this);

}

/***********************************************************************//**
 * \brief Destructor
 *
 * Releases all members, clears the Listener Map
 *
 * \return n/a
 *//************************************************************************/
tclSystemSetHandler::~tclSystemSetHandler()
{
    _BP_TRY_BEGIN
    {
        oListenerMap.clear(); // Lint Warning 1551: Function may throw exception '...' in destructor
        if(OSAL_NULL != poDefaultListener)
        {
            OSAL_DELETE poDefaultListener;
        }
    }
    _BP_CATCH_ALL
    {

    }
    _BP_CATCH_END

   poManager = OSAL_NULL;
   poService = OSAL_NULL;
   poDefaultListener = OSAL_NULL;
}

/***********************************************************************//**
 * \brief Sends a response to a SystemSet request
 *
 * Forwards the response on a SystemSet request to the service handler.
 * the only check done is, if the MessageContext pointer is valid
 *
 * \param[in] enSystemSetResult Result of the systemset request
 * \param[in] oErrorVector Vector containing error code in case of failure
 * \param[in] MsgContext Context that was issued with the request
 *
 * \return tBool Indicates if sending of message was successful
 *//************************************************************************/
tBool tclSystemSetHandler::bSendSystemSetResult (
                                        tenSystemSetResult enSystemSetResult,
                                        const tclParameterVector& oErrorVector,
                                        tContext MsgContext
                                      )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_SYSTEMSET_RESULT);
   if(poService == OSAL_NULL )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return FALSE;
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_OUTGOING_MESSAGE_ADV_START, poService->u32GetAppIdentifier());
   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_MESSAGE_TYPE_SYSSET_MR);
   tBool bResult;


   if(OSAL_NULL == MsgContext)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      bResult =  false;
   }
   else
   {
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_FORWARDING_RSP_WITH_CTXT, static_cast<tU32>(MsgContext));
      bResult = poService->bSendSystemSetMethodResult ( enSystemSetResult,
                                                        oErrorVector,
                                                        MsgContext
                                                      );
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_OUTGOING_MESSAGE_ADV_STOP, poService->u32GetAppIdentifier());


   return bResult;
}

tBool tclSystemSetHandler::bSendSystemSetCheckResult (
                                                         tenSystemSetResult enSystemSetResult,
                                                         const tclParameterVector& oValueList,
                                                         tContext MsgContext
                                                       )
{
   // diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_SYSTEMSET_RESULT);
   DIA_PARAMETER_INTENTIONALLY_UNUSED( oValueList );
   DIA_PARAMETER_INTENTIONALLY_UNUSED( enSystemSetResult );
   if(poService == OSAL_NULL )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return FALSE;
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_OUTGOING_MESSAGE_ADV_START, poService->u32GetAppIdentifier());
   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_MESSAGE_TYPE_SYSSET_MR);
   tBool bResult = FALSE;


   if(OSAL_NULL == MsgContext)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      bResult =  false;
   }
   else
   {
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_FORWARDING_RSP_WITH_CTXT, static_cast<tU32>(MsgContext));
      bResult = poService->bSendSystemSetCheckResult ( enSystemSetResult,
                                                       oValueList,
                                                       MsgContext
                                                     );
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_OUTGOING_MESSAGE_ADV_STOP, poService->u32GetAppIdentifier());


   return bResult;
}

/***********************************************************************//**
 * \brief Sends a response to a SystemSetPrepare request
 *
 * Forwards the response on a SystemSetPrepare request to the service handler.
 * the only check done is, if the MessageContext pointer is valid.
 *
 * \param[in] enSystemSetResult Result of the systemsetPrepare request
 * \param[in] oErrorVector Vector containing error code in case of failure
 * \param[in] MsgContext Context that was issued with the request
 *
 * \return tBool Indicates if sending of message was successful
 *//************************************************************************/
tBool tclSystemSetHandler::bSendSystemSetPrepareResult (
                                               tenSystemSetResult enSystemSetResult,
                                               const tclParameterVector& oErrorVector,
                                               tContext MsgContext
                                             )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_SYSTEMSET_PREPARE_RESULT);

   if(poService == OSAL_NULL )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return FALSE;
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_OUTGOING_MESSAGE_ADV_START, poService->u32GetAppIdentifier() );
   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_MESSAGE_TYPE_SYSSET_PREPARE_MR);
   tBool bResult;

   if(OSAL_NULL == MsgContext)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      bResult =  false;
   }
   else
   {
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_FORWARDING_RSP_WITH_CTXT, static_cast<tU32>(MsgContext));
      bResult = poService->bSendSystemSetPrepareMethodResult ( enSystemSetResult,
                                                               oErrorVector,
                                                               MsgContext
                                                             );
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_OUTGOING_MESSAGE_ADV_STOP, poService->u32GetAppIdentifier());


   return bResult;
}

/***********************************************************************//**
 * \brief Sends a acknowledge message to a SystemSet Finished message
 *
 * Forwards the response on a SystemSetFinished request to the service handler.
 * the only check done is, if the MessageContext pointer is valid.
 *
 * \param[in] MsgContext Context that was issued with the request
 *
 * \return tBool Indicates if sending of message was successful
 *//************************************************************************/
tBool tclSystemSetHandler::bAcknowledgeSystemSetFinished (
                                                 tContext MsgContext
                                               )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_ACK_SYSTEMSET_FINISHED);
   if(poService == OSAL_NULL )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return FALSE;
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_OUTGOING_MESSAGE_ADV_START, poService->u32GetAppIdentifier() );
   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_MESSAGE_TYPE_SYSSET_FINISHED_MR);
   tBool bResult;


   if(OSAL_NULL == MsgContext)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      bResult =  false;
   }
   else
   {
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_FORWARDING_RSP_WITH_CTXT, static_cast<tU32>(MsgContext));
      bResult = poService->bSendSystemSetFinishedMethodResult ( MsgContext );
   }

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_OUTGOING_MESSAGE_ADV_STOP, poService->u32GetAppIdentifier() );

   return bResult;
}

tBool tclSystemSetHandler::bSendDiaglibError (
                                       tenInternalError enErrorCode,
                                       tContext MsgContext
                                   )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_DIAGLIB_ERROR);

   tclParameterVector oResultVector;
   oResultVector = oCreateErrorVector(enErrorCode);


   return bSendSystemSetPrepareResult( EN_SYSTEMSET_NOT_OK,
                                       oResultVector,
                                       MsgContext
                                     );
}

/***********************************************************************//**
 * \brief Register listener that handles incoming (Diag->Comp) messages.
 *
 * Method to register a listener, that is called on every incoming systemset
 * message of the specified type and ID. This version of the two overloaded
 * methods allows distinction between the different types of incoming
 * systemsets. So components can register listeners for every type of systemset.
 * The function checks if the pointer seems to be valid and then stores it
 * in a map using a key built from the ID and the SystemSet type.
 *
 * \param[in]      u32Id ID the given listener should be called for
 * \param[in]      u32SystemSetType SystemSet type the provided listener
 *                 should be called for.
 * \param[in]      SystemSetListener Pointer to listener that is called
 *                 on incoming messages
 *
 * \return         none
 *//*************************************************************************/
tVoid tclSystemSetHandler::vRegisterListener (
                                     tU32 u32Id,
                                     tenSystemSetType enSystemSetType,
                                     tclSystemSetListenerIF* poSystemsetListener
                                   )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_REGISTER_LISTENER);

   if(OSAL_NULL == poService)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_APPLICATION_ID, poService->u32GetAppIdentifier());

   tclSystemListenerMapKey oKey(u32Id, enSystemSetType);

   if(au32MainId[enSystemSetType] == SYSSETID_ALL_COMP)
   {
      au32MainId[enSystemSetType] = u32Id;
   }

   if(OSAL_NULL == poSystemsetListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   oListenerMap[oKey] = poSystemsetListener;
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_ADDED_LISTENER_ID, u32Id);
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_ADDED_LISTENER_SYSSET_TYPE, static_cast<tU32>(enSystemSetType));


}

/***********************************************************************//**
 * \brief Register listener that handles incoming messages.
 *
 * Method to register a listener, that is called on every incoming systemset
 * message of the specified type and ID. This version of the two overloaded
 * methods does not allow distinction between the different types of incoming
 * systemsets. So components can register only one listener for all types.
 * The function checks if the pointer seems to be valid and then stores it
 * in a map using a key built from the ID and the SystemSet type.
 *
 * \param[in]      u32Id - ID the given listener should be called for
 * \param[in]      SystemSetListener Pointer to listener that is called
 *                 on incoming messages
 *
 * \return         none
 *//*************************************************************************/
tVoid tclSystemSetHandler::vRegisterListener (
                                     tU32 u32Id,
                                     tclSystemSetListenerIF* poSystemsetListener
                                   )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_REGISTER_LISTENER);

   if(OSAL_NULL == poSystemsetListener || OSAL_NULL == poService)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_APPLICATION_ID, poService->u32GetAppIdentifier());

   for(tU32 i = DIAGLIB_SYSTEM_SET_TYPE_MIN; i <= DIAGLIB_SYSTEM_SET_TYPE_MAX; ++i)
   {
      if(au32MainId[i] == SYSSETID_ALL_COMP)
      {
         au32MainId[i] = u32Id;
      }
      tclSystemListenerMapKey oKey(u32Id, static_cast<tenSystemSetType>(i));

      oListenerMap[oKey] = poSystemsetListener;
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_ADDED_LISTENER_ID, u32Id);
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_ADDED_LISTENER_SYSSET_TYPE, i);
   }


}

/***********************************************************************//**
 * \brief Handles incoming (Diag->Comp) SystemSet requests
 *
 * It is checked if the component registered a listener for this kind of
 * request. If it has, the message is forwarded to the correct listener. If
 * it has not, a error message is sent back, stating that the requested
 * functionality is not supported by this component.
 *
 * \param[in] u32SystemSetID - ID the request was sent for
 * \param[in] enSystemSetType - SystemSet type the request was sent for
 *
 * \return tU32 Diaglib return value
 *//*************************************************************************/
tU32 tclSystemSetHandler::vOnSystemSet (
                                                 tU32 u32SystemSetID,
                                                 tenSystemSetType enSystemSetType,
                                                 tContext MsgContext
                                               )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_ON_SYSTEMSET);
   tclSystemListenerMapKey oKey(u32SystemSetID, enSystemSetType);
   tU32 u32Ret = U32_DIAGLIB_RETURN_OK;
   tBool bUseComponentListener = true;

   // Lint 1702 -> comparison of std iterators. We have no influence on implementation
   tclSystemListenerMap::iterator iter = oListenerMap.find(oKey);
   if(iter == oListenerMap.end())//lint !e1702
   {
      if( u32SystemSetID != SYSSETID_ALL_COMP ||               // If it is no broadcast request
          au32MainId[enSystemSetType] == SYSSETID_ALL_COMP)    // or if the component did not register a handler at all
      {
         if(poDefaultListener == OSAL_NULL )
         {
            DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
            return U32_DIAGLIB_RETURN_OK;
         }
         bUseComponentListener = false;
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_DEF_FORWARD_MSG_WITH_CTXT, static_cast<tU32>(MsgContext));
         DIAGLIB_TRACE_WARN(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_MESSAGE_HANDLED_BY_DEF_HANDLER);
         poDefaultListener->vOnSystemSet (
                                             u32SystemSetID,
                                             enSystemSetType,
                                             MsgContext
                                         );
      }
      else
      {
         tclSystemListenerMapKey oNewKey(au32MainId[enSystemSetType], enSystemSetType);
         oKey = oNewKey;
      }
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_APPID, poService->u32GetAppIdentifier());
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_ID, u32SystemSetID);
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_TYPE, static_cast<tU32>(enSystemSetType));
      //vSendSystemSetIdNotSupported(MsgContext);
      u32Ret = U32_DIAGLIB_RETURN_NOT_OK;
   }

   if(bUseComponentListener)
   {
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_FORWARDING_MSG_WITH_CTXT, static_cast<tU32>(MsgContext));
      u32Ret = oListenerMap[oKey]->vOnSystemSet (
                                                   u32SystemSetID,
                                                   enSystemSetType,
                                                   MsgContext
                                                );
   }

   return u32Ret;
}

/***********************************************************************//**
 * \brief Handles incoming (Diag->Comp) SystemSetFinished messages
 *
 * It is checked if the component registered a listener for this kind of
 * request. If it has, the message is forwarded to the correct listener. If
 * it has not, a error message is sent back, stating that the requested
 * functionality is not supported by this component.
 *
 * \param[in] u32SystemSetID - ID the request was sent for
 * \param[in] enSystemSetType - SystemSet type the request was sent for
 *
 * \return tU32 Diaglib return value
 *//*************************************************************************/
tU32 tclSystemSetHandler::vOnSystemSetFinished (
                                                         tU32 u32SystemSetID,
                                                         tenSystemSetType enSystemSetType,
                                                         tContext MsgContext
                                                       )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_ON_SYSTEMSET_FINISHED);
   tclSystemListenerMapKey oKey(u32SystemSetID, enSystemSetType);
   tU32 u32Ret = U32_DIAGLIB_RETURN_OK;
   tBool bUseComponentListener = true;

   // Lint 1702 -> comparison of std iterators. We have no influence on implementation
   tclSystemListenerMap::iterator iter = oListenerMap.find(oKey);
   if(iter == oListenerMap.end())//lint !e1702
   {
      if( u32SystemSetID != SYSSETID_ALL_COMP ||               // If it is no broadcast request
          au32MainId[enSystemSetType] == SYSSETID_ALL_COMP)    // or if the component did not register a handler at all

      {
         if(poDefaultListener == OSAL_NULL )
         {
            DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
            return U32_DIAGLIB_RETURN_NOT_OK;
         }
         bUseComponentListener = false;
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_DEF_FORWARD_MSG_WITH_CTXT, static_cast<tU32>(MsgContext));
         DIAGLIB_TRACE_WARN(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_MESSAGE_HANDLED_BY_DEF_HANDLER);
         poDefaultListener->vOnSystemSetFinished (
                                                u32SystemSetID,
                                                enSystemSetType,
                                                MsgContext
                                             );
      }
      else
      {
         tclSystemListenerMapKey oNewKey(au32MainId[enSystemSetType], enSystemSetType);
         oKey = oNewKey;
      }
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_APPID, poService->u32GetAppIdentifier());
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_ID, u32SystemSetID);
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_TYPE, static_cast<tU32>(enSystemSetType));
      //vSendSystemSetIdNotSupported(MsgContext);
      u32Ret = U32_DIAGLIB_RETURN_OK;
   }

   if(bUseComponentListener)
   {
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_FORWARDING_MSG_WITH_CTXT, static_cast<tU32>(MsgContext));
      u32Ret = oListenerMap[oKey]->vOnSystemSetFinished (
                                                   u32SystemSetID,
                                                   enSystemSetType,
                                                   MsgContext
                                                );
   }


   return u32Ret;
}

/***********************************************************************//**
 * \brief Handles incoming (Diag->Comp) SystemSetPrepare requests
 *
 * It is checked if the component registered a listener for this kind of
 * request. If it has, the message is forwarded to the correct listener. If
 * it has not, a error message is sent back, stating that the requested
 * functionality is not supported by this component.
 *
 * \param[in] u32SystemSetID - ID the request was sent for
 * \param[in] u32SystemSetType - SystemSet type the request was sent for
 *
 * \return tU32 Diaglib return value
 *//*************************************************************************/
tU32 tclSystemSetHandler::vOnSystemSetPrepare (
                                                         tU32 u32SystemSetID,
                                                         tenSystemSetType enSystemSetType,
                                                         tContext MsgContext
                                                         )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_ON_SYSTEMSET_PREPARE);
   tclSystemListenerMapKey oKey(u32SystemSetID, enSystemSetType);
   tU32 u32Ret = U32_DIAGLIB_RETURN_OK;
   tBool bUseComponentListener = true;

   // Lint 1702 -> comparison of std iterators. We have no influence on implementation
   tclSystemListenerMap::iterator iter = oListenerMap.find(oKey);
   if(iter == oListenerMap.end())//lint !e1702
   {
      if( u32SystemSetID != SYSSETID_ALL_COMP ||               // If it is no broadcast request
          au32MainId[enSystemSetType] == SYSSETID_ALL_COMP)    // or if the component did not register a handler at all

      {
         if(poDefaultListener == OSAL_NULL )
         {
            DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
            return U32_DIAGLIB_RETURN_NOT_OK;
         }
         bUseComponentListener = false;
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_DEF_FORWARD_MSG_WITH_CTXT, static_cast<tU32>(MsgContext));
         DIAGLIB_TRACE_WARN(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_MESSAGE_HANDLED_BY_DEF_HANDLER);
         poDefaultListener->vOnSystemSetPrepare (
                                                   u32SystemSetID,
                                                   enSystemSetType,
                                                   MsgContext
                                                );
      }
      else
      {
         tclSystemListenerMapKey oNewKey(au32MainId[enSystemSetType], enSystemSetType);
         oKey = oNewKey;
      }
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_APPID, poService->u32GetAppIdentifier());
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_ID, u32SystemSetID);
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_TYPE, static_cast<tU32>(enSystemSetType));
      //vSendSystemSetIdNotSupported(MsgContext);
      u32Ret = U32_DIAGLIB_RETURN_OK;
   }

   if(bUseComponentListener)
   {
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_FORWARDING_MSG_WITH_CTXT, static_cast<tU32>(MsgContext));
      u32Ret = oListenerMap[oKey]->vOnSystemSetPrepare (
                                                   u32SystemSetID,
                                                   enSystemSetType,
                                                   MsgContext
                                                );
   }


   return u32Ret;
}

tU32 tclSystemSetHandler::vOnSystemSetCheck (
                                    tU32 u32SystemSetID,
                                    tenSystemSetType u32SystemSetType,
                                    tContext MsgContext
                                   )
{
   // diaglib_FuncTrace fnctrc(I_DIAGLIB_F_ON_SYSTEMSET_PREPARE);
   tclSystemListenerMapKey oKey(u32SystemSetID, u32SystemSetType);
   tU32 u32Ret = U32_DIAGLIB_RETURN_OK;
   tBool bUseComponentListener = true;

   // Lint 1702 -> comparison of std iterators. We have no influence on implementation
   tclSystemListenerMap::iterator iter = oListenerMap.find(oKey);
   if(iter == oListenerMap.end())//lint !e1702
   {
      if( u32SystemSetID != SYSSETID_ALL_COMP ||               // If it is no broadcast request
          au32MainId[u32SystemSetType] == SYSSETID_ALL_COMP)    // or if the component did not register a handler at all

      {
         if(poDefaultListener == OSAL_NULL )
         {
            DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
            return U32_DIAGLIB_RETURN_NOT_OK;
         }
         bUseComponentListener = false;
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_DEF_FORWARD_MSG_WITH_CTXT, static_cast<tU32>(MsgContext));
         DIAGLIB_TRACE_WARN(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_MESSAGE_HANDLED_BY_DEF_HANDLER);
         poDefaultListener->vOnSystemSetCheck (
                                                   u32SystemSetID,
                                                   u32SystemSetType,
                                                   MsgContext
                                                );
      }
      else
      {
         tclSystemListenerMapKey oNewKey(au32MainId[u32SystemSetType], u32SystemSetType);
         oKey = oNewKey;
      }
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_APPID, poService->u32GetAppIdentifier());
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_ID, u32SystemSetID);
      //DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_ID_NOT_SUPPORTED_TYPE, static_cast<tU32>(u32SystemSetType));
      //vSendSystemSetIdNotSupported(MsgContext);
      u32Ret = U32_DIAGLIB_RETURN_OK;
   }

   if(bUseComponentListener)
   {
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_FORWARDING_MSG_WITH_CTXT, static_cast<tU32>(MsgContext));
      u32Ret = oListenerMap[oKey]->vOnSystemSetCheck (
                                                         u32SystemSetID,
                                                         u32SystemSetType,
                                                         MsgContext
                                                      );
   }


   return u32Ret;
}

/***********************************************************************//**
 * \brief Returns the requested interface
 *
 * Delegates the request for an interface to the manager, as he is aware of
 * all interfaces.
 *
 * \param[in] enInterfaceId Interface ID
 *
 * \return tclDiagnosisIF* Pointer to requested interface.
 *//************************************************************************/
tclDiagnosisIF* tclSystemSetHandler::poGetInterface(tenDiagHandlerInterfaces enInterfaceId)
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_GETINTERFACE);

   tclDiagnosisIF* poReturnIF;

   poReturnIF = poManager->poGetInterface(enInterfaceId);


   return poReturnIF;
}

}
