/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief Service Handler CCA - Diagnosis Library - Introduced in MFD Nav
*
* (c) 2008 Blaupunkt Werke GmbH
*//****************************************************************************/

#ifdef DIAGLIB_FILE_NUMBER
#undef DIAGLIB_FILE_NUMBER
#endif
#define DIAGLIB_FILE_NUMBER F_DIAGLIB_CCA_SERVICE_HANDLER

#ifndef __INCLUDED_DIAGLIB_COMMON__
#include "../diaglib_common.h"
#endif

#ifndef __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_TYPES
#define OSAL_S_IMPORT_INTERFACE_MEMORY
#include "osal_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard
#endif


// MIDW_FI DEFINES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_FUNCTIONIDS

// FI DEFINES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "midw_fi_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define CCA_S_IMPORT_INTERFACE_GENERIC
#include "cca_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

// If the Service ID is not defined in ccaservice.h use the default one
#ifndef CCA_C_U16_SRV_DIAGLIB
#define CCA_C_U16_SRV_DIAGLIB 0x0071
#endif

#include "tclCCAHelper.h"
#include "../Enumerations.h"
#include "tclServiceHandlerCCA.h"
#include "../tclMessageContextManager.h"
#include "tclMessageContextCCA.h"
#include "../tclSystemSetListenerIF.h"

#include "../Trace.h"

namespace diaglib {

midw_fi_tcl_e8_SystemSetType SystemSetType2Fi(tenSystemSetType enSystemSetType)
{
  midw_fi_tcl_e8_SystemSetType result;
  switch (enSystemSetType)
  {
    case EN_TYPE_DEFAULT_HMI:
      result.enType = midw_fi_tcl_e8_SystemSetType::FI_EN_SYSTEMSET_TYPE_HMI;
      break;

    case EN_TYPE_DEFAULT_TEF:
      result.enType = midw_fi_tcl_e8_SystemSetType::FI_EN_SYSTEMSET_TYPE_TEF;
      break;

    case EN_TYPE_CODING:
      result.enType = midw_fi_tcl_e8_SystemSetType::FI_EN_SYSTEMSET_TYPE_CODING;
      break;

    case EN_TYPE_CALIBRATION:
      result.enType = midw_fi_tcl_e8_SystemSetType::FI_EN_SYSTEMSET_TYPE_CALIBRATION;
      break;

    default:
       DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_ENUM_VALUE);
       break;
  }

  return result;
}

midw_fi_tcl_e8_SystemSetResult SystemSetResult2Fi(tenSystemSetResult enSystemSetResult)
{
  midw_fi_tcl_e8_SystemSetResult result;
  switch (enSystemSetResult)
  {
    case EN_SYSTEMSET_OK:
      result.enType = midw_fi_tcl_e8_SystemSetResult::FI_EN_SYSTEMSET_OK;
      break;

    case EN_SYSTEMSET_NOT_OK:
      result.enType = midw_fi_tcl_e8_SystemSetResult::FI_EN_SYSTEMSET_NOT_OK;
      break;

    default:
      result.enType = midw_fi_tcl_e8_SystemSetResult::FI_EN_SYSTEMSET_NOT_OK;
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_ENUM_VALUE);
      break;
   }

   return result;
}

/***********************************************************************//**
 * \brief Send System Set MethodResult Message
 *
 * Construct and send CCA message by doing the following steps:
 * \li Retrieve the message context using the provided Context pointer
 * \li Extract messaging data out of the MessageContext
 * \li Convert diaglib internal types to CCA specific midw_fi types
 * \li Put everything into a CCA message object
 * \li Send it using the application message send functionality
 *
 * \param[in] enResult - Result of the SystemSet process
 * \param[in] oVector - Result vector containing error reasons in case of
 *                      failure
 * \param[in] context - MessageContext handle of incoming message
 *
 * \return  boolean if sending of response was successful or failed
 *//************************************************************************/
tBool tclServiceHandlerCCA::bSendSystemSetMethodResult (
                                              tenSystemSetResult enResult,
                                              const tclParameterVector& oVector,
                                              tContext context
                                            )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_SYSTEMSET_MR);

   midw_diaglibfi_tclMsgSystemSetMethodResult oResultMsgData;
   tclMessageContextCCA* poMessageContextCca = OSAL_NULL;
   tenSystemSetType enSystemSetType;
   tBool bResult = TRUE;

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_GET_BACK_CTXT_FOR_HANDLE , static_cast<tU32>(context) );
   // poMessageContextCca = static_cast<tclMessageContextCCA*> ( tclMessageContextManager::bGetContextBack(context) );

   if(
       tclMessageContextManager::bGetContextBack<tclMessageContextCCA>(context, &poMessageContextCca) == FALSE ||
       tclMessageContext::EN_MESSAGE_SYSTEM_SET != poMessageContextCca->getMessageGroup()
     )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_CANT_SEND_MSG_DUE_TO_CTXT);
      bResult = FALSE;
   }
   else
   {
      // Fill the CCA result message from diaglib internal data
      poMessageContextCca->getSystemSetType(enSystemSetType);

      oResultMsgData.SystemSetId              = poMessageContextCca->getId();
      oResultMsgData.SystemSetMode     = SystemSetType2Fi(enSystemSetType);
      oResultMsgData.SystemSetResult   = SystemSetResult2Fi(enResult);
      tclCCAHelper::vFillCcaValueListFromVector(oVector, oResultMsgData.ErrorList);

      fi_tclVisitorMessage oResultMsg( oResultMsgData );

      tclCCAHelper::vInitMethodResultServiceData(oResultMsg, poMessageContextCca, u16OurAppId);

      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_SENDING_MESSAGE_ID, oResultMsgData.SystemSetId);
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_SENDING_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));
      bResult = bSendMessage(oResultMsg);
   }

   if(OSAL_NULL != poMessageContextCca)
      OSAL_DELETE poMessageContextCca;

   return bResult;
}

/***********************************************************************//**
 * \brief Send System Set Prepare MethodResult Message
 *
 * Construct and send CCA message by doing the following steps:
 * \li Retrieve the message context using the provided Context pointer
 * \li Extract messaging data out of the MessageContext
 * \li Convert diaglib internal types to CCA specific midw_fi types
 * \li Put everything into a CCA message object
 * \li Send it using the application message send functionality
 *
 * \param[in] enResult - Result of the SystemSet prepare process
 * \param[in] oVector - Result vector containing error reasons in case of
 *                      failure
 * \param[in] context - MessageContext handle of incoming message
 *
 * \return  boolean if sending of response was successful or failed
 *//************************************************************************/
tBool tclServiceHandlerCCA::bSendSystemSetPrepareMethodResult (
                                                          tenSystemSetResult enResult,
                                                          const tclParameterVector& oVector,
                                                          tContext context
                                                         )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_SYSTEMSET_PREPARE_MR);

   midw_diaglibfi_tclMsgSystemSetPrepareMethodResult oResultMsgData;
   tclMessageContextCCA* poMessageContextCca = OSAL_NULL;
   tenSystemSetType enSystemSetType;
   tBool bResult = TRUE;

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_GET_BACK_CTXT_FOR_HANDLE , static_cast<tU32>(context) );
   if(
       tclMessageContextManager::bGetContextBack<tclMessageContextCCA>(context, &poMessageContextCca) == FALSE ||
       tclMessageContext::EN_MESSAGE_SYSTEM_SET != poMessageContextCca->getMessageGroup()
     )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_CANT_SEND_MSG_DUE_TO_CTXT);
      bResult = FALSE;
   }
   else
   {
      // Fill the CCA result message
      poMessageContextCca->getSystemSetType(enSystemSetType);

      oResultMsgData.SystemSetId              = poMessageContextCca->getId();
      oResultMsgData.SystemSetMode.enType     = static_cast<midw_fi_tcl_e8_SystemSetType::tenType>(enSystemSetType);
      oResultMsgData.SystemSetResult.enType   = static_cast<midw_fi_tcl_e8_SystemSetResult::tenType>(enResult);
      tclCCAHelper::vFillCcaValueListFromVector(oVector, oResultMsgData.ErrorList);

      fi_tclVisitorMessage oResultMsg( oResultMsgData );

      tclCCAHelper::vInitMethodResultServiceData(oResultMsg, poMessageContextCca, u16OurAppId);

      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_SENDING_MESSAGE_ID, oResultMsgData.SystemSetId);
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_SENDING_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));

      bResult = bSendMessage(oResultMsg);
   }

   if(OSAL_NULL != poMessageContextCca)
      OSAL_DELETE poMessageContextCca;

   return bResult;
}

/***********************************************************************//**
 * \brief Send System Set Finished MethodResult Message
 *
 * Construct and send CCA message by doing the following steps:
 * \li Retrieve the message context using the provided Context pointer
 * \li Extract messaging data out of the MessageContext
 * \li Convert diaglib internal types to CCA specific midw_fi types
 * \li Put everything into a CCA message object
 * \li Send it using the application message send functionality
 * As there is no need to report success or failure on a finish message, only
 * the MessageContext handle is a parameter to this message.
 *
 * \param[in] context - MessageContext handle of incoming message
 *
 * \return  boolean if sending of response was successful or failed
 *//************************************************************************/
tBool tclServiceHandlerCCA::bSendSystemSetFinishedMethodResult (
                                                      tContext context
                                                    )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_SYSTEMSET_FINISHED_MR);

   midw_diaglibfi_tclMsgSystemSetFinishedMethodResult oResultMsgData;
   tclMessageContextCCA* poMessageContextCca = OSAL_NULL;
   tBool bResult = TRUE;

   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_GET_BACK_CTXT_FOR_HANDLE , static_cast<tU32>(context) );

   if(
       tclMessageContextManager::bGetContextBack<tclMessageContextCCA>(context, &poMessageContextCca) == FALSE ||
       tclMessageContext::EN_MESSAGE_SYSTEM_SET != poMessageContextCca->getMessageGroup()
     )
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_CANT_SEND_MSG_DUE_TO_CTXT);
      bResult = FALSE;
   }
   else
   {
      // Fill the CCA result message
      oResultMsgData.SystemSetId = poMessageContextCca->getId();

      fi_tclVisitorMessage oResultMsg( oResultMsgData );

      tclCCAHelper::vInitMethodResultServiceData(oResultMsg, poMessageContextCca, u16OurAppId);

      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_SENDING_MESSAGE_ID, oResultMsgData.SystemSetId);
      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_SENDING_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));

      bResult = bSendMessage(oResultMsg);
   }

   if(OSAL_NULL != poMessageContextCca)
      OSAL_DELETE poMessageContextCca;

   return bResult;
}

/***********************************************************************//**
 * \brief Send System Set Check MethodResult Message
 *
 * Construct and send CCA message by doing the following steps:
 * \li Retrieve the message context using the provided Context pointer
 * \li Extract messaging data out of the MessageContext
 * \li Convert diaglib internal types to CCA specific midw_fi types
 * \li Put everything into a CCA message object
 * \li Send it using the application message send functionality
 *
 * \param[in] enResult - Result of the SystemSet check process
 * \param[in] oVector - Result vector containing error reasons in case of
 *                      failure or the value IDs that differ
 * \param[in] context - MessageContext handle of incoming message
 *
 * \return  boolean if sending of response was successful or failed
 *//************************************************************************/

tBool tclServiceHandlerCCA::bSendSystemSetCheckResult (
                                                        tenSystemSetResult /*enResult*/,
                                                        const tclParameterVector& /*oVector*/,
                                                        tContext /*context*/
                                                       )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_SYSTEMSET_CHECK_MR);

   // as this method is no longer supported we are returning TRUE here
   return TRUE;

//   midw_diaglibfi_tclMsgSystemSetVerifyMethodResult oResultMsgData;
//   tclMessageContextCCA* poMessageContextCca = OSAL_NULL;
//   tenSystemSetType enSystemSetType;
//   tBool bResult = TRUE;
//
//   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_GET_BACK_CTXT_FOR_HANDLE , static_cast<tU32>(context) );
//   if(
//       tclMessageContextManager::bGetContextBack<tclMessageContextCCA>(context, &poMessageContextCca) == FALSE ||
//       tclMessageContext::EN_MESSAGE_SYSTEM_SET != poMessageContextCca->getMessageGroup()
//     )
//   {
//      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_CANT_SEND_MSG_DUE_TO_CTXT);
//      bResult = FALSE;
//   }
//   else
//   {
//      // Fill the CCA result message
//      poMessageContextCca->getSystemSetType(enSystemSetType);
//
//      oResultMsgData.SystemSetId              = poMessageContextCca->getId();
//      oResultMsgData.SystemSetMode.enType     = static_cast<midw_fi_tcl_e8_SystemSetType::tenType>(enSystemSetType);
//      oResultMsgData.SystemSetResult.enType   = static_cast<midw_fi_tcl_e8_SystemSetResult::tenType>(enResult);
//      tclCCAHelper::vFillCcaValueListFromVector(oVector, oResultMsgData.ValueList);
//
//      fi_tclVisitorMessage oResultMsg( oResultMsgData );
//
//      tclCCAHelper::vInitMethodResultServiceData(oResultMsg, poMessageContextCca, u16OurAppId);
//
//      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_SENDING_MESSAGE_ID, oResultMsgData.SystemSetId);
//      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_SENDING_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));
//
//      bResult = bSendMessage(oResultMsg);
//   }
//
//   if(OSAL_NULL != poMessageContextCca)
//      OSAL_DELETE poMessageContextCca;
//
//   return bResult;
}

/***********************************************************************//**
 * \brief Register the system set listener
 *
 * Register the pointer to the diaglib INTERNAL listener for system set
 * requests.
 *
 * \param[in] poListener - Pointer to Listener
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vRegisterSystemSetListener      (
                                                   tclSystemSetListenerIF* poListener
                                                 )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_REGISTER_SYSTEMSET_LISTENER);
   if(OSAL_NULL == poListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   poSystemSetListener = poListener;

}

/***********************************************************************//**
 * \brief Handles incoming System Set Prepare messages
 *
 * On incoming request the following steps are executed
 * \li Check whether Message and registered listener are valid
 * \li On correct OpCode use midw_fi types to extract payload
 * \li Convert message payload to diaglib internal types
 * \li Create a message context to be able to send back a response
 * \li Call the registered listener using diaglib internal types
 *
 * \param[in] poMessage The incoming message
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vHandleSystemSetPrepare   ( amt_tclServiceData* poMessage )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_HANDLE_SYSTEMSET_PREPARE);
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_INCOMING_MESSAGE_ADV_START, static_cast<tU32>(u16OurAppId));

   if(OSAL_NULL == poMessage || OSAL_NULL == poSystemSetListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   tclMessageContextCCA* poMsgContext;
   midw_diaglibfi_tclMsgSystemSetPrepareMethodStart oMsgData;
   tenSystemSetType enSystemSetType;
   tContext context;

   switch (poMessage->u8GetOpCode())
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
      {
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_MESSAGE_TYPE_SYSSET_PREPARE_MS);
         // Create concrete msg object
         fi_tclVisitorMessage oMsg( poMessage );
         if (oMsg.s32GetData(oMsgData) == OSAL_ERROR)
         {
            NORMAL_M_ASSERT_ALWAYS ();
         }

         // Cast CCA enum to internal enum
         enSystemSetType = static_cast<tenSystemSetType>(oMsgData.SystemSetMode.enType);

         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_RECEIVED_MESSAGE_ID, oMsgData.SystemSetId);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_RECEIVED_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));

         // Create Message context and store it in MessageContext Manager
         poMsgContext = OSAL_NEW tclMessageContextCCA (
                                                           poMessage,
                                                           tclMessageContext::EN_MESSAGE_SYSTEM_SET,
                                                           oMsgData.SystemSetId,
                                                           (tU8)enSystemSetType
                                                         );
         context = tclMessageContextManager::poAddContext(poMsgContext);

         poSystemSetListener->vOnSystemSetPrepare(oMsgData.SystemSetId, enSystemSetType, context);

         break;
      }

      default:
      {
         DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_OPCODE_NOT_SUPPORTED , poMessage->u8GetOpCode() );
         vSendError ( poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         break;
      }
   }
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_INCOMING_MESSAGE_ADV_STOP, static_cast<tU32>(u16OurAppId));

}

/***********************************************************************//**
 * \brief Handles incoming System Set Finished messages
 *
 * On incoming request the following steps are executed
 * \li Check whether Message and registered listener are valid
 * \li On correct OpCode use midw_fi types to extract payload
 * \li Convert message payload to diaglib internal types
 * \li Create a message context to be able to send back a response
 * \li Call the registered listener using diaglib internal types
 *
 * \param[in] poMessage The incoming message
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vHandleSystemSetFinished  ( amt_tclServiceData* poMessage )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_HANDLE_SYSTEMSET_FINISHED);
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_INCOMING_MESSAGE_ADV_START, static_cast<tU32>(u16OurAppId));

   if(OSAL_NULL == poMessage || OSAL_NULL == poSystemSetListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   tclMessageContextCCA* poMsgContext;
   midw_diaglibfi_tclMsgSystemSetFinishedMethodStart oMsgData;
   tenSystemSetType enSystemSetType;
   tContext context;

   switch (poMessage->u8GetOpCode())
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
      {
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_MESSAGE_TYPE_SYSSET_FINISHED_MS);
         // Create concrete msg object
         fi_tclVisitorMessage oMsg( poMessage );
         if (oMsg.s32GetData(oMsgData) == OSAL_ERROR)
         {
            NORMAL_M_ASSERT_ALWAYS ();
         }

         // Cast CCA enum to internal enum
         enSystemSetType = static_cast<tenSystemSetType>(oMsgData.SystemSetMode.enType);

         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_RECEIVED_MESSAGE_ID, oMsgData.SystemSetId);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_RECEIVED_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));

         // Create Message context and store it in MessageContext Manager
         poMsgContext = OSAL_NEW tclMessageContextCCA (
                                                           poMessage,
                                                           tclMessageContext::EN_MESSAGE_SYSTEM_SET,
                                                           oMsgData.SystemSetId,
                                                           (tU8)enSystemSetType
                                                         );
         context = tclMessageContextManager::poAddContext(poMsgContext);

         poSystemSetListener->vOnSystemSetFinished(oMsgData.SystemSetId, enSystemSetType, context);

         break;
      }

      default:
      {
         DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_OPCODE_NOT_SUPPORTED , poMessage->u8GetOpCode() );
         vSendError ( poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         break;
      }
   }
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_INCOMING_MESSAGE_ADV_STOP, static_cast<tU32>(u16OurAppId));

}

/***********************************************************************//**
 * \brief Handles incoming System Set messages
 *
 * On incoming request the following steps are executed
 * \li Check whether Message and registered listener are valid
 * \li On correct OpCode use midw_fi types to extract payload
 * \li Convert message payload to diaglib internal types
 * \li Create a message context to be able to send back a response
 * \li Call the registered listener using diaglib internal types
 *
 * \param[in] poMessage The incoming message
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vHandleSystemSet          ( amt_tclServiceData* poMessage )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_HANDLE_SYSTEMSET);
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_INCOMING_MESSAGE_ADV_START, static_cast<tU32>(u16OurAppId));

   if(OSAL_NULL == poMessage || OSAL_NULL == poSystemSetListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_SYSTEMSET, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   tclMessageContextCCA* poMsgContext;
   midw_diaglibfi_tclMsgSystemSetMethodStart oMsgData;
   tenSystemSetType enSystemSetType;
   tContext context;

   switch (poMessage->u8GetOpCode())
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:
      {
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_MESSAGE_TYPE_SYSSET_MS);
         // Create concrete msg object
         fi_tclVisitorMessage oMsg( poMessage );
         if (oMsg.s32GetData(oMsgData) == OSAL_ERROR)
         {
            NORMAL_M_ASSERT_ALWAYS ();
         }

         // Cast CCA enum to internal enum
         enSystemSetType = static_cast<tenSystemSetType>(oMsgData.SystemSetMode.enType);

         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_RECEIVED_MESSAGE_ID, oMsgData.SystemSetId);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_RECEIVED_MESSAGE_APPID, static_cast<tU32>(u16OurAppId));

         // Create Message context and store it in MessageContext Manager
         poMsgContext = OSAL_NEW tclMessageContextCCA (
                                                           poMessage,
                                                           tclMessageContext::EN_MESSAGE_SYSTEM_SET,
                                                           oMsgData.SystemSetId,
                                                           (tU8)enSystemSetType
                                                         );

         context = tclMessageContextManager::poAddContext(poMsgContext);
         poSystemSetListener->vOnSystemSet(oMsgData.SystemSetId, enSystemSetType, context);

         break;
      }

      default:
      {
         DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_SYSTEMSET, W_DIAGLIB_OPCODE_NOT_SUPPORTED , poMessage->u8GetOpCode() );
         vSendError ( poMessage, AMT_C_U16_ERROR_OPCODE_NOT_SUPPORTED);
         break;
      }
   }
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_SYSTEMSET, I_DIAGLIB_INCOMING_MESSAGE_ADV_STOP, static_cast<tU32>(u16OurAppId));

}

}
