/**************************************************************************//**
* \file
* \author         AI/PJ-FO45 - Evers
* \date           2008-07-07
*
* \brief Service Handler CCA - Diagnosis Library - Introduced in MFD Nav
*
* (c) 2008 Blaupunkt Werke GmbH
*//****************************************************************************/

#ifdef DIAGLIB_FILE_NUMBER
#undef DIAGLIB_FILE_NUMBER
#endif

#include "tclServiceHandlerCCA.h"
#ifndef __INCLUDED_DIAGLIB_COMMON__
#define __INCLUDED_DIAGLIB_COMMON__
#include "../diaglib_common.h"
#endif

#define DIAGLIB_FILE_NUMBER F_DIAGLIB_CCA_SERVICE_HANDLER

#ifndef __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define __INCLUDED_DIA_EXTERN_OSAL_INTERFACE__
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_TYPES
#define OSAL_S_IMPORT_INTERFACE_MEMORY
#include "osal_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard
#endif


// MIDW_FI DEFINES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_DIAGLIBFI_FUNCTIONIDS

// FI DEFINES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "midw_fi_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

#define CCA_S_IMPORT_INTERFACE_GENERIC
#include "cca_if.h" //lint !e451 !e537 repeatedly included header file without standard include guard

// If the Service ID is not defined in ccaservice.h use the default one
#ifndef CCA_C_U16_SRV_DIAGLIB
#define CCA_C_U16_SRV_DIAGLIB 0x0071
#endif
#define DIAGLIB_U16_CCA_SERVICE_MAJOR_VERSION 0x0001
#define DIAGLIB_U16_CCA_SERVICE_MINOR_VERSION 0x0000
#define DIAGLIB_U16_CCA_SERVICE_PATCH_VERSION 0x0000

#include "tclCCAHelper.h"
#include "../Enumerations.h"
#include "../tclMessageContextManager.h"
#include "tclMessageContextCCA.h"

#include "../Trace.h"

namespace diaglib {

// Wshift-negative-value is disabled here because there seems to be a bug in gcc.
// For one there is no shift operation here to be found.
// Additionally the warning shows up only when building on Ubuntu 16.04, while it does not show up when compiling on Ubuntu 14.04, using the same version of gcc (OSTC 5.3 - gcc 6.2.0).
// This warning is valid for gcc version 6 and up since lower versions don't know it.
#if __GNUC__>5
# pragma GCC diagnostic push
# pragma GCC diagnostic ignored "-Wshift-negative-value"
#endif //__GNUC__>5
BEGIN_MSG_MAP(tclServiceHandlerCCA, ahl_tclBaseWork)
   ON_MESSAGE(MIDW_DIAGLIBFI_C_U16_ROUTINECONTROL,      vHandleRoutineControl)
   ON_MESSAGE(MIDW_DIAGLIBFI_C_U16_GETROUTINERESULT,    vHandleGetRoutineResult)
   ON_MESSAGE(MIDW_DIAGLIBFI_C_U16_IOCONTROL,           vHandleIOControl)
   ON_MESSAGE(MIDW_DIAGLIBFI_C_U16_SYSTEMSETPREPARE,    vHandleSystemSetPrepare)
   ON_MESSAGE(MIDW_DIAGLIBFI_C_U16_SYSTEMSETFINISHED,   vHandleSystemSetFinished)
   ON_MESSAGE(MIDW_DIAGLIBFI_C_U16_SYSTEMSET,           vHandleSystemSet)
   ON_MESSAGE(MIDW_DIAGLIBFI_C_U16_DIAGNOSISDATA,       vHandleDiagnosisData)
END_MSG_MAP()
#if __GNUC__>5
# pragma GCC diagnostic pop
#endif //__GNUC__>5

/***********************************************************************//**
 * \brief CCA Service Handler constructor
 *
 * Initializes all members, provided void* application context ist casted
 * to ail_tclAppInterface* .
 *
 * \param[in] poApplicationContext - Pointer to components application
 *
 * \return  n/a
 *//************************************************************************/
tclServiceHandlerCCA::tclServiceHandlerCCA(tVoid* poApplicationContext)
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_CONSTRUCTOR);
   poIoControlListener = OSAL_NULL;
   poRoutineControlListener = OSAL_NULL;
   poDiagDataListener = OSAL_NULL;
   poSystemSetListener = OSAL_NULL;
   poGWMain = OSAL_NULL;
   u16OurAppId = 0;

   // Catch errors during casting, so nothing breaks.
   _BP_TRY_BEGIN
   {
      if(OSAL_NULL != poApplicationContext)
      {
         // Evers 01.04.2011
         // This is an obsolete way to provide the application pointer. Use typesafe
         // method vSetApplication() instead
         poGWMain = static_cast<ail_tclAppInterfaceRestricted*>(poApplicationContext);
         u16OurAppId = poGWMain->u16GetAppId();
      }
   }
   _BP_CATCH_ALL
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_PARAMETER_WRONG_TYPE);
      poGWMain = OSAL_NULL;
   }
   _BP_CATCH_END

      DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_APPLICATION_ID, u16OurAppId);
   DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_CREATE_CCA_SVC_HANDLER);
}

/***********************************************************************//**
 * \brief CCA Service Handler destructor
 *
 * Invalidates all members. As this object does not own other dynamic objects
 * no memory is freed, Only aggregation pointers are set to NULL.
 *
 * \return  n/a
 *//************************************************************************/
tclServiceHandlerCCA::~tclServiceHandlerCCA()
{
   poIoControlListener = OSAL_NULL;
   poRoutineControlListener = OSAL_NULL;
   poDiagDataListener = OSAL_NULL;
   poSystemSetListener = OSAL_NULL;
   poGWMain = OSAL_NULL;
}

tVoid tclServiceHandlerCCA::vSetApplication(ail_tclIAppAdapt* poApp)
{
    if(OSAL_NULL != poApp)
    {
        poGWMain = static_cast<ail_tclAppInterfaceRestricted*>(poApp);
        u16OurAppId = poGWMain->u16GetAppId();
    }
    else
    {
        NORMAL_M_ASSERT_ALWAYS();
    }
}

/***********************************************************************//**
 * \brief Register the diagnosis data listener
 *
 * Register the pointer to the diaglib INTERNAL listener for diagnosis data
 * requests.
 *
 * \param[in] poListener - Pointer to Listener
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vRegisterDiagDataListener     (
                                                            tclDiagnosisDataListenerIF* poListener
                                                          )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_REGISTER_DIAGDATA_LISTENER);
   if(OSAL_NULL == poListener)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   poDiagDataListener = poListener;

}

/***********************************************************************//**
 * \brief Returns the ServiceID of this service (all diaglib services)
 *
 * Just returns the u16 Service ID. As all diaglib services have the same
 * Service ID, which is constant, this is hardcoded in the function.
 *
 * \param[in] none
 *
 * \return tU16 - ServiceID of diaglib services
 *//************************************************************************/
tU16 tclServiceHandlerCCA::u16GetServiceId()
{
   // diaglib_FuncTrace fnctrc(I_DIAGLIB_F_GET_SERVICE_ID);

   return CCA_C_U16_SRV_DIAGLIB;
}

/***********************************************************************//**
 * \brief Apply a new application state to the service
 *
 * Changes service behaviour in order to fit the new application state.
 *
 * \param[in] u32OldState - Previous application state
 * \param[in] u32NewState - New application state
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vOnNewAppState(tU32 u32OldState, tU32 u32NewState)
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_ON_NEW_APP_STATE);
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_APPLICATION_ID, u16OurAppId);

   if(u32OldState == u32NewState)
   {
      DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_NEW_STATE_EQUAL_OLD_STATE, u32NewState);
      return;
   }

   if(OSAL_NULL == poGWMain)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   switch(u32NewState)
   {
      case AMT_C_U32_STATE_NORMAL:
      {
         poGWMain->vServiceAvailabilityChanged(CCA_C_U16_SRV_DIAGLIB, AMT_C_U8_SVCSTATE_AVAILABLE);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_ENTERED_NEW_STATE, u32NewState);
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_SERVICE_AVAILABLE);
         break;
      }
      case AMT_C_U32_STATE_DIAGNOSIS:
      {
         poGWMain->vServiceAvailabilityChanged(CCA_C_U16_SRV_DIAGLIB, AMT_C_U8_SVCSTATE_AVAILABLE);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_ENTERED_NEW_STATE, u32NewState);
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_SERVICE_AVAILABLE);
         break;
      }
      case AMT_C_U32_STATE_PAUSE:
      {
         poGWMain->vServiceAvailabilityChanged(CCA_C_U16_SRV_DIAGLIB, AMT_C_U8_SVCSTATE_NOT_AVAILABLE);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_ENTERED_NEW_STATE, u32NewState);
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_SERVICE_UNAVAILABLE);
         break;
      }
      case AMT_C_U32_STATE_OFF:
      {
         poGWMain->vServiceAvailabilityChanged(CCA_C_U16_SRV_DIAGLIB, AMT_C_U8_SVCSTATE_NOT_AVAILABLE);
         DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_ENTERED_NEW_STATE, u32NewState);
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_SERVICE_UNAVAILABLE);
         break;
      }

      default:
      {
         DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNKNOWN_APP_STATE);
         break;
      }
   }

}

/***********************************************************************//**
 * \brief Inform this service of the status change of another service
 *
 * This is not applicable as the diaglib is never client to other services.
 * So this function will never be called, and when it is called it is an
 * error
 *
 * \param[in] Multiple Identifiers stating the service its status ...
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vOnServiceState(tU16 u16Val1, tU16 u16Val2, tU16 u16Val3, tU8 u8Val4, tU16 u16Val5)
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_ON_SERVICE_STATE);

   DIA_PARAMETER_INTENTIONALLY_UNUSED( u16Val1 );
   DIA_PARAMETER_INTENTIONALLY_UNUSED( u16Val2 );
   DIA_PARAMETER_INTENTIONALLY_UNUSED( u16Val3 );
   DIA_PARAMETER_INTENTIONALLY_UNUSED( u8Val4 );
   DIA_PARAMETER_INTENTIONALLY_UNUSED( u16Val5 );

   DIAGLIB_TRACE_WARN(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_NOT_IMPLEMENTED);

   return;
}

/***********************************************************************//**
 * \brief Handles messages with unknown function IDs
 *
 * A call to this function indicates a message with wrong/unknown function
 * ID as wrong service IDs won't be forwarded to this service. We send back
 * an error message stating this and delete the message.
 *
 * \param[in] poMessage The message
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vOnUnknownMessage(amt_tclBaseMessage* poMessage)
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_ON_UNKNOWN_MESSAGE);
   DIAGLIB_TRACE_INFO_U32(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_APPLICATION_ID, u16OurAppId);
   DIAGLIB_TRACE_WARN(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_UNKNOWN_MSG_RECEIVED);

   if(OSAL_NULL == poMessage)
   {
      DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_UNEXPECTED_NULLPOINTER);
      return;
   }

   if(poMessage->u8GetType() == AMT_C_U8_CCAMSGTYPE_SVCDATA)
   {
      /* ---
      cast the received message to a AMT Service data message so we
      can access the standard members
      --- */
      amt_tclServiceData oServiceData(poMessage);

      DIAGLIB_TRACE_WARN_U32(TR_CLASS_DIAGLIB_GENERAL, W_DIAGLIB_UNKNOWN_FUNCTION_ID, oServiceData.u16GetFunctionID());

      vSendError
      (
         &oServiceData,
         AMT_C_U16_ERROR_UNKNOWN_FCT_ID
      );
   }

   // We demand deletion of incoming messages in application
   // poMessage->bDelete();


}



/***********************************************************************//**
 * \brief Sends back an error message with the given ErrorCode
 *
 * Sends an error message to the sender of the provided message using
 * the provided OpCode
 *
 * \param[in] poMessage The incoming message that caused the error
 * \param[in] u16ErrorCode The error code to send in the message
 *
 * \return none
 *//************************************************************************/
tVoid tclServiceHandlerCCA::vSendError ( const amt_tclServiceData* poMessage, tU16 u16ErrorCode )
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_ERROR);

   gm_tclU16Message oErrorMsg (
                                poMessage->u16GetTargetAppID(),
                                poMessage->u16GetSourceAppID(),
                                0,
                                poMessage->u16GetCmdCounter(),
                                poMessage->u16GetServiceID(),
                                poMessage->u16GetFunctionID(),
                                AMT_C_U8_CCAMSG_OPCODE_ERROR
                              );

   oErrorMsg.vSetWord(u16ErrorCode);

   bSendMessage(oErrorMsg);


}

tU32 tclServiceHandlerCCA::u32GetAppIdentifier()
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_GET_APP_IDENTIFIER);

   tU32 u32Return;

   u32Return = static_cast<tU32>(u16OurAppId);


   return u32Return;
}

tBool tclServiceHandlerCCA::bGetServiceVersion ( tU16& rfu16MajorVersion,
                                                 tU16& rfu16MinorVersion,
                                                 tU16& rfu16PatchVersion) const
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_GET_SERVICE_VERSION);
   rfu16MajorVersion = DIAGLIB_U16_CCA_SERVICE_MAJOR_VERSION;
   rfu16MinorVersion = DIAGLIB_U16_CCA_SERVICE_MINOR_VERSION;
   rfu16PatchVersion = DIAGLIB_U16_CCA_SERVICE_PATCH_VERSION;

   return TRUE;
}

tBool tclServiceHandlerCCA::bSendMessage(amt_tclServiceData& oMsg)
{
   diaglib_FuncTrace fnctrc(I_DIAGLIB_F_SEND_MESSAGE);
   tBool bResult = FALSE;

   if (OSAL_NULL != poGWMain)
   {
      ail_tenCommunicationError enError = poGWMain->enPostMessage( &oMsg );

      if( AIL_EN_N_NO_ERROR == enError )
      {
         DIAGLIB_TRACE_INFO(TR_CLASS_DIAGLIB_GENERAL, I_DIAGLIB_MESSAGE_SEND_SUCCESSFUL);
         bResult = TRUE;
      }
      else
      {
         DIAGLIB_TRACE_ERROR_U32(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_ERROR_WHILE_SENDING_MESSAGE, enError);

         if (oMsg.bDelete() == FALSE)
         {
            DIAGLIB_TRACE_ERROR(TR_CLASS_DIAGLIB_GENERAL, E_DIAGLIB_ERROR_WHILE_DELETING_MESSAGE);
         }
      }
   }

   return bResult;
}

}
