/************************************************************************
* FILE:         dia_tclDiagSessionUds.cpp
* PROJECT:      ARION Prototyp
* SW-COMPONENT: Diagnostic application
*----------------------------------------------------------------------
*
* DESCRIPTION: Class for the diagnostic session of the UDS protocol
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2004 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 16.02.05  | 3SOFT Drenkhahn    | initial version
* 19.06.02  | John W P Elkins    | removal of trace statements
*
*************************************************************************/

#ifndef __INCLUDED_DIA_COMMON__
#include <common/framework/application/dia_common.h>
#endif

#ifndef __INCLUDED_DIA_APPLICATION__
#include "common/framework/application/dia_Application.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_MANAGER__
#include "common/framework/engine/dia_EngineManager.h"
#endif

#ifndef __INCLUDED_DIA_ENGINE_SERVER__
#include "common/framework/engine/dia_EngineServer.h"
#endif

#ifndef __DIA_UNIT_TESTING__

// application control
//#ifndef __INCLUDED_DIA_APPCONTROLLER__
//#include "common/framework/application/dia_AppController.h"
//#endif

#ifndef __INCLUDED_DIA_THREAD_MONITOR__
#include "common/framework/application/dia_ThreadMonitor.h"
#endif

#endif

#ifndef __INCLUDED_DIA_MESSAGE_BUFFER_UDS__
#include "common/framework/protocols/uds/dia_MessageBufferUDS.h"
#endif

#ifndef __INCLUDED_DIA_DIAGSESSION_UDS__
#include "common/depricated/dia_tclDiagSessionUds.h"
#endif

#ifndef __INCLUDED_DIA_DEFS_CONFIG__
#include "common/framework/config/dia_defsConfig.h"
#endif

#ifndef __INCLUDED_DIA_CONFIG_MANAGER__
#include "common/framework/config/dia_ConfigManager.h"
#endif

#ifndef __INCLUDED_DIA_SERVICE_DISPATCHER__
#include "common/framework/engine/dia_ServiceDispatcher.h"
#endif

#include "common/depricated/dia_tclDiagSessionBodyUds.h"

#include <common/framework/application/dia_ApplicationLock.h>


// static const attributes
tCString dia_tclDiagSessionUds::strQueueName  = "DIA_UDS_Session_Queue";
tCString dia_tclDiagSessionUds::strThreadName = "DIA_UDS";

#define UDS_REQUEST_QUEUE_SIZE            100

DIA_IMPL_SINGLETON_WITH_SETUP_AND_TEARDOWN(dia_tclDiagSessionUds)


#ifndef __DIA_UNIT_TESTING__
dia_tclDiagSessionUds*
getInstanceOfDiagSessionUDS ( void )
{
   return dia_tclDiagSessionUds::getInstance();
}

void
releaseInstanceOfDiagSessionUDS ( void )
{
   dia_tclDiagSessionUds::deleteInstance();
}
#endif

//-----------------------------------------------------------------------------

dia_tclDiagSessionUds::dia_tclDiagSessionUds ( void )
   : dia_ActiveObject(strThreadName,DIA_PROP_ENGINE_UDS_THREAD_PRIO,DIA_PROP_ENGINE_UDS_THREAD_STACK_SIZE),
     mpBody(0),
     mpMsgQueue(0)
//,
//     hMsgQueue(OSAL_C_INVALID_HANDLE)
{
   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::dia_tclDiagSessionUds()");
}

//------------------------------------------------------------------------------

tDiaResult
dia_tclDiagSessionUds::setup ( void )
{
   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::setup()");

   // create the state machine object
   mpBody = dia_tclDiagSessionBodyUds::getInstance();
   if ( mpBody )
   {
      mpBody->vInit();
   }

//   //
//   // create message queue
//   //
//   tS32 s32Result = OSAL_s32MessageQueueCreate (
//         strQueueName,
//         UDS_REQUEST_QUEUE_SIZE,               /* no of msgs */
//         sizeof(tclEvent*),
//         OSAL_EN_READWRITE,
//         &hMsgQueue
//   );
//
//   // Check the creation of the queue was successful
//   DIA_ASSERT(s32Result == OSAL_OK);

   if (0==mpMsgQueue)
   {
      DIA_TR_INF("CREATING MESSAGE QUEUE this=0x%p", this);
      mpMsgQueue = OSAL_NEW dia_Queue<tclEvent>(strQueueName,UDS_REQUEST_QUEUE_SIZE);
      if ( !mpMsgQueue || (mpMsgQueue->open() != DIA_SUCCESS) )
      {
         DIA_TR_INF("dia_tclDiagSessionUds::setup() returned DIA_FAILED");
         return DIA_FAILED;
      }
   }
   else
   {
      DIA_TR_ERR("dia_tclDiagSessionUds::setup() MESSAGE QUEUE already exist. this=0x%p", this);
      DIA_ASSERT_ALWAYS();
   }

#ifndef VARIANT_S_FTR_DIAG_DELAYED_START_DIA_UDS
   // if VARIANT_S_FTR_DIAG_DELAYED_START_DIA_UDS is enabled thread is started once Runlevel 1 is reached
   startThread();
#endif /* VARIANT_S_FTR_DIAG_DELAYED_START_DIA_UDS */


   DIA_TR_INF("dia_tclDiagSessionUds::setup() returned DIA_SUCCESS");
   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

dia_tclDiagSessionUds::~dia_tclDiagSessionUds ( void )
{
   _BP_TRY_BEGIN
   {
      // Please use releaseInstanceOfDiagSessionUDS() only.
      //(void) dia_tclDiagSessionUds::tearDown();

      if ( mpMsgQueue )
      {
         OSAL_DELETE mpMsgQueue;
         mpMsgQueue = 0;
      }

#if 0 /* it was already done in tearDown() */
      // delete the body object
      if ( mpBody )
      {
         OSAL_DELETE mpBody;
         mpBody = 0;
      }
#else
      mpBody = 0;
#endif

   }
   _BP_CATCH_ALL
   {
      DIA_TR_ERR("EXCEPTION CAUGHT: dia_tclDiagSessionUds::~dia_tclDiagSessionUds !!!");
      DIA_ASSERT_ALWAYS();
   }
   _BP_CATCH_END
}

//-----------------------------------------------------------------------------

tDiaResult
dia_tclDiagSessionUds::tearDown ( void )
{
   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::tearDown()");

//      tCU8 dummy=0;
//
//      // send data to queue
//      tS32 s32Result = OSAL_s32MessageQueuePost (
//            hMsgQueue,
//            &dummy,
//            sizeof(dummy),
//            OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST
//            );
//
//      // Check the returned result
//      DIA_ASSERT(s32Result == OSAL_OK);
//
//      tS32 s32WaitCount = 0;
//      while((hMsgQueue != OSAL_C_INVALID_HANDLE) && (s32WaitCount < 10))
//      {
//         OSAL_s32ThreadWait(100);
//         ++s32WaitCount;
//      }
//
//      // Close the message queue
//      DIA_TR_INF("--- +++ About to CLOSE the UDS msg queue deletion +++ ---");
//      tS32 s32CloseResult = OSAL_s32MessageQueueClose(hMsgQueue);
//      DIA_ASSERT(s32CloseResult == OSAL_OK);
//
//      // Now delete the message queue
//      DIA_TR_INF("--- +++ About to DELETE the UDS msg queue deletion +++ ---");
//      s32Result = OSAL_s32MessageQueueDelete(strQueueName);
//      DIA_ASSERT(s32Result == OSAL_OK);
//
//      // to signal to the destructor that we have finished clearing up
//      hMsgQueue = OSAL_C_INVALID_HANDLE;

   // end thread and close msg queue by sending 1 byte long pseudo-event to the diag session thread


   //terminate thread
#ifndef VARIANT_S_FTR_DIAG_DELAYED_START_DIA_UDS
   terminateThread();
#endif

   tU32 dummy = 0;
   if ( mpMsgQueue )
   {
      mpMsgQueue->addElement((tclEvent*) &dummy); //lint !e826 !e740 Info: Suspicious pointer-to-pointer conversion (area too small); Unusual pointer cast (incompatible indirect types); Correct but here we need to pass a null pointer to the OSAL queue --> crazy, but if we don't do it this way we are getting a reset
      for ( tS32 waitCount=0; waitCount < 10; waitCount++ )
      {
         DIA_TR_INF("dia_tclDiagSessionUds::tearDown Sleep for 100ms");
#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
         OSAL_s32ThreadWait(100);
#else
         usleep(100*1000);   // DiagOmitSleepWarning
#endif
      }
      mpMsgQueue->close();
   }

   dia_tclDiagSessionBodyUds::deleteInstance();

   return DIA_SUCCESS;
}

//-----------------------------------------------------------------------------

dia_MessageBufferUDS& dia_tclDiagSessionUds::oDiagMsgBuffer() const
{
#ifdef __DIA_UNIT_TESTING__
   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::oDiagMsgBuffer()");
#endif

   // Declare pointer to UDS Msg Buffer
   dia_MessageBufferUDS *poDiagMsgBufferUds = OSAL_NULL;

   // check fo Null pointer
   if ( poDiagMsgBuffer != OSAL_NULL )
   {
#ifdef __DIA_UNIT_TESTING__
      DIA_TR_INF("dia_tclDiagSessionUds::oDiagMsgBuffer poDiagMsgBuffer is not NULL");
#endif
      // Not Null so perform the cast
      poDiagMsgBufferUds = static_cast<dia_MessageBufferUDS*>(poDiagMsgBuffer);
   }

   if(poDiagMsgBufferUds == OSAL_NULL)
   {
      DIA_ASSERT_ALWAYS();
      // this code should never execute and it's here to get round a lint warning
      // a better solution would be to change the function to return a pointer
      // but there are 3100 references in about 200 files so let's do it this way
      // anyone offended by this solution is free edit the 3000 references
      static dia_MessageBufferUDS oDiagMsgBufferUds(0, 0, 0);
      return oDiagMsgBufferUds;
   }

   return *poDiagMsgBufferUds;
}

//-----------------------------------------------------------------------------


dia_MessageBufferUDS *dia_tclDiagSessionUds::poUdsMsgBuffer() //const
{
   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::poUdsMsgBuffer()");

   // dynamic cast will automatically return 0 if downcast fails
   // no need to explicitly check null-ness of poDiagMsgBuffer
   return static_cast<dia_MessageBufferUDS*>(poDiagMsgBuffer);
}


//-----------------------------------------------------------------------------


void
dia_tclDiagSessionUds::vLookupSrvcHandler ( void )
{
   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::vLookupSrvcHandler()");

   // access was granted or the decision has to be taken in the service handler

   // try to find the handler in the map based repository

   dia_EngineManager* pEngineMgr = getInstanceOfEngineManager();
   if ( pEngineMgr )
   {
      dia_EngineServer* pEngine = 0;
      if ( pEngineMgr->queryEngineServer(mEngineID,&pEngine) == DIA_SUCCESS )
      {
         dia_ServiceDispatcher* pDispatcher = pEngine->getDispatcher();
         if ( pDispatcher && poDiagMsgBuffer )
         {
            pDispatcher->lookup(oDiagMsgBuffer());
         }
      }
   }
}


//-----------------------------------------------------------------------------

#ifdef VARIANT_S_FTR_DIAG_DELAYED_START_DIA_UDS
void dia_tclDiagSessionUds::vOnRunLevelChanged ( dia_enRunlevel newLevel, dia_enRunlevel oldLevel )
{
   DIA_TR_INF("--> dia_tclDiagSessionUds::vOnRunLevelChanged (%d -> %d)", oldLevel, newLevel);

   if ((oldLevel == DIA_EN_RUNLEVEL_UNDEFINED) && (newLevel == DIA_EN_RUNLEVEL_1))
   {
      (void) startThread();
   }

   DIA_TR_INF("<-- dia_tclDiagSessionUds::vOnRunLevelChanged");
}
#else
void
dia_tclDiagSessionUds::vOnRunLevelChanged ( dia_enRunlevel /*newLevel*/, dia_enRunlevel /*oldLevel*/ )
{}
#endif /* VARIANT_S_FTR_DIAG_DELAYED_START_DIA_UDS */

//-----------------------------------------------------------------------------

void
dia_tclDiagSessionUds::vThreadEntrypointObject ( void )
{
   DIA_TR_INF( "---> Entering  dia_tclDiagSessionUds::vThreadEntrypointObject");

   tclEvent* pEvent;
   bool isDone = false;

   do
   {
      pEvent = 0;
      if ( mpMsgQueue && mpMsgQueue->getElement(&pEvent) == DIA_SUCCESS )
      {
         if (NULL==pEvent)    DIA_TR_INF("pEvent is NULL");

         if ( pEvent )
         {
            DIA_TR_INF("##### UDS_WORKER: PROCESS EVENT #####");
            DIA_TR_INF("Accept tclEvent Id=%d, size=%zu", pEvent->getEventID(), sizeof(tclEvent*));

            bool needToDestroy = true;
            if( (pEvent->getEventID() == (tS32) dia_tclDiagSession::enEvIdReqRx) || (pEvent->getEventID() == (tS32) dia_tclDiagSession::enEvIdIntMsgRx) )
            {
               needToDestroy = false;
            }

            if ( mpBody )
            {
               #if !defined(VARIANT_S_FTR_DONOTUSE_APPLICATION_LOCK) && !defined(__DIA_UNIT_TESTING__)
               dia::LockScope rLock((dia::getInstanceOfApplicationLock()->getAppLock()));
               #endif
               DIA_TR_INF("Forward Message to FSM ...");
               mpBody->vAcceptEvent(*pEvent);
               DIA_TR_INF("Message Processed !!");
            }

            if ( needToDestroy )
            {
               // enEvIdReqRx and enEvIdIntMsgRx events are handled separately, because they might get deferred
               OSAL_DELETE pEvent;
               pEvent = 0;
            }
         }
         else
         {
            isDone = true;
         }
      }
      else
      {
         // retrieving the message has failed some reason
         isDone = true;
      }
   }
   // we keep the thread running until we have received the stop message (pEvent is null pointer) or message reception has failed
   while ( !isDone ) ;

   DIA_TR_ERR("############################################################");
   DIA_TR_ERR("#");
   DIA_TR_ERR("# UDS THREAD IS GETTING TERMINATED !!!!!!!!!!!!!!!!!!!!!!!!!");
   DIA_TR_ERR("#");
   DIA_TR_ERR("############################################################");

#ifndef VARIANT_S_FTR_ENABLE_THREAD_AS_PTHREAD_THREAD
   // Ask OSAL what the last ERROR to occur was
   tU32 u32ErrorReason = OSAL_u32ErrorCode();
   if ( u32ErrorReason != OSAL_E_NOERROR ) DIA_TR_ERR("msg queue error: %d", u32ErrorReason);
#endif
}

//----------------------------------------------------------------------------


void
dia_tclDiagSessionUds::vNullResponse ( const tU8*, tU16, tCookieType /*cookie*/ )
{
   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::vNullResponse()");

   dia_tclDiagSessionUds* pUdsEngine = getInstanceOfDiagSessionUDS();
   if ( pUdsEngine )
   {
      // here we know that everything is fine with the configuration, so we can stop the timer
      pUdsEngine->oTimer.s32SetTime(0,0);
      // post message to the session queue
      pUdsEngine->vEmit(OSAL_NEW dia_tclDiagSession::tclEventConfTxOk());
   }
}


//-----------------------------------------------------------------------------

void
dia_tclDiagSessionUds::vEmit ( tclEvent* pEvent) const
{
   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::vEmit()");

   if ( !pEvent ) return;

   tDiaResult retCode = ( mpMsgQueue ) ? mpMsgQueue->addElement(pEvent) : DIA_E_INVALID_POINTER;

   //trace
   if (NULL!=mpMsgQueue)
   {
      tU32 queueMaxSize = UDS_REQUEST_QUEUE_SIZE;
      tU32 queueCurrentElemNum = mpMsgQueue->getSize();
      float queueUsage = (float)queueCurrentElemNum / (float)queueMaxSize;

      //print info if queue usage is greater than 80%.
      if (queueUsage > 0.8f)
      {
         DIA_TR_ERR("dia_tclDiagSessionUds::vEmit(): queueMaxSize=%d CurrElemNum=%d", queueMaxSize, queueCurrentElemNum);
      }
   }

   if (DIA_SUCCESS != retCode)
   {
      DIA_TR_INF("dia_tclDiagSessionUds::vEmit(): retCode=0x%08X mpMsgQueue=0x%p", retCode, mpMsgQueue);
   }

   DIA_ASSERT(retCode == DIA_SUCCESS);
}

//void
//dia_tclDiagSessionUds::vEmit ( dia_tclDiagSession::tclEvent* pEvent) const
//{
//   dia_tclFnctTrace oTrace("dia_tclDiagSessionUds::vEmit()");
//
//#ifndef __DIA_UNIT_TESTING__
//
//   if ( !pEvent ) return;
//
//   tU32 ptr = (tU32) pEvent;
//
//   DIA_ASSERT(hMsgQueue != OSAL_NULL);
//
//   // send event to the own diag session thread
//   tS32 s32Result = OSAL_s32MessageQueuePost (
//         hMsgQueue,
//         (tU8*) &ptr, //reinterpret_cast<tPCU8>(&pEvent),
//         sizeof(tclEvent*),
//         OSAL_C_U32_MQUEUE_PRIORITY_HIGHEST
//         );
//
//   if ( OSAL_OK != s32Result )
//   {
//      tU32 u32ErrorCode = OSAL_u32ErrorCode();
//      DIA_TR_ERR("Error by posting message, OSAL_u32ErrorCode = ", u32ErrorCode);
//   }
//
//   DIA_ASSERT(s32Result == OSAL_OK);
//
//#endif
//}
