/**
 * @file WifiSetupService.cpp
 * @author RBEI/ECO32 Karthikeyan Madeswaran
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "WifiSetupService.h"
#include "ConflictManagerFactory.h"
#include "WBLServiceInterfaces.h"
#include "DbusIfTypes.h"
#include "DbusParser.h"
#include "DbusHelper.h"
#include "WBLDefines.h"
#include "WBLTypeProperties.h"
#include "WBLASFComponent.h"
#include <boost/shared_ptr.hpp>

namespace org
{
namespace bosch
{

static ChannelList carPlayChannels_2_4GHz = {1, 6, 11};
static ChannelList carPlayChannels_5GHz = {36, 40, 44, 48, 149, 153, 157, 161};

static DbusParser<WBLServiceInterfaces> wblDbusParser;

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/WBLServices", WifiSetupService, Debug);

WifiSetupService::WifiSetupService(::std::string portName) :
      WifisetupStub(portName)
{
   LOG_INFO("WifiSetupService:Constructor");
   wblDbusParser.setTraces(true);
}//WifiSetupService::WifiSetupService

WifiSetupService::~WifiSetupService()
{

}//WifiSetupService::~WifiSetupService

void WifiSetupService::onDeAuthenticateStationRequest(
      const ::boost::shared_ptr< DeAuthenticateStationRequest >& request)
{
   LOG_INFO("WifiSetupService:onDeAuthenticateStationRequest");

   if(_currentItem.property.mode == WIFI_MODE_STA)
   {
      LOG_DEBUG("WifiSetupService:onDeAuthenticateStationRequest is Invalid for STA mode");
      onDeAuthenticateStationResponse(WBL_ERR_INVALID_ARGS,static_cast<uint>(request->getAct()));
      return;
   }
   ::std::map<::std::string, ::asf::dbus::DBusVariant> stationMap = request->getStation();
   asf::dbus::DBusVariant valVariant;
   const ::boost::shared_ptr < DeAuthenticateStationMsg > deAuthenticateMsg(new DeAuthenticateStationMsg);
   DeAuthenticateStationItem deAuthItem;
   ::std::string macAddress,type,asPath;

   for (auto stationIter = stationMap.begin(); stationIter != stationMap.end(); stationIter++)
   {
      valVariant = stationIter->second;
      DBusMessageIter* valIter = valVariant.getReadIterator();

      if (stationIter->first.compare("Macaddress") == 0)
      {
         macAddress = ::std::string((char*)getDbusIterValue(valIter));
         LOG_DEBUG("WifiSetupService:onDeAuthenticateStationRequest MacAddress:%s", macAddress.c_str());
         deAuthItem.macAddress = macAddress;
      }
      else if (stationIter->first.compare("DeAuthenticate") == 0)
      {
         type = ::std::string((char*)getDbusIterValue(valIter));
         LOG_DEBUG("WifiSetupService:onDeAuthenticateStationRequest Type: %s", type.c_str());
         deAuthItem.type = type;
      }
      else
      {
         LOG_DEBUG("WifiSetupService:onDeAuthenticateStationRequest Invalid Arguments");
         onDeAuthenticateStationResponse(WBL_ERR_INVALID_ARGS,static_cast<uint>(request->getAct()));
      }
   }
   if (!isAssociatedStationExist(macAddress))
   {
      LOG_DEBUG("WifiSetupService:onDeAuthenticateStationRequest Invalid Arguments");
      onDeAuthenticateStationResponse(WBL_ERR_INVALID_ARGS,static_cast<uint>(request->getAct()));
   }
   else
   {
      deAuthenticateMsg->setMsgToken(static_cast<uint>(request->getAct()));
      deAuthItem.objPath = _currentItem.nativeObjPath;
      deAuthenticateMsg->setDeAuthStation(deAuthItem);
      ConflictManagerFactory* factory = ConflictManagerFactory::getInstance();
      ICmdConflictManager* conflictManagerIf = (factory) ? (factory->getConflictManagerRequestIf()) : (nullptr);
      if (conflictManagerIf)
      {
         conflictManagerIf->onProcessRequest(deAuthenticateMsg);
      }
   }
}//WifiSetupService::onDeAuthenticateStationRequest

void WifiSetupService::onDeAuthenticateStationResponse(WBLErrorCode errcode, uint32 actToken)
{
   LOG_INFO("WifiSetupService::onDeAuthenticateStationResponse");
   ::std::string errname = "";
   ::std::string errmsg = "";
   if (errcode == WBL_ERR_NOERROR)
   {
      LOG_DEBUG("No error");
      sendDeAuthenticateStationResponse(actToken);
   }
   else
   {
      errname = wblErrorCode.getErrorCode2String(errcode);
      LOG_INFO("WifiSetupService::onDeAuthenticateStationResponse %s", errname.c_str());
      sendDeAuthenticateStationError(errname, errmsg, actToken);
   }
} //WifiSetupService::onDeAuthenticateStationResponse

void WifiSetupService::onManageBandwidthRequest(
      const ::boost::shared_ptr< ManageBandwidthRequest >& request)
{
   LOG_INFO("WifiSetupService::onManageBandwidthRequest");
   (void) request;
} //WifiSetupService::onManageBandwidthRequest

void WifiSetupService::onSetRestrictionsRequest(const ::boost::shared_ptr< SetRestrictionsRequest >& request)
{
   LOG_INFO("WifiSetupService::onSetRestrictionsRequest");
   svMap mapSPI,mapTechnology;
   ::std::string strRestrict,strTech;
   asf::dbus::DBusVariant valVariant,techVariant;
   DBusMessageIter* valIter;
   WBLTechnology tech = WBL_TECH_UNKNOWN;
   bool bRet = false;

   mapSPI = request->getTechnology();
   strRestrict = request->getRestrict();
   mapTechnology["Technology"] = mapSPI.find("SPITechnology")->second;
   for(auto iter=mapTechnology.begin();iter != mapTechnology.end();++iter)
   {
      techVariant = iter->second;
      valIter= techVariant.getReadIterator();
      if(!strcmp(iter->first.c_str(),"Technology"))
      {
         strTech = ::std::string((char*)getDbusIterValue(valIter));
         tech = wblTechRestriction.getWBLTechnology2Enum(strTech);
         LOG_DEBUG("WifiSetupService::onSetRestrictionsRequest Technology:%s",strTech.c_str());
      }
   }
   LOG_DEBUG("WifiSetupService::onSetRestrictionsRequest Restrict:%s",strRestrict.c_str());
   updateRestrictions(tech,WBL_RESTRICT_REQUESTED);
   LCMFactory* lcmFactory = LCMFactory::getInstance();
   ILCMClient* poLCMClient = (lcmFactory) ? (lcmFactory->getLCMClientIf()) : (nullptr);
   if(WBL_TECH_UNKNOWN != tech)
   {
      if(0 == strRestrict.compare("ACTIVATE"))
      {
         LOG_DEBUG("WifiSetupService::onSetRestrictionsRequest ACTIVATE");
         updateRestrictions(tech,WBL_RESTRICT_ACTIVATING);
         bRet = poLCMClient->setTechnologyRestrictions(true);
         updateRestrictions(tech,(bRet?WBL_RESTRICT_ACTIVATED:WBL_RESTRICT_FAILED));
      }
      else if(0 == strRestrict.compare("DEACTIVATE"))
      {
         LOG_DEBUG("WifiSetupService::onSetRestrictionsRequest DEACTIVATE");
         updateRestrictions(tech,WBL_RESTRICT_DEACTIVATING);
         bRet = poLCMClient->setTechnologyRestrictions(false);
         updateRestrictions(tech,(bRet?WBL_RESTRICT_DEACTIVATED:WBL_RESTRICT_FAILED));
      }
      else
      {
         LOG_DEBUG("WifiSetupService:onDeAuthenticateStationRequest Invalid Arguments");
         onSetRestrictionsResponse(WBL_ERR_INVALID_ARGS,static_cast<uint>(request->getAct()));
      }
      onSetRestrictionsResponse(WBL_ERR_NOERROR,static_cast<uint>(request->getAct()));
   }
   else
   {
      LOG_DEBUG("WifiSetupService:onDeAuthenticateStationRequest Invalid Arguments");
      onSetRestrictionsResponse(WBL_ERR_INVALID_ARGS,static_cast<uint>(request->getAct()));
   }
}//WifiSetupService::onSetRestrictionsRequest

void WifiSetupService::onSetRestrictionsResponse(WBLErrorCode errcode, uint32 actToken)
{
   LOG_INFO("WifiSetupService::onSetRestrictionsResponse");
   ::std::string errname = "";
   ::std::string errmsg = "";
   if (errcode == WBL_ERR_NOERROR)
   {
      LOG_DEBUG("No error");
      sendSetRestrictionsResponse(actToken);
   }
   else
   {
      errname = wblErrorCode.getErrorCode2String(errcode);
      LOG_INFO("WifiSetupService::onSetRestrictionsResponse %s", errname.c_str());
      sendSetRestrictionsError(errname, errmsg, actToken);
   }
} //WifiSetupService::onSetRestrictionsResponse

void WifiSetupService::onRestoreFactorySettingsRequest(
      const ::boost::shared_ptr<RestoreFactorySettingsRequest>& request)
{
   LOG_INFO("WifiSetupService::onRestoreFactorySettingsRequest: %s", WIFI_MODE_TO_STR(_currentItem.property.mode));

   const ::boost::shared_ptr<RestoreFactorySettingsMsg> factoryReset(new RestoreFactorySettingsMsg);
   factoryReset->setWifiMode(_currentItem.property.mode);
   factoryReset->setMsgID(WBL_MSG_RESTORE_SETTINGS_REQ);
   factoryReset->setMsgToken(static_cast<uint>(request->getAct()));

   ConflictManagerFactory* factory = ConflictManagerFactory::getInstance();
   ICmdConflictManager* conflictManagerIf = (factory) ? (factory->getConflictManagerRequestIf()) : (nullptr);
   if (conflictManagerIf)
   {
      conflictManagerIf->onProcessRequest(factoryReset);
   }
} //WifiSetupService::onRestoreFactorySettingsRequest

void WifiSetupService::onRestoreFactorySettingsResponse(const WBLErrorCode errcode, const uint32 actToken)
{
   LOG_INFO("WifiSetupService::onRestoreFactorySettingsResponse");

   if (errcode == WBL_ERR_NOERROR)
   {
      LOG_DEBUG("No error");
      sendRestoreFactorySettingsResponse(actToken);
   }
   else
   {
      const ::std::string errname = wblErrorCode.getErrorCode2String(errcode);
      LOG_INFO("WifiSetupService::onRestoreFactorySettingsResponse %s", errname.c_str());
      sendRestoreFactorySettingsError(errname, "", actToken);
   }

} //WifiSetupService::onRestoreFactorySettingsResponse

void WifiSetupService::updateMode(WifiMode wifiMode)
{
   LOG_INFO("WifiSetupService::updateMode");
   _currentItem.property.mode = wifiMode;
   std::string mode = wblWifiMode.getWifiMode2String(wifiMode);
   setMode(mode);
} //WifiSetupService::updateMode

void WifiSetupService::updateInterface(::std::string interface)
{
   LOG_INFO("WifiSetupService::updateInterface");
   _currentItem.property.interface = interface;
   setInterface(interface);
}//WifiSetupService::updateInterface

void WifiSetupService::updatePowerState(WBLPowerStateType powerState)
{
   LOG_INFO("WifiSetupService::updatePowerState");
   _currentItem.property.powerState = powerState;
   std::string strPowerState = wblPowerState.getPowerState2String(powerState);
   setPowerState(strPowerState);
}//WifiSetupService::updatePowerState

void WifiSetupService::updateAPMacAddress(::std::string macAddress)
{
   LOG_INFO("WifiSetupService::updateAPMacAddress");
   _currentItem.property.apSetup.macAddress = macAddress;
   setMacAddress(macAddress);
}//WifiSetupService::updateAPMacAddress

void WifiSetupService::updateFrequency(WBLFrequencyBand frequency)
{
   LOG_INFO("WifiSetupService::updateFrequency");
   _currentItem.property.frequency = frequency;
   std::string strFrequency = wblFrequency.getFrequency2String(frequency);
   setFrequency(strFrequency);
}//WifiSetupService::updateFrequency

void WifiSetupService::updatePowerFailureReason(WBLPowerFailureType powerFailurType)
{
   LOG_INFO("WifiSetupService::updatePowerFailureReason");
   _currentItem.property.powerFailure = powerFailurType;
   std::string strPowerFailure = wblPowerFailure.getPowerFailure2String(powerFailurType);
   setPowerFailureReason(strPowerFailure);
}//WifiSetupService::updatePowerFailureReason

void WifiSetupService::updateCurrentOperatingChannel(int currentOperatingChannel)
{
   LOG_INFO("WifiSetupService::updateCurrentOperatingChannel");
   _currentItem.property.currChannel = static_cast<uint16>(currentOperatingChannel);
   setCurrentOperatingChannel(currentOperatingChannel);
}//WifiSetupService::updateCurrentOperatingChannel

void WifiSetupService::updateConnectionStatus(WBLConnectionStatus &connectionStatus)
{
   LOG_INFO("WifiSetupService::updateConnectionStatus");
   svMap connStatus;
   dbus_bool_t bConnectedVal = connectionStatus.isConnected;
   dbus_uint32_t uHealthIndicatorVal = connectionStatus.healthIndicatorValue;
   insert_map_element(connStatus,"Connected",DBUS_TYPE_BOOLEAN,&bConnectedVal);
   insert_map_element(connStatus,"HealthIndicator",DBUS_TYPE_UINT32,&uHealthIndicatorVal);
   _currentItem.property.connectionStatus = connectionStatus;
   svMap &connectStatus = getConnectionStatusMutable();
   if(!connectStatus.empty())
   {
      connectStatus = connStatus;
      sendConnectionStatusUpdate();
   }
   else
   {
      setConnectionStatus(connStatus);
   }
}//WifiSetupService::updatePowerState


void WifiSetupService::updateAllProperties(WifiSetUpItem& item)
{
   LOG_INFO("WifiSetupService::updateAllProperties");
   updateMode(item.property.mode);
   updateInterface(item.property.interface);
   updateFrequency(item.property.frequency);
   updateCurrentOperatingChannel(item.property.currChannel);
   updatePowerState(item.property.powerState);
   updateConnectionStatus(item.property.connectionStatus);
   if(item.property.mode == WIFI_MODE_AP1 ||
         item.property.mode == WIFI_MODE_AP2)
   {
      updateAPMacAddress(item.property.apSetup.macAddress);
      processApSetupUpdate(item.property.apSetup);
      updateSupportedFrequencies(item.property.apSetup);
      updateIPv4Configuration(item.property.apSetup);
      if(!_techRestrictions.empty())
         sendRestrictionsUpdate();
   }
}//WifiSetupService::updateAllProperties

void WifiSetupService::updateChangedProperties(WifiSetUpItem& item)
{
   LOG_INFO("WifiSetupService::updateChangedProperties");
   if(_currentItem.property.mode != item.property.mode)
      updateMode(item.property.mode);
   if(_currentItem.property.interface != item.property.interface)
      updateInterface(item.property.interface);
   if(_currentItem.property.frequency != item.property.frequency)
      updateFrequency(item.property.frequency);
   if(_currentItem.property.currChannel != item.property.currChannel)
      updateCurrentOperatingChannel(item.property.currChannel);
   if(_currentItem.property.powerState != item.property.powerState)
      updatePowerState(item.property.powerState);
   if(!(_currentItem.property.connectionStatus == item.property.connectionStatus))
      updateConnectionStatus(item.property.connectionStatus);
   if(item.property.mode == WIFI_MODE_AP1 ||
         item.property.mode == WIFI_MODE_AP2)
   {
      updateAPMacAddress(item.property.apSetup.macAddress);
      processApSetupUpdate(item.property.apSetup);
      updateSupportedFrequencies(item.property.apSetup);
      updateIPv4Configuration(item.property.apSetup);
   }
}//WifiSetupService::updateAllProperties

void WifiSetupService::updateSetupProperties(WifiSetUpItem& item)
{
   LOG_INFO("WifiSetupService::updateSetupProperties");
   if(_currentItem.isEmpty())
   {
      _currentItem = item;
      updateAllProperties(item);
   }
   else
   {
      updateChangedProperties(item);
   }
}//WifiSetupService::updateSetupProperties

void WifiSetupService::insertAssociatedStation(const AssociatedStationItem &staItem,
      DBusMessageIter* iterSta)
{
   LOG_INFO("WifiSetupService::insertAssociatedStation");

   DBusMessageIter iterEntry, iterEntryElement, iterStaDet;
   std::string macAddress, interface, ipAddress, apPath, hostname;

   macAddress = staItem.property.macAddress;
   interface = staItem.property.interface;
   ipAddress = staItem.property.ipAddress;
   hostname = staItem.property.hostName;
   apPath = staItem.property.apPath;

   (void) dbus_message_iter_open_container(iterSta, DBUS_TYPE_DICT_ENTRY, NULL, &iterStaDet);
   const char* strPath = staItem.objPath.c_str();
   (void) dbus_message_iter_append_basic(&iterStaDet, DBUS_TYPE_STRING, &strPath);

   (void) dbus_message_iter_open_container(&iterStaDet, DBUS_TYPE_VARIANT, "a{sv}", &iterEntry);

   (void) dbus_message_iter_open_container(&iterEntry, DBUS_TYPE_ARRAY, "{sv}", &iterEntryElement);

   dict_append_entry(&iterEntryElement, "HostName", DBUS_TYPE_STRING, &hostname);

   dict_append_entry(&iterEntryElement, "MacAddress", DBUS_TYPE_STRING, &macAddress);

   dict_append_entry(&iterEntryElement, "Interface", DBUS_TYPE_STRING, &interface);

   dict_append_entry(&iterEntryElement, "IPAddress", DBUS_TYPE_STRING, &ipAddress);

   dict_append_entry(&iterEntryElement, "AccessPoint", DBUS_TYPE_STRING, &apPath);

   (void) dbus_message_iter_close_container(&iterEntry, &iterEntryElement);
   (void) dbus_message_iter_close_container(&iterStaDet, &iterEntry);
   (void) dbus_message_iter_close_container(iterSta, &iterStaDet);

}//WifiSetupService::insertAssociatedStation

void WifiSetupService::insertSSID(const APSetup &apSetup,::std::map< ::std::string, ::asf::dbus::DBusVariant> &setupsMap)
{
   LOG_INFO("WifiSetupService::insertSSID");
   uint8 index = 0;
   ::asf::dbus::DBusVariant varSSID;
   DBusMessageIter *iterSSID,array;
   int ssidlength =  static_cast<int>(apSetup.SSID.size());
   uint8* ssid = (uint8*) getMemAlloc(ssidlength);

   iterSSID = varSSID.getWriteIterator();
   for (auto it = apSetup.SSID.begin(); it != apSetup.SSID.end(); ++it, index++)
      ssid[index] = *it;

   if ((!dbus_message_iter_open_container(iterSSID, DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE_AS_STRING, &array))
         || (!dbus_message_iter_append_fixed_array(&array, DBUS_TYPE_BYTE, &ssid, ssidlength))
         || (!dbus_message_iter_close_container(iterSSID, &array)))
   {
      LOG_INFO("WifiSetupService::insertSSID SSID Insertion Fails");
   }
   else
   {
      setupsMap.insert(std::pair<::std::string, ::asf::dbus::DBusVariant>("SSID", varSSID));
   }
   free(ssid);
}//WifiSetupService::insertSSID

void WifiSetupService::insertAPConfigElements(const APSetup &apSetup,
      ::std::map< ::std::string, ::asf::dbus::DBusVariant> &setupsMap)
{
   LOG_INFO("WifiSetupService::insertAPConfigElements");
   DBusMessageIter *iterStaElement, dict, iterSta;
   ::asf::dbus::DBusVariant varAssociatedSta;
   dbus_bool_t visibility;
   std::string security, passphrase, type;
   security = wblSecurity.getSecurity2String(apSetup.security);
   passphrase = apSetup.passphrase;
   visibility = apSetup.isVisible;
   type = wblAPType.getAPType2String(apSetup.type);
   if(!apSetup.SSID.empty())
      insertSSID(apSetup,setupsMap);
   insert_map_element(setupsMap, "Security", DBUS_TYPE_STRING, &security);
   insert_map_element(setupsMap, "Passphrase", DBUS_TYPE_STRING, &passphrase);
   insert_map_element(setupsMap, "Visible", DBUS_TYPE_BOOLEAN, &visibility);
   LOG_DEBUG("WifiSetupService::insertAPConfigElements No. Associated Stations %d ",apSetup.stationList.size());
   iterStaElement = varAssociatedSta.getWriteIterator();
   if (apSetup.stationList.size())
   {
      (void) dbus_message_iter_open_container(iterStaElement, DBUS_TYPE_VARIANT, "a{sv}", &dict);
      (void) dbus_message_iter_open_container(&dict, DBUS_TYPE_ARRAY, "{sv}", &iterSta);
      for (auto itr = apSetup.stationList.begin(); itr != apSetup.stationList.end(); itr++)
      {
         AssociatedStationItem staItem = *itr;
         insertAssociatedStation(staItem, &iterSta);
      }
      (void) dbus_message_iter_close_container(&dict, &iterSta);
      (void)  dbus_message_iter_close_container(iterStaElement, &dict);
      setupsMap.insert(std::pair<::std::string, ::asf::dbus::DBusVariant>("AssociatedStations", varAssociatedSta));
   }
   insert_map_element(setupsMap, "Type", DBUS_TYPE_STRING, &type);
}//WifiSetupService::insertAPConfigElements

void WifiSetupService::processApSetupUpdate(APSetup &apSetup)
{
   LOG_INFO("WifiSetupService::processApSetupUpdate");
   ::std::map< ::std::string, ::asf::dbus::DBusVariant> setupsMap;
   insertAPConfigElements(apSetup,setupsMap);
   setAPConfig(setupsMap);
}//WifiSetupService::processApSetupUpdate

void WifiSetupService::updateIPv4Configuration(APSetup &apSetup)
{
   LOG_INFO("WifiSetupService::updateIPv4Configuration");
   ::std::string strAddress,strNetMask;
   svMapVector ipv4ConfigList;
   for (auto it = apSetup.ipv4ConfigurationList.begin();
         it != apSetup.ipv4ConfigurationList.end(); it++)
   {
      svMap mapIPConfig;
      strAddress =  it->network;
      strNetMask =  it->netMask;
      insert_map_element(mapIPConfig,"Address",DBUS_TYPE_STRING, &strAddress);
      insert_map_element(mapIPConfig,"NetMask",DBUS_TYPE_STRING, &strNetMask);
      ipv4ConfigList.push_back(mapIPConfig);
   }

   if(!ipv4ConfigList.empty())
   setIPv4Configuration(ipv4ConfigList);
 /** void setIPv4Configuration(const ::std::vector< ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& iPv4Configuration);*/
}//WifiSetupService::updatePowerFailureReason

void WifiSetupService::addRestrictionsListItem(svMapVector &vectorRestrictItems,::std::string tech,::std::string state)
{
   LOG_INFO("WifiSetupService::addRestrictionsListItem");
   ::asf::dbus::DBusVariant varTech;
   DBusMessageIter *iterTech,iterTechElement;
   svMap mapRestrictions;
   iterTech = varTech.getWriteIterator();
   (void) dbus_message_iter_open_container(iterTech, DBUS_TYPE_ARRAY, "{sv}", &iterTechElement);
   dict_append_entry(&iterTechElement,"Technology",DBUS_TYPE_STRING, &tech);
   (void) dbus_message_iter_close_container(iterTech, &iterTechElement);
   mapRestrictions.insert(std::pair<::std::string, ::asf::dbus::DBusVariant>("SPITechnology", varTech));
   insert_map_element(mapRestrictions, "State", DBUS_TYPE_STRING, &state);
   vectorRestrictItems.push_back(mapRestrictions);
}//WifiSetupService::addRestrictionsListItem

void WifiSetupService::updateRestrictions(WBLTechnology tech, WBLRestrictState state)
{
   LOG_INFO("WifiSetupService::updateRestrictions");
   svMapVector restrictions;
   ::std::string strTech,strState;
   _techRestrictions[tech]=state;
   for(auto it = _techRestrictions.begin(); it != _techRestrictions.end(); it++ )
   {
      strTech = wblTechRestriction.getWBLTechnology2String(it->first);
      strState = wblRestrictionState.getWBLRestriction2String(it->second);
      addRestrictionsListItem(restrictions,strTech,strState);
   }
   if(!restrictions.empty())
   setRestrictions(restrictions);
}//WifiSetupService::updateRestrictions

TechnologyRestrictions WifiSetupService::getTechnologyRestrictions()
{
   return _techRestrictions;
}
void WifiSetupService::addSupportedFrequenciesListItem(const APSetup &apSetup,
        const SPITechnologies &technologies,const WBLFrequencyBand &frequencyBand,
        svMapVector &supportedFrequencies)
{
    LOG_INFO("WifiSetupService::%s",__func__);
    ::asf::dbus::DBusVariant varSPI,varChannels;
    DBusMessageIter *iterSPI, iterSPIElement,*iterChannel,iterChannelElement;

    ::std::map<::std::string, ::asf::dbus::DBusVariant> mapFrequency;
    ::std::string technology, techSupport,strfrequency,channel;

    strfrequency = wblFrequency.getFrequency2String(frequencyBand);
    //Insert Frequency
    insert_map_element(mapFrequency, "Frequency", DBUS_TYPE_STRING, &strfrequency);
    //Insert Technologies
    iterSPI = varSPI.getWriteIterator();
    (void) dbus_message_iter_open_container(iterSPI, DBUS_TYPE_ARRAY, "{sv}",&iterSPIElement);
    technology = "CarPlayWireless";
    //Update CPW technology
    techSupport = (technologies.isCPWSupport) ? "ALLOWED" : "NOT_ALLOWED";
    LOG_INFO("WifiSetupService::%s CPW technology:%s", __func__,techSupport.c_str());
    dict_append_entry(&iterSPIElement, technology.c_str(), DBUS_TYPE_STRING,&techSupport);
    //Update AAW technology
    technology = "AndroidAutoWireless";
    techSupport = (technologies.isAAWSupport) ? "ALLOWED" : "NOT_ALLOWED";
    LOG_INFO("WifiSetupService::%s AAW technology:%s", __func__,techSupport.c_str());
    dict_append_entry(&iterSPIElement, technology.c_str(), DBUS_TYPE_STRING,&techSupport);
    (void) dbus_message_iter_close_container(iterSPI, &iterSPIElement);
    mapFrequency.insert(std::pair<::std::string, ::asf::dbus::DBusVariant>("SPITechnologies", varSPI));
    LOG_INFO("WifiSetupService::%s insertion of tech succcessful",__func__);
    //Insert Channels
    iterChannel = varChannels.getWriteIterator();
    (void) dbus_message_iter_open_container(iterChannel, DBUS_TYPE_ARRAY, "{sv}", &iterChannelElement);
    for (auto it=apSetup.chanList.begin(); it != apSetup.chanList.end(); it++ )
    {
        if (it->second >= frequencyBand)
        {
            if((frequencyBand == WBL_FREQ_BAND_2_4_GHZ && it->second <= WBL_FREQ_BAND_5_GHZ) ||
                   (frequencyBand == WBL_FREQ_BAND_5_GHZ))
            {
                channel = std::to_string(it->first);
                uint16 freq = it->second;
                dict_append_entry(&iterChannelElement, channel.c_str(),DBUS_TYPE_UINT16, &freq);
            }
        }
    }
    (void) dbus_message_iter_close_container(iterChannel, &iterChannelElement);
    LOG_INFO("WifiSetupService::%s insertion of channels succcessful",__func__);
    mapFrequency.insert(std::pair<::std::string, ::asf::dbus::DBusVariant>("Channels", varChannels));
    supportedFrequencies.push_back(mapFrequency);
}//WifiSetupService::addSupportedFrequenciesListItem

void WifiSetupService::updateSupportedFrequencies(APSetup &apSetup)
{
   LOG_INFO("WifiSetupService::updateSupportedFrequencies");
   svMapVector supportedFrequencies;
   svMap mapFrequency;
   SPITechnologies tech;
   ::std::string technology,techSupport;
   ::asf::dbus::DBusVariant varSPI;
   WBLFrequencyBand frequency;

   //Pack the SupportedFrequencies
   for(auto it = apSetup.supportedFrequencies.begin();it != apSetup.supportedFrequencies.end(); ++it )
   {
      frequency = it->first;
      tech = (SPITechnologies) it->second;
      addSupportedFrequenciesListItem(apSetup,tech,frequency,supportedFrequencies);
   }
   setSupportedFrequencies(supportedFrequencies);
}//WifiSetupService::updateSupportedFrequencies

bool WifiSetupService::isAssociatedStationExist(const ::std::string macAddress)
{
   LOG_INFO("WifiSetupService::isAssociatedStationExist");
   AssociatedStationList list = _currentItem.property.apSetup.stationList;
   auto iter = list.begin();
   for( ;iter != list.end(); ++iter)
   {
      LOG_DEBUG("WifiSetupService::isAssociatedStationExist current Address %s",iter->property.macAddress.c_str());
      if(iter->property.macAddress.compare(macAddress)==0)
      {
         LOG_DEBUG("WifiSetupService::isAssociatedStationExist Address Exist");
         return true;
      }
   }
   LOG_DEBUG("WifiSetupService::isAssociatedStationExist Address Not Exist");
   return false;
}//WifiSetupService::isAssociatedStationExist

void WifiSetupService::onWBLServiceResponse(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("WifiSetupService::onWBLServiceResponse");

   WBLBaseMsgID msgID = msg->getMsgID();
   WBLErrorCode errCode = msg->getErrorCode();
   uint32 actToken = msg->getMsgToken();

   switch (msgID)
   {
      case WBL_MSG_DEAUTHENTICATE_STATION_REQ:
         onDeAuthenticateStationResponse(errCode, actToken);
      break;
      default:
         LOG_ERROR("Unhandled WBL Response");
      break;
   }
}//WifiSetupService::onWBLServiceResponse

void WifiSetupService::onActiveWifiSetupsChanged(const ActiveWifiSetupsMsg& msg)
{
   LOG_INFO("WifiSetupService::onActiveWifiSetupsChanged");
   WBL_INTENTIONALLY_UNUSED(msg);
}//WifiSetupService::onActiveWifiSetupsChanged

void WifiSetupService::clearEmptyProperties(::std::map< ::std::string, ::asf::dbus::DBusVariant >& propertiesList)
{
   LOG_INFO("WifiSetupService::clearEmptyProperties Entered");

   if (_currentItem.property.mode == WIFI_MODE_UNKNOWN)
   {
      if (propertiesList.find("MODE") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("MODE"));
      }
   }
   if (_currentItem.property.interface.empty())
   {
      if (propertiesList.find("Interface") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("Interface"));
      }
   }
   if (_currentItem.property.apSetup.macAddress.empty())
   {
      if (propertiesList.find("MacAddress") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("MacAddress"));
      }
   }
   if (_currentItem.property.powerState == WIFI_STATE_UNKNOWN)
   {
      if (propertiesList.find("PowerState") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("PowerState"));
      }
   }
   if (_currentItem.property.powerFailure == WIFI_POWER_FAILURE_REASON_UNKNOWN)
   {
      if (propertiesList.find("PowerFailureReason") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("PowerFailureReason"));
      }
   }
   if (_currentItem.property.frequency == WBL_FREQ_BAND_UNKNOWN)
   {
      if (propertiesList.find("Frequency") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("Frequency"));
      }
   }
   if (_currentItem.property.currChannel == 0)
   {
      if (propertiesList.find("CurrentOperatingChannel") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("CurrentOperatingChannel"));
      }
   }
   if (_currentItem.property.apSetup.type == WBL_AP_TYPE_UNKNOWN)
   {
      if (propertiesList.find("APConfig") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("APConfig"));
      }
   }
   if (_currentItem.property.apSetup.supportedFrequencies.empty())
   {
      if (propertiesList.find("SupportedFrequencies") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("SupportedFrequencies"));
      }
   }
   if (_techRestrictions.empty())
   {
      if (propertiesList.find("Restrictions") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("Restrictions"));
      }
   }
   if (_currentItem.property.apSetup.ipv4ConfigurationList.empty())
   {
      if (propertiesList.find("IPv4Configuration") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("IPv4Configuration"));
      }
   }
   if (_currentItem.property.mode != WIFI_MODE_STA)
   {
      if (propertiesList.find("ConnectionStatus") != propertiesList.end())
      {
         propertiesList.erase(propertiesList.find("ConnectionStatus"));
      }
   }

}//WifiSetupService::clearEmptyProperties

void WifiSetupService::getAllProperties(::std::map< ::std::string, ::asf::dbus::DBusVariant >& propertiesMap)
{
   LOG_INFO("WifiSetupService::getAllProperties");
   ::std::map< ::std::string, ::asf::dbus::DBusVariant > propList;
   _stubDelegate->getAllProperties(propList);
   propertiesMap.insert(propList.begin(),propList.end());
   clearEmptyProperties(propertiesMap);
}//WifiSetupService::getAllProperties

void WifiSetupService::getDBusObjectPath(::std::string& objPath)
{
   LOG_INFO("WifiSetupService::getDBusObjectPath");
   objPath = _stubDelegate->getDBusObjectPath();
}//WifiSetupService::getDBusObjectPath

void WifiSetupService::getInterfaceName(::std::string& interfaceName)
{
   LOG_INFO("WifiSetupService::getInterfaceName");
   interfaceName = _stubDelegate->getInterfaceName();
}//WifiSetupService::getInterfaceName


} //namespace bosch
} //namespace org

/** @} */
