/**
 * @file RegCountryCodeService.cpp
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2018 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_business_logic
 *
 * @brief
 *
 * @{
 */

#include "RegCountryCodeService.h"
#include "WBLPortsDefines.h"
#include "CountryCodeClientInfo.h"
#include "RegulatoryMethodFactory.h"
#include "DbusHelper.h"
#include "WBLTypeProperties.h"

#define SERAVAIL_INVAL_CHECK(x)		(x > 1)
#define CC_TYPE_INVAL_CHECK(x)		(x > 4 || x == 0)
#define CC_SOURCE_INVAL_CHECK(x)	(x > 5 || x == 0)

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/WBLServices", cRegCountryCodeInfoService, Debug);

cRegCountryCodeInfoService::cRegCountryCodeInfoService() : cRegulationsStub (sWblRegCCInfoPortName)

{
	LOG_INFO("Creating the Regulatory Country Code Information stub");
}

cRegCountryCodeInfoService::~cRegCountryCodeInfoService()
{
}

void cRegCountryCodeInfoService::extractSetCCInfoInternals(::std::map< ::std::string, ::asf::dbus::DBusVariant > &ccinfo,
			uint16_t &sa, uint16_t &cc_type, uint16_t &cc_src)
{
	::DBusMessageIter* iter;
	::std::string key;
	::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it;

	for (it = ccinfo.begin(); it != ccinfo.end(); ++it) {

		key = it->first;
		iter = it->second.getReadIterator();

		if (!iter)
			continue;

		if (!key.compare("ServiceAvailability") &&
				(dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_UINT16))
			dbus_message_iter_get_basic(iter, &sa);
		else if (!key.compare("CountryCodeType") &&
				(dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_UINT16))
			dbus_message_iter_get_basic(iter, &cc_type);
		else if (!key.compare("CountryCodeSource") &&
				(dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_UINT16))
			dbus_message_iter_get_basic(iter, &cc_src);
	}
}

int cRegCountryCodeInfoService::isUpperAlpha (const char *alpha)
{
	bool invalid = false;
	size_t length = 0, index;

	if (!alpha)
		return -EINVAL;

	length = strlen (alpha);
	for (index = 0; index < length; ++index) {
		if (alpha[index] >= 65 && alpha[index] <= 90) { }
		else {
			invalid = true;
			break;
		}
	}

	return invalid == false ? 0 : -EINVAL;
}

int cRegCountryCodeInfoService::isWorldRegDom(const char *alpha)
{
	if (alpha && strlen(alpha) == 2)
		if (alpha[0] == 48 && alpha[1] == 48)
			return 0;
	return -EINVAL;
}

int cRegCountryCodeInfoService::extractCountryCode(::DBusMessageIter *iter, uint16_t &cc_type,
		char **data, uint16_t &udata)
{
	int iRet = -EINVAL;
	cc_type_t type = eCC_TYPE_INVAL;

	type = static_cast<cc_type_t>(cc_type);
	if (iter) {
		if ((eCC_TYPE_ALPHA2 == type || eCC_TYPE_ALPHA3 == type) &&
				(dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_STRING)) {
			dbus_message_iter_get_basic(iter, &(*data));
			if (*data) {
				if (eCC_TYPE_ALPHA2 == type) {
					if (strlen(*data) == 2 && (isUpperAlpha(*data) == 0
							|| isWorldRegDom(*data) == 0))
						iRet = 0;
				}
				else if (eCC_TYPE_ALPHA3 == type && (strlen(*data) == 3 &&
						(isUpperAlpha(*data) == 0)))
					iRet = 0;
			}
		}
		else if ((eCC_TYPE_MCC == type || eCC_TYPE_NUMERIC == type) &&
				(dbus_message_iter_get_arg_type(iter) == DBUS_TYPE_UINT16)) {
			iRet = 0;
			dbus_message_iter_get_basic(iter, &udata);
		}
	}

	return iRet;
}

int cRegCountryCodeInfoService::performSanityCheck(uint16_t &sa, uint16_t &type, uint16_t &src)
{
	bool invalid = false;
	cc_type_t cc_type = static_cast <cc_type_t> (type);
	cc_source_t cc_src = static_cast <cc_source_t> (src);

	if (SERAVAIL_INVAL_CHECK(sa) || CC_TYPE_INVAL_CHECK(type) ||
			CC_SOURCE_INVAL_CHECK(src))
		invalid = true;

	if (!invalid) {
		if ((eCC_SOURCE_GNSS == cc_src && (eCC_TYPE_ALPHA2 != cc_type &&
				eCC_TYPE_ALPHA3 != cc_type)) ||
				(eCC_SOURCE_MCC == cc_src && eCC_TYPE_MCC != cc_type) ||
				(eCC_SOURCE_IEEE80211D == cc_src && eCC_TYPE_ALPHA2 != cc_type) ||
				(eCC_SOURCE_LASTMODE == cc_src) || (eCC_SOURCE_IPADDR == cc_src))
			invalid = true;
	}

	return invalid == false ? 0 : -EINVAL;
}

void cRegCountryCodeInfoService::onSetCountryCodeRequest (const ::boost::shared_ptr< cSetCountryCodeRequest >& request)
{
	cRegulatoryMethod *method;
	::std::string country_code, key;
	uint16_t serviceAvailability, numeric_cc, utype, usrc;
	::std::string invalid_args("InvalidArguments");
	::DBusMessageIter* iter;
	int iRet = 0;
	char *alpha = NULL;
	::std::map< ::std::string, ::asf::dbus::DBusVariant > ccinfo;
	::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it;

	ccinfo = request->getCountrycode();
	serviceAvailability = numeric_cc = utype = usrc = 0;
	extractSetCCInfoInternals(ccinfo, serviceAvailability, utype, usrc);

	LOG_INFO("ServiceAvailability: \"%u\", CC Type: \"%u\", CC src: \"%u\"",
			serviceAvailability, utype, usrc);

	iRet = performSanityCheck(serviceAvailability, utype, usrc);
	if (iRet < 0) {
		sendSetCountryCodeError(wblErrorCode.convertEnum2String(WBL_ERR_INVALID_ARGS),
				invalid_args, request->getAct());
		return;
	}

	iRet = -EINVAL;
	for (it = ccinfo.begin(); it != ccinfo.end(); ++it) {
		key = it->first;
		iter = it->second.getReadIterator();
		if (iter && !key.compare("CountryCode")) {
			iRet = extractCountryCode(iter, utype, &alpha, numeric_cc);
			break;
		}
	}

	if (iRet == 0) {
		LOG_INFO("Set requested for Country Code: \"%s\" \"%u\"",
				alpha ? alpha : "numeric", numeric_cc);
		if (alpha)
			country_code = alpha;
		method = cRegulatoryMethodFactory::getInstance()->getRegulatoryMethodfromSrc(static_cast<cc_source_t>(usrc));
		if (method) {
			method->setCountryCodeInfo(static_cast<cc_availability_t>(serviceAvailability),
					country_code, numeric_cc);
			sendSetCountryCodeResponse(request->getAct());
		} else {
			invalid_args = "NotSupported";
			sendSetCountryCodeError(wblErrorCode.convertEnum2String(WBL_ERR_NOTSUPPORTED),
					invalid_args, request->getAct());
		}
	} else {
		sendSetCountryCodeError(wblErrorCode.convertEnum2String(WBL_ERR_INVALID_ARGS),
				invalid_args, request->getAct());
	}
}

void cRegCountryCodeInfoService::updateCountryCodeInfo(const ::std::string &alpha2, const uint16_t source)
{
	::std::map< ::std::string, ::asf::dbus::DBusVariant > ccinfo;

	insert_map_element(ccinfo,
			"CountryCode",
			DBUS_TYPE_STRING,
			static_cast<void *>(const_cast<::std::string *>(&alpha2)));

	insert_map_element(ccinfo,
			"CountryCodeSource",
			DBUS_TYPE_UINT16,
			static_cast<void *>(const_cast<uint16_t *>(&source)));

	setCountryCode (ccinfo);
}

	}
}

/** @} */
