/**
 * @file OrgFreedesktopDBusPropertiesProxyManager.cpp.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <string.h>
#include "asf/core/Proxy.h"
#include <dbus/dbus.h>
#include "WBLPortsDefines.h"
#include "OrgFreedesktopDBusPropertiesProxyManager.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/WBLClients", cOrgFreedesktopDBusPropertiesProxyManager, Info);

cOrgFreedesktopDBusPropertiesProxyManager::cOrgFreedesktopDBusPropertiesProxyManager()
{
}

cOrgFreedesktopDBusPropertiesProxyManager::~cOrgFreedesktopDBusPropertiesProxyManager()
{
}

int cOrgFreedesktopDBusPropertiesProxyManager::createProxy(const ::std::string &portName, const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const eProxyType eType,
		cIOrgFreedesktopDBusPropertiesNotifCallbackIf *cb)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cDBusPropertiesProxy >  oProxy;

	if (busName.empty() || objPath.empty()
			|| portName.empty() || PROXY_TYPE_INVAL == eType || !cb)
		return -EINVAL;

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (m_oProxyManager.isProxyAvailable (oProxy, oProxyData))
		return -EEXIST;

	LOG_INFO ("Creating Proxy to the dbus remote object [PortName: %s]"
			"[busName: %s] [objPath: %s]", portName.c_str(), busName.c_str(),
			objPath.c_str());

	if (PROXY_TYPE_STATIC == eType)
		oProxy = cDBusPropertiesProxy::createProxy(portName, *this);
	else if (PROXY_TYPE_WIRED_AT_RUNTIME == eType)
		oProxy = cDBusPropertiesProxy::createProxy(portName, busName, objPath, busType, *this);

	oProxy->sendPropertiesChangedRegister(*cb);
	m_oProxyManager.addProxyInstance(oProxyData, oProxy);
	return 0;
}

int cOrgFreedesktopDBusPropertiesProxyManager::destroyProxy(const eProxyDestroy eType, const ::std::string &busName,
		const ::DBusBusType &busType, const ::std::string objPath)
{
	unsigned int iIndex = 0;
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cDBusPropertiesProxy > oProxy;

	if (PROXY_DESTROY_INVAL == eType || (PROXY_DESTROY_ONE == eType &&
			objPath.empty()) || (PROXY_DESTROY_ALL == eType && !objPath.empty()))
		return -EINVAL;

	LOG_INFO ("Destroying the proxy to : %s", (eType == PROXY_DESTROY_ONE) ?
			objPath.c_str() : "All Objects");

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (PROXY_DESTROY_ONE == eType) {
		if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
			return -ENOENT;
		oProxy->sendDeregisterAll();
		m_oProxyManager.removeProxyInstance(oProxyData);
	}
	else if (PROXY_DESTROY_ALL == eType) {
		for( ; iIndex < m_oProxyManager.getNumberOfProxyInstances(); ++iIndex) {
			oProxy = m_oProxyManager[iIndex].proxy;
			oProxy->sendDeregisterAll();
		}
		m_oProxyManager.resetAllProxiesAndClear();
	}

	return 0;
}

act_t cOrgFreedesktopDBusPropertiesProxyManager::sendGetRequest(const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const ::std::string &interface_name,
		const ::std::string &property_name, cDBusPropGetCbIf &cb)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cDBusPropertiesProxy > oProxy;

	if (busName.empty() || objPath.empty()
			|| interface_name.empty() || property_name.empty())
		return DEFAULT_ACT;

	LOG_INFO ("[busName: %s] [objPath: %s] Get Property : %s [interface : %s]", busName.c_str(),
			objPath.c_str(), property_name.c_str(), interface_name.c_str());

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
		return DEFAULT_ACT;

	return oProxy->sendGetRequest(cb, interface_name, property_name);
}

act_t cOrgFreedesktopDBusPropertiesProxyManager::sendGetAllRequest(const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const ::std::string &interface_name,
		cDBusPropGetAllCbIf &cb)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cDBusPropertiesProxy > oProxy;

	if (busName.empty() || objPath.empty()
			|| interface_name.empty())
		return DEFAULT_ACT;

	LOG_INFO ("[busName: %s] [objPath: %s] Get All: %s", busName.c_str(),
			objPath.c_str(), interface_name.c_str());

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
		return DEFAULT_ACT;

	return oProxy->sendGetAllRequest(cb, interface_name);
}

act_t cOrgFreedesktopDBusPropertiesProxyManager::sendSetRequest(const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const ::std::string &interface_name,
		const ::std::string &property_name, const ::asf::dbus::DBusVariant &value,
		cDBusPropSetCbIf &cb)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cDBusPropertiesProxy > oProxy;

	if (busName.empty() || objPath.empty()
			|| interface_name.empty() || property_name.empty())
		return DEFAULT_ACT;

	LOG_INFO ("[busName: %s] [objPath: %s] [interface: %s] property: %s", busName.c_str(),
			objPath.c_str(), interface_name.c_str(), property_name.c_str());

	oProxyData.busName = busName;
	oProxyData.objPath = objPath;
	oProxyData.busType = busType;

	if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
		return DEFAULT_ACT;

	return oProxy->sendSetRequest(cb, interface_name, property_name, value);
}

void cOrgFreedesktopDBusPropertiesProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy > &proxy,
      const ::asf::core::ServiceStateChange &stateChange)
{
   	(void) stateChange;
	::boost::shared_ptr< cDBusPropertiesProxy > oProxy;
	oProxy = ::boost::static_pointer_cast< cDBusPropertiesProxy > (proxy);

	LOG_INFO ("Service Available for the Proxy: [PortName: %s], "
			"[busName: %s], [ObjPath: %s], [IfName: %s]", oProxy->getPortName().c_str(),
			oProxy->getDBusBusName().c_str(), oProxy->getDBusObjectPath().c_str(),
			oProxy->getInterfaceName().c_str());
}

void cOrgFreedesktopDBusPropertiesProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy > &proxy,
      const ::asf::core::ServiceStateChange &stateChange)
{
   	(void) stateChange;
	::boost::shared_ptr< cDBusPropertiesProxy > oProxy;
	oProxy = ::boost::static_pointer_cast< cDBusPropertiesProxy > (proxy);

	LOG_INFO ("Service Unavailable for the Proxy: [PortName: %s], "
			"[busName: %s], [ObjPath: %s], [IfName: %s]", oProxy->getPortName().c_str(),
			oProxy->getDBusBusName().c_str(), oProxy->getDBusObjectPath().c_str(),
			oProxy->getInterfaceName().c_str());
}

}
}

/** @} */
