/**
 * @file WapdmgrManagerProxyManager.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */


#include "ProxyTypes.h"
#include "WapdmgrDefines.h"
#include "WBLPortsDefines.h"
#include "WapdmgrManagerProxyManager.h"

namespace org 
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", WapdmgrManagerProxyManager, Info);

static ProxyMetadata sWapdmgrManagerProxyMetadata;

WapdmgrManagerProxyManager::WapdmgrManagerProxyManager()
{
   LOG_INFO(" WapdmgrManagerProxyManager() entered ");

   sWapdmgrManagerProxyMetadata.busName = sWapdmgrBusName;
   sWapdmgrManagerProxyMetadata.objPath = sWapdmgrManagerObjectPath;
   sWapdmgrManagerProxyMetadata.busType = DBUS_BUS_SYSTEM;
}

WapdmgrManagerProxyManager::~WapdmgrManagerProxyManager()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   //LOG_INFO(" ~WapdmgrManagerProxyManager() entered ");
}

void WapdmgrManagerProxyManager::createProxy()
{
   LOG_INFO(" WapdmgrManagerProxyManager::createProxy() entered ");

   ::boost::shared_ptr< WapdmgrManagerProxy > managerProxy;
   if(false == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata))
   {
      managerProxy = WapdmgrManagerProxy::createProxy(sWapdmgrManagerPortName, *this);
      _proxyManager.addProxyInstance(sWapdmgrManagerProxyMetadata, managerProxy);
   }
}

void WapdmgrManagerProxyManager::destroyProxy()
{
   LOG_INFO(" WapdmgrManagerProxyManager::destroyProxy() entered ");

   ::boost::shared_ptr< WapdmgrManagerProxy > managerProxy;
   if ((true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata)) && (managerProxy))
   {
      managerProxy->sendDeregisterAll();
   }

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _proxyManager.resetAllProxiesAndClear();
   _proxyCbManager.removeAllCallbackIfs();
}

bool WapdmgrManagerProxyManager::isProxyServiceAvailable()
{
   return (_proxyManager.isProxyServiceAvailable(
         sWapdmgrManagerProxyMetadata.busName,
         sWapdmgrManagerProxyMetadata.objPath,
         sWapdmgrManagerProxyMetadata.busType));
}

void WapdmgrManagerProxyManager::setCallbackIf(const ProxyUser user,
      IWapdmgrManagerNotifCallbackIf* callbackIf)
{
   LOG_INFO(" WapdmgrManagerProxyManager::setCallbackIf(): user=%d ", user);

   ProxyMetadata metadata(sWapdmgrManagerProxyMetadata);
   metadata.user = user;
   
   if (callbackIf)
   {
      _proxyCbManager.addCallbackIf(metadata, callbackIf);
   }
   else
   {
      _proxyCbManager.removeCallbackIf(metadata);
   }
}

act_t WapdmgrManagerProxyManager::sendFindAccessPointRequest(WapdmgrFindAccessPointCbIf& callbackIf,
      const ::std::string& pattern)
{
   LOG_INFO(" WapdmgrManagerProxyManager::sendFindAccessPointRequest(): pattern=%s ", pattern.c_str());

   ::boost::shared_ptr< WapdmgrManagerProxy > managerProxy;
   if (true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata))
   {
      return managerProxy->sendFindAccessPointRequest(callbackIf, pattern);
   }
   //Lint Prio1 Warning 533: function should return a value
   return DEFAULT_ACT;
}

act_t WapdmgrManagerProxyManager::sendGetAccessPointsRequest(WapdmgrGetAccessPointsCbIf& callbackIf)
{
   LOG_INFO(" WapdmgrManagerProxyManager::sendGetAccessPointsRequest() entered ");

   ::boost::shared_ptr< WapdmgrManagerProxy > managerProxy;
   if (true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata))
   {
      return managerProxy->sendGetAccessPointsRequest(callbackIf);
   }
   return DEFAULT_ACT;
}

act_t WapdmgrManagerProxyManager::sendRegisterAgentRequest(WapdmgrRegisterAgentCbIf& callbackIf,
      const ::std::string& path)
{
   LOG_INFO(" WapdmgrManagerProxyManager::sendRegisterAgentRequest(): path=%s ", path.c_str());

   ::boost::shared_ptr< WapdmgrManagerProxy > managerProxy;
   if (true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata))
   {
      return managerProxy->sendRegisterAgentRequest(callbackIf, path);
   }
   return DEFAULT_ACT;
}

act_t WapdmgrManagerProxyManager::sendUnregisterAgentRequest(WapdmgrUnregisterAgentCbIf& callbackIf,
      const ::std::string& path)
{
   LOG_INFO(" WapdmgrManagerProxyManager::sendUnregisterAgentRequest(): path=%s ", path.c_str());

   ::boost::shared_ptr< WapdmgrManagerProxy > managerProxy;
   if (true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata))
   {
      return managerProxy->sendUnregisterAgentRequest(callbackIf, path);
   }
   return DEFAULT_ACT;
}

// ServiceAvailableIF implementation
void WapdmgrManagerProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" WapdmgrManagerProxyManager::onAvailable() entered ");

   ::boost::shared_ptr< WapdmgrManagerProxy > managerProxy;
   if ((true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata)) &&
         (managerProxy) && (proxy == managerProxy))
   {
      const ::std::string& busName = managerProxy->getDBusBusName();
      const ::std::string& objectPath = managerProxy->getDBusObjectPath();
      ::DBusBusType busType = managerProxy->getBusType();

      _proxyManager.setProxyServiceAvailability(busName, objectPath, busType, true);

      std::vector<IWapdmgrManagerNotifCallbackIf*> callbacksList =
            _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
      for (std::vector<IWapdmgrManagerNotifCallbackIf*>::iterator it = callbacksList.begin();
            it != callbacksList.end(); ++it)
      {
         LOG_DEBUG(" onAvailable: Registering for AP signals ");
         managerProxy->sendAccessPointAddedRegister(**it);
         managerProxy->sendAccessPointRemovedRegister(**it);
         (*it)->onWapdmgrManagerServiceAvailable(busName, objectPath, busType,
               stateChange.getPreviousState(), stateChange.getCurrentState());
      } //for (std::vector<IWapdmgrManagerNotifCallbackIf*>::iterator it = callbacksList.begin();...)
   } //if ((true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata)) &&...)
}

void WapdmgrManagerProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
      const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" WapdmgrManagerProxyManager::onUnavailable() entered ");

   ::boost::shared_ptr< WapdmgrManagerProxy > managerProxy;
   if ((true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata)) &&
         (managerProxy) && (proxy == managerProxy))
   {
      LOG_DEBUG(" onUnavailable: Deregistering for AP signals ");
      managerProxy->sendDeregisterAll();

      const ::std::string& busName = managerProxy->getDBusBusName();
      const ::std::string& objectPath = managerProxy->getDBusObjectPath();
      ::DBusBusType busType = managerProxy->getBusType();

      _proxyManager.setProxyServiceAvailability(busName, objectPath, busType, false);

      std::vector<IWapdmgrManagerNotifCallbackIf*> callbacksList =
            _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
      for (std::vector<IWapdmgrManagerNotifCallbackIf*>::iterator it = callbacksList.begin();
            it != callbacksList.end(); ++it)
      {
         (*it)->onWapdmgrManagerServiceUnavailable(busName, objectPath, busType,
               stateChange.getPreviousState(), stateChange.getCurrentState());
      } //for (std::vector<IWapdmgrManagerNotifCallbackIf*>::iterator it = callbacksList.begin();...)
   } //if ((true == _proxyManager.isProxyAvailable(managerProxy, sWapdmgrManagerProxyMetadata)) &&...)
}


} // namespace bosch
} // namespace org

/** @} */
