/**
 * @file LoopbackProxyManager.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <string.h>
#include "asf/core/Proxy.h"
#include "asf/core/Logger.h"
#include <dbus/dbus.h>
#include "WBLPortsDefines.h"
#include "LoopbackProxyManager.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", cLoopbackProxyManager, Info);

cLoopbackProxyManager::cLoopbackProxyManager()
{
}

cLoopbackProxyManager::~cLoopbackProxyManager()
{
}

int cLoopbackProxyManager::createProxy(const ::std::string &portName,
		cILoopbackNotifCallbackIf *cb)
{
	(void) cb;
	ProxyMetadata oProxyData;
	::boost::shared_ptr<cWBLASFServiceProxy> oProxy;

	if (portName.empty())
		return -EINVAL;

	oProxyData.busName = portName;

	if (m_oProxyManager.isProxyAvailable (oProxy, oProxyData))
		return -EEXIST;

	LOG_INFO ("Creating Proxy to the ASF CMS object [PortName: %s]", portName.c_str());

	oProxy = cWBLASFServiceProxy::createProxy(portName, *this);
	LOG_INFO ("Successfully created loopback service Use Count: %d", oProxy.use_count());
	m_oProxyManager.addProxyInstance(oProxyData, oProxy);
	return 0;
}

int cLoopbackProxyManager::destroyProxy(const ::std::string &portName)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr<cWBLASFServiceProxy> oProxy;

	if (portName.empty())
		return -EINVAL;

	LOG_INFO ("Destroying the proxy to : %s", portName.c_str());

	oProxyData.busName = portName;

	if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
		return -ENOENT;

	LOG_INFO ("Use Count: %d", oProxy.use_count());
	oProxy->sendDeregisterAll();
	m_oProxyManager.removeProxyInstance(oProxyData);
	return 0;
}

int cLoopbackProxyManager::setCallbackIf(const ::std::string &portName,
		cILoopbackNotifCallbackIf *cb)
{
	ProxyMetadata oProxyData;

	if (portName.empty())
		return -EINVAL;

	oProxyData.busName = portName;

	LOG_INFO ("%s the callback %p to the dbus remote object [portName: %s]"
			, cb ? "Adding" : "Removing", cb, portName.c_str());

	if (!cb) {
		m_pProxyCbManager.removeCallbackIf(oProxyData);
	} else {
		if (m_pProxyCbManager.bIsCallbackAvailable(oProxyData, cb))
			return -EEXIST;
		m_pProxyCbManager.addCallbackIf(oProxyData, cb);
	}

	return 0;
}

act_t cLoopbackProxyManager::sendStarteventloopRequest(const ::std::string &portname,
		cStarteventloopCallbackIF &cb)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr<cWBLASFServiceProxy> oProxy;

	if (portname.empty())
		return DEFAULT_ACT;

	LOG_INFO ("[portName: %s] SendStarteventLoop", portname.c_str());

	oProxyData.busName = portname;

	if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
		return DEFAULT_ACT;

	return oProxy->sendStarteventloopRequest(cb);
}

act_t cLoopbackProxyManager::sendCurrentRegSettingsGet(const ::std::string &portname,
		cCurrentRegSettingsGetCallbackIF &cb)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr<cWBLASFServiceProxy> oProxy;

	if (portname.empty())
		return DEFAULT_ACT;

	LOG_INFO ("[portName: %s] Get Current Reg Settings", portname.c_str());

	oProxyData.busName = portname;

	if (!m_oProxyManager.isProxyAvailable(oProxy, oProxyData))
		return DEFAULT_ACT;

	return oProxy->sendCurrentRegSettingsGet(cb);
}

void cLoopbackProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy > &proxy,
      const ::asf::core::ServiceStateChange &stateChange)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cWBLASFServiceProxy > oProxy;
	std::vector<cILoopbackNotifCallbackIf*> cblist;
	std::vector<cILoopbackNotifCallbackIf*>::iterator it;

	oProxy = ::boost::static_pointer_cast< cWBLASFServiceProxy > (proxy);
	oProxyData.busName = oProxy->getPortName();

	LOG_INFO ("Service Available for the Loopback Proxy");

	cblist = m_pProxyCbManager.getAllCallbackIf(oProxyData);
	for (it = cblist.begin(); it != cblist.end(); ++it) {
		oProxy->sendCurrentRegSettingsRegister(**it);
		oProxy->sendMediaConnectedRegister(**it);
		oProxy->sendMediaDisconnectedRegister(**it);
		(*it)->onLoopbackServiceAvailable(oProxy, stateChange.getPreviousState(),
				stateChange.getCurrentState());
	}
}

void cLoopbackProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy > &proxy,
      const ::asf::core::ServiceStateChange &stateChange)
{
	ProxyMetadata oProxyData;
	::boost::shared_ptr< cWBLASFServiceProxy > oProxy;
	std::vector<cILoopbackNotifCallbackIf*> cblist;
	std::vector<cILoopbackNotifCallbackIf*>::iterator it;

	oProxy = ::boost::static_pointer_cast< cWBLASFServiceProxy > (proxy);
	oProxyData.busName = oProxy->getPortName();

	LOG_INFO ("Service UnAvailable for the Loopback Proxy");

	oProxy->sendDeregisterAll();

	cblist = m_pProxyCbManager.getAllCallbackIf(oProxyData);
	for (it = cblist.begin(); it != cblist.end(); ++it) {
		(*it)->onLoopbackServiceUnAvailable(oProxy, stateChange.getPreviousState(),
				stateChange.getCurrentState());
	}
}

	}
}

/** @} */
