/**
 * @file ConnmanManagerProxyManager.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

#include "ProxyTypes.h"
#include "ConnmanDefines.h"
#include "WBLPortsDefines.h"
#include "ConnmanManagerProxyManager.h"

namespace org 
{
namespace bosch 
{

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", ConnmanManagerProxyManager, Info);

static ProxyMetadata sConnmanManagerProxyMetadata;

ConnmanManagerProxyManager::ConnmanManagerProxyManager()
{
   LOG_INFO(" ConnmanManagerProxyManager() entered ");

   sConnmanManagerProxyMetadata.busName = sConnmanBusName;
   sConnmanManagerProxyMetadata.objPath = sConnmanManagerObjectPath;
   sConnmanManagerProxyMetadata.busType = DBUS_BUS_SYSTEM;
}

ConnmanManagerProxyManager::~ConnmanManagerProxyManager()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   // LOG_INFO(" ~ConnmanManagerProxyManager() entered ");
}

void ConnmanManagerProxyManager::createProxy()
{
   LOG_INFO(" ConnmanManagerProxyManager::createProxy() ");

   ::boost::shared_ptr< ConnmanManagerProxy > managerProxy;
   if(false == _proxyManager.isProxyAvailable(managerProxy, sConnmanManagerProxyMetadata))
   {
      managerProxy = ConnmanManagerProxy::createProxy(sConnmanManagerPortName, *this);
      _proxyManager.addProxyInstance(sConnmanManagerProxyMetadata, managerProxy);
   }
}

void ConnmanManagerProxyManager::destroyProxy()
{
   LOG_INFO(" ConnmanManagerProxyManager::destroyProxy() entered ");

   ::boost::shared_ptr< ConnmanManagerProxy > managerProxy;
   if ((true == _proxyManager.isProxyAvailable(managerProxy, sConnmanManagerProxyMetadata)) && (managerProxy))
   {
      managerProxy->sendDeregisterAll();
   }

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _proxyManager.resetAllProxiesAndClear();
   _proxyCbManager.removeAllCallbackIfs();
}

bool ConnmanManagerProxyManager::isProxyServiceAvailable()
{
   return (_proxyManager.isProxyServiceAvailable(
         sConnmanManagerProxyMetadata.busName,
         sConnmanManagerProxyMetadata.objPath,
         sConnmanManagerProxyMetadata.busType));
}

void ConnmanManagerProxyManager::setCallbackIf(ProxyUser user, IConnmanManagerNotifCallbackIf* callbackIf)
{
   LOG_INFO(" ConnmanManagerProxyManager::setCallbackIf() entered ");

   ProxyMetadata metadata(sConnmanManagerProxyMetadata);
   metadata.user = user;

   if (callbackIf)
   {
      _proxyCbManager.addCallbackIf(metadata, callbackIf);
   }
   else
   {
      _proxyCbManager.removeCallbackIf(metadata);
   }
}

act_t ConnmanManagerProxyManager::sendGetTechnologiesRequest(ConnmanGetTechnologiesCbIf& callbackIf)
{
   LOG_INFO(" ConnmanManagerProxyManager::sendGetTechnologiesRequest() entered ");

   ::boost::shared_ptr< ConnmanManagerProxy > managerProxy;
   if ((true == _proxyManager.isProxyAvailable(managerProxy, sConnmanManagerProxyMetadata)) && (managerProxy))
   {
      return managerProxy->sendGetTechnologiesRequest(callbackIf);
   }

   return DEFAULT_ACT;
}

// ServiceAvailableIF implementation
void ConnmanManagerProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
		const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" ConnmanManagerProxyManager::onAvailable() entered ");

   ::boost::shared_ptr< ConnmanManagerProxy > managerProxy;
   if ((true == _proxyManager.isProxyAvailable(managerProxy, sConnmanManagerProxyMetadata)) &&
         (managerProxy) && (proxy == managerProxy))
   {
      const ::std::string& busName = managerProxy->getDBusBusName();
      const ::std::string& objectPath = managerProxy->getDBusObjectPath();
      ::DBusBusType busType = managerProxy->getBusType();

      _proxyManager.setProxyServiceAvailability(busName, objectPath, busType, true);

      std::vector<IConnmanManagerNotifCallbackIf*> callbacksList =
            _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
      for (std::vector<IConnmanManagerNotifCallbackIf*>::iterator it = callbacksList.begin();
            it != callbacksList.end(); ++it)
      {
         LOG_DEBUG(" onAvailable: Registering for STA signals ");
         managerProxy->sendTechnologyAddedRegister(**it);
         managerProxy->sendTechnologyRemovedRegister(**it);
         (*it)->onConnmanManagerServiceAvailable(busName, objectPath, busType,
               stateChange.getPreviousState(), stateChange.getCurrentState());
      } //for (std::vector<IConnmanManagerNotifCallbackIf*>::iterator it = callbacksList.begin();...)
   } //if ((true == _proxyManager.isProxyAvailable(managerProxy, sConnmanManagerProxyMetadata)) &&...)
}

void ConnmanManagerProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
		const ::asf::core::ServiceStateChange& stateChange)
{
   LOG_INFO(" ConnmanManagerProxyManager::onUnavailable() entered ");

   ::boost::shared_ptr< ConnmanManagerProxy > managerProxy;
   if ((true == _proxyManager.isProxyAvailable(managerProxy, sConnmanManagerProxyMetadata)) &&
         (managerProxy) && (proxy == managerProxy))
   {
      managerProxy->sendDeregisterAll();

      const ::std::string& busName = managerProxy->getDBusBusName();
      const ::std::string& objectPath = managerProxy->getDBusObjectPath();
      ::DBusBusType busType = managerProxy->getBusType();

      _proxyManager.setProxyServiceAvailability(busName, objectPath, busType, false);

      std::vector<IConnmanManagerNotifCallbackIf*> callbacksList =
            _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
      for (std::vector<IConnmanManagerNotifCallbackIf*>::iterator it = callbacksList.begin();
            it != callbacksList.end(); ++it)
      {
         (*it)->onConnmanManagerServiceUnavailable(busName, objectPath, busType,
               stateChange.getPreviousState(), stateChange.getCurrentState());
      } //for (std::vector<IConnmanManagerNotifCallbackIf*>::iterator it = callbacksList.begin();...)
   } //if ((true == _proxyManager.isProxyAvailable(managerProxy, sConnmanManagerProxyMetadata)) &&...)
}


} // namespace bosch
} // namespace org

/** @} */
