/**
 * @file ProxyManager.h
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 *
 * @brief 
 *
 * @{
 */

#ifndef _PROXY_MANAGER_H
#define _PROXY_MANAGER_H

#include <string>
#include <map>
#include "boost/shared_ptr.hpp"

#include "ProxyMetadata.h"

namespace org 
{
namespace bosch 
{

//! Template structure to store Proxy data
template < typename tProxy >
struct ProxyData
{
   typedef ::boost::shared_ptr< tProxy > tProxySptr;

   tProxySptr proxy;
   bool isServiceAvailable;

   ProxyData():
      isServiceAvailable(false)
   {

   }
};


/**
 * Type definition for DBUS proxy manager.
 * HINT: No locking mechanisms are implemented. Therefore access to this object is only allowed from same thread or
 *       locking mechanisms are applied outside.
 */
template < typename tProxy >
class ProxyManager
{
public:
   typedef ::boost::shared_ptr< tProxy > tProxySptr;
   typedef ProxyData< tProxy > tProxyData;

   ProxyManager():
      _maxNumberOfProxyInstances(0)
   {}

   ProxyManager(const ProxyManager& ref):
         _proxyMetaData2ProxyInstance(ref._proxyMetaData2ProxyInstance),
         _maxNumberOfProxyInstances(ref._maxNumberOfProxyInstances)
   {}

   virtual ~ProxyManager() {}

   ProxyManager& operator=(const ProxyManager& ref)
   {
      if(this == &ref)
      {
         return *this;
      }

      _proxyMetaData2ProxyInstance = ref._proxyMetaData2ProxyInstance;
      _maxNumberOfProxyInstances = ref._maxNumberOfProxyInstances;

      return *this;
   }

   bool operator==(const ProxyManager& ref) const
   {
      bool result = true;

      result = (true == result) && (_proxyMetaData2ProxyInstance == ref._proxyMetaData2ProxyInstance);
      result = (true == result) && (_maxNumberOfProxyInstances == ref._maxNumberOfProxyInstances);

      return result;
   }

   bool operator!=(const ProxyManager& ref) const
   {
      return !(operator==(ref));
   }

   void setMaxNumberOfProxyInstances(const unsigned int number)
   {
      // 0 means unlimited
      _maxNumberOfProxyInstances = number;
   }

   unsigned int getNumberOfProxyInstances() const
   {
      return static_cast<unsigned int>(_proxyMetaData2ProxyInstance.size());
   }

   void setProxyServiceAvailability(const ::std::string& busName, const ::std::string& objPath,
         const ::DBusBusType busType, bool isServiceAvailable)
   {
      ProxyMetadata data1;
      ProxyMetadata data2;

      data1.busName = busName;
      data1.objPath = objPath;
      data1.busType = busType;
      data1.user = e8PROXY_USER_UNKNOWN;

      // do not use find because user is set to NULL; first found entry is used;
      for (typename ::std::map< ProxyMetadata, tProxyData >::iterator it = _proxyMetaData2ProxyInstance.begin();
            it != _proxyMetaData2ProxyInstance.end(); ++it)
      {
         data2 = it->first;
         data2.user = e8PROXY_USER_UNKNOWN;

         if (data1 == data2)
         {
            (it->second).isServiceAvailable = isServiceAvailable;
         }
      }
   }

   bool isProxyServiceAvailable(const ::std::string& busName, const ::std::string& objPath,
         const ::DBusBusType busType)
   {
      ProxyMetadata data1;
      ProxyMetadata data2;

      data1.busName = busName;
      data1.objPath = objPath;
      data1.busType = busType;
      data1.user = e8PROXY_USER_UNKNOWN;

      // do not use find because user is set to NULL; first found entry is used;
      for (typename ::std::map< ProxyMetadata, tProxyData >::iterator it = _proxyMetaData2ProxyInstance.begin();
            it != _proxyMetaData2ProxyInstance.end(); ++it)
      {
         data2 = it->first;
         data2.user = e8PROXY_USER_UNKNOWN;

         if (data1 == data2)
         {
            return (it->second).isServiceAvailable;
         }
      }

      return false;
   }

   bool isProxyAvailable(tProxySptr& instance, const ProxyMetadata& metaData)
   {
      ProxyMetadata data1;
      ProxyMetadata data2;

      data1 = metaData;
      data1.user = e8PROXY_USER_UNKNOWN;

      // do not use find because user is set to NULL; first found entry is used;
      for (typename ::std::map< ProxyMetadata, tProxyData >::iterator it = _proxyMetaData2ProxyInstance.begin();
            it != _proxyMetaData2ProxyInstance.end(); ++it)
      {
         data2 = it->first;
         data2.user = e8PROXY_USER_UNKNOWN;

         if (data1 == data2)
         {
            // related proxy already exists
            instance = (it->second).proxy;
            return true;
         }
      }

      return false;
   }

   bool isProxyAvailable(tProxySptr& instance, const ::std::string& objPath)
   {
      ProxyMetadata metaData;
      metaData.objPath = objPath;

      return isProxyAvailable(instance, metaData);
   }

   bool isProxyAvailable(tProxySptr& instance, const ::std::string& busName,
         const ::std::string& objPath, const ::DBusBusType busType, ProxyUser user)
   {
      ProxyMetadata metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;
      metaData.user = user;

      return isProxyAvailable(instance, metaData);
   }

   void addProxyInstance(const ProxyMetadata& metaData, tProxySptr& instance)
   {
      typename ::std::map<ProxyMetadata, tProxyData>::iterator it = _proxyMetaData2ProxyInstance.find(metaData);

      if(_proxyMetaData2ProxyInstance.end() != it)
      {
         // entry is already stored => can happen because function is called after reception of different signals
      }
      else
      {
         // entry is not stored
         _proxyMetaData2ProxyInstance[metaData].proxy = instance;

         // check size
         if(0 < _maxNumberOfProxyInstances)
         {
            if(_proxyMetaData2ProxyInstance.size() > _maxNumberOfProxyInstances)
            {
               // should never happen - forgot to remove instance?
               // #error_indication
            }
         }
      }
   }

   void addProxyInstance(const ::std::string& objPath, tProxySptr& instance)
   {
      ProxyMetadata metaData;
      metaData.objPath = objPath;

      addProxyInstance(metaData, instance);
   }

   void addProxyInstance(const ::std::string& busName, const ::std::string& objPath,
         const ::DBusBusType busType, ProxyUser user, tProxySptr& instance)
   {
      ProxyMetadata metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;
      metaData.user = user;

      addProxyInstance(metaData, instance);
   }

   void removeProxyInstance(const ProxyMetadata& metaData)
   {
      typename ::std::map<ProxyMetadata, tProxyData>::iterator it = _proxyMetaData2ProxyInstance.find(metaData);

      if(_proxyMetaData2ProxyInstance.end() == it)
      {
         // entry is already erased => can happen
      }
      else
      {
         // entry is not erased
         (it->second).proxy.reset();
         _proxyMetaData2ProxyInstance.erase(it);
      }
   }

   void removeProxyInstance(const ::std::string& objPath)
   {
      ProxyMetadata metaData;
      metaData.objPath = objPath;

      removeProxyInstance(metaData);
   }

   void removeProxyInstance(const ::std::string& busName, const ::std::string& objPath,
         const ::DBusBusType busType, ProxyUser user)
   {
      ProxyMetadata metaData;
      metaData.busName = busName;
      metaData.objPath = objPath;
      metaData.busType = busType;
      metaData.user = user;

      removeProxyInstance(metaData);
   }

   const tProxyData& operator[](const unsigned int index) const
   {
      if(_proxyMetaData2ProxyInstance.size() <= index)
      {
         // should never happen - forgot to check size before?
         // #error_indication
      }

      typename ::std::map<ProxyMetadata, tProxyData>::const_iterator it = _proxyMetaData2ProxyInstance.begin() + index;

      return (it->second);
   }

   tProxyData& operator[](const unsigned int index)
   {
      if(_proxyMetaData2ProxyInstance.size() <= index)
      {
         // should never happen - forgot to check size before?
         // #error_indication
      }

      typename ::std::map<ProxyMetadata, tProxyData>::iterator it = _proxyMetaData2ProxyInstance.begin();

      for(unsigned int i = 0; i < index; i++)
      {
         ++it;
      }

      return (it->second);
   }

   void resetAllProxiesAndClear()
   {
      for(typename ::std::map<ProxyMetadata, tProxyData>::iterator it = _proxyMetaData2ProxyInstance.begin();
            it != _proxyMetaData2ProxyInstance.end(); ++it)
      {
         if ((it->second).proxy)
         {
            (it->second).proxy.reset();
         }
      }
      _proxyMetaData2ProxyInstance.clear();
   }

protected:

   ::std::map< ProxyMetadata, tProxyData > _proxyMetaData2ProxyInstance;
   unsigned int _maxNumberOfProxyInstances;
};

} // namespace bosch
} // namespace org

#endif //_PROXY_MANAGER_H
