/**
 * @file BluezAdapterProxyManager.cpp
 * @author RBEI/ECO3 Karthikeyan Madeswaran
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "ProxyTypes.h"
#include "BluezAdapterDefines.h"
#include "WBLPortsDefines.h"
#include "BluezAdapterProxyManager.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/WBLClients", BluezAdapterProxyManager, Info);

static ProxyMetadata sBluezAdapterProxyMetadata;
BluezAdapterProxyManager::BluezAdapterProxyManager()
{
    LOG_INFO(" BluezAdapterProxyManager() entered ");

    sBluezAdapterProxyMetadata.busName = sBluezAdapterBusName;
    sBluezAdapterProxyMetadata.objPath = sBluezAdpaterObjectPath;
    sBluezAdapterProxyMetadata.busType = DBUS_BUS_SYSTEM;
}

BluezAdapterProxyManager::~BluezAdapterProxyManager()
{
    // Commented the Log since it produces the Uncaught exception from Coverity
    // LOG_INFO(" ~BluezAdapterProxyManager() entered ");
}

void BluezAdapterProxyManager::createProxy()
{
    LOG_INFO(" BluezAdapterProxyManager::createProxy() ");

    ::boost::shared_ptr< BluezAdpaterProxy > managerProxy;
    if(false == _proxyManager.isProxyAvailable(managerProxy, sBluezAdapterProxyMetadata))
    {
        managerProxy = BluezAdpaterProxy::createProxy(sBluezAdapterPortName, *this);
        _proxyManager.addProxyInstance(sBluezAdapterProxyMetadata, managerProxy);
    }
}

void BluezAdapterProxyManager::destroyProxy()
{
    LOG_INFO(" BluezAdapterProxyManager::destroyProxy() entered ");

    ::boost::shared_ptr< BluezAdpaterProxy > managerProxy;
    if ((true == _proxyManager.isProxyAvailable(managerProxy, sBluezAdapterProxyMetadata)) && (managerProxy))
    {
        managerProxy->sendDeregisterAll();
    }

    // these proxies have to be destroyed during runtime
    // destroy all proxies now if still available
    _proxyManager.resetAllProxiesAndClear();
    _proxyCbManager.removeAllCallbackIfs();
}

bool BluezAdapterProxyManager::isProxyServiceAvailable()
{
    return (_proxyManager.isProxyServiceAvailable(
            sBluezAdapterProxyMetadata.busName,
            sBluezAdapterProxyMetadata.objPath,
            sBluezAdapterProxyMetadata.busType));
}

void BluezAdapterProxyManager::setCallbackIf(ProxyUser user, IBluezAdapterNotifCallbackIf* callbackIf)
{
    LOG_INFO(" BluezAdapterProxyManager::setCallbackIf() entered ");

    ProxyMetadata metadata(sBluezAdapterProxyMetadata);
    metadata.user = user;

    if (callbackIf)
    {
        _proxyCbManager.addCallbackIf(metadata, callbackIf);
    }
    else
    {
        _proxyCbManager.removeCallbackIf(metadata);
    }
}


act_t BluezAdapterProxyManager::sendGetAddressRequest(IBluezAdapterNotifCallbackIf& callbackIf)
{
    LOG_INFO(" BluezAdapterProxyManager::sendGetAddressRequest() entered ");

    ::boost::shared_ptr< BluezAdpaterProxy > adapterProxy;

    if ((true == _proxyManager.isProxyAvailable(adapterProxy, sBluezAdapterProxyMetadata)) && (adapterProxy))
    {
        return adapterProxy->sendAddressGet(callbackIf);
    }

    return DEFAULT_ACT;
}

act_t BluezAdapterProxyManager::sendGetAliasRequest(IBluezAdapterNotifCallbackIf& callbackIf)
{
    LOG_INFO(" BluezAdapterProxyManager::sendGetAliasRequest() entered ");

    ::boost::shared_ptr< BluezAdpaterProxy > adapterProxy;

    if ((true == _proxyManager.isProxyAvailable(adapterProxy, sBluezAdapterProxyMetadata)) && (adapterProxy))
    {
        return adapterProxy->sendAliasGet(callbackIf);
    }

    return DEFAULT_ACT;
}
// ServiceAvailableIF implementation
void BluezAdapterProxyManager::onAvailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
        const ::asf::core::ServiceStateChange& stateChange)
{
    LOG_INFO(" BluezAdapterProxyManager::onAvailable() entered ");

    ::boost::shared_ptr< BluezAdpaterProxy > managerProxy;
    if ((true == _proxyManager.isProxyAvailable(managerProxy, sBluezAdapterProxyMetadata)) &&
            (managerProxy) && (proxy == managerProxy))
    {
        const ::std::string& busName = managerProxy->getDBusBusName();
        const ::std::string& objectPath = managerProxy->getDBusObjectPath();
        ::DBusBusType busType = managerProxy->getBusType();

        _proxyManager.setProxyServiceAvailability(busName, objectPath, busType, true);

        std::vector<IBluezAdapterNotifCallbackIf*> callbacksList =
                _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
        for (std::vector<IBluezAdapterNotifCallbackIf*>::iterator it = callbacksList.begin();
                it != callbacksList.end(); ++it)
        {
            LOG_DEBUG(" onAvailable: Registering for Bluez Properties & signals ");

            managerProxy->sendAddressRegister(**it);
            managerProxy->sendAliasRegister(**it);
            managerProxy->sendAdapterInitializationRegister(**it);
            (*it)->onBluezAdapterServiceAvailable(busName, objectPath, busType,
                    stateChange.getPreviousState(), stateChange.getCurrentState());
        }
    }
}

void BluezAdapterProxyManager::onUnavailable(const boost::shared_ptr< ::asf::core::Proxy >& proxy,
        const ::asf::core::ServiceStateChange& stateChange)
{
    LOG_INFO(" BluezAdapterProxyManager::onUnavailable() entered ");

    ::boost::shared_ptr< BluezAdpaterProxy > managerProxy;
    if ((true == _proxyManager.isProxyAvailable(managerProxy, sBluezAdapterProxyMetadata)) &&
            (managerProxy) && (proxy == managerProxy))
    {
        const ::std::string& busName = managerProxy->getDBusBusName();
        const ::std::string& objectPath = managerProxy->getDBusObjectPath();
        ::DBusBusType busType = managerProxy->getBusType();

        _proxyManager.setProxyServiceAvailability(busName, objectPath, busType, false);

        std::vector<IBluezAdapterNotifCallbackIf*> callbacksList =
                _proxyCbManager.getAllCallbackIf(busName, objectPath, busType);
        for (std::vector<IBluezAdapterNotifCallbackIf*>::iterator it = callbacksList.begin();
                it != callbacksList.end(); ++it)
        {
            managerProxy->sendDeregisterAll();
            (*it)->onBluezAdapterServiceUnAvailable(busName, objectPath, busType,
                    stateChange.getPreviousState(), stateChange.getCurrentState());
        }
    }
}

void BluezAdapterProxyManager::setBluezAdapterAddress(::std::string& address)
{
    LOG_INFO(" BluezAdapterProxyManager::setBluezAdapterAddress() entered ");
    _bluezAdapterAddress = address;
}
void BluezAdapterProxyManager::setBluezAdapterAlias(::std::string& alias)
{
    LOG_INFO(" BluezAdapterProxyManager::setBluezAdapterAlias() entered ");
    _bluezAdapterAlias = alias;
}
bool BluezAdapterProxyManager::getBluezAdapterAddress(::std::string& address)
{
    LOG_INFO(" BluezAdapterProxyManager::getBluezAdapterAddress() entered ");
    ::boost::shared_ptr< BluezAdpaterProxy > adapterProxy;
    if(!_bluezAdapterAddress.empty())
    {
        address = _bluezAdapterAddress;
        return true;
    }
    return false;
}
bool BluezAdapterProxyManager::getBluezAdapterAlias(::std::string& alias)
{
    LOG_INFO(" BluezAdapterProxyManager::getBluezAdapterAlias() entered ");
    ::boost::shared_ptr< BluezAdpaterProxy > adapterProxy;
    if(!_bluezAdapterAlias.empty())
    {
        alias = _bluezAdapterAlias;
        return true;
    }
    return false;
}
} // namespace bosch
} // namespace org

/** @} */
