/**
 * @file GenericDbusParser.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "GenericDbusParser.h"
#include "IDbusProperty2EnumMapping.h"

namespace org 
{
namespace bosch 
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Utils",GenericDbusParser, Info);

GenericDbusParser::GenericDbusParser()
{
   _tracesEnabled = false;
}

GenericDbusParser::~GenericDbusParser()
{
}

void GenericDbusParser::parseProperties(::std::vector<DbusVariantProperty>& outProperties,
      const ::std::vector< int >& matchingInterfaces,
      ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
      const ::DBusBusType busType, const ::std::string& busName,
      const ::std::string& objPath, const ::std::string& interfaceName,
      const IDbusProperty2EnumMapping* mapping, const bool parseDictItemsAsProperties) const
{
   if(!mapping)
   {
      LOG_ERROR(" parseProperties: invalid mapping ");
      return;
   }

   if(true == _tracesEnabled)
   {
      if(DBUS_BUS_SYSTEM == busType)
      {
         LOG_DEBUG(" parseProperties: SYSTEM : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str());
      }
      else
      {
         LOG_DEBUG(" parseProperties: SESSION : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str());
      }
   }

   parsePropertiesInternal(outProperties, matchingInterfaces, inProperties, mapping, parseDictItemsAsProperties);
}

void GenericDbusParser::parseIntrospection(const ::std::string& data,
      const ::DBusBusType busType, const ::std::string& busName,
      const ::std::string& objPath, const ::std::string& interfaceName,
      const IDbusProperty2EnumMapping* mapping) const
{
   if(!mapping)
   {
      LOG_ERROR(" parseIntrospection: invalid mapping ");
      return;
   }

   if(true == _tracesEnabled)
   {
      if(DBUS_BUS_SYSTEM == busType)
      {
         LOG_DEBUG(" parseIntrospection: SYSTEM : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str());
      }
      else
      {
         LOG_DEBUG(" parseIntrospection: SESSION : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str());
      }
      LOG_DEBUG(" parseIntrospection: ___start___");
      printf("%s", data.c_str()); fflush(stdout);
      LOG_DEBUG(" parseIntrospection: ___end___");
   }
}

void GenericDbusParser::parseInterfacesAdded(const ::std::string& addedObjectPath,
		::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
		 const ::DBusBusType busType, const ::std::string& busName,
		 const ::std::string& objPath, const ::std::string& interfaceName,
		 const IDbusProperty2EnumMapping* mapping) const
{
   if(!mapping)
   {
      LOG_ERROR(" parseInterfacesAdded: invalid mapping ");
      return;
   }

   if(true == _tracesEnabled)
   {
      if(DBUS_BUS_SYSTEM == busType)
      {
         LOG_DEBUG(" parseInterfacesAdded: SYSTEM : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str());
      }
      else
      {
         LOG_DEBUG(" parseInterfacesAdded: SESSION : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str());
      }
   }

#if 0
   <signal name="InterfacesAdded">
      <arg name="object" type="o"/>
      <arg name="interfaces" type="a{sa{sv}}"/>
   </signal>
#endif

   if(true == _tracesEnabled) { LOG_DEBUG(" parseInterfacesAdded: addedObjectPath=%s", addedObjectPath.c_str()); }

   size_t i = 0;
   ::std::vector<DbusVariantProperty> outProperties;
   ::std::vector< int > matchingInterfaces;
   for(::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >::iterator it = inInterfaces.begin();
		   it != inInterfaces.end(); ++it)
   {
      if(true == _tracesEnabled) { LOG_DEBUG(" parseInterfacesAdded: [%u] interface=%s", i, it->first.c_str()); }

      parsePropertiesInternal(outProperties, matchingInterfaces, it->second, mapping);

      outProperties.clear();
      ++i;
   }
}

void GenericDbusParser::parseInterfacesRemoved(const ::std::string& removedObjectPath,
		const ::std::vector< ::std::string >& inInterfaces, const ::DBusBusType busType,
		const ::std::string& busName, const ::std::string& objPath,
		const ::std::string& interfaceName, const IDbusProperty2EnumMapping* mapping) const
{
   if(!mapping)
   {
      LOG_ERROR(" parseInterfacesRemoved: invalid mapping ");
      return;
   }

   if(true == _tracesEnabled)
   {
      if(DBUS_BUS_SYSTEM == busType)
      {
         LOG_DEBUG(" parseInterfacesRemoved: SYSTEM : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str());
      }
      else
      {
         LOG_DEBUG(" parseInterfacesRemoved: SESSION : %30s : %50s : %s", busName.c_str(), interfaceName.c_str(), objPath.c_str());
      }
   }

#if 0
   <signal name="InterfacesRemoved">
      <arg name="object" type="o"/>
      <arg name="interfaces" type="as"/>
   </signal>
#endif

   if(true == _tracesEnabled) { LOG_DEBUG(" parseInterfacesRemoved: removedObjectPath=%s", removedObjectPath.c_str()); }

   for(size_t i = 0; i < inInterfaces.size(); ++i)
   {
      if(true == _tracesEnabled) { LOG_DEBUG(" parseInterfacesRemoved: [%u] interface=%s", i, inInterfaces[i].c_str()); }
   }
}

bool GenericDbusParser::findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList,
		::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
		 const ::std::string& propertyName, const int interface,
		 const IDbusProperty2EnumMapping* mapping) const
{
   ::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it = inProperties.find(propertyName);

   if(inProperties.end() != it)
   {
      parsePropertyValue(outPropertyList, it->second, propertyName, interface, mapping);

      return true;
   }

   return false;
}

void GenericDbusParser::parsePropertyValue(::std::vector<DbusVariantProperty>& outPropertyList,
		::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName,
		 const int interface, const IDbusProperty2EnumMapping* mapping) const
{
   if(!mapping)
   {
      LOG_ERROR(" parsePropertyValue: invalid mapping ");
      return;
   }

   // 1. get interfaces
   ::std::vector<const ::std::map< ::std::string, int >*> interfaceList;
   ::std::vector< int > matchingInterfaces;
   matchingInterfaces.push_back(interface);
   mapping->getInterfaces(interfaceList, matchingInterfaces);

   // 2. parse properties
   ::std::string signature;
   ::std::string dummy;

   propertyValue.getSignature(signature);

   parseProperty(outPropertyList, interfaceList, propertyValue.getReadIterator(), signature, propertyName, 0, 0, false, dummy);

   if(0 == outPropertyList.size())
   {
      LOG_ERROR(" parsePropertyValue: error parsing property value ");
   }
}

void GenericDbusParser::addDictionaryItem(::std::map< ::std::string, ::asf::dbus::DBusVariant >& dictionary,
		const ::std::string& key, const DbusVariant& value) const
{
   if(true == key.empty())
   {
      LOG_ERROR(" addDictionaryItem: key is empty ");
      return;
   }

   ::std::map< ::std::string, ::asf::dbus::DBusVariant >::const_iterator it = dictionary.find(key); // else lint complains: Info 864: prio3: Expression involving variable 'dictionary' possibly depends on order of evaluation
   if(dictionary.end() != it)
   {
      LOG_ERROR(" addDictionaryItem: key not found ");
      return;
   }

   ::asf::dbus::DBusVariant& item = dictionary[key];

   switch(value.getType())
   {
      case VARIANT_INT8:
      {
         // transfer as unsigned char
         unsigned char data = (unsigned char)value.getInt8();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_BYTE, &data);
         break;
      }
      case VARIANT_UINT8:
      {
         unsigned char data = value.getUInt8();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_BYTE, &data);
         break;
      }
      case VARIANT_INT16:
      {
         dbus_int16_t data = value.getInt16();
        (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_INT16, &data);
         break;
      }
      case VARIANT_UINT16:
      {
         dbus_uint16_t data = value.getUInt16();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_UINT16, &data);
         break;
      }
      case VARIANT_INT32:
      {
         dbus_int32_t data = value.getInt32();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_INT32, &data);
         break;
      }
      case VARIANT_UINT32:
      {
         dbus_uint32_t data = value.getUInt32();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_UINT32, &data);
         break;
      }
      case VARIANT_INT64:
      {
         dbus_int64_t data = value.getInt64();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_INT64, &data);
         break;
      }
      case VARIANT_UINT64:
      {
         dbus_uint64_t data = value.getUInt64();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_UINT64, &data);
         break;
      }
      case VARIANT_ENUM:
      {
         LOG_ERROR(" addDictionaryItem: error adding enum ");
         break;
      }
      case VARIANT_BOOL:
      {
         dbus_bool_t data = (dbus_bool_t)value.getBool();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_BOOLEAN, &data);
         break;
      }
      case VARIANT_STRING:
      {
         const char* data = value.getString().c_str();
         (void) dbus_message_iter_append_basic(item.getWriteIterator(), DBUS_TYPE_STRING, &data);
         break;
      }
      case VARIANT_STRING_ARRAY:
      {
         // TODO: to be checked with ASF team
#if 1
         // this implementation creates a variant entry with type array but array is always empty
         int type = DBUS_TYPE_ARRAY;
         const char* subType = "s";
         DBusMessageIter containerWriteIterator;
         (void) dbus_message_iter_open_container(item.getWriteIterator(), type, subType, &containerWriteIterator);

         const ::std::vector<std::string>& arrayValue = value.getStringArray();
         // printf("##### arrayValue.size()=%u\n", arrayValue.size());
         for(size_t i = 0; i < arrayValue.size(); i++)
         {
            const char* data = arrayValue[i].c_str();
            (void) dbus_message_iter_append_basic(&containerWriteIterator, DBUS_TYPE_STRING, &data);
            // printf("##### data=%s\n", data);
         }

         (void) dbus_message_iter_close_container(item.getWriteIterator(), &containerWriteIterator);
#endif

#if 0
         // copy from generated ASF code
         void serializeDBus (const ::org::bluez::Adapter1::TraceCategoryUpdate& in, DBusMessageIter* out) {
            DBusMessageIter o61ArrayIter;
            DBUS_ASSERT(dbus_message_iter_open_container(out, DBUS_TYPE_ARRAY, "s", &o61ArrayIter));
            const ::std::vector< ::std::string >* o62 = &in.getTraceCategory();
            ::std::vector< ::std::string >::const_iterator o63 = o62->end();
            for (::std::vector< ::std::string >::const_iterator o64 = o62->begin(); o64 != o63; ++o64) {
               const char* o65 = (*o64).c_str();
               DBUS_ASSERT(dbus_message_iter_append_basic(&o61ArrayIter, DBUS_TYPE_STRING, &o65));
            }
            DBUS_ASSERT(dbus_message_iter_close_container(out, &o61ArrayIter));
         }
#endif

#if 0
         // this implementation creates a variant entry with type array but array is always empty
         DBusMessageIter variantWriteIterator;
         DBusMessageIter containerWriteIterator;
         char typeSignature[2];
         char arraySignature[3];
         arraySignature[0] = (char)DBUS_TYPE_ARRAY;
         arraySignature[1] = typeSignature[0] = (char)DBUS_TYPE_STRING;
         arraySignature[2] = typeSignature[1] = '\0';
         dbus_message_iter_open_container(item.getWriteIterator(), DBUS_TYPE_VARIANT, arraySignature, &variantWriteIterator);
         dbus_message_iter_open_container(&variantWriteIterator, DBUS_TYPE_ARRAY, typeSignature, &containerWriteIterator);

         const ::std::vector<std::string>& arrayValue = value.getStringArray();
         // printf("##### arrayValue.size()=%u\n", arrayValue.size());
         for(size_t i = 0; i < arrayValue.size(); i++)
         {
            const char* data = arrayValue[i].c_str();
            dbus_message_iter_append_basic(&containerWriteIterator, DBUS_TYPE_STRING, &data);
            // printf("##### data=%s\n", data);
         }

         dbus_message_iter_close_container(&variantWriteIterator, &containerWriteIterator);
         dbus_message_iter_close_container(item.getWriteIterator(), &variantWriteIterator);
#endif

#if 0
         setTraces(true);
         ::std::vector<DbusVariantProperty> outProperties;
         ::std::vector<const ::std::map< ::std::string, int >*> interfaceList;
         ::std::string signature;
         ::std::string dummy;
         item.getSignature(signature);
         parseProperty(outProperties, interfaceList, item.getReadIterator(), signature, key, 0, 0, false, dummy);
#endif
         break;
      }
      case VARIANT_NONE:
      case VARIANT_UNKNOWN:
      default:
      {
         LOG_ERROR(" addDictionaryItem: variant type unknown ");
         break;
      }
   }
}

void GenericDbusParser::parsePropertiesInternal(::std::vector<DbusVariantProperty>& outProperties,
		const ::std::vector< int >& matchingInterfaces,
		::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
		 const IDbusProperty2EnumMapping* mapping, const bool parseDictItemsAsProperties) const
{
   if(!mapping)
   {
      LOG_ERROR(" parsePropertiesInternal: invalid mapping ");
      return;
   }

   // 1. get interfaces
   ::std::vector<const ::std::map< ::std::string, int >*> interfaceList;
   mapping->getInterfaces(interfaceList, matchingInterfaces);

   // 2. parse properties
   ::std::string signature;
   unsigned int i = 0;
   ::std::string dummy;
   outProperties.reserve(inProperties.size() + 20);

   for(::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it = inProperties.begin(); it != inProperties.end(); ++it)
   {
      it->second.getSignature(signature);
      parseProperty(outProperties, interfaceList, it->second.getReadIterator(),
            signature, it->first, i, 0, false, dummy, parseDictItemsAsProperties);
      ++i;
   }
}

void GenericDbusParser::parseProperty(::std::vector<DbusVariantProperty>& outProperties,
		const ::std::vector<const ::std::map< ::std::string, int >*>& interfaceList,
		::DBusMessageIter* iter, const ::std::string& signature,
		 const ::std::string& property, const unsigned int counter,
		 const unsigned int depth, const bool isDict,
		 const ::std::string& dictKey, const bool parseDictItemsAsProperties) const
{
   LOG_DEBUG("GenericDbusParser::parseProperty() entered: signature = %s", signature.c_str());
//   LOG_DEBUG("GenericDbusParser::parseProperty() entered: parseDictItemsAsProperties = %d, property = %s",
//         parseDictItemsAsProperties, property.c_str());
//   LOG_DEBUG("GenericDbusParser::parseProperty() entered: isDict = %d, dictKey = %s", isDict, dictKey.c_str());

   if(!iter)
   {
      LOG_ERROR(" parseProperty: invalid dbus msg iterator ");
      return;
   }

   DbusVariantProperty variantEntry;
   unsigned int i = counter;
   bool add = true;

   bool storeDictItemsAsDict = !parseDictItemsAsProperties;

   // BYTE (uint8)
   if(0 == signature.compare("y"))
   {
      unsigned char value = 0;
      dbus_message_iter_get_basic(iter, &value);
      (isDict && storeDictItemsAsDict)? (variantEntry.propData.setUInt8DictElement(dictKey, value)) : (variantEntry.propData.setUInt8(value));
      if(true == _tracesEnabled)
      {
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%u", i, property.c_str(), dictKey.c_str(), value); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%u", i, property.c_str(), value); }
      }
   }
   // BOOLEAN (bool)
   else if(0 == signature.compare("b"))
   {
      dbus_bool_t value = FALSE;
      dbus_message_iter_get_basic(iter, &value);
      (isDict && storeDictItemsAsDict)? (variantEntry.propData.setBoolDictElement(dictKey, value)) : (variantEntry.propData.setBool((bool)value));
      if(true == _tracesEnabled)
      {
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%d", i, property.c_str(), dictKey.c_str(), value); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%d", i, property.c_str(), value); }
      }
   }
   // INT16 (int16)
   else if(0 == signature.compare("n"))
   {
      dbus_int16_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      (isDict && storeDictItemsAsDict)? (variantEntry.propData.setInt16DictElement(dictKey, value)) : (variantEntry.propData.setInt16(value));
      if(true == _tracesEnabled)
      {
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%d", i, property.c_str(), dictKey.c_str(), value); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%d", i, property.c_str(), value); }
      }
   }
   // UINT16 (uint16)
   else if(0 == signature.compare("q"))
   {
      dbus_uint16_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      (isDict && storeDictItemsAsDict)? (variantEntry.propData.setUInt16DictElement(dictKey, value)) : (variantEntry.propData.setUInt16(value));
      if(true == _tracesEnabled)
      {
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%u", i, property.c_str(), dictKey.c_str(), value); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%u", i, property.c_str(), value); }
      }
   }
   // INT32 (int32)
   else if(0 == signature.compare("i"))
   {
      dbus_int32_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      (isDict && storeDictItemsAsDict)? (variantEntry.propData.setInt32DictElement(dictKey, value)) : (variantEntry.propData.setInt32(value));
      if(true == _tracesEnabled)
      {
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%d", i, property.c_str(), dictKey.c_str(), value); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%d", i, property.c_str(), value); }
      }
   }
   // UINT32 (uint32)
   else if(0 == signature.compare("u"))
   {
      dbus_uint32_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      (isDict && storeDictItemsAsDict)? (variantEntry.propData.setUInt32DictElement(dictKey, value)) : (variantEntry.propData.setUInt32(value));
      if(true == _tracesEnabled)
      {
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%u", i, property.c_str(), dictKey.c_str(), value); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%u", i, property.c_str(), value); }
      }
   }
   // INT64 (int64)
   else if(0 == signature.compare("x"))
   {
      dbus_int64_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      (isDict && storeDictItemsAsDict)? (variantEntry.propData.setInt64DictElement(dictKey, value)) : (variantEntry.propData.setInt64(value));

      //TODO
      /*if(true == _tracesEnabled)
      {
         ::fw::FormattedOutputS64 printValue(value);
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), printValue.c_str()); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), printValue.c_str()); }
      }*/
   }
   // UINT64 (uint64)
   else if(0 == signature.compare("t"))
   {
      dbus_uint64_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      (isDict && storeDictItemsAsDict)? (variantEntry.propData.setUInt64DictElement(dictKey, value)) : (variantEntry.propData.setUInt64(value));

      //TODO
      /*if(true == _tracesEnabled)
      {
         ::fw::FormattedOutputU64 printValue(value);
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), printValue.c_str()); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), printValue.c_str()); }
      }*/
   }
   // DOUBLE (double)
   else if(0 == signature.compare("d"))
   {
      double value = 0.0;
      dbus_message_iter_get_basic(iter, &value);
      // not supported at the moment
      LOG_ERROR(" parseProperty: signature %s not supported ", signature.c_str());
      add = false;
      if(true == _tracesEnabled)
      {
         //::fw::FormattedOutputDouble printValue(value);
         //if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), printValue.c_str()); }
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%f", i, property.c_str(), dictKey.c_str(), value); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%f", i, property.c_str(), value); }
      }
   }
   // STRING (::std::string)
   else if(0 == signature.compare("s"))
   {
      const char* value = NULL;
      dbus_message_iter_get_basic(iter, &value);
      if(value)
      {
         variantEntry.propData.setString(value);
         if(true == _tracesEnabled)
         {
            if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), value); }
            else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), value); }
         }
      }
      else
      {
         variantEntry.propData.setString("");
         if(true == _tracesEnabled)
         {
            if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), "<NULL>"); }
            else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), "<NULL>"); }
         }
         LOG_ERROR(" parseProperty: error reading string ");
      }
   }
   // OBJECT_PATH (::std::string)
   else if(0 == signature.compare("o"))
   {
      const char* value = NULL;
      dbus_message_iter_get_basic(iter, &value);
      if(value)
      {
         variantEntry.propData.setString(value);
         if(true == _tracesEnabled)
         {
            if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), value); }
            else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), value); }
         }
      }
      else
      {
         variantEntry.propData.setString("");
         if(true == _tracesEnabled)
         {
            if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), "<NULL>"); }
            else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), "<NULL>"); }
         }
         LOG_ERROR(" parseProperty: object path is empty ");
      }
   }
   // SIGNATURE (::std::string)
   else if(0 == signature.compare("g"))
   {
      const char* value = NULL;
      dbus_message_iter_get_basic(iter, &value);
      if(value)
      {
         // not supported at the moment
         LOG_ERROR(" parseProperty: signature %s not supported ", signature.c_str());
         add = false;
         if(true == _tracesEnabled)
         {
            if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), value); }
            else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), value); }
         }
      }
      else
      {
         // not supported at the moment
         LOG_ERROR(" parseProperty: object path is empty ");
         add = false;
         if(true == _tracesEnabled)
         {
            if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%s", i, property.c_str(), dictKey.c_str(), "<NULL>"); }
            else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), "<NULL>"); }
         }
      }
   }
   // UNIX_FD (uint32)
   else if(0 == signature.compare("h"))
   {
      dbus_uint32_t value = 0;
      dbus_message_iter_get_basic(iter, &value);
      // not supported at the moment
      LOG_ERROR(" parseProperty: signature %s not supported ", signature.c_str());
      add = false;
      if(true == _tracesEnabled)
      {
         if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value=%u", i, property.c_str(), dictKey.c_str(), value); }
         else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%u", i, property.c_str(), value); }
      }
   }
   // ARRAY of STRING (::std::string)
   else if(0 == signature.compare("as"))
   {
      ::DBusMessageIter subIter;
      ::std::string subSignature;

      dbus_message_iter_recurse(iter, &subIter);
      char* sigPtr = dbus_message_iter_get_signature(&subIter);
      subSignature.assign(sigPtr);
      dbus_free(sigPtr);

      if(0 == subSignature.compare("s"))
      {
         ::std::vector<std::string> stringList;
         stringList.reserve(20);
         size_t j = 0;
         const char* value;
         bool next = true;
         while(true == next)
         {
            value = nullptr;
            dbus_message_iter_get_basic(&subIter, &value);
            if(value)
            {
               stringList.push_back(value);
               if(true == _tracesEnabled)
               {
                  if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value[%u]=%s", i, property.c_str(), dictKey.c_str(), j, value); }
                  else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value[%u]=%s", i, property.c_str(), j, value); }
               }
            }
            else
            {
               stringList.push_back("");
               if(true == _tracesEnabled)
               {
                  if(true == isDict) { LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value[%u]=%s", i, property.c_str(), dictKey.c_str(), j, "<NULL>"); }
                  else               { LOG_DEBUG(" parseProperty: [%u] property=%20s value[%u]=%s", i, property.c_str(), j, "<NULL>"); }
               }
               LOG_ERROR(" parseProperty: string is empty ");
            }

            ++j;

            if(FALSE == dbus_message_iter_next(&subIter))
            {
               next = false;
            }
         }
         variantEntry.propData.setStringArray(stringList);
      }
      else
      {
         LOG_ERROR(" parseProperty: signature %s not supported ", signature.c_str());
         add = false;
      }
   }
   else if (0 == signature.compare("ay"))
   {
      ::DBusMessageIter subIter;
      ::std::string subSignature;

      dbus_message_iter_recurse(iter, &subIter);
      char* sigPtr = dbus_message_iter_get_signature(&subIter);
      subSignature.assign(sigPtr);
      dbus_free(sigPtr);
      if (0 == subSignature.compare("y"))
      {
         ::std::vector < uint8_t > byteList;
         size_t j = 0;
         unsigned char value = 0;
         while (dbus_message_iter_get_arg_type(&subIter) != DBUS_TYPE_INVALID)
         {
            dbus_message_iter_get_basic(&subIter, &value);
            byteList.push_back(value);
            if (true == _tracesEnabled)
            {
               if (true == isDict)
               {
                  LOG_DEBUG(" parseProperty: [%u] property=%20s : key=%20s value[%u]=%u", i, property.c_str(),
                        dictKey.c_str(), j, value);
               }
               else
               {
                  LOG_DEBUG(" parseProperty: [%u] property=%20s value[%u]=%u", i, property.c_str(), j, value);
               }
            }
            ++j;
            dbus_message_iter_next(&subIter);
         }
         variantEntry.propData.setByteArray(byteList);
      }
   }
   // DICTIONARY (std::map)
   else if(0 == signature.compare("a{sv}"))
   {
      // do not parse any property now
      add = false;

      // if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT: signature=%s", signature.c_str()); }

         if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT: depth=%u (start)", depth); }

         ::DBusMessageIter subIter;
         ::std::string subSignature;
         char* sigPtr;
         ::std::string dummy;

         // recurse into container value
         dbus_message_iter_recurse(iter, &subIter);

         size_t j = 0;
         bool next = true;
         while(true == next)
         {
            sigPtr = dbus_message_iter_get_signature(&subIter);
            subSignature.assign(sigPtr);
            dbus_free(sigPtr);

            if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT: [%u] subSignature=%s", j, subSignature.c_str()); }

            parseProperty(outProperties, interfaceList, &subIter, subSignature,
                  property, i, (depth + 1), false, dummy, parseDictItemsAsProperties);

            ++j;

            if(FALSE == dbus_message_iter_next(&subIter))
            {
               next = false;
            }
         }

         if(FALSE == dbus_message_iter_next(&subIter))
         {
            if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT: no more fields (depth=%u)", depth); }
         }
         else
         {
            sigPtr = dbus_message_iter_get_signature(&subIter);
            subSignature.assign(sigPtr);
            dbus_free(sigPtr);
            LOG_ERROR(" parseProperty: DICT: more fields (depth=%u) subSignature=%s", depth, subSignature.c_str());
         }

         if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT: depth=%u (end)", depth); }
   }
   // DICTIONARY ENTRY
   else if(0 == signature.compare("{sv}"))
   {
      // do not parse any property now
      add = false;

      // if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT ENTRY: signature=%s", signature.c_str()); }

         if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT ENTRY: depth=%u (start)", depth); }

         ::DBusMessageIter subIter;
         ::std::string subSignature;
         char* sigPtr;

         // recurse into container value
         dbus_message_iter_recurse(iter, &subIter);

         // get key
         sigPtr = dbus_message_iter_get_signature(&subIter);
         subSignature.assign(sigPtr);
         dbus_free(sigPtr);

         if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT ENTRY: key: subSignature=%s", subSignature.c_str()); }

         if(0 == subSignature.compare("s"))
         {
            const char* value = NULL;
            dbus_message_iter_get_basic(&subIter, &value);
            if(value)
            {
               if(TRUE == dbus_message_iter_next(&subIter))
               {
                  // get value
                  sigPtr = dbus_message_iter_get_signature(&subIter);
                  subSignature.assign(sigPtr);
                  dbus_free(sigPtr);
                  // if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT ENTRY: value: subSignature=%s", subSignature.c_str()); }

                  if(0 == subSignature.compare("v"))
                  {
                     ::DBusMessageIter valueIter;
                     ::std::string valueSignature;
                     char* sigValuePtr;
                     ::std::string key(value);

                     // recurse into container value
                     dbus_message_iter_recurse(&subIter, &valueIter);
                     sigValuePtr = dbus_message_iter_get_signature(&valueIter);
                     valueSignature.assign(sigValuePtr);
                     dbus_free(sigValuePtr);

                     if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT ENTRY: value: subSignature=%s", valueSignature.c_str()); }

                     parseProperty(outProperties, interfaceList, &valueIter, valueSignature,
                           property, i, (depth + 1), true, key, parseDictItemsAsProperties);
                  }
                  else
                  {
                     LOG_ERROR(" parseProperty: DICT ENTRY: signature %s not supported ", subSignature.c_str());
                  }
               }
               else
               {
                  LOG_ERROR(" parseProperty: DICT ENTRY: value not found ");
               }
            }
            else
            {
               LOG_ERROR(" parseProperty: DICT ENTRY: value is empty ");
            }
         }
         else
         {
            LOG_ERROR(" parseProperty: DICT ENTRY: signature %s not supported ", subSignature.c_str());
         }

         if(FALSE == dbus_message_iter_next(&subIter))
         {
            if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT ENTRY: no more fields (depth=%u)", depth); }
         }
         else
         {
            sigPtr = dbus_message_iter_get_signature(&subIter);
            subSignature.assign(sigPtr);
            dbus_free(sigPtr);
            LOG_ERROR(" parseProperty: DICT ENTRY: more fields (depth=%u) subSignature=%s", depth, subSignature.c_str());
         }

         if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: DICT ENTRY: depth=%u (end)", depth); }
    }
   // invalid data (e.g. empty ARRAY)
   else if(0 == signature.compare(""))
   {
      // according DBUS specification an empty string indicates an invalid type
      // this was observed during testing for e.g. an empty string array
      // this means that the related property is empty
      // therefore add to property list to indicate an empty value
      // HINT: if property is not found related entry is not added to property list
      variantEntry.propData.setNone();
      if(true == _tracesEnabled) { LOG_DEBUG(" parseProperty: [%u] property=%20s value=%s", i, property.c_str(), "\"\""); }
   }
   // others
   else
   {
      LOG_ERROR(" parseProperty: [%u] property=%20s signature=\"%s\" not supported", i, property.c_str(), signature.c_str());
      add = false;
   }

   if(true == add)
   {
      // get matching property
      bool found = false;

      for(size_t k = 0; (false == found) && (k < interfaceList.size()); k++)
      {
         if(interfaceList[k])
         {
            const ::std::map< ::std::string, int >& propList = *interfaceList[k];

            ::std::map< ::std::string, int >::const_iterator it;

            if(isDict && parseDictItemsAsProperties)
            {
               it = propList.find(dictKey);
            }
            else
            {
               it = propList.find(property);
            }

            if(propList.end() != it)
            {
               variantEntry.propEnum = it->second;
               found = true;
               LOG_DEBUG(" parseProperty: Stored Variant entry enum %d", variantEntry.propEnum);
            }
            else
            {
               LOG_ERROR(" parseProperty: No match found in interfaces list ");
            }
         }
         else
         {
            LOG_ERROR(" parseProperty: interfaces list is empty ");
         }
      }

      if(true == found)
      {
         outProperties.push_back(variantEntry);
      }
   }
}

}
//namespace bosch
} //namespace org

/** @} */
