/**
 * @file DBusHelper.cpp
 * @author RBEI/ECO32 Karthikeyan Madeswaran
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 * @brief
 *
 * @{
 */

#include "DbusHelper.h"

namespace org
{
namespace bosch
{

bool dict_append_entry(DBusMessageIter *dict, const char *key, int type, void *val)
{
   DBusMessageIter entry;
   bool isSuccess = true;

   if (type == DBUS_TYPE_STRING)
   {
      const char *str = *((const char **) val);
      if (str == NULL)
         isSuccess = false;
   }

   if ((!dbus_message_iter_open_container(dict, DBUS_TYPE_DICT_ENTRY, nullptr, &entry)||
       (!dbus_message_iter_append_basic(&entry, DBUS_TYPE_STRING, &key))||
       (!append_variant(&entry, type, val)||
       (!dbus_message_iter_close_container(dict, &entry)))))
   {
      isSuccess = false;
   }
   return isSuccess;
}

bool append_variant(DBusMessageIter *iter, int type, void *val)
{
   DBusMessageIter value;
   bool isSuccess = true;
   char sig[2] =
      { char(type), '\0' };

   if ((!dbus_message_iter_open_container(iter, DBUS_TYPE_VARIANT, sig, &value))||
       (!dbus_message_iter_append_basic(&value, type, val)) ||
       (!dbus_message_iter_close_container(iter, &value)))
   {
      isSuccess = false;
   }
   return isSuccess;
}

bool append_dict_entry_fixed_array(DBusMessageIter *iter, const char *property, void *value, int length)
{
   DBusMessageIter dict_entry, variant, array;
   bool isSuccess = true;

   if ((!dbus_message_iter_open_container(iter, DBUS_TYPE_DICT_ENTRY, nullptr,&dict_entry))||
         (!dbus_message_iter_append_basic(&dict_entry, DBUS_TYPE_STRING, &property))||
         (!dbus_message_iter_open_container(&dict_entry, DBUS_TYPE_VARIANT, DBUS_TYPE_ARRAY_AS_STRING DBUS_TYPE_BYTE_AS_STRING,&variant))||
         (!dbus_message_iter_open_container(&variant, DBUS_TYPE_ARRAY,
               DBUS_TYPE_BYTE_AS_STRING, &array))||
         (!dbus_message_iter_append_fixed_array(&array, DBUS_TYPE_BYTE, value, length))||
         (!dbus_message_iter_close_container(&variant, &array))||
         (!dbus_message_iter_close_container(&dict_entry, &variant))||
         (!dbus_message_iter_close_container(iter, &dict_entry)))
   {
      isSuccess = false;
   }
   return isSuccess;
}

bool insert_map_element(::std::map<::std::string, ::asf::dbus::DBusVariant>& map, const ::std::string& key, int type,
      void *val)
{
   ::asf::dbus::DBusVariant variant;
   bool isSuccess = true;

   DBusMessageIter *iterVariant = variant.getWriteIterator();

   if(!dbus_message_iter_append_basic(iterVariant, type, val))
   {
      isSuccess = false;
   }
   map.insert(std::pair<::std::string, ::asf::dbus::DBusVariant>(key, variant));
   return isSuccess;
}

void *getMemAlloc(size_t size)
{
   void* memPtr = malloc(size);
   if (memPtr)
   {
      memset(memPtr, 0, size);
   }
   return memPtr;
}

void* getDbusIterValue(DBusMessageIter* iter)
{
   void* pVal = NULL;
   int type = dbus_message_iter_get_arg_type(iter);
   if (type == DBUS_TYPE_INVALID)
   {
      return NULL;
   }
   switch (type)
   {
      case DBUS_TYPE_STRING:
      case DBUS_TYPE_SIGNATURE:
      case DBUS_TYPE_OBJECT_PATH:
      {
         char* val;
         dbus_message_iter_get_basic(iter, &val);
         if (val)
         {
            char* cpVal = (char*) getMemAlloc(strlen(val) + 1);
            if (cpVal)
            {
               strcpy(cpVal, val);
               return cpVal;
            }
         }
         break;
      }
      case DBUS_TYPE_INT16:
      case DBUS_TYPE_UINT16:
      {
         dbus_uint16_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = getMemAlloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_INT32:
      case DBUS_TYPE_UINT32:
      {
         dbus_uint32_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = getMemAlloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_INT64:
      case DBUS_TYPE_UINT64:
      {
         dbus_uint64_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = getMemAlloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_DOUBLE:
      {
         double val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = getMemAlloc(sizeof(val));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(val));
         }
         break;
      }
      case DBUS_TYPE_BYTE:
      {
         unsigned char val;
         dbus_message_iter_get_basic(iter, &val);
         int iVal = static_cast< int >(val);
         pVal = getMemAlloc(sizeof(int));
         if (pVal != NULL)
         {
            memcpy(pVal, &iVal, sizeof(val));
         }
         break;
      }

      case DBUS_TYPE_BOOLEAN:
      {
         dbus_bool_t val;
         dbus_message_iter_get_basic(iter, &val);
         pVal = getMemAlloc(sizeof(bool));
         if (pVal != NULL)
         {
            memcpy(pVal, &val, sizeof(bool));
         }
         return pVal;
         break;
      }
      case DBUS_TYPE_VARIANT:
      {
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         void* vpVal = getDbusIterValue(&subiter);
         return vpVal;
      }
      case DBUS_TYPE_DICT_ENTRY:
      {
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         getDbusIterValue(&subiter);
         dbus_message_iter_next(&subiter);
         void* vpVal = getDbusIterValue(&subiter);
         return vpVal;
      }
      case DBUS_TYPE_ARRAY:
      case DBUS_TYPE_STRUCT:
      {
         int current_type;
         DBusMessageIter subiter;
         dbus_message_iter_recurse(iter, &subiter);
         void* vpVal = NULL;
         while ((current_type = dbus_message_iter_get_arg_type(&subiter))
               != DBUS_TYPE_INVALID)
         {
            vpVal = getDbusIterValue(&subiter);
            dbus_message_iter_next(&subiter);
            if (dbus_message_iter_get_arg_type(&subiter) != DBUS_TYPE_INVALID)
            {
               //TO DO:
            }
         }
         return vpVal;
         break;
      }

      default:
      {
         return pVal;
      }
   }
   return pVal;
}

bool init_variant(::asf::dbus::DBusVariant& variant, bool value)
{
   DBusMessageIter *iter = variant.getWriteIterator();
   bool isSuccess = false;
   dbus_bool_t val = value;
   isSuccess = dbus_message_iter_append_basic(iter, DBUS_TYPE_BOOLEAN, &val);
   return isSuccess;
}

} //namespace bosch
} //namespace org

/** @} */
