/**
 * @file WpaSupplicantClient.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include <errno.h>
#include "DbusIfTypes.h"
#include "asf/dbus/DBusVariant.h"
#include "DBusProxyFactory.h"
#include "IWpaSupplicantIfaceProxyIf.h"
#include "IWpaSupplicantProxyIf.h"
#include "IWpaSupplicantBssProxyIf.h"
#include "IOrgFreedesktopDBusProxyIf.h"
#include "ILoopbackProxyIf.h"
#include "BasicServiceSet.h"
#include "WBLPortsDefines.h"
#include "WpaSupplicantClient.h"
#include "ISO3166CountryInfo.h"

#define WLAN_EID_COUNTRY	07

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/Regulation", cWpaSupplicantClient, Debug);

static const std::string sWlanIface = "wlan0";
static const std::string sWpaSuppBusName = "fi.w1.wpa_supplicant1";
static const std::string sWpaSuppInterfaceIfName = "fi.w1.wpa_supplicant1.Interface";
static const std::string sWpaSuppBSSIfName = "fi.w1.wpa_supplicant1.BSS";
static const std::string sWpaSuppObjPath = "/fi/w1/wpa_supplicant1";
static const std::string sOrgFreedesktopDBusPropIfName = "org.freedesktop.DBus.Properties";

cWpaSupplicantClient::cWpaSupplicantClient() :
		cRegulatoryChannel(REGULATORY_CHANNEL_WPA_SUPPLICANT)
{
	_wpaSuppIfaceServiceAvailability = false;
	createWpaSuppProxy(sWpaSuppObjPath, DBUS_BUS_SYSTEM, PROXY_TYPE_STATIC);
}

int cWpaSupplicantClient::deregisterBSSProxy(cBasicServiceSet &bss)
{
	int iRet;
	cIWpaSupplicantBssProxyIf *bssProxy;
	cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;

	LOG_DEBUG ("DeRegistering the BSS [objpath: %s, ifName: %s] [type: %d]: %s",
			bss.GetObjPath().c_str(), bss.GetIfName().c_str(), bss.GetBusType(), bss.GetCountry().c_str());

	bssProxy = DBusProxyFactory::getInstance()->getWpaSupplicantBssProxyIf();
	iRet = bssProxy->destroyProxy(PROXY_DESTROY_ONE, sWpaSuppBusName, bss.GetBusType(), bss.GetObjPath());
	if (iRet < 0) {
		LOG_ERROR ("Failed to destroy proxy to object: [objPath: %s] [BusType: %d] %s/%d",
				bss.GetObjPath().c_str(), bss.GetBusType(), strerror(-iRet), -iRet);
	} else {

		orgPropProxy = DBusProxyFactory::getInstance()->getOrgFreedesktopDBusPropProxyIf();
		iRet = orgPropProxy->destroyProxy(PROXY_DESTROY_ONE, sWpaSuppBusName, bss.GetBusType(), bss.GetObjPath());
		if (iRet < 0) {
			LOG_ERROR ("Failed to destroy proxy to the interface [%s] of object: [objPath: %s] "
					"[BusType: %d] %s/%d", sOrgFreedesktopDBusPropIfName.c_str(), bss.GetObjPath().c_str(),
					bss.GetBusType(), strerror(-iRet), -iRet);
		}

		iRet = bssProxy->setCallbackIf(sWpaSuppBusName, bss.GetObjPath(), bss.GetBusType(), nullptr);
		if (iRet < 0) {
			LOG_ERROR ("Failed to remove the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
					bss.GetObjPath().c_str(), bss.GetBusType(), strerror(-iRet), -iRet);
		}
	}

	return iRet;
}

int cWpaSupplicantClient::deregisterWpaSupplicantProxy(const cWpaSupplicantIfaceData &iface)
{
	int iRet;
	cIWpaSupplicantIfaceProxyIf *ifProxy;
	cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;

	LOG_DEBUG ("DeRegistering the wpa_supp iface [objpath: %s, ifName: %s] [type: %d]: %s",
			iface._sObjPath.c_str(), iface._sIfName.c_str(), iface._BusType,
			iface._sCountry.c_str());

	ifProxy = DBusProxyFactory::getInstance()->getWpaSupplicantIfaceProxyIf();
	iRet = ifProxy->destroyProxy(PROXY_DESTROY_ONE, sWpaSuppBusName, iface._BusType, iface._sObjPath);
	if (iRet < 0) {
		LOG_ERROR ("Failed to destroy proxy to object: [objPath: %s] [BusType: %d] %s/%d",
				iface._sObjPath.c_str(), iface._BusType, strerror(-iRet), -iRet);
	} else {

		orgPropProxy = DBusProxyFactory::getInstance()->getOrgFreedesktopDBusPropProxyIf();
		iRet = orgPropProxy->destroyProxy(PROXY_DESTROY_ONE, sWpaSuppBusName, iface._BusType, iface._sObjPath);
		if (iRet < 0) {
			LOG_ERROR ("Failed to destroy proxy to the interface [%s] of object: [objPath: %s] "
					"[BusType: %d] %s/%d", sOrgFreedesktopDBusPropIfName.c_str(), iface._sObjPath.c_str(),
					iface._BusType, strerror(-iRet), -iRet);
		}

		iRet = ifProxy->setCallbackIf(sWpaSuppBusName, iface._sObjPath, iface._BusType, nullptr);
		if (iRet < 0) {
			LOG_ERROR ("Failed to remove the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
					iface._sObjPath.c_str(), iface._BusType, strerror(-iRet), -iRet);
		}
	}

	return iRet;
}

cWpaSupplicantClient::~cWpaSupplicantClient()
{
   try
   {

      int iRet;
      cIWpaSupplicantProxyIf *wpaSuppProxy;
      cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;
      std::vector< cBasicServiceSet >::iterator bssit;
      std::vector< cWpaSupplicantIfaceData >::iterator ifaceit;

      for (bssit = _wpaSupplicantBssProxies.begin(); bssit < _wpaSupplicantBssProxies.end();
            bssit++)
         deregisterBSSProxy(*bssit);

      for (ifaceit = _wpaSupplicantIfaceProxies.begin(); ifaceit < _wpaSupplicantIfaceProxies.end();
            ifaceit++)
         deregisterWpaSupplicantProxy(*ifaceit);

      wpaSuppProxy = DBusProxyFactory::getInstance()->getWpaSupplicantProxyIf();
      iRet = wpaSuppProxy->destroyProxy(PROXY_DESTROY_ONE, sWpaSuppBusName, DBUS_BUS_SYSTEM, sWpaSuppObjPath);
      if (iRet == 0) {
         iRet = wpaSuppProxy->setCallbackIf(sWpaSuppBusName, sWpaSuppObjPath, DBUS_BUS_SYSTEM, this);
         if (iRet < 0) {
            LOG_ERROR ("Failed to remove the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
                  sWpaSuppObjPath.c_str(), DBUS_BUS_SYSTEM, strerror(-iRet), -iRet);
         }

         orgPropProxy = DBusProxyFactory::getInstance()->getOrgFreedesktopDBusPropProxyIf();
         iRet = orgPropProxy->destroyProxy(PROXY_DESTROY_ONE, sWpaSuppBusName, DBUS_BUS_SYSTEM, sWpaSuppObjPath);
         if (iRet < 0) {
            LOG_ERROR ("Failed to delete the Proxy for: %s [Error : %s/%d]", sWpaSuppObjPath.c_str(),
                  strerror (-iRet), -iRet);
         }
      } else {
         LOG_ERROR ("Failed to create proxy to object: [objPath: %s] [BusType: %d] %s/%d",
               sWpaSuppObjPath.c_str(), DBUS_BUS_SYSTEM, strerror(-iRet), -iRet);
      }

   }
   catch(...){}
}

int cWpaSupplicantClient::registerWpaConnectionNotification(cWpaSupplicantConnectionNotifier *listener)
{
	std::vector<cWpaSupplicantConnectionNotifier *>::iterator it;

	if (!listener)
		return -EINVAL;

	LOG_INFO ("Registering the Wpa Connection Observer: %p", listener);
	for (it = _wpaConnectionNotifierClients.begin(); it < _wpaConnectionNotifierClients.end(); it++)
		if (listener == *it)
			return -EALREADY;

	_wpaConnectionNotifierClients.push_back (listener);
	return 0;
}

int cWpaSupplicantClient::unRegisterWpaConnectionNotification(cWpaSupplicantConnectionNotifier *listener)
{
	bool exist = false;
	std::vector<cWpaSupplicantConnectionNotifier *>::iterator it;

	if (!listener)
		return -EINVAL;

	LOG_INFO ("UnRegistering the Wpa Connection Observer: %p", listener);
	for (it = _wpaConnectionNotifierClients.begin(); it < _wpaConnectionNotifierClients.end(); it++)
		if (listener == *it) {
			exist = true;
			break;
		}

	if (exist) {
		_wpaConnectionNotifierClients.erase(it);
		return 0;
	}

	return -ENOENT;
}

int cWpaSupplicantClient::registerBSSNotification(cBSSNotifier *listener)
{
	std::vector<cBSSNotifier *>::iterator it;

	if (!listener)
		return -EINVAL;

	LOG_INFO ("Registering the BSS Observer: %p", listener);
	for (it = _bssNotifierClients.begin(); it < _bssNotifierClients.end(); it++)
		if (listener == *it)
			return -EALREADY;

	_bssNotifierClients.push_back (listener);
	return 0;
}

int cWpaSupplicantClient::unRegisterBSSNotification(cBSSNotifier *listener)
{
	bool exist = false;
	std::vector<cBSSNotifier *>::iterator it;

	if (!listener)
		return -EINVAL;

	LOG_INFO ("UnRegistering the BSS Observer: %p", listener);
	for (it = _bssNotifierClients.begin(); it < _bssNotifierClients.end(); it++)
		if (listener == *it) {
			exist = true;
			break;
		}

	if (exist) {
		_bssNotifierClients.erase(it);
		return 0;
	}

	return -ENOENT;
}

void cWpaSupplicantClient::updateWpaConnectionStatus(eWpaSuppConnectionState_t state, const ::std::string &ifname,
			const ::std::string &associatedBss)
{
	std::vector<cWpaSupplicantConnectionNotifier *>::iterator it;
	for (it = _wpaConnectionNotifierClients.begin(); it < _wpaConnectionNotifierClients.end(); it++)
		(*it)->wpaConnectionNotification(state, ifname, associatedBss);
}

void cWpaSupplicantClient::updateWpaDisconnectionStatus(int status, const ::std::string &ifName)
{
	std::vector<cWpaSupplicantConnectionNotifier *>::iterator it;
	for (it = _wpaConnectionNotifierClients.begin(); it < _wpaConnectionNotifierClients.end(); it++)
		(*it)->wpaDisconnectionStatus(status, ifName);
}

void cWpaSupplicantClient::updateBSSNotification(eBssEvent_t event, const ::std::string &objPath,
		const cBasicServiceSet &bss)
{
	std::vector<cBSSNotifier *>::iterator it;

	if (objPath.empty())
		return;

	for (it = _bssNotifierClients.begin(); it < _bssNotifierClients.end(); it++)
		(*it)->BSSNotification(event, objPath, bss);
}

int cWpaSupplicantClient::changeRegulatory(const ::std::string &country)
{
	act_t token;
	bool channel = false;
	std::string property = "Country";
	const char* data = NULL;
	::asf::dbus::DBusVariant value;
	cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;
	std::vector< cWpaSupplicantIfaceData >::iterator it;

	if (country.empty())
		return -EINVAL;

	LOG_INFO ("Changing the Regulatory information to: %s", country.c_str());
	for (it = _wpaSupplicantIfaceProxies.begin(); it < _wpaSupplicantIfaceProxies.end(); it++) {
		if (!(*it)._sObjPath.empty()) {
			channel = true;
			break;
		}
	}

	if (channel) {
		LOG_INFO ("Using the WLAN interface \"%s\" objpath: %s", (*it)._sIfName.c_str(),
				(*it)._sObjPath.c_str());
		data = country.c_str();
		value.init();
		dbus_message_append_args(value.getMessage(), DBUS_TYPE_STRING, &data, DBUS_TYPE_INVALID);
		orgPropProxy = DBusProxyFactory::getInstance()->getOrgFreedesktopDBusPropProxyIf();
		token = orgPropProxy->sendSetRequest(sWpaSuppBusName, (*it)._sObjPath, (*it)._BusType,
				sWpaSuppInterfaceIfName, property, value, *this);
		if (token == DEFAULT_ACT)
			return -ENOTCONN;
		return 0;
	}

	return -ENOTCONN;
}

void cWpaSupplicantClient::notifyWpaSupplicantAvailabilityStatus()
{
	bool channel = false;
	std::vector< cWpaSupplicantIfaceData >::iterator it;

	for (it = _wpaSupplicantIfaceProxies.begin(); it < _wpaSupplicantIfaceProxies.end(); it++) {
		if (!(*it)._sObjPath.empty()) {
			channel = true;
			break;
		}
	}

	if (channel) {
		if (!_wpaSuppIfaceServiceAvailability) {
			_wpaSuppIfaceServiceAvailability = channel;
			notifyRegChanStatus(channel);
		}
	} else {
		if (_wpaSuppIfaceServiceAvailability) {
			_wpaSuppIfaceServiceAvailability = channel;
			notifyRegChanStatus(channel);
		}
	}
}

void cWpaSupplicantClient::onWpaSupplicantServiceAvailable(const ::std::string& busName,
		const ::std::string& objPath, const ::DBusBusType busType,
		const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState)
{
   (void) previousState;
   (void) currentState;
	act_t token;
	cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;

	orgPropProxy = createDbusPropProxy(sWpaSuppIfacePort, sWpaSuppBusName, objPath, busType,
			PROXY_TYPE_WIRED_AT_RUNTIME);
	if (orgPropProxy) {
		token = orgPropProxy->sendGetRequest(busName, objPath, busType, sWpaSuppBusName, "Interfaces",
				*this);
		if (DEFAULT_ACT == token)
			LOG_ERROR ("Failed to post the Get Interfaces request for: %s", objPath.c_str());
	}
}

void cWpaSupplicantClient::onWpaSupplicantServiceUnavailable(const ::std::string& busName,
		const ::std::string& objPath, const ::DBusBusType busType,
		const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState)
{
	(void)previousState;
	(void)currentState;
	int iRet;
	cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;

	orgPropProxy = DBusProxyFactory::getInstance()->getOrgFreedesktopDBusPropProxyIf();
	iRet = orgPropProxy->destroyProxy(PROXY_DESTROY_ONE, busName, busType, objPath);
	if (iRet < 0) {
		LOG_ERROR ("Failed to delete the Proxy for: %s [Error : %s/%d]", objPath.c_str(),
				strerror (-iRet), -iRet);
	}
}

void cWpaSupplicantClient::onInterfaceAddedError(const ::boost::shared_ptr< cWpaSupplicantProxy > &proxy,
		const ::boost::shared_ptr< cIfaceAddedError > &error)
{
	LOG_ERROR ("Interface Added Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

int cWpaSupplicantClient::getBssFromMacAddr(const std::string &macaddr,
		::std::string &objpath)
{
	std::vector< cBasicServiceSet >::iterator it;

	for (it = _wpaSupplicantBssProxies.begin(); it < _wpaSupplicantBssProxies.end(); it++)
		if (!(*it).getMacAddress().compare(macaddr)) {
			objpath = (*it).GetObjPath();
			return 0;
		}

	return -ENOENT;
}

cBasicServiceSet *cWpaSupplicantClient::getBSSFromObjectPath(const ::std::string &path)
{
	std::vector< cBasicServiceSet >::iterator it;

	for (it = _wpaSupplicantBssProxies.begin(); it < _wpaSupplicantBssProxies.end(); it++)
		if (!(*it).GetObjPath().compare(path))
			return &(*it);

	return NULL;
}

cWpaSupplicantIfaceData *cWpaSupplicantClient::getWpaSupplicantfromIfName(const ::std::string &name)
{
	std::vector< cWpaSupplicantIfaceData >::iterator it;

	for (it = _wpaSupplicantIfaceProxies.begin(); it < _wpaSupplicantIfaceProxies.end(); it++)
		if (!(*it)._sIfName.compare(name))
			return &(*it);

	return NULL;
}

::std::vector< cBasicServiceSet > &cWpaSupplicantClient::getBSSes()
{
	return _wpaSupplicantBssProxies;
}

int cWpaSupplicantClient::getWpaSupplicantBssData(const ::std::string &objPath, const ::DBusBusType &busType,
		cBasicServiceSet **bssData)
{
	std::vector< cBasicServiceSet >::iterator it;

	if (!bssData)
		return -EINVAL;

	LOG_DEBUG ("Get BSS Data for Object: %s [BusType: %d]", objPath.c_str(),
			busType);

	for (it = _wpaSupplicantBssProxies.begin(); it < _wpaSupplicantBssProxies.end(); it++) {
		if (!(*it).GetObjPath().compare(objPath) && (*it).GetBusType() == busType) {
			*bssData = &(*it);
			return 0;
		}
	}

	return -ENOENT;
}

int cWpaSupplicantClient::getWpaSupplicantData(const ::std::string &objPath, const ::DBusBusType &busType,
		cWpaSupplicantIfaceData **wpaData)
{
	std::vector< cWpaSupplicantIfaceData >::iterator it;

	if (!wpaData)
		return -EINVAL;

	LOG_DEBUG ("Get WpaSupp Data for Object: %s [BusType: %d]", objPath.c_str(),
			busType);

	for (it = _wpaSupplicantIfaceProxies.begin(); it < _wpaSupplicantIfaceProxies.end(); it++) {
		if ((!(*it)._sObjPath.compare(objPath) && (*it)._BusType == busType) ||
				(!(*it)._sIfName.compare(objPath) && (*it)._BusType == busType)) {
			*wpaData = &(*it);
			return 0;
		}
	}

	return -ENOENT;
}

void cWpaSupplicantClient::onInterfaceAddedSignal(const ::boost::shared_ptr< cWpaSupplicantProxy > &proxy,
		const ::boost::shared_ptr< cIfaceAddedSignal > &signal)
{
	int iRet, disconnectReason = 0;
	const char *sIfname = NULL, *sCountry = NULL,
			*currentBss = NULL, *currentNetwork = NULL, *state = NULL;
	::asf::dbus::DBusVariant variant;
	cWpaSupplicantIfaceData oData, *temp = NULL;
	cBasicServiceSet *bss = NULL;
	::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it;
	::std::map< ::std::string, ::asf::dbus::DBusVariant > properties;

	LOG_INFO ("Interface Added by wpa_supplicant: %s", signal->getInterface().c_str());

	iRet = getWpaSupplicantData(signal->getInterface(), proxy->getBusType(), &temp);
	if (iRet < 0) {

		properties = signal->getProperties();
		for(it = properties.begin(); it != properties.end(); ++it) {
			if (!it->first.compare("Ifname"))
				dbus_message_iter_get_basic(it->second.getReadIterator(), &sIfname);
			else if (!it->first.compare("Country"))
				dbus_message_iter_get_basic(it->second.getReadIterator(), &sCountry);
			else if (!it->first.compare("State"))
				dbus_message_iter_get_basic(it->second.getReadIterator(), &state);
			else if (!it->first.compare("CurrentBSS"))
				dbus_message_iter_get_basic(it->second.getReadIterator(), &currentBss);
			else if (!it->first.compare("CurrentNetwork"))
				dbus_message_iter_get_basic(it->second.getReadIterator(), &currentNetwork);
			else if (!it->first.compare("DisconnectReason"))
				dbus_message_iter_get_basic(it->second.getReadIterator(), &disconnectReason);
		}

		LOG_INFO ("Managed Wlan Interface: \"%s\", "
				"Country Information: \"%s\" "
				"state: \"%s\" "
				"currentBss: \"%s\" "
				"currentNetwork: \"%s\" "
				"disconnectReason: %d",
				sIfname, sCountry, state, currentBss, currentNetwork, disconnectReason);

		oData._sIfName = sIfname ? sIfname : "";
		oData._sCountry = sCountry ? sCountry : "";
		oData._currentBSS = currentBss ? currentBss : "";
		oData._disconnectReason = disconnectReason;
		oData._currentNetwork = currentNetwork ? currentNetwork : "";
		oData._state = state ? state : "";
		oData._sObjPath = signal->getInterface();
		oData._BusType = proxy->getBusType();

		iRet = createWpaSuppIfaceProxy(signal->getInterface(), proxy->getBusType(),
				PROXY_TYPE_WIRED_AT_RUNTIME);
		if (!iRet)
			_wpaSupplicantIfaceProxies.push_back(oData);

		if (oData._currentBSS.compare("/")) {
			iRet = getWpaSupplicantBssData(oData._currentBSS, proxy->getBusType(), &bss);
			if (iRet == 0 && bss->getFrequency() != 0) {
				LOG_INFO("Notifying the change in connection for the interface [%s]: %s", oData._sIfName.c_str(),
						oData._currentBSS.c_str());
				updateWpaConnectionStatus(cWpaSupplicantConnectionNotifier::wpaStateToEvent(oData._state),
						oData._sIfName, oData._currentBSS);
			} else {
				LOG_INFO("Interface: %s connection status due for : %s", oData._sIfName.c_str(),
						oData._currentBSS.c_str());
				oData._wpaConnStatusDue = true;
			}
		}

	} else {
		LOG_INFO ("Already an existing proxy entry to the object: [objPath: %s] [BusType: %d]",
				proxy->getDBusObjectPath().c_str(), proxy->getBusType());
	}
}

void cWpaSupplicantClient::onInterfaceRemovedError(const ::boost::shared_ptr<cWpaSupplicantProxy> &proxy,
		const ::boost::shared_ptr<cIfaceRemovedError> &error)
{
	LOG_ERROR ("Interface Removed Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

void cWpaSupplicantClient::onInterfaceRemovedSignal(const ::boost::shared_ptr<cWpaSupplicantProxy> &proxy,
		const ::boost::shared_ptr<cIfaceRemovedSignal> &signal)
{
	int iRet;
	cWpaSupplicantIfaceData *temp = NULL, tempobj;

	LOG_INFO ("Interface Removed by wpa_supplicant: %s", signal->getInterface().c_str());

	iRet = getWpaSupplicantData(signal->getInterface(), proxy->getBusType(), &temp);
	if (iRet == 0) {
		tempobj = *temp;
		iRet = deregisterWpaSupplicantProxy(*temp);
		if (iRet < 0) {
			LOG_ERROR ("Failed to remove the proxy for : %s [bustype: %d], error: %s/%d",
					signal->getInterface().c_str(), proxy->getBusType(), strerror(-iRet), -iRet);
		}
		iRet = removeWpaSuppIfacedata(tempobj);
		if (iRet < 0) {
			LOG_ERROR ("Failed to remove the wpa supp iface object : %s [bus type: %d]",
					signal->getInterface().c_str(), proxy->getBusType());
		}

		notifyWpaSupplicantAvailabilityStatus();
	} else {
		LOG_INFO ("No Proxy created for the object: [objPath: %s] [BusType: %d]",
				signal->getInterface().c_str(), proxy->getBusType());
	}
}

void cWpaSupplicantClient::onGetInterfaceError(const ::boost::shared_ptr< cWpaSupplicantProxy > &proxy,
		const ::boost::shared_ptr<cGetIfaceError> &error)
{
	LOG_ERROR ("Get Interface Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

void cWpaSupplicantClient::onGetInterfaceResponse(const ::boost::shared_ptr< cWpaSupplicantProxy > &proxy,
		const ::boost::shared_ptr<cGetIfaceResponse> &response)
{
	int iRet;
	cWpaSupplicantIfaceData oData, *temp = NULL;

	LOG_INFO ("Get Interface Response: %s", response->getPath().c_str());

	iRet = getWpaSupplicantData(response->getPath(), proxy->getBusType(), &temp);
	if (iRet < 0) {

		oData._BusType = proxy->getBusType();
		oData._sIfName = sWlanIface;
		oData._sObjPath = response->getPath();

		iRet = createWpaSuppIfaceProxy(response->getPath(), proxy->getBusType(),
				PROXY_TYPE_WIRED_AT_RUNTIME);
		if (!iRet)
			_wpaSupplicantIfaceProxies.push_back(oData);
	} else {
		LOG_INFO ("A Proxy exists already for the object: [objPath: %s] [BusType: %d]",
				proxy->getDBusObjectPath().c_str(), proxy->getBusType());
	}

	LOG_INFO ("get exit");
}

void cWpaSupplicantClient::onWpaSupplicantIfaceServiceAvailable(const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const ::asf::core::ServiceState previousState,
		const ::asf::core::ServiceState currentState)
{
   (void) previousState;
   (void) currentState;
	act_t token;
	cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;

	orgPropProxy = createDbusPropProxy(sWpaSuppIfacePort, busName, objPath, busType,
			PROXY_TYPE_WIRED_AT_RUNTIME);
	if (orgPropProxy) {
		token = orgPropProxy->sendGetAllRequest(busName, objPath, busType, sWpaSuppInterfaceIfName, *this);
		if (DEFAULT_ACT == token)
			LOG_ERROR ("Failed to get the currently available scan results for interface: %s",
					objPath.c_str());
		notifyWpaSupplicantAvailabilityStatus();
	}
}

void cWpaSupplicantClient::onWpaSupplicantIfaceServiceUnavailable(const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const ::asf::core::ServiceState previousState,
		const ::asf::core::ServiceState currentState)
{
   (void) previousState;
   (void) currentState;
	LOG_INFO ("onWpaSupplicantIfaceServiceUnavailable [busName: %s] [objpath: %s] [busType: %d]",
			busName.c_str(), objPath.c_str(), busType);
	notifyWpaSupplicantAvailabilityStatus();
}

void cWpaSupplicantClient::onBSSAddedError(const ::boost::shared_ptr<cWpaSupplicantIfaceProxy> &proxy,
		const ::boost::shared_ptr<cBssAddedError> &error)
{
	LOG_ERROR ("BSS Added Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

void cWpaSupplicantClient::onBSSAddedSignal(const ::boost::shared_ptr<cWpaSupplicantIfaceProxy> &proxy,
		const ::boost::shared_ptr<cBssAddedSignal> &signal)
{
	int iRet;
	std::string bss;
	cBasicServiceSet oBss, *bssTemp = NULL;
	cWpaSupplicantIfaceData *temp = NULL;

	LOG_INFO ("BSS Added Signal [proxy object: %s, interface: %s] [type: %d]: %s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), signal->getPath().c_str());

	iRet = getWpaSupplicantData(proxy->getDBusObjectPath(), proxy->getBusType(), &temp);
	if (iRet < 0) {
		LOG_ERROR ("No Proxy created for object: [objPath: %s] [BusType: %d]",
				proxy->getDBusObjectPath().c_str(), proxy->getBusType());
		return;
	}

	bss = signal->getPath();
	iRet = getWpaSupplicantBssData(bss, proxy->getBusType(), &bssTemp);
	if (iRet < 0) {

		oBss.SetIfName(temp->_sIfName);
		oBss.SetBusType(proxy->getBusType());
		oBss.SetObjPath(bss);

		iRet = createWpaSuppBssProxy(bss, proxy->getBusType(), PROXY_TYPE_WIRED_AT_RUNTIME);
		if (!iRet) {
			_wpaSupplicantBssProxies.push_back(oBss);
//			updateBSSNotification(EVENT_BSS_ADDED, bss, oBss);
		}
	} else {
		LOG_INFO ("A Proxy exists already for the object: [objPath: %s] "
				"[BusType: %d]", bss.c_str(), proxy->getBusType());
	}
}

void cWpaSupplicantClient::onBSSRemovedError(const ::boost::shared_ptr<cWpaSupplicantIfaceProxy> &proxy,
		const ::boost::shared_ptr<cBssRemovedError> &error)
{
	LOG_ERROR ("BSS Removed Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

int cWpaSupplicantClient::removeBSSdata(cBasicServiceSet &bss)
{
	int iRet = -ENOENT;
	bool found = false;
	std::vector< cBasicServiceSet >::iterator it;

	for (it = _wpaSupplicantBssProxies.begin(); it < _wpaSupplicantBssProxies.end(); it++) {
		if (!(*it).GetObjPath().compare(bss.GetObjPath()) && (*it).GetBusType() == bss.GetBusType()) {
			found = true;
			break;
		}
	}

	if (found) {
		_wpaSupplicantBssProxies.erase(it);
		iRet = 0;
	} else {
		LOG_ERROR("No Entry for \"%s\" [bustype: %s] in the local BSS table", bss.GetObjPath().c_str(),
				bss.GetBusType());
	}

	return iRet;
}

int cWpaSupplicantClient::removeWpaSuppIfacedata(cWpaSupplicantIfaceData &iface)
{
	int iRet = -ENOENT;
	bool found = false;
	std::vector< cWpaSupplicantIfaceData >::iterator it;

	for (it = _wpaSupplicantIfaceProxies.begin(); it < _wpaSupplicantIfaceProxies.end(); it++) {
		if (!(*it)._sObjPath.compare(iface._sObjPath) && (*it)._BusType == iface._BusType) {
			found = true;
			break;
		}
	}

	if (found) {
		_wpaSupplicantIfaceProxies.erase(it);
		iRet = 0;
	} else {
		LOG_ERROR("No Entry for \"%s\" [bustype: %s] in the local iface table", iface._sObjPath.c_str(),
				iface._BusType);
	}

	return iRet;
}

void cWpaSupplicantClient::onBSSRemovedSignal(const ::boost::shared_ptr<cWpaSupplicantIfaceProxy> &proxy,
		const ::boost::shared_ptr<cBssRemovedSignal> &signal)
{
	int iRet;
	std::string bss;
	cBasicServiceSet *bssTemp = NULL, temp;

	LOG_INFO ("BSS Removed Signal [proxy object: %s, interface: %s] [type: %d]: %s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), signal->getPath().c_str());

	bss = signal->getPath();
	iRet = getWpaSupplicantBssData(bss, proxy->getBusType(), &bssTemp);
	if (iRet == 0) {
		temp = *bssTemp;
		iRet = deregisterBSSProxy(*bssTemp);
		if (iRet < 0) {
			LOG_ERROR ("Failed to remove the proxy for : %s [bustype: %d], error: %s/%d",
					signal->getPath().c_str(), proxy->getBusType(), strerror(-iRet), -iRet);
		}
		iRet = removeBSSdata(temp);
		if (iRet == 0)
			updateBSSNotification(EVENT_BSS_REMOVED, bss, temp);
		else {
			LOG_ERROR ("Failed to remove the bss object : %s [bus type: %d]", signal->getPath().c_str(),
					proxy->getBusType());
		}
	} else {
		LOG_INFO ("No Proxy created for the object: [objPath: %s] [BusType: %d]",
				bss.c_str(), proxy->getBusType());
	}
}

void cWpaSupplicantClient::onGetError(const ::boost::shared_ptr< cDBusPropertiesProxy > &proxy,
		const ::boost::shared_ptr< cDBusPropGetError > &error)
{
	LOG_ERROR ("On Get Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

int cWpaSupplicantClient::createWpaSuppIfaceProxy(const ::std::string &objPath, const ::DBusBusType &busType,
			eProxyType eType)
{
	int iRet;
	cIWpaSupplicantIfaceProxyIf *wpaSuppIfaceProxy;

	wpaSuppIfaceProxy = DBusProxyFactory::getInstance()->getWpaSupplicantIfaceProxyIf();
	iRet = wpaSuppIfaceProxy->createProxy(sWpaSuppIfacePort, sWpaSuppBusName, objPath,
			busType, eType);
	if (iRet == 0) {
		iRet = wpaSuppIfaceProxy->setCallbackIf(sWpaSuppBusName, objPath, busType, this);
		if (iRet < 0) {
			LOG_ERROR ("Failed to add the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
					objPath.c_str(), busType, strerror(-iRet), -iRet);
		}
	} else {
		LOG_ERROR ("Failed to create proxy to object: [objPath: %s] [BusType: %d] %s/%d",
				objPath.c_str(), busType, strerror(-iRet), -iRet);
	}

	return iRet;
}

int cWpaSupplicantClient::createWpaSuppProxy(const ::std::string &objPath, const ::DBusBusType &busType,
				eProxyType eType)
{
	int iRet;
	cIWpaSupplicantProxyIf *wpaSuppProxy;

	wpaSuppProxy = DBusProxyFactory::getInstance()->getWpaSupplicantProxyIf();
	iRet = wpaSuppProxy->createProxy(sWpaSuppPort, sWpaSuppBusName, objPath,
			busType, eType);
	if (iRet == 0) {
		iRet = wpaSuppProxy->setCallbackIf(sWpaSuppBusName, objPath, busType, this);
		if (iRet < 0) {
			LOG_ERROR ("Failed to add the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
					objPath.c_str(), busType, strerror(-iRet), -iRet);
		}
	} else {
		LOG_ERROR ("Failed to create proxy to object: [objPath: %s] [BusType: %d] %s/%d",
				objPath.c_str(), busType, strerror(-iRet), -iRet);
	}

	return iRet;
}

int cWpaSupplicantClient::createWpaSuppBssProxy(const ::std::string &objPath, const ::DBusBusType &busType,
					eProxyType eType)
{
	int iRet;
	cIWpaSupplicantBssProxyIf *bssProxy;

	bssProxy = DBusProxyFactory::getInstance()->getWpaSupplicantBssProxyIf();
	iRet = bssProxy->createProxy(sWpaSuppBssPort, sWpaSuppBusName, objPath,
			busType, eType);
	if (iRet == 0) {
		iRet = bssProxy->setCallbackIf(sWpaSuppBusName, objPath, busType, this);
		if (iRet < 0) {
			LOG_ERROR ("Failed to add the callback to the proxy [objPath: %s] [BusType: %d] %s/%d",
					objPath.c_str(), busType, strerror(-iRet), -iRet);
		}
	} else {
		LOG_ERROR ("Failed to create proxy to object: [objPath: %s] [BusType: %d] %s/%d",
				objPath.c_str(), busType, strerror(-iRet), -iRet);
	}

	return iRet;
}

cIOrgFreedesktopDBusPropertiesProxyIf *cWpaSupplicantClient::createDbusPropProxy(const ::std::string &portName,
		const ::std::string &busName, const ::std::string &objPath, const ::DBusBusType &busType, eProxyType eType)
{
	int iRet;
	cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;

	orgPropProxy = DBusProxyFactory::getInstance()->getOrgFreedesktopDBusPropProxyIf();
	iRet = orgPropProxy->createProxy(portName, busName, objPath, busType, eType, this);
	if (iRet < 0) {
		LOG_ERROR ("Failed to create proxy to the interface [%s] of object: [objPath: %s] "
				"[BusType: %d] %s/%d", sOrgFreedesktopDBusPropIfName.c_str(), objPath.c_str(),
				busType, strerror(-iRet), -iRet);
		return NULL;
	}

	return orgPropProxy;
}

void cWpaSupplicantClient::onGetResponse(const ::boost::shared_ptr< cDBusPropertiesProxy > &proxy,
		const ::boost::shared_ptr< cDBusPropGetResponse > &response)
{
	int iRet;
	bool next = true;
	cWpaSupplicantIfaceData oData, *temp = NULL;
	cBasicServiceSet oBss, *bssTemp = NULL;
	char *bss = NULL;
    ::DBusMessageIter subIter;
    ::std::string signature;
    ::asf::dbus::DBusVariant reply;

    LOG_INFO ("Get Response: %s [interface: %s]", proxy->getDBusObjectPath().c_str(),
    		proxy->getInterfaceName().c_str());

    if (proxy->getDBusObjectPath().compare(sWpaSuppObjPath)) {
		iRet = getWpaSupplicantData(proxy->getDBusObjectPath(), proxy->getBusType(), &temp);
		if (iRet < 0) {
			LOG_ERROR ("No Proxy created for object: [objPath: %s] [BusType: %d]",
					proxy->getDBusObjectPath().c_str(), proxy->getBusType());
			return;
		}
    }

    reply = response->getValue();
    reply.getSignature(signature);

    LOG_DEBUG ("Get Response Signature: %s", signature.c_str());
    if (!signature.compare("ao")) {

    	dbus_message_iter_recurse(reply.getReadIterator(), &subIter);
    	while (true == next) {
    		dbus_message_iter_get_basic(&subIter, &bss);

    		LOG_INFO ("Available Wpa_Supplicant Object: %s", bss);
    		if (!proxy->getDBusObjectPath().compare(sWpaSuppObjPath)) {

        		iRet = getWpaSupplicantData(bss, proxy->getBusType(), &temp);
        		if (iRet < 0) {

            		oData._BusType = proxy->getBusType();
            		oData._sObjPath = bss;
            		iRet = createWpaSuppIfaceProxy(bss, proxy->getBusType(), PROXY_TYPE_WIRED_AT_RUNTIME);
            		if (!iRet)
            			_wpaSupplicantIfaceProxies.push_back(oData);
        		} else {
        			LOG_INFO ("A Proxy exists already for the object: [objPath: %s] "
        					"[BusType: %d]", bss, proxy->getBusType());
        		}

    		} else {
        		iRet = getWpaSupplicantBssData(bss, proxy->getBusType(), &bssTemp);
        		if (iRet < 0) {

            		oBss.SetIfName(temp->_sIfName);
            		oBss.SetBusType(proxy->getBusType());
            		oBss.SetObjPath(bss);

            		iRet = createWpaSuppBssProxy(bss, proxy->getBusType(), PROXY_TYPE_WIRED_AT_RUNTIME);
            		if (!iRet) {
            			LOG_INFO("Pushing: %s", bss);
            			_wpaSupplicantBssProxies.push_back(oBss);
//            			updateBSSNotification(EVENT_BSS_ADDED, bss, oBss);
            		}
        		} else {
        			LOG_INFO ("A Proxy exists already for the object: [objPath: %s] "
        					"[BusType: %d]", bss, proxy->getBusType());
        		}
    		}

    		if (!dbus_message_iter_next(&subIter))
    			next = false;
    	}
    }
}

void cWpaSupplicantClient::onWpaSupplicantBssServiceAvailable(const ::std::string &busName,
		const ::std::string &objPath, const ::DBusBusType busType, const ::asf::core::ServiceState previousState,
		const ::asf::core::ServiceState currentState)
{
	(void) previousState;
	(void) currentState;
	act_t token;
	cIOrgFreedesktopDBusPropertiesProxyIf *orgPropProxy;

	orgPropProxy = createDbusPropProxy(sWpaSuppBssPort, busName, objPath, busType,
			PROXY_TYPE_WIRED_AT_RUNTIME);
	if (orgPropProxy) {
		token = orgPropProxy->sendGetAllRequest(busName, objPath, busType, sWpaSuppBSSIfName, *this);
		if (DEFAULT_ACT == token)
			LOG_ERROR ("Failed to get the elements of the available BSS: %s", objPath.c_str());
	}
}

void cWpaSupplicantClient::onWpaSupplicantBssServiceUnAvailable(const ::std::string &busName, const ::std::string &objPath,
		const ::DBusBusType busType, const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState)
{
	(void) previousState;
	(void) currentState;
	LOG_INFO ("onWpaSupplicantBssServiceUnAvailable [busName: %s] [objpath: %s] [busType: %d]",
				busName.c_str(), objPath.c_str(), busType);
}

void cWpaSupplicantClient::onGetAllError(const ::boost::shared_ptr< cDBusPropertiesProxy > &proxy,
		const ::boost::shared_ptr< cDBusPropGetAllError >& error)
{
	LOG_ERROR ("Get All Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

int cWpaSupplicantClient::getCountryElemfromIEs(const uint8 *ielements, const int &length,
		const int element_id, ::std::string &sCountry)
{
	char country[3];
	const uint8 *pos, *end;

	pos = ielements;
	end = ielements + length;

	memset(country, 0, sizeof (country));
	while (end - pos > 1) {
		if (2 + pos[1] > end - pos)
			break;
		if (pos[0] == element_id && pos[1] >= 3) {
			memcpy(country, pos+2, 2);
			break;
		}
		pos += 2 + pos[1];
	}

	if (strlen(country)) {
		if (cISO3166CountryInfo::getInstance()->isValidAlpha2(country) == 0) {
			sCountry.assign(country);
			return 0;
		}
	}

	return -ENOENT;
}

int cWpaSupplicantClient::getBssKeyManagement(::DBusMessageIter *iter, uint32_t *keyMgmt)
{
	const char *ikey = NULL, *keyvalue = NULL;
	::DBusMessageIter dict, entry, array, value;

	if (!iter || !keyMgmt || dbus_message_iter_get_arg_type(iter)
			!= DBUS_TYPE_ARRAY)
		return -EINVAL;

	*keyMgmt = 0;
	dbus_message_iter_recurse(iter, &dict);
	while (dbus_message_iter_get_arg_type(&dict) != DBUS_TYPE_INVALID) {

		dbus_message_iter_recurse(&dict, &entry);
		dbus_message_iter_get_basic(&entry, &ikey);
		if (ikey && !strcmp(ikey, "KeyMgmt")) {

			dbus_message_iter_next(&entry);
			if (dbus_message_iter_get_arg_type(&entry) != DBUS_TYPE_VARIANT)
				return -EINVAL;

			dbus_message_iter_recurse(&entry, &array);
			if (dbus_message_iter_get_arg_type(&array) != DBUS_TYPE_ARRAY)
				return -EINVAL;

			dbus_message_iter_recurse(&array, &value);
			while (dbus_message_iter_get_arg_type(&value) != DBUS_TYPE_INVALID) {
				dbus_message_iter_get_basic(&value, &keyvalue);

				if (keyvalue) {
					if (!strcmp("wpa-psk", keyvalue))
						*keyMgmt |= IEEE80211_NETWORK_KEYMGMT_WPA_PSK;
					else if (!strcmp("wpa-ft-psk", keyvalue))
						*keyMgmt |= IEEE80211_NETWORK_KEYMGMT_WPA_FT_PSK;
					else if (!strcmp("wpa-psk-sha256", keyvalue))
						*keyMgmt |= IEEE80211_NETWORK_KEYMGMT_WPA_PSK_256;
					else if (!strcmp("wpa-eap", keyvalue))
						*keyMgmt |= IEEE80211_NETWORK_KEYMGMT_WPA_EAP;
					else if (!strcmp("wpa-ft-eap", keyvalue))
						*keyMgmt |= IEEE80211_NETWORK_KEYMGMT_WPA_FT_EAP;
					else if (!strcmp("wpa-eap-sha256", keyvalue))
						*keyMgmt |= IEEE80211_NETWORK_KEYMGMT_WPA_EAP_256;
				}

				dbus_message_iter_next(&value);
			}
		}

		dbus_message_iter_next(&dict);
	}

	return 0;
}

void cWpaSupplicantClient::fillWpaSuppData(cWpaSupplicantIfaceData *data, int type, const void *response)
{
	int disconnectReason = 0;
	const char *sIfname = NULL, *sCountry = NULL, *state = NULL,
				*currentBss = NULL, *currentNetwork = NULL;
	::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it;
	::std::map< ::std::string, ::asf::dbus::DBusVariant > prop;
	uint32_t setprops = 0;

	if (!data)
		return;

	if (type)
		prop = (*(const_cast<::boost::shared_ptr<
				cDBusPropGetAllResponse >*>(reinterpret_cast<const ::boost::shared_ptr<
						cDBusPropGetAllResponse >*>(response))))->getProperties();
	else {
		prop = (*(const_cast<::boost::shared_ptr<
				cDBusPropChangedSignal >*>(reinterpret_cast<const ::boost::shared_ptr<
						cDBusPropChangedSignal >*>(response))))->getChanged_properties();
	}

	for (it = prop.begin(); it != prop.end(); ++it) {
		if (!it->first.compare("Ifname")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &sIfname);
			setprops |= WPA_SUPP_IFNAME_SET;
		}
		else if (!it->first.compare("Country")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &sCountry);
			setprops |= WPA_SUPP_ALPHA2_SET;
		}
		else if (!it->first.compare("State")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &state);
			setprops |= WPA_SUPP_STATE_SET;
		}
		else if (!it->first.compare("CurrentBSS")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &currentBss);
			setprops |= WPA_SUPP_CURRENT_BSS_SET;
		}
		else if (!it->first.compare("CurrentNetwork")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &currentNetwork);
			setprops |= WPA_SUPP_CURRENT_NETWORK_SET;
		}
		else if (!it->first.compare("DisconnectReason")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &disconnectReason);
			setprops |= WPA_SUPP_DISCONNECT_REASON_SET;
		}
	}

	LOG_DEBUG ("set Properties: 0x%02x", setprops);
	if (setprops & WPA_SUPP_ALPHA2_SET)
		data->_sCountry.assign(sCountry ? sCountry : "");
	if (setprops & WPA_SUPP_IFNAME_SET)
		data->_sIfName.assign(sIfname ? sIfname : "");
	if (setprops & WPA_SUPP_STATE_SET)
		data->_state.assign(state ? state : "");
	if (setprops & WPA_SUPP_CURRENT_BSS_SET)
		data->_currentBSS.assign(currentBss ? currentBss : "");
	if (setprops & WPA_SUPP_CURRENT_NETWORK_SET)
		data->_currentNetwork.assign(currentNetwork ? currentNetwork : "");
	if (setprops & WPA_SUPP_DISCONNECT_REASON_SET)
		data->_disconnectReason = disconnectReason;

	/* purely for debugging */
	LOG_DEBUG ("****************Wpa Object Dump Start: \"%s\"************", data->_sObjPath.c_str());
	LOG_INFO ("Wireless Interface: \"%s\"",data->_sIfName.c_str());
	LOG_INFO ("Alpha2 code: \"%s\"",data->_sCountry.c_str());
	LOG_INFO ("state: \"%s\"",data->_state.c_str());
	LOG_INFO ("currentBss: \"%s\"",data->_currentBSS.c_str());
	LOG_INFO ("currentNetwork: \"%s\"",data->_currentNetwork.c_str());
	LOG_INFO ("disconnectReason: \"%d\"",data->_disconnectReason);
	LOG_DEBUG ("****************Wpa Object Dump End: \"%s\"************", data->_sObjPath.c_str());
}

void cWpaSupplicantClient::fillBssData(cBasicServiceSet *bss, int type, const void *response)
{
	int iRet, length = 0, ssid_length = 0, bssid_length = 0;
	bool ieee80211d = false;
	const char *mode = NULL;
	uint8 *data = NULL;
	uint8 *ssid = NULL;
	uint8 *bssid = NULL;
	::DBusMessageIter subIter;
	uint16_t frequency = 0;
	uint32_t rsnkeymgmt = 0, wpakeymgmt = 0, setprops = 0;
	dbus_bool_t privacy = FALSE;
	::std::string sCoun;
	::std::map< ::std::string, ::asf::dbus::DBusVariant >::iterator it;
	::std::map< ::std::string, ::asf::dbus::DBusVariant > prop;
	char macaddress[3 * ETH_ALEN];

	if (!bss)
		return;

	if (type)
		prop = (*(const_cast<::boost::shared_ptr<
				cDBusPropGetAllResponse >*>(reinterpret_cast<const ::boost::shared_ptr<
						cDBusPropGetAllResponse >*>(response))))->getProperties();
	else {
		prop = (*(const_cast<::boost::shared_ptr<
				cDBusPropChangedSignal >*>(reinterpret_cast<const ::boost::shared_ptr<
						cDBusPropChangedSignal >*>(response))))->getChanged_properties();
	}

	memset(macaddress, 0, sizeof (macaddress));
	for (it = prop.begin(); it != prop.end(); ++it) {
		if (!it->first.compare("IEs")) {
			if (dbus_message_iter_get_arg_type(it->second.getReadIterator()) == DBUS_TYPE_ARRAY) {
				dbus_message_iter_recurse(it->second.getReadIterator(), &subIter);
				dbus_message_iter_get_fixed_array(&subIter, &data, &length);
				iRet = getCountryElemfromIEs (data, length, WLAN_EID_COUNTRY, sCoun);
				if (!iRet) {
					ieee80211d = true;
					setprops |= BSS_IES_SET;
				}
			} else {
				LOG_ERROR("IEs is not of type ARRAY or it is empty/invalid");
			}
		}
		else if (!it->first.compare("SSID")) {
			if (dbus_message_iter_get_arg_type(it->second.getReadIterator()) == DBUS_TYPE_ARRAY) {
				dbus_message_iter_recurse(it->second.getReadIterator(), &subIter);
				dbus_message_iter_get_fixed_array(&subIter, &ssid, &ssid_length);
				if (ssid_length > 0 && ssid_length < 33 && ssid[0] != 0)
					setprops |= BSS_SSID_SET;
			} else {
				LOG_ERROR("SSID is not of type ARRAY or it is empty/invalid");
			}
		}
		else if (!it->first.compare("BSSID")) {
			if (dbus_message_iter_get_arg_type(it->second.getReadIterator()) == DBUS_TYPE_ARRAY) {
				dbus_message_iter_recurse(it->second.getReadIterator(), &subIter);
				dbus_message_iter_get_fixed_array(&subIter, &bssid, &bssid_length);
				if (bssid_length == ETH_ALEN) {
					cUtils::macToString(macaddress, bssid);
					setprops |= BSS_BSSID_SET;
				}
			} else {
				LOG_ERROR("BSSID is not of type ARRAY or it is empty/invalid");
			}
		}
		else if (!it->first.compare("Frequency")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &frequency);
			setprops |= BSS_FREQUENCY_SET;
		}
		else if (!it->first.compare("Privacy")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &privacy);
			setprops |= BSS_PRIVACY_SET;
		}
		else if (!it->first.compare("Mode")) {
			dbus_message_iter_get_basic(it->second.getReadIterator(), &mode);
			setprops |= BSS_MODE_SET;
		}
		else if (!it->first.compare("WPA") || !it->first.compare("RSN")) {
			if (!it->first.compare("WPA")) {
				getBssKeyManagement(it->second.getReadIterator(), &wpakeymgmt);
				setprops |= BSS_WPA_KEYMGMT_SET;
			} else {
				getBssKeyManagement(it->second.getReadIterator(), &rsnkeymgmt);
				setprops |= BSS_RSN_KEYMGMT_SET;
			}
		}
	}

	LOG_DEBUG ("set properties: \"0x%02x\"",setprops);
	if (setprops & BSS_FREQUENCY_SET)
		bss->setFrequency(frequency);
	if (setprops & BSS_SSID_SET)
		bss->setSSID(ssid, ssid_length);
	if (setprops & BSS_PRIVACY_SET)
		bss->setPrivacy(static_cast<bool>(privacy));
	if (setprops & BSS_MODE_SET)
		bss->setMode(mode);
	if (setprops & BSS_WPA_KEYMGMT_SET)
		bss->setKeyMgmt(IEEE80211_NETWORK_KEYMGMT_WPA, wpakeymgmt);
	if (setprops & BSS_RSN_KEYMGMT_SET)
		bss->setKeyMgmt(IEEE80211_NETWORK_KEYMGMT_RSN, rsnkeymgmt);
	if (setprops & BSS_IES_SET) {
		bss->Set80211d(ieee80211d);
		bss->SetCountry(sCoun);
	}
	if (setprops & BSS_BSSID_SET)
		bss->setMacAddress(macaddress);

	bss->computeBssSecurity();

	/* purely for debugging */
	LOG_DEBUG ("****************BSS Dump Start: \"%s\" ************",bss->GetObjPath().c_str());
	LOG_INFO ("MAC Address: \"%s\"",bss->getMacAddress().c_str());
	LOG_INFO ("SSID: \"%s\"",bss->convertSSIDToHexString().c_str());
	LOG_INFO ("802.11D element: \"%s\"",bss->Get80211d() ? bss->GetCountry().c_str() : "unavailable");
	LOG_INFO ("Frequency: \"%d\"",bss->getFrequency());
	LOG_INFO ("Privacy: \"%s\"",bss->getPrivacy() ? "Y" : "N");
	LOG_INFO ("Mode: \"%s\"",cBasicServiceSet::bssModeToString(bss->getMode()).c_str());
	LOG_INFO ("Rsn KeyMgmt: \"0x%02x\"",bss->getKeyMgmt(IEEE80211_NETWORK_KEYMGMT_RSN));
	LOG_INFO ("WPA KeyMgmt: \"0x%02x\"",bss->getKeyMgmt(IEEE80211_NETWORK_KEYMGMT_WPA));
	LOG_INFO ("Security: \"%s\"",cBasicServiceSet::bssSecurityToString(bss->getSecurityType()).c_str());
	LOG_DEBUG ("****************BSS Dump end: \"%s\" ***************", bss->GetObjPath().c_str());
}

void cWpaSupplicantClient::onGetAllResponse(const ::boost::shared_ptr< cDBusPropertiesProxy > &proxy,
		const ::boost::shared_ptr< cDBusPropGetAllResponse > &response)
{
	int iRet;
	act_t token;
	bool wpaIfObj = false, wpaBssObj = false;
	cWpaSupplicantIfaceData *oData = NULL;
	cBasicServiceSet *bssTemp = NULL;
	cIOrgFreedesktopDBusPropertiesProxyIf *propProxy;

	LOG_INFO ("GetAll Response: %s", proxy->getDBusObjectPath().c_str());

	iRet = getWpaSupplicantData(proxy->getDBusObjectPath(), proxy->getBusType(), &oData);
	if (!iRet)
		wpaIfObj = true;
	else {
		iRet = getWpaSupplicantBssData(proxy->getDBusObjectPath(), proxy->getBusType(),
				&bssTemp);
		if (!iRet)
			wpaBssObj = true;
	}

	if (!wpaBssObj && !wpaIfObj) {
		LOG_INFO ("No proxy entry to the object: [objPath: %s] [BusType: %d]",
				proxy->getDBusObjectPath().c_str(), proxy->getBusType());
		return;
	}

	if (wpaIfObj) {
		fillWpaSuppData(oData, 1, reinterpret_cast<const void*>(&response));
		propProxy = DBusProxyFactory::getInstance()->getOrgFreedesktopDBusPropProxyIf();
		token = propProxy->sendGetRequest(proxy->getDBusBusName(), proxy->getDBusObjectPath(),
				proxy->getBusType(), sWpaSuppInterfaceIfName, "BSSs", *this);
		if (DEFAULT_ACT == token)
			LOG_ERROR ("Failed to get the BSSs of interface: %s", proxy->getDBusObjectPath().c_str());

		if (oData->_currentBSS.compare("/") &&
				WPA_SUPP_STATE_CONNECTED == cWpaSupplicantConnectionNotifier::wpaStateToEvent(oData->_state)) {
			iRet = getWpaSupplicantBssData(oData->_currentBSS, proxy->getBusType(), &bssTemp);
			if (iRet == 0 && bssTemp->getFrequency() != 0) {
				LOG_INFO("Notifying the change in connection for the interface [%s]: %s", oData->_sIfName.c_str(),
						oData->_currentBSS.c_str());
				updateWpaConnectionStatus(cWpaSupplicantConnectionNotifier::wpaStateToEvent(oData->_state),
						oData->_sIfName, oData->_currentBSS);
			} else {
				LOG_INFO("Interface: %s connection status due for : %s", oData->_sIfName.c_str(),
						oData->_currentBSS.c_str());
				oData->_wpaConnStatusDue = true;
			}
		}
	}
	else if (wpaBssObj) {
		fillBssData(bssTemp, 1, reinterpret_cast<const void*>(&response));
		updateBSSNotification(EVENT_BSS_ADDED, proxy->getDBusObjectPath(), *bssTemp);

		/* Check if there is something connected and any connection status is due
		 * to the regulatory core */
		iRet = getWpaSupplicantData(bssTemp->GetIfName(), proxy->getBusType(), &oData);
		if (!iRet && oData->_wpaConnStatusDue == true) {
			if (!bssTemp->GetObjPath().compare(oData->_currentBSS)) {
				LOG_INFO("Notifying the change in connection for the interface [%s]: %s", oData->_sIfName.c_str(),
						oData->_currentBSS.c_str());
				updateWpaConnectionStatus(cWpaSupplicantConnectionNotifier::wpaStateToEvent(oData->_state),
						oData->_sIfName, oData->_currentBSS);
				oData->_wpaConnStatusDue = false;
			}
		}
	}
}

void cWpaSupplicantClient::onPropertiesChangedError(const ::boost::shared_ptr< cDBusPropertiesProxy > &proxy,
			const ::boost::shared_ptr< cDBusPropChangedError > &error)
{
	LOG_ERROR ("Properties Changed Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

void cWpaSupplicantClient::onPropertiesChangedSignal(const ::boost::shared_ptr<cDBusPropertiesProxy> &proxy,
		const ::boost::shared_ptr< cDBusPropChangedSignal > &signal)
{
	int iRet;
	::std::string sCoun, sInterface;
	cBasicServiceSet *bssTemp = NULL;
	bool wpaIfObj = false, wpaBssObj = false,
			disconnected = false;
	cWpaSupplicantIfaceData *oData = NULL;
	::std::string curBss, curCountry;
	eWpaSuppConnectionState_t state = WPA_SUPP_STATE_INVAL;

	LOG_DEBUG ("Properties Changed Signal for object: %s", proxy->getDBusObjectPath().c_str());

	iRet = getWpaSupplicantData(proxy->getDBusObjectPath(), proxy->getBusType(), &oData);
	if (!iRet)
		wpaIfObj = true;
	else {
		iRet = getWpaSupplicantBssData(proxy->getDBusObjectPath(), proxy->getBusType(),
				&bssTemp);
		if (!iRet)
			wpaBssObj = true;
	}

	if (!wpaBssObj && !wpaIfObj) {
		LOG_INFO ("No proxy entry to the object: [objPath: %s] [BusType: %d]",
				proxy->getDBusObjectPath().c_str(), proxy->getBusType());
		return;
	}

	sInterface = signal->getInterface();
	LOG_DEBUG ("Changed interface: %s", sInterface.c_str());

	if (wpaIfObj) {
		if (!sInterface.compare(sWpaSuppInterfaceIfName)) {

			curBss = oData->_currentBSS;
			fillWpaSuppData(oData, 0, reinterpret_cast<const void*>(&signal));

			LOG_INFO ("old bss: %s current bss: %s", curBss.c_str(),
					oData->_currentBSS.c_str());

			if (!oData->_currentBSS.compare("/"))
				disconnected = true;

			if (oData->_wpaConnStatusDue) {
				if (disconnected)
					oData->_wpaConnStatusDue = false;
			} else {
				state = cWpaSupplicantConnectionNotifier::wpaStateToEvent(oData->_state);
				if (WPA_SUPP_STATE_CONNECTED == state || WPA_SUPP_STATE_DISCONNECTED == state) {
					iRet = getWpaSupplicantBssData(disconnected ? curBss : oData->_currentBSS,
							proxy->getBusType(), &bssTemp);
					if (iRet == 0 && bssTemp->getFrequency() != 0) {
						LOG_INFO("Notifying the change in connection for the interface [%s]: %s",
								oData->_sIfName.c_str(), oData->_currentBSS.c_str());
						updateWpaConnectionStatus(state, oData->_sIfName, disconnected ? curBss : oData->_currentBSS);
					} else {
						if (!disconnected) {
							LOG_INFO("Interface: %s connection status due for : %s", oData->_sIfName.c_str(),
									oData->_currentBSS.c_str());
							oData->_wpaConnStatusDue = true;
						}
					}
				}
			}
		}
	}
	else if (wpaBssObj) {
		if (!sInterface.compare(sWpaSuppBSSIfName)) {
			curCountry = bssTemp->GetCountry();
			fillBssData(bssTemp, 0, reinterpret_cast<const void*>(&signal));
			if (curCountry.compare(bssTemp->GetCountry()))
				updateBSSNotification(EVENT_BSS_CHANGED, proxy->getDBusObjectPath(), *bssTemp);
		}
	}
}

void cWpaSupplicantClient::onSetError(const ::boost::shared_ptr< cDBusPropertiesProxy > &proxy,
		const ::boost::shared_ptr< cDBusPropSetError >& error)
{
	LOG_ERROR ("On Set Error [proxy object: %s, interface: %s] [type: %d]: %s/%s",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
			proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());
}

void cWpaSupplicantClient::onSetResponse(const ::boost::shared_ptr< cDBusPropertiesProxy > &proxy,
		const ::boost::shared_ptr< cDBusPropSetResponse >& response)
{
	LOG_INFO ("On Set Response [proxy object: %s, interface: %s] [type: %d], Act: %d",
			proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(), proxy->getBusType(),
			response->getAct());
}

int cWpaSupplicantClient::disconnectBss(const ::std::string &bss)
{
	act_t token;
	cBasicServiceSet *bssTemp = NULL;
	cWpaSupplicantIfaceData *oData = NULL;
	cIWpaSupplicantIfaceProxyIf *proxy;

	if (bss.empty())
		return -EINVAL;

	bssTemp = getBSSFromObjectPath(bss);
	if (!bssTemp)
		return -ENODATA;

	oData = getWpaSupplicantfromIfName(bssTemp->GetIfName());
	if (!oData)
		return -ENODATA;

	proxy = DBusProxyFactory::getInstance()->getWpaSupplicantIfaceProxyIf();
	if (!proxy)
		return -ENOMEDIUM;

	token = proxy->sendDisconnectRequest(sWpaSuppBusName, oData->_sObjPath,
			oData->_BusType, *this);
	if (token == DEFAULT_ACT)
		return -EIO;

	return 0;
}

void cWpaSupplicantClient::onDisconnectError(const ::boost::shared_ptr< cWpaSupplicantIfaceProxy >& proxy,
		const ::boost::shared_ptr< cWpaDisconnectError >& error)
{
	int iRet;
	cWpaSupplicantIfaceData *oData = NULL;

	LOG_ERROR ("Failed to Disconnect [proxy object: %s, interface: %s] [type: %d]: %s/%s",
				proxy->getDBusObjectPath().c_str(), proxy->getInterfaceName().c_str(),
				proxy->getBusType(), error->getName().c_str(), error->getMessage().c_str());

	iRet = getWpaSupplicantData(proxy->getDBusObjectPath(), proxy->getBusType(), &oData);
	if (iRet < 0) {
		LOG_ERROR("Failed to get the wpa supplicant local data for the object: %s",
				proxy->getDBusObjectPath().c_str());
	} else {
		updateWpaDisconnectionStatus(1, oData->_sIfName);
	}
}

void cWpaSupplicantClient::onDisconnectResponse(const ::boost::shared_ptr< cWpaSupplicantIfaceProxy >& proxy,
		const ::boost::shared_ptr< cWpaDisconnectResponse >& response)
{
	(void) response;

	int iRet;
	cWpaSupplicantIfaceData *oData = NULL;

	iRet = getWpaSupplicantData(proxy->getDBusObjectPath(), proxy->getBusType(), &oData);
	if (iRet < 0) {
		LOG_ERROR("Failed to get the wpa supplicant local data for the object: %s",
				proxy->getDBusObjectPath().c_str());
	} else {
	   LOG_DEBUG("successful disconnection for the interface [%s]: %s",
	         oData->_sIfName.c_str(), oData->_currentBSS.c_str());
		updateWpaDisconnectionStatus(0, oData->_sIfName);
	}
}

::std::string cWpaSupplicantClient::getConnectedBss(const ::std::string ifName)
{
	::std::string bss;
	cBasicServiceSet *temp;

	cWpaSupplicantIfaceData *data = getWpaSupplicantfromIfName(ifName);
	if (data && (WPA_SUPP_STATE_CONNECTED ==
			cWpaSupplicantConnectionNotifier::wpaStateToEvent(data->_state))) {
		 temp = getBSSFromObjectPath(data->_currentBSS);
		 if (temp && temp->getFrequency() != 0)
			 bss = temp->GetObjPath();
	}
	return bss;
}

	}
}

/** @} */
