/**
 * @file RegulatoryCore.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include <linux/if_ether.h>
#include <linux/nl80211.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <net/if_arp.h>
#include "RegulatoryGNSSMethod.h"
#include "Regulatory80211dMethod.h"
#include "WpaSupplicantClient.h"
#include "DBManagerFactory.h"
#include "RegulatoryMethodFactory.h"
#include "IEEE80211ClientFactory.h"
#include "BasicServiceSet.h"
#include "WifiRadio.h"
#include "WifiChannel.h"
#include "GenlEvent.h"
#include "RegulatoryCore.h"
#include "WBLServiceFactory.h"
#include "IRegCountryCodeServiceIf.h"

namespace org {
    namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cRegulatoryCore, Debug);

/*
 * Currently implemented regulatory methods. Will be directly used
 * by the regulatory core. Dont forget to add here if there is a new
 * addition (implementation) to the regulatory methods */
static void *regMethods[] = {
      (void *) ((long) (REGULATORY_METHOD_GNSS)),
      (void *) ((long) (REGULATORY_METHOD_MCC)),
      nullptr
};

cRegulatoryCore::cRegulatoryCore()
{
    _readkernelReg = false;
    _validRegSources = false;
    _regulatorychannel = 0;
    _currentmethod = REGULATORY_METHOD_MAX;
    _currentRegSettings = nullptr;
    _reg_source = eREG_SOURCE_DEFAULT;
    _watchDog = new cRegulatoryWatchDog();

    (void) cWpaSupplicantClient::getInstance()->Subscribe(this);
    (void) cWpaSupplicantClient::getInstance()->registerWpaConnectionNotification(this);
    (void) IEEE80211ClientFactory::getInstance()->getSTAClientIf()->
            subscribeIEEE80211STAConnectionStatus(this);
    (void) cWifiMlmeUpdates::getInstance()->registerForMlmeUpdates(this);
    (void) IEEE80211ClientFactory::getInstance()->getAPClientIf()->Subscribe(this);
    (void) _watchDog->subscribe(this);

    configureRegulatorySources();
    WBLServiceFactory::getInstance()->createRegCCInfoService();
}

inline int cRegulatoryCore::validateRegulatorySource(const char *source)
{
    int iRet = -EINVAL;

    if (source && (!strcmp(source, "GNSS") || !strcmp(source, "MCC") || !strcmp(source, "80211D")
            || !strcmp(source, "gnss") || !strcmp(source, "mcc") || !strcmp(source, "80211d")))
        iRet =  0;

    return iRet;
}

int cRegulatoryCore::validateRegulatorySources()
{
    int iRet = -EINVAL;
    bool valid = false;
    char **sources = NULL, **temp;

    sources = g_strsplit (_regSources.c_str(), ",", -1);
    if (sources) {

        for (temp = sources; *temp; temp++) {
            valid = false;
            if (0 == validateRegulatorySource(*temp))
                valid = true;
            if (!valid)
                break;
        }

        if (valid) {
            _validRegSources = true;
            iRet = 0;
        }

        g_strfreev (sources);
    } else {
        /* Probably only one source */
        if (0 == validateRegulatorySource(_regSources.c_str())) {
            _validRegSources = true;
            iRet  = 0;
        }
    }

    return iRet;
}

void cRegulatoryCore::createRegulatorySource(const eRegulatoryMethod_t &method)
{
    int iRet;
    cRegulatoryMethod *regMethod;

    regMethod = cRegulatoryMethodFactory::getInstance()->getRegulatoryMethod(method);
    if (regMethod) {
        iRet = regMethod->Register(this);
        if (iRet < 0) {
            LOG_ERROR ("Failed to register to the regulatory method: %s [err: %s/%d]",
                    regMethodToString(method).c_str(), strerror(-iRet), -iRet);
        }
    } else {
        LOG_ERROR ("Failed to create the regulatory method: %s",
                regMethodToString(method).c_str());
    }
}

void cRegulatoryCore::createRegulatorySources(void *regsources, bool configured)
{
    char **temp;
    void **methods;
    eRegulatoryMethod_t method;

    if (configured) {
        for (temp = static_cast<char **> (regsources); *temp; temp++) {
            method = srcToRegMethod(*temp);
            if (method != REGULATORY_METHOD_MAX) {
                LOG_INFO("Creating Reg Method: %s", regMethodToString(method).c_str());
                createRegulatorySource(method);
            }
        }
    } else {
        methods = static_cast<void **> (regsources);
        for (int index = 0; methods[index]; index++) {
            method = static_cast<eRegulatoryMethod_t> ((long)methods[index]);
            LOG_INFO("Creating Reg Method: %s", regMethodToString(method).c_str());
            createRegulatorySource(method);
        }
    }
}

void cRegulatoryCore::configureRegulatorySources()
{
    int iRet;
    char **sources = NULL;
    ::std::string group("RegulatorySources"), key("Sources");
    bool configure_static_methods = true;

    /* The priority is (configured regulatory sources > static regulatory sources).
     * This is to give a chance to add or remove the regulatory sources on demand.
     * First, WBL will see if there are any configured regulatory sources provided
     * and will use them only if they are valid. Else the static regulatory sources
     * will be used. */

    iRet = DBManagerFactory::getInstance()->getDBManagerIf()->getRegulatorySources(group, key, _regSources);
    if (0 == iRet) {
        LOG_INFO("Configured Regulatory sources: %s", _regSources.c_str());
        if (!_regSources.empty())
            (void) validateRegulatorySources();
        else {
            /* We have the key but the value is empty i.e., explicit
             * request to disable all regulatory sources */
            configure_static_methods = false;
        }
    }

    LOG_INFO("Supplied Regulatory sources validity: %s, static methods: %s", _validRegSources ? "Y" : "N",
            configure_static_methods ? "Y" : "N");
    if (_validRegSources) {
        sources = g_strsplit (_regSources.c_str(), ",", -1);
        createRegulatorySources(static_cast<void *> (sources), true);
        g_strfreev (sources);
    } else {
        if (configure_static_methods)
            createRegulatorySources(static_cast<void *> (regMethods), false);
    }
}

void cRegulatoryCore::destroyRegulatorySource(void *source, bool configured)
{
    eRegulatoryMethod_t method;

    if (configured && source) {
        method = srcToRegMethod(static_cast<char *>(source));
        if (method != REGULATORY_METHOD_MAX)
            (void) cRegulatoryMethodFactory::getInstance()->destroyRegMethod(method);
    } else {
        method = static_cast<eRegulatoryMethod_t> ((long) (source));
        (void) cRegulatoryMethodFactory::getInstance()->destroyRegMethod(method);
    }
}

void cRegulatoryCore::destroyRegulatorySources()
{
    char **sources = NULL, **temp;

    if (_validRegSources) {
        sources = g_strsplit (_regSources.c_str(), ",", -1);
        if (sources) {
            for (temp = sources; *temp; temp++)
                destroyRegulatorySource(static_cast<void *> (*temp), true);
            g_strfreev (sources);
        } else
            destroyRegulatorySource(static_cast<void *> (const_cast<char *>(_regSources.c_str())), true);
    } else {
        for (int index = 0; regMethods[index]; index++)
            destroyRegulatorySource(regMethods[index], false);
    }
}

int cRegulatoryCore::eraseRadioSettings(int index)
{
    int available = 0;
    ::std::vector<cWifiRadio*> radios;
    ::std::vector<cWifiRadio*>::iterator it;

    if (!_currentRegSettings)
        return -ENODATA;

    radios = _currentRegSettings->getWifiRadios();
    for (it = radios.begin(); it < radios.end(); ++it) {
        if ((*it)->getIndex() == index) {
            available = 1;
            break;
        }
    }

    if (available) {
        delete *it;
        radios.erase(it);
        return 0;
    }

    return -ENOENT;
}

cRegulatoryCore::~cRegulatoryCore()
{
   try
   {
      destroyRegulatorySources();

      (void) cWpaSupplicantClient::getInstance()->UnSubscribe(this);
      (void) cWpaSupplicantClient::getInstance()->unRegisterWpaConnectionNotification(this);
      (void) IEEE80211ClientFactory::getInstance()->getSTAClientIf()->
            unSubscribeIEEE80211STAConnectionStatus(this);
      (void) cWifiMlmeUpdates::getInstance()->unRegisterForMlmeUpdates(this);
      (void) _watchDog->unsubscribe(this);

      genlEventCleanup(&_currentRegSettings);
      _watchDog->stop(false);
      delete _watchDog;
      WBLServiceFactory::getInstance()->deleteRegCCInfoService();
   }catch(...){}
}

int cRegulatoryCore::changeRegulatory(const ::std::string &country)
{
    int iRet = -ENOTCONN;

    if (_regulatorychannel & REGULATORY_CHANNEL_WAPDMAN)
        iRet = IEEE80211ClientFactory::getInstance()->getAPClientIf()->changeRegulatory(country);
    else if (_regulatorychannel & REGULATORY_CHANNEL_WPASUPPLICANT)
        iRet = cWpaSupplicantClient::getInstance()->changeRegulatory(country);
    else if (_regulatorychannel & REGULATORY_CHANNEL_GNETLINK)
        LOG_ERROR ("%s channel is currently not handled", "GenericNetlink");

    if (iRet < 0) {
        LOG_ERROR ("Failed to Change the regulatory domain: %s/%d",
                strerror(-iRet), -iRet);
    } else {
        LOG_DEBUG ("Successfully initiated the regulatory domain change request to: %s",
                country.c_str());
    }

    return iRet;
}

void cRegulatoryCore::watchDogExpired(void *data)
{
    ::std::string country_code = *(static_cast<::std::string *>(data));

    LOG_INFO("Watch Dog expired, restoring to: %s", country_code.c_str());
    notifyRegChange(country_code);
    _watchDog->stop(true);
}

int cRegulatoryCore::notifyRegChange(const std::string &country)
{
    int iRet = 0;
    bool change = false, update_cc_prop = false;

    if (country.empty())
        return -EINVAL;

    LOG_INFO ("New notification: %s "
            "_country: %s , "
            "_regchangeinprogress: %s, "
            "_regtobechanged: %s",
            country.c_str(),
            _country.empty() ? "Empty" : _country.c_str(),
            _regchangeinprogress.empty() ? "Empty" :_regchangeinprogress.c_str(),
            _regtobechanged.empty() ? "Empty" :_regtobechanged.c_str());

    /* We have the country information from one of the
     * regulatory sources */
    if (_reg_source != eREG_SOURCE_OTHERS) {
        _reg_source = eREG_SOURCE_OTHERS;
        update_cc_prop = true;
    }
    if (!country.compare("00"))
        _reg_source = eREG_SOURCE_INVAL;

    if (!_regchangeinprogress.empty()) {
        if (_regchangeinprogress.compare(country))
            _regtobechanged = country;
        update_cc_prop = false;
    } else {
        if (_country.compare (country)) {

            /* We have received an alpha2 update from from one of our
             * regulatory methods.
             *
             * 1) We check if there is a disconnection sequence in progress.
             *    we expect that after a disconnection the kernel will restore
             *    the previous reg domain settings and therefore
             *      a) if a disconnection sequence is in progress (and is initiated
             *         by reg core) we update "_regdomainchange" so that it will be
             *         set once the disconnection sequence gets completed
             *      b) If there is a disconnection in progress and is not initiated by
             *         core, then we update _regtobechanged so that it will be
             *         set once the disconnection gets completed
             * */

            if (_disconnectionstatus._disconnectrequested &&
                    !_disconnectionstatus._completed) {
                if (!_disconnectionstatus._cfg80211events.isRegDomChangeRequired()) {
                    _disconnectionstatus._regdomainchange = country;
                    _disconnectionstatus._cfg80211events.regdomChangeRequired();
                } else {
                    if (!_disconnectionstatus._cfg80211events.isRegDomChangeInProgress()) {
                        if (_disconnectionstatus._regdomainchange.compare(country))
                            _disconnectionstatus._regdomainchange = country;

                        if (_disconnectionstatus._cfg80211events.isCoreUpdateReceived()) {
                            if (_disconnectionstatus._cfg80211events.isSetByUserReceived())
                                change = true;
                            else if (!_disconnectionstatus._cfg80211events.isSetByUserExpected())
                                change = true;
                        }
                    } else {
                        _regtobechanged = country;
                    }
                }
            }
            else if (_cfg80211regevents.isCoreUpdateExpected() ||
                    _cfg80211regevents.isSetByUserExpected()) {
                _regtobechanged = country;
            } else {
                change = true;
            }

            dumpInternals();
            if (change) {
                iRet = changeRegulatory(country);
                if (!iRet)
                    _regchangeinprogress = country;
                else if (iRet == -ENOTCONN)
                    _regtobechanged = country;
            }
            update_cc_prop = false;
        } else {
            _regtobechanged.clear();
        }
    }

    if (update_cc_prop)
        updateCountryCode(country);

    return iRet;
}

int cRegulatoryCore::flushOldRegMethod()
{
    int iRet = 0;
    cRegulatoryMethod *method;

    method = cRegulatoryMethodFactory::getInstance()->getRegulatoryMethod(_currentmethod);
    if (method) {
        LOG_DEBUG ("Initiating Stop to the old Reg Method: %s",
                regMethodToString(_currentmethod).c_str());
        iRet = method->stop();
        if (iRet == 0)
            _currentmethod = REGULATORY_METHOD_MAX;
        else {
            LOG_ERROR ("Failed to Stop the old Reg Method: %s",
                    regMethodToString(_currentmethod).c_str());
        }
    }

    return iRet;
}

inline eRegulatoryMethod_t &cRegulatoryCore::getCurrentRegSource()
{
    return _currentmethod;
}

cc_source_t cRegulatoryCore::regMethodToCCSource(const eRegulatoryMethod_t method)
{
    cc_source_t cc_src = eCC_SOURCE_INVAL;

    switch (method) {
    case REGULATORY_METHOD_GNSS:
        cc_src = eCC_SOURCE_GNSS;
        break;
    case REGULATORY_METHOD_MCC:
        cc_src = eCC_SOURCE_MCC;
        break;
    case REGULATORY_METHOD_80211D:
        cc_src = eCC_SOURCE_IEEE80211D;
        break;
    default:
        break;
    }

    return cc_src;
}

inline void cRegulatoryCore::stopWatchDog()
{
    if (_watchDog)
        _watchDog->stop(true);
}

inline void cRegulatoryCore::startWatchDog()
{
    if (_currentmethod == REGULATORY_METHOD_MAX && _watchDog)
        _watchDog->start();
}

void cRegulatoryCore::serviceAvailability(const eRegulatoryMethod_t method,
        const bool available)
{
    int iRet, index = 0;
    cRegulatoryMethod *regmethod;
    eRegulatoryMethod_t eRegMethod;
    std::vector<cRegulatoryMethod *> methods;

    LOG_INFO("Current Regulatory Method: %s, Changed Regulatory Method: %s Availability: %s",
            regMethodToString(_currentmethod).c_str(), regMethodToString(method).c_str(),
            (true == available) ? "Available" : "UnAvailable");

    for ( ; regMethods[index]; index++) {
        eRegMethod = static_cast<eRegulatoryMethod_t> ((long)regMethods[index]);
        regmethod = cRegulatoryMethodFactory::getInstance()->getRegulatoryMethod(eRegMethod);
        if (regmethod && regmethod->getServiceAvailability()) {
            LOG_INFO("Regulatory Method \"%s\" is available", regMethodToString(eRegMethod).c_str());
            methods.push_back(regmethod);
        }
    }

    if (methods.size() > 0) {
        std::sort(methods.begin(), methods.end(), cRegulatoryMethod::RegPrioComparison);
        regmethod = methods.front();

        if (regmethod) {
            if (_currentmethod != regmethod->getRegMethod()) {
                LOG_INFO("Choosing the Regulatory Method \"%s\"",
                        regMethodToString(regmethod->getRegMethod()).c_str());
                flushOldRegMethod();
                _currentmethod = regmethod->getRegMethod();
                iRet = regmethod->start();
                if (iRet == 0) {
                    _watchDog->stop(true);
                    updateCountryCode(_country);
                } else {
                    LOG_ERROR ("Failed to start the regulatory method: %s [error: %s/%d]",
                            regMethodToString(regmethod->getRegMethod()).c_str(), strerror(-iRet), -iRet);
                }
            }
        }
    } else {
        flushOldRegMethod();
        _watchDog->start();
    }
}

void cRegulatoryCore::dumpInternals()
{
    LOG_INFO ("_country: %s , "
            "_regchangeinprogress: %s, "
            "_regtobechanged: %s",
            _country.empty() ? "Empty" : _country.c_str(),
            _regchangeinprogress.empty() ? "Empty" :_regchangeinprogress.c_str(),
            _regtobechanged.empty() ? "Empty" :_regtobechanged.c_str());

    if (_disconnectionstatus._disconnectrequested) {
        LOG_INFO("Disconnecting Bss: \"%s\", "
                "Disconnection status: \"%s\","
                "reg dom change: \"%s\" "
                "core update expected: \"%s\" "
                "core update received: \"%s\" "
                "set by user expected: \"%s\" "
                "set by user received: \"%s\" "
                "disconnection interface: \"%s\"",
                _disconnectionstatus._disconnectingbss.c_str(),
                IEEE80211STAConnectionStatusNotifier::IEEE80211StatusToString(
                        _disconnectionstatus._status).c_str(),
                _disconnectionstatus._regdomainchange.empty() ? "N" :
                        _disconnectionstatus._regdomainchange.c_str(),
                _disconnectionstatus._cfg80211events.isCoreUpdateExpected() ? "Y" : "N",
                _disconnectionstatus._cfg80211events.isCoreUpdateReceived() ? "Y" : "N",
                _disconnectionstatus._cfg80211events.isSetByUserExpected() ? "Y" : "N",
                _disconnectionstatus._cfg80211events.isSetByUserReceived() ? "Y" : "N",
                _disconnectionstatus._ifname.c_str());
    }

    if (_cfg80211regevents.isCoreUpdateExpected() ||
            _cfg80211regevents.isSetByUserExpected()) {
        LOG_INFO("Disconnecting in progress but not initiated by regulatory core "
                "core update expected: \"%s\" "
                "core update received: \"%s\" "
                "set by user expected: \"%s\" "
                "set by user received: \"%s\" ",
                _cfg80211regevents.isCoreUpdateExpected() ? "Y" : "N",
                _cfg80211regevents.isCoreUpdateReceived() ? "Y" : "N",
                _cfg80211regevents.isSetByUserExpected() ? "Y" : "N",
                _cfg80211regevents.isSetByUserReceived() ? "Y" : "N");
    }
}

int cRegulatoryCore::handleNormalDisconnectUpdates(cRegulatoryUpdate *regUpdate)
{
    if (!regUpdate)
        return 0;

    if (_cfg80211regevents.isCoreUpdateExpected() ||
            _cfg80211regevents.isSetByUserExpected()) {

        if (regUpdate->getInitiator() ==
                static_cast<uint8_t>(NL80211_REGDOM_SET_BY_CORE)) {
            _cfg80211regevents.coreUpdateReceived();
            if (_cfg80211regevents.isSetByUserExpected())
                return -1;
        }
        else if (regUpdate->getInitiator() ==
                static_cast<uint8_t>(NL80211_REGDOM_SET_BY_USER)) {
            if (_cfg80211regevents.isCoreUpdateExpected() &&
                    !_cfg80211regevents.isCoreUpdateReceived()) {
                _cfg80211regevents.coreUpdateReceived();
            } else {
                _cfg80211regevents.setByUserReceived();
            }

            if (_cfg80211regevents.isSetByUserExpected())
                return -1;
        }
        LOG_INFO("Disconnection but not initiated by regulatory core "
                "is completed");
    }

    _cfg80211regevents.reset();
    return 0;
}

int cRegulatoryCore::handleDisconnectionUpdates(cRegulatoryUpdate *regUpdate)
{
    int iRet;
    bool changeregdom = false;
    ::std::string country, path;
    cBasicServiceSet *bss;

    if (!regUpdate)
        return -EINVAL;

    if (!_disconnectionstatus._disconnectrequested)
        return 0;

    if (_disconnectionstatus._completed)
        return 0;

    /**
     * When a disconnection happens, the Linux kernel thinks that all the
     * stale hints regarding the regulations are invalid anymore and therefore
     * restores the previous settings if was configured by the userspace which
     * is nothing but us. The sequence is
     *
     * 1) If there was no alpha2 set by the user before connection, then cfg80211
     *    will restore to the safe regulatory domain i.e., "00"
     * 2) If there was already an alpha2 set by userspace before connection, then
     *    there are two updates expected
     *      a) one for the safe regulatory domain
     *      b) one to restore the previous user settings
     *
     * Therefore a disconnection sequence is completed when the STA gets
     * disassociated from the AP followed by the regulatory updates from the kernel
     *
     * 1) One update with "00" if there was no userspace setting before
     * 2) Two updates if there was an userspace setting before ("00" and the previous
     *    userspace settings)  */

    country = regUpdate->getCountry();
    if (_currentRegSettings->getInitiator() ==
            static_cast<uint8_t>(NL80211_REGDOM_SET_BY_CORE)) {
        if (_disconnectionstatus._cfg80211events.isCoreUpdateExpected()) {
            _disconnectionstatus._cfg80211events.coreUpdateReceived();
            if (!_disconnectionstatus._cfg80211events.isSetByUserExpected())
                changeregdom = true;
        }
    }
    else if (_currentRegSettings->getInitiator() ==
            static_cast<uint8_t>(NL80211_REGDOM_SET_BY_USER)) {
        if (_disconnectionstatus._cfg80211events.isSetByUserExpected()) {
            if (_disconnectionstatus._cfg80211events.isCoreUpdateExpected() &&
                    !_disconnectionstatus._cfg80211events.isCoreUpdateReceived()) {
                _disconnectionstatus._cfg80211events.coreUpdateReceived();
            } else {
                _disconnectionstatus._cfg80211events.setByUserReceived();
                changeregdom = true;
            }
        }
    }

    if (changeregdom) {
        if (_disconnectionstatus._cfg80211events.isRegDomChangeRequired()) {
            notifyRegChange(_disconnectionstatus._regdomainchange);
            _disconnectionstatus._cfg80211events.regdomChangeInProgress();
            if (!country.compare(_country))
                return -1;
        } else _disconnectionstatus._completed = true;
    } else {
        if (_disconnectionstatus._cfg80211events.isRegDomChangeInProgress() &&
                !_disconnectionstatus._regdomainchange.compare(country))
            _disconnectionstatus._completed = true;
    }

    if (_disconnectionstatus._completed) {
        LOG_INFO("Disconnection sequence to %s is completed",
                _disconnectionstatus._disconnectingbss.c_str());

        if (IEEE80211_NETWORK_DISCONNECT_CHAN_CONNMAN ==
                _disconnectionstatus._disconnectchannel) {
            LOG_DEBUG ("Removing the connman's service proxy to: %s",
                    _disconnectionstatus._disconnectingbss.c_str());
            bss = cWpaSupplicantClient::getInstance()->getBSSFromObjectPath(
                    _disconnectionstatus._disconnectingbss);
            if (bss) {
                iRet = computeObjPath(_disconnectionstatus._ifname, bss, path);
                if (!iRet) {
                    IEEE80211ClientFactory::getInstance()->getSTAClientIf()->destroyConnManServiceProxy(path);
                }
            } else {
                LOG_ERROR ("Wpa supplicant does not know about: %s",
                        _disconnectionstatus._disconnectingbss.c_str());
            }
        }

        _disconnectionstatus.reset();
        /*if (_regchangeinprogress.empty()) {
            return (country.compare(_country) == 0) ? -1 : 0;
        } else*/ return 0;
    }

    return -1;
}

void cRegulatoryCore::updateDb(const ::std::string &country, ::std::string &group)
{
    int iRet;

    iRet = DBManagerFactory::getInstance()->getDBManagerIf()->setRegulatoryInfo(country, group);
    if (iRet < 0) {
        LOG_ERROR ("Failed to update the Regulatory DB: %s/%d",
                strerror(-iRet), -iRet);
    }
}

::std::string cRegulatoryCore::regSourceToString(const regsource_t source)
{
    ::std::string src;

    switch (source) {
    default:
    case eREG_SOURCE_INVAL:
        src = "R_SOURCE_INVAL";
        break;
    case eREG_SOURCE_DEFAULT:
        src = "R_SOURCE_DEF";
        break;
    case eREG_SOURCE_LASTMODE:
        src = "R_SOURCE_LASTMODE";
        break;
    case eREG_SOURCE_OTHERS:
        src = "R_SOURCE_OTHERS";
        break;
    }

    return src;
}

void cRegulatoryCore::updateCountryCode(const ::std::string &country)
{
    cRegCountryCodeInfoIf *service;
    cc_source_t source = eCC_SOURCE_INVAL;

    if (country.empty())
        return;

    switch (_reg_source) {
    default:
        break;
    case eREG_SOURCE_LASTMODE:
        source = eCC_SOURCE_LASTMODE;
        break;
    case eREG_SOURCE_OTHERS:
        source = regMethodToCCSource(getCurrentRegSource());
        break;
    }

    LOG_INFO ("CountryCode Update, Reg Source: \"%s\" country: \"%s\" cc_source: \"%s\"",
            regSourceToString(_reg_source).c_str(),
            country.c_str(), cCountryCodeClientInfo::CCSourceToString(source).c_str());

    service = WBLServiceFactory::getInstance()->getRegCCInfoService();
    if (service)
        service->updateCountryCodeInfo(country, static_cast <uint16_t> (source));
}

void cRegulatoryCore::loopbacknotification(cRegulatoryUpdate *regUpdate)
{
    int iRet;
    bool global = false, validate = true,
            updatedb = true, stopwatchdog(false);
    ::std::string reginfo, country, radioname,
                 group = "Global";
    ::boost::ptr_vector<WifiRadio> radios;

    if (!regUpdate)
        return;

    global = regUpdate->getGlobal();

    if (global)
        country = regUpdate->getCountry();
    else {
        radios = regUpdate->getRadios();
        if (!radios.size())
            return;

        country = radios.at(0).getRegDomain();
        radioname = radios.at(0).getRadio();
    }

    LOG_INFO ("Regulatory Change Notification "
            "Alpha2: %s "
            "[change for: %s]",
            country.c_str(),
            regUpdate->getGlobal() ? "Global" : radioname.c_str());

    /**
     * Little bit of background:) The design is as generic as possible. It is
     * possible that there could be multiple wifi radios attached to the host
     * and they could have their own custom reg settings (although it is discouraged
     * but keep in mind that it is possible, the Generic Netlink system of WBL would
     * take care of informing the right updates to this regulatory core (Well, the core
     * here in the sense of WBL). Therefore we maintain radio based settings which
     * would help in taking appropriate decisions.
     *
     * However. currently we don't take care of the radios which are self managed and
     * therefore the updates are ignored. This is by considering the fact that in an
     * automotive environment when the vehicle is stamped as an OUTDOOR device and
     * therefore it does not matter if there are one or more radios attached to the host,
     * all of them shall have the same regulatory settings and therefore shall comply
     * to the regulatory rules of WBL (Read
     * "G3G_BT-WIFI_UGKZ7_CRDA_Setup_Scenarios.0-3based.ConceptSummary.0-3-6.docx"
     * by Mr. Volker Schade-Buensow
     *
     * (Note: Currently the Gen3 infotainment systems have one 802.11 radio (UGKZ7)
     * attached). */

    genlEventCleanup(&_currentRegSettings);
    _currentRegSettings = RegulatoryUpdateToGenlEvent(regUpdate);
    if (!_currentRegSettings) {
        LOG_ERROR("Failed to create a local cache of the regulatory update");
    }

    if (!global)
        return;

    dumpInternals();
    if (handleDisconnectionUpdates(regUpdate) != 0)
        return;

    if (handleNormalDisconnectUpdates(regUpdate) != 0)
        return;

    if (!_readkernelReg) {

        reginfo = DBManagerFactory::getInstance()->getDBManagerIf()->getRegulatoryInfo(group);
        LOG_INFO ("Regulatory information read from DB: %s", reginfo.c_str());

        if (!reginfo.empty()) {
            if (reginfo.compare(country)) {
                /* This is the first update from the Generic netlink client
                 * after WBL's startup but there is some reg change which is
                 * already in progress. We should not set the value again
                 * from db!! */
                if (_regchangeinprogress.empty()) {
                    updatedb = false;
                    iRet = changeRegulatory(reginfo);
                    if (!iRet) {
                        _reg_source = eREG_SOURCE_LASTMODE;
                        _regchangeinprogress = reginfo;
                    }
                    else if (iRet == -ENOTCONN)
                        _regtobechanged = reginfo;
                }
            } else {
                if (country.compare("00"))
                    _reg_source = eREG_SOURCE_LASTMODE;
                else {
                    stopwatchdog = true;
                }
                _country = country;
            }
        } else {
            _country = country;
            if (!country.compare("00"))
                stopwatchdog = true;
        }

        /* This is the first update i.e., on boot, the current country code is
         * updated. We start the regulatory timer on the below cases
         * 1) The country code does not match WBL's persistent country code
         *   a) Anyway WBL tries to restore but also start a watchdog to set
         *      safe domain if there is no regulatory sources available
         *      Note: The regulatory timer will be stopped if one of the regulatory
         *      sources is available
         * 2) The country code match WBL's persistent country code (Last Mode)
         * 3) WBL does not have have anything stored but the system seems to have
         *    a different country code than the safe regulatory domain */
        if (stopwatchdog) {
            LOG_INFO ("Stopping the watch dog timer");
            stopWatchDog();
        }

        /* If the Component thread is fast enough than the Service thread,
         * it is possible that the Navigation client is created first and
         * therefore it receives an update from the Nav module and triggers a
         * change which is perfectly fine. But if the regulatory settings
         * are as same as the current settings (which will be known to this
         * thread only if there is a loopback update), the variable
         * _regchangeinprogress will be hanging and therefore reset this */
        if (updatedb) {
            if (!_regchangeinprogress.compare(country)) {
                _country = country;
                _regchangeinprogress.clear();
                LOG_INFO("Updating the regulatory info in db to: %s", country.c_str());
                updateDb(country, group);
            }
        }

        updateCountryCode(country);
        _readkernelReg = true;
    } else {

        if (!_regchangeinprogress.empty() && !_regchangeinprogress.compare(country)) {
            LOG_INFO("Regulatory Domain changed Successfully to: %s", country.c_str());
            _country = country;
            _regchangeinprogress.clear();
            updateDb(country, group);
        } else {
            /* Regulatory has been changed without Core's intention, Would abort
             * the app and reset? */
            if (_country.compare(country)) {
                /* This should never happen */
                LOG_ERROR ("Mismatch Detected in the Regulatory Domain between the core and kernel");
                _country = country;
            }
        }

        if (country.compare("00"))
            startWatchDog();
        updateCountryCode(country);

        if (!_regtobechanged.empty()) {
            if (_country.compare(_regtobechanged)) {
                iRet = changeRegulatory(_regtobechanged);
                if (!iRet) {
                    _regchangeinprogress = _regtobechanged;
                    _regtobechanged.clear();
                    validate = false;
                }
            } else {
                _regtobechanged.clear();
            }
        }
    }

    if (!validate)
        return;

    iRet = validateNextConnections();
    if (!iRet) {
        /* A dry run to check if all the radios are operating legally */
        iRet = validateRadios();
        if (!iRet) {
            LOG_INFO("All attached radios are in legal operations according to the "
                    "current Reg settings");
        }
    }
}

int cRegulatoryCore::validateRadios()
{
    ::std::vector<cWifiNetworkInterface*> interfaces;
    ::std::vector<cWifiNetworkInterface*>::iterator it2;
    ::std::string bss;
    int iRet;
    ::std::vector<cWifiRadio*> radios;
    ::std::vector<cWifiRadio*>::iterator it;

    if (!_currentRegSettings)
        return -ENODATA;

    radios = _currentRegSettings->getWifiRadios();
    for (it = radios.begin(); it < radios.end(); ++it) {

        if ((*it)->getSelfManaged())
            continue;

        interfaces = (*it)->getNetDevices();
        for (it2 = interfaces.begin(); it2 < interfaces.end();
                ++it2) {
            bss = cWpaSupplicantClient::getInstance()->getConnectedBss(
                    (*it2)->getIfName());
            if (bss.empty())
                continue;

            iRet = disconnectBss(bss, (*it2)->getIfName());
            if (!iRet) {
                LOG_INFO("Disconnection initiated to the bss %s [iface: %s] "
                        "as it is associated in a invalid channel",
                        bss.c_str(), (*it2)->getIfName().c_str());
                return -EINPROGRESS;
            }
        }
    }

    return 0;
}

cPendingConnection *cRegulatoryCore::getNextConnection()
{
    if (_pendingvalidations.empty())
        return NULL;
    return &_pendingvalidations.front();
}

int cRegulatoryCore::validateNextConnections()
{
    int iRet;
    cPendingConnection *next;
    ::std::string bss;

    while ((next = getNextConnection()) != NULL) {
        LOG_INFO("validateNextConnection -> bss: %s ifname: %s",
                next->_bss.c_str(), next->_ifName.c_str());
        bss = cWpaSupplicantClient::getInstance()->getConnectedBss(next->_ifName);
        if (bss.empty())
            removePendingConnection(next->_ifName);
        else {
            iRet = disconnectBss(bss, next->_ifName);
            if (iRet < 0 && (iRet == -ENODATA ||
                    iRet == -ECANCELED)) {
                removePendingConnection(next->_ifName);
            } else {
                return -EINPROGRESS;
            }
        }
    }

    return 0;
}

void cRegulatoryCore::notifyRegChannelAvailabilty(const eRegChannelType_t type,
                const bool availability)
{
    int iRet;
    tU8 channel = 0;

    LOG_INFO ("Regulatory Channel type: %s, "
            "Availability: %s",
            (REGULATORY_CHANNEL_WPA_SUPPLICANT == type) ?
                    "wpa_supplicant" :
            (REGULATORY_CHANNEL_WIFI_AP_DIRECT_MANAGER == type) ?
                    "WiFi_AP_Direct_Manager" :
            (REGULATORY_CHANNEL_GENERIC_NETLINK == type) ?
                    "GenericNetlink"
            :"Unknown", availability ? "AVAILABLE" : "UNAVAILABLE");

    if (REGULATORY_CHANNEL_WPA_SUPPLICANT == type)
        channel = REGULATORY_CHANNEL_WPASUPPLICANT;
    else if (REGULATORY_CHANNEL_WIFI_AP_DIRECT_MANAGER == type)
        channel = REGULATORY_CHANNEL_WAPDMAN;
    else if (REGULATORY_CHANNEL_GENERIC_NETLINK == type)
        channel = REGULATORY_CHANNEL_GNETLINK;
    else {
        return;
    }

    if (availability) {
        _regulatorychannel |= channel;
    } else {
        _regulatorychannel &= (tU8) ~channel;
    }

    if (availability && !_regtobechanged.empty()) {
        LOG_INFO ("There is a pending Regulatory Change: %s [current: %s]",
                _regtobechanged.c_str(), _country.c_str());
        if (_country.compare(_regtobechanged)) {
            iRet = changeRegulatory(_regtobechanged);
            if (!iRet) {
                _regchangeinprogress = _regtobechanged;
                _regtobechanged.clear();
            }
        } else {
            _regtobechanged.clear();
        }
    }
}

cWifiRadio *cRegulatoryCore::getRadioFromIfname(const ::std::string &ifname)
{
    ::std::vector<cWifiRadio*> wifiRadios;
    ::std::vector<cWifiRadio*>::iterator it;
    ::std::vector<cWifiNetworkInterface*> interfaces;
    ::std::vector<cWifiNetworkInterface*>::iterator it1;

    if (ifname.empty() || !_currentRegSettings)
        return NULL;

    wifiRadios = _currentRegSettings->getWifiRadios();
    for (it = wifiRadios.begin(); it < wifiRadios.end(); ++it) {
        interfaces = (*it)->getNetDevices();
        for (it1 = interfaces.begin(); it1 != interfaces.end(); ++it1)
            if (!(*it1)->getIfName().compare(ifname))
                return *it;
    }

    return NULL;
}

int cRegulatoryCore::getMacAddress(const ::std::string &ifname,
        ::std::string &macaddress)
{
    int index = 0, iRet, sk;
    const char *name;
    struct ifreq ifr;
    char mac[ETH_ALEN], ifrname[IFNAMSIZ],
            address[20];

    if (ifname.empty())
        return -EINVAL;

    name = ifname.c_str();

    memset(mac, 0, ETH_ALEN);
    memset(&ifr, 0, sizeof(struct ifreq));
    memset(ifrname, 0, sizeof(ifrname));
    memset(address, 0, sizeof(address));

    for (index = 0; *name; name++, index++)
        ifrname[index] = *name;

    sk = socket(PF_INET, SOCK_DGRAM | SOCK_CLOEXEC, 0);
    if (sk < 0)
        return -errno;

    strncpy(ifr.ifr_name, ifrname, sizeof(ifr.ifr_name) - 1);
    iRet = ioctl(sk, SIOCGIFHWADDR, &ifr);
    close(sk);

    if (iRet < 0)
        return -errno;

    memcpy(mac, ifr.ifr_hwaddr.sa_data, ETH_ALEN);
    sprintf(address, "%02x%02x%02x%02x%02x%02x",
            mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
    macaddress.assign(address);

    return 0;
}

const ::std::string cRegulatoryCore::getMacAddressFromIfname(const ::std::string &ifname)
{
    ::std::string macaddress;
    size_t index = 0, tempindex = 0;
    bool acquired = false;
    char tempaddr[15];
    const char *name;
    ::std::vector<cWifiRadio*> wifiRadios;
    ::std::vector<cWifiRadio*>::iterator it;
    ::std::vector<cWifiNetworkInterface*> interfaces;
    ::std::vector<cWifiNetworkInterface*>::iterator it1;

    if (ifname.empty() || !_currentRegSettings)
        return macaddress;

    memset(tempaddr, 0, sizeof(tempaddr));
    wifiRadios = _currentRegSettings->getWifiRadios();
    for (it = wifiRadios.begin(); it < wifiRadios.end(); ++it) {
        interfaces = (*it)->getNetDevices();
        for (it1 = interfaces.begin(); it1 < interfaces.end(); ++it1)
            if (!(*it1)->getIfName().compare(ifname)) {
                for (name = (*it1)->getMacAddress().c_str();
                        index < (*it1)->getMacAddress().length(); ++index)
                    if (name[index] != ':')
                        tempaddr[tempindex++] = name[index];
                macaddress.assign(tempaddr);
                acquired = true;
                break;
            }
    }

    if (!acquired)
        getMacAddress(ifname, macaddress);

    return macaddress;
}

int cRegulatoryCore::computeObjPath(const ::std::string &ifName,
        const cBasicServiceSet *bss, ::std::string &path)
{
    ::std::string macaddress, ssid;

    if (!bss || ifName.empty())
        return -EINVAL;

    path = "/net/connman/service/wifi_";
    macaddress = getMacAddressFromIfname(ifName);
    if (macaddress.empty()) {
        LOG_ERROR("Regulatory core is not aware of the Mac address of "
                "the interface: %s", ifName.c_str());
        return -ENODATA;
    }

    path.append(macaddress);
    path.append("_");

    ssid = bss->convertSSIDToHexString();
    if (ssid.empty())
        ssid = "hidden";

    path.append(ssid);
    path.append("_");
    path.append(cBasicServiceSet::bssModeToString(bss->getMode()));
    path.append("_");
    path.append(cBasicServiceSet::bssSecurityToString(bss->getSecurityType()));

    return 0;
}

int cRegulatoryCore::isAssociatedChannelValid(cWifiRadio *radio, uint16_t frequency)
{
    uint32_t flags;
    ::std::vector<cWifiChannel*> channels;
    ::std::vector<cWifiChannel*>::iterator it;

    if (!radio)
        return -ENODATA;

    channels = radio->getChannels();
    for (it = channels.begin(); it < channels.end(); ++it)
        if ((*it)->getFrequency() == frequency) {
            flags = (*it)->getFlags();
            IS_VALID_CHANNEL(flags);
        }

    return -EINVAL;
}

void cRegulatoryCore::setDisconnectionStatus(bool requested,
        eIeee80211NetworkDisconnectChannel_t channel,
        const ::std::string &bss, const ::std::string &regdomain,
        const ::std::string &ifname, bool expect)
{
    _disconnectionstatus._disconnectchannel = channel;
    _disconnectionstatus._disconnectingbss = bss;
    _disconnectionstatus._cfg80211events.expectedRegUpdates(expect);
    _disconnectionstatus._disconnectrequested = requested;
    _disconnectionstatus._ifname = ifname;

    if (!regdomain.empty()) {
        _disconnectionstatus._regdomainchange = regdomain;
        _disconnectionstatus._cfg80211events.regdomChangeRequired();
    }
}

int cRegulatoryCore::disconnectBss(const ::std::string &bss,
        const ::std::string &ifName)
{
    int iRet;
    cBasicServiceSet *ap;
    ::std::string path, alpha2byap;
    cWifiRadio *radio;
    bool setbyuser = false;
    uint16_t frequency = 0;

    if (bss.empty() || ifName.empty())
        return -ENODATA;

    radio = getRadioFromIfname(ifName);
    /* Currently we don't react on the events from
     * those "self-managed" radios */
    if (!radio || radio->getSelfManaged())
        return -ENODATA;

    LOG_INFO("Radio %s has exposed the interface: %s [regdom: %s]",
            radio->getPhyName().c_str(), ifName.c_str(),
            radio->getRegDomain().c_str());

    ap = cWpaSupplicantClient::getInstance()->getBSSFromObjectPath(bss);
    if (!ap) {
        LOG_ERROR("WpaSupplicant does not know anything about: %s",
                bss.c_str());
        return -ENODATA;
    }

    alpha2byap = ap->GetCountry();
    frequency = ap->getFrequency();
    iRet = isAssociatedChannelValid(radio, frequency);
    if (iRet != -EINVAL)
        return -ECANCELED;

    LOG_INFO("Radio \"%s\" in station mode is associated to an external "
            "AP \"%s\" in an invalid channel: %d", radio->getPhyName().c_str(),
            ap->convertSSIDToHexString().c_str(), frequency);

    if (radio->getRegDomain().compare("00"))
        setbyuser = true;

    /**
     * Set the 802.11D reported by the BSS only when the module
     * has the safe regulatory settings, otherwise we just disconnect
     * and continue with the country settings
     */
    if (!radio->getRegDomain().compare(alpha2byap) || setbyuser)
        alpha2byap.clear();

    iRet = computeObjPath(ifName, ap, path);
    if (iRet < 0) {
        LOG_ERROR("Failed to compute the object path for the bss %s "
                "connected via %s", ap->convertSSIDToHexString().c_str(),
                ifName.c_str());
        goto wpa_supplicant;
    }

    LOG_INFO("Computed Object path: %s", path.c_str());
    iRet = IEEE80211ClientFactory::getInstance()->getSTAClientIf()->createConnManServiceProxy(path);
    if (iRet < 0) {
        LOG_ERROR("Failed to create dbus proxy to connman's object: %s [%s/%d]",
                path.c_str(), strerror(-iRet), -iRet);
        goto wpa_supplicant;
    }

    setDisconnectionStatus(true, IEEE80211_NETWORK_DISCONNECT_CHAN_CONNMAN,
            ap->GetObjPath(), alpha2byap, ifName, setbyuser);

    return iRet;

wpa_supplicant:

    /*
     * Right now we are not going through kernel to disconnect if the
     * disconnection request fails to be posted via wpa_supplicant. Because WBL
     * needs NET_ADMIN capability in order to change any network configuration */
    iRet = cWpaSupplicantClient::getInstance()->disconnectBss(ap->GetObjPath());
    if (iRet < 0) {
        LOG_ERROR("Failed to initiate a disconnection via wpa_supplicant "
                "to Bss: %s [%s/%d]", ap->GetObjPath().c_str(),
                strerror(-iRet), -iRet);
    } else {
        setDisconnectionStatus(true, IEEE80211_NETWORK_DISCONNECT_CHAN_WPA_SUPP,
                ap->GetObjPath(), alpha2byap, ifName, setbyuser);
        _disconnectionstatus._status = IEEE80211_NETWORK_DISCONNECTING;
    }

    return iRet;
}

cPendingConnection *cRegulatoryCore::getPendingConnection(const ::std::string &ifName)
{
    ::std::vector<cPendingConnection>::iterator it;
    for (it = _pendingvalidations.begin(); it < _pendingvalidations.end(); ++it)
        if (!(*it)._ifName.compare(ifName))
            return &(*it);
    return NULL;
}

int cRegulatoryCore::removePendingConnection(const ::std::string &ifName)
{
    bool avail = false;
    ::std::vector<cPendingConnection>::iterator it;

    for (it = _pendingvalidations.begin(); it < _pendingvalidations.end();
            ++it)
        if (!(*it)._ifName.compare(ifName)) {
            avail = true;
            break;
        }

    if (avail) {
        _pendingvalidations.erase(it);
        return 0;
    }

    return -ENOENT;
}

void cRegulatoryCore::wpaConnectionNotification(eWpaSuppConnectionState_t state, const ::std::string &ifName,
                const ::std::string &associatedbss)
{
    LOG_INFO("wpa connection status "
            "state: %s "
            "interface: %s "
            "connected bss: %s",
            cWpaSupplicantConnectionNotifier::wpaStateToString(state).c_str(),
            ifName.c_str(), associatedbss.c_str());
}

void cRegulatoryCore::wpaDisconnectionStatus(int status, const ::std::string &ifName)
{
    LOG_INFO("Disconnection to interface \"%s\" is: %s", ifName.c_str(),
            status ? "successful" : "failed");

    if (_disconnectionstatus._disconnectrequested &&
            IEEE80211_NETWORK_DISCONNECT_CHAN_WPA_SUPP ==
                    _disconnectionstatus._disconnectchannel) {
        if (!status)
            _disconnectionstatus._status = IEEE80211_NETWORK_DISCONNECTED;
        else {
            _disconnectionstatus._status = IEEE80211_NETWORK_DISCONNECT_FAILED;
        }
    }
}

void cRegulatoryCore::connectionStatusChange (const IEEE80211Network &network,
        const eIeee80211NetworkStatus_t status)
{
    LOG_INFO("Disconnection status to \"%s\" [ssid: \"%s\"] is: %s",
            network.getObjectpath().c_str(),
            network.getHidden() ? "hidden" : network.getNetworkSsid(),
            IEEE80211STAConnectionStatusNotifier::IEEE80211StatusToString(status).c_str());

    if (_disconnectionstatus._disconnectrequested &&
            IEEE80211_NETWORK_DISCONNECT_CHAN_CONNMAN ==
                    _disconnectionstatus._disconnectchannel)
        _disconnectionstatus._status = status;
}

void cRegulatoryCore::addPendingConnection(const ::std::string &ifName, const ::std::string &bss)
{
    cPendingConnection *connection = NULL, newconn;

    connection = getPendingConnection(ifName);
    if (connection) {
        connection->_bss = bss;
    } else {
        newconn._ifName = ifName;
        newconn._bss = bss;
        _pendingvalidations.push_back(newconn);
    }

    LOG_INFO("_pendingvalidations size: %u", _pendingvalidations.size());
}

/**
 * Initially the plan was to get the connection states from wpa_supplicant and act
 * based on that but it was discovered the state handling from wpa_supplicant is not
 * so clean and there was a mismatch between the kernel and wpa_supplicant when there
 * is a flood of connect/disconnect requests from ConnMan to wpa_supplicant i.e.,
 * wpa_supplicant internally updates the states to clients (like WBL) as soon as the
 * request to nl80211 gets succeeded in some cases which also introduces a clash between
 * WBL's regulatory code and the kernel regulatory code.
 *
 *
 * Now, the idea (I would term it as a solution) is that WBL will monitor the MLME events
 * from kernel instead of wpa_supplicant to avoid this clash.
 *
 * 1) MLME Connected --> Basic 802.11 association (MAC layer (both L1 and L2 in Linux terms))
 *    succeeded and the pending thing is the WPA2-RSN protocol which is performed by the
 *    userspace daemon like wpa_supplicant Linux world but WBL does not really have to
 *    wait for it to initiate a disconnection if the wifi radio is already associated in
 *    an illegal channel. This would also prevent few Mgmt frames being exchanged.
 * 2) MLME Disconnected --> The MAC layer is disconnected
 * */

void cRegulatoryCore::MlmeConnected(const ::std::string &ifName, const ::std::string &associatedbss,
      const unsigned int &ifindex, const ::std::string &macaddr)
{
   (void) ifindex;
   (void) macaddr;
   int iRet;
    bool disconnect = true;

    LOG_INFO("MLME Connected for %s: %s", ifName.c_str(), associatedbss.c_str());

    /**
     * Here is the funda:) Connectivity script loads the wifi module and then sets
     * safe reg domain as default and WBL has to restore the correct regulatory
     * settings from the previous run on every power cycle. But for the very first
     * run (i.e., when the module is virgin) and if there is no GNSS equipped, unclear
     * environment (i.e., 802.11D based evaluation fails to decide on the correct reg
     * settings), the module will still have have the safe regulatory domain set. In
     * such case the module will be capable to see who (AP) is available even in case
     * if the channel is stamped as NO-IR.
     *
     * If the user tries to connect the AP in NO-IR, WBL shall detect that and disconnect
     * the AP and the module
     *  1) set the reg domain as broadcasted by the AP if the 802.11D information is
     *     available. Now this will make sure that the AP is no more available in the
     *     scan results once the reg domain is successfully updated
     *  2) If the 802.11D information is not available then we are kind of looped (maybe
     *     connect --> disconnect --> connect), the user will for sure see and dont
     *     understand why the disconnection happens to an existing network. But do we have
     *     an option? */

    if (!_regchangeinprogress.empty()) {
        LOG_INFO("Connection to \"%s\" will be validated once the regulatory domain "
                "change to \"%s\" gets completed", associatedbss.c_str(),
                _regchangeinprogress.c_str());
        addPendingConnection(ifName, associatedbss);
    } else {

        if (_disconnectionstatus._disconnectrequested) {
            if (!_disconnectionstatus._completed &&
                    _disconnectionstatus._disconnectingbss.compare(associatedbss)) {
                LOG_INFO("Connection to \"%s\" will be validated once the current "
                        "disconnection sequence to \"%s\" gets completed [ifname: %s]",
                         associatedbss.c_str(),
                        _disconnectionstatus._disconnectingbss.c_str(),
                        _disconnectionstatus._ifname.c_str());
                addPendingConnection(ifName, associatedbss);
                disconnect = false;
            }
        }

        if (disconnect) {
            iRet = disconnectBss(associatedbss, ifName);
            if (iRet < 0 && iRet != -ENODATA &&
                    iRet != -ECANCELED) {
                LOG_ERROR("Failed to validate and disconnect the associated bss: %s [%s/%d]",
                        associatedbss.c_str(), strerror(-iRet), -iRet);
            }
        }
    }
}

void cRegulatoryCore::MlmeDisConnected(const ::std::string &ifName, const ::std::string &disconnectedBss)
{
    int iRet;
    cWifiRadio *radio = NULL;
    cPendingConnection *connection = NULL;

    LOG_INFO("MLME Disconnected for %s: %s", ifName.c_str(), disconnectedBss.c_str());

    connection = getPendingConnection(ifName);
    if (connection) {
        LOG_INFO("Connection to \"%s\" will be removed before validation as it "
                "is already disconnected [ifname: %s]", disconnectedBss.c_str(),
                ifName.c_str());
        iRet = removePendingConnection(ifName);
        if (iRet < 0) {
            LOG_ERROR ("Failed to remove the pending connection in disconncted "
                    "state to:%s [ifname: %s]", disconnectedBss.c_str(),
                    ifName.c_str());
        }
    }

    if (!_disconnectionstatus._disconnectrequested) {
        radio = getRadioFromIfname(ifName);
        if (radio && !radio->getSelfManaged()) {
            _cfg80211regevents.expectedRegUpdates(radio->getRegDomain().compare("00") == 0 ?
                    false : true);
            LOG_INFO("Disconnection happened on \"%s\" to \"%s\" but not initiated by "
                    "the regulatory core", ifName.c_str(), disconnectedBss.c_str());
        }
    }
}

    }
}

/** @} */
