/**
 * @file Poller.cpp
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief Main file for Wifi_Business_Logic.
 *
 *
 * @{
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#include <sys/epoll.h>
#include "asf/core/Logger.h"
#include "Poller.h"

#define EPOLL_DEF_MAX_FD_COUNTS 4

namespace org {
    namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cPoller, Debug);

int cIOChannel::iSetPollEvent(const ePollEvent eEvt)
{
    if (m_ePollEvent == eEvt)
        return -EALREADY;

    m_ePollEvent = eEvt;

    if (eGetPollState() == POLL_STATE_REGISTERED)
        iSetPollState(POLL_STATE_MODIFY);
    return 0;
}

int cIOChannel::iSetPollState(const ePollFDState eState)
{
    if (m_ePollFdState == eState)
        return -EALREADY;

    m_ePollFdState = eState;
    return 0;
}

const ePollEvent &cIOChannel::eGetPollEvent()
{
    return m_ePollEvent;
}

int cIOChannel::iSetPollFd(const int iIOFd)
{
    if (m_iFileDesc == iIOFd)
        return -EALREADY;

    m_iFileDesc = iIOFd;
    return 0;
}

const int &cIOChannel::iGetPollFd()
{
    return m_iFileDesc;
}

const ePollFDState &cIOChannel::eGetPollState()
{
    return m_ePollFdState;
}

int cIOChannel::iGetEpollEvent(const ePollFDState eEvent)
{
    int iEvent = -1;

    switch (eEvent) {
    default:
        break;
    case POLL_STATE_ADD:
        iEvent = EPOLL_CTL_ADD;
        break;
    case POLL_STATE_MODIFY:
        iEvent = EPOLL_CTL_MOD;
        break;
    case POLL_STATE_REMOVE:
        iEvent = EPOLL_CTL_DEL;
        break;
    }

    return iEvent;
}


int cIOChannel::iGetEpollType(const ePollEvent eEvent)
{
    int iEvent = -1;

    switch (eEvent) {
    default:
        break;
    case POLL_EVENT_READ:
        iEvent = EPOLLIN;
        break;
    case POLL_EVENT_WRITE:
        iEvent = EPOLLOUT;
        break;
    }

    return iEvent;
}

cIOChannel::cIOChannel(const ePollEvent eEvent)
{
    m_ePollEvent = eEvent;
    m_ePollFdState = POLL_STATE_ADD;
    m_iFileDesc = -1;
}

cIOChannel::~cIOChannel()
{
}

int cPoller::iRegisterForPoll(cIOChannel *pListener)
{
    int iRet = 0, iWrite, iData;
    std::vector<cIOChannel *>::iterator it;

    LOG_INFO ("Registering the iochannel: %p", pListener);

    if (!pListener || pListener->iGetPollFd() == -1)
        return -EINVAL;

    if (!m_pPipe)
        return -ENOTCONN;

    m_oLock.s16Lock();
    for (it = m_lListeners.begin(); it < m_lListeners.end(); it++)
        if (pListener == *it) {
            m_oLock.vUnlock();
            return -EALREADY;
        }

    m_lListeners.push_back (pListener);
    iWrite = m_pPipe->iGetPipe(PIPE_END_WRITE);
    if (iWrite != -1) {
        iData = (int) PIPE_CMD_FD_ADDED;
        iRet = static_cast<int> (write (iWrite, &iData, sizeof(iData)));
        if (iRet < 0) {
            iRet = -errno;
            LOG_ERROR ("Failed to write to the pipe: %s/%d", strerror (-iRet),
                    -iRet);
        }
    }

    m_oLock.vUnlock();
    return iRet;
}

int cPoller::iUnRegisterFromPoll(cIOChannel *pListener)
{
    int iFound = 0, iWrite,
            iData, iRet = -ENOENT;
    std::vector<cIOChannel *>::iterator it;

    LOG_INFO ("UnRegistering the iochannel: %p", pListener);

    if (!pListener || pListener->iGetPollFd() == -1)
        return -EINVAL;

    if (!m_pPipe)
        return -ENOTCONN;

    m_oLock.s16Lock();
    for (it = m_lListeners.begin(); it < m_lListeners.end(); it++)
        if (pListener == *it) {
            m_lListeners.erase (it);
            iFound = 1;
            break;
        }

    if (iFound) {
        iRet = 0;
        iWrite = m_pPipe->iGetPipe(PIPE_END_WRITE);
        if (iWrite != -1) {
            iData = (int) PIPE_CMD_FD_DELETED;
            iRet = static_cast<int> (write (iWrite, &iData,sizeof(iData)));
            if (iRet < 0){
                iRet = -errno;
                LOG_ERROR ("Failed to write to the pipe: %s/%d", strerror (-iRet),
                        -iRet);
            }
        }
    }

    m_oLock.vUnlock();
    return iRet;
}

int cPoller::vRunEventLoop()
{
    bool bContinue = false;
    int iRet, iEvent, iIndex;
    struct epoll_event event, *oldbuf;
    std::vector<cIOChannel *>::iterator it;

    if (m_iPollFd == -1) {
        LOG_ERROR ("Epoll instance is not created");
        return -ENOTCONN;
    }

    LOG_INFO ("++++++Running Event loop++++++");

    while (true) {

        if (_terminate) {
            LOG_INFO("Terminating the WBL Service component");
            break;
        }

        m_oLock.s16Lock();
        LOG_INFO ("listeners size : %d", m_lListeners.size());
        for (it = m_lListeners.begin(); it < m_lListeners.end();
                ++it) {

            if (!*it || (*it)->eGetPollState () == POLL_STATE_REGISTERED)
                continue;
            iRet = (*it)->iGetEpollType ((*it)->eGetPollEvent ());
            if (iRet < 0)
                continue;

            event.data.fd = (*it)->iGetPollFd ();
            event.events = iRet;

            iEvent = (*it)->iGetEpollEvent ((*it)->eGetPollState ());
            if (iEvent < 0) {
                LOG_ERROR ("Invalid Poll event for the file descriptor: %d",
                        (*it)->iGetPollFd ());
                continue;
            }

            iRet = epoll_ctl (m_iPollFd, iEvent, (*it)->iGetPollFd (), &event);
            if (iRet < 0) {
                LOG_ERROR ("Failed to Add the fd [%d] from %p to the poller: %s/%d",
                        (*it)->iGetPollFd (), *it, strerror (errno), errno);
            } else {

                if (iEvent == EPOLL_CTL_ADD)
                    m_iCount++;
                else if (iEvent == EPOLL_CTL_DEL)
                    m_iCount--;

                (*it)->iSetPollState (POLL_STATE_REGISTERED);
                LOG_INFO ("Successfully added the fd [%d] to the poller",
                        (*it)->iGetPollFd ());
            }
        }
        m_oLock.vUnlock();

        if (m_iCount > m_iMaxFds) {
            LOG_INFO ("Reallocating the event buffer: %d --> %d", m_iMaxFds,
                    m_iMaxFds * 2);
            oldbuf = m_pEvents;
            m_iMaxFds = m_iMaxFds * 2;
            m_pEvents = (struct epoll_event*) realloc (m_pEvents,
                    m_iMaxFds * sizeof (*m_pEvents));
            if (!m_pEvents) {
                free (oldbuf);
                return -1;
            }
        }

        iRet = epoll_wait (m_iPollFd, m_pEvents, m_iCount, -1);
        for (iIndex = 0; iIndex < iRet; iIndex++) {
            for (it = m_lListeners.begin(); it < m_lListeners.end(); ++it) {
                if (!*it || m_pEvents[iIndex].data.fd != (*it)->iGetPollFd ())
                    continue;
                LOG_INFO ("Notifying Listener: %p", *it);
                bContinue = (*it)->bProcessEvent (m_pEvents[iIndex].data.fd,
                        m_pEvents[iIndex].events);
                if (!bContinue) {
                    LOG_ERROR ("Removing [%d] from poller due to failure",
                            (*it)->iGetPollFd ());
                }
            }
        }
    }

    return 0;
}

bool cPoller::bProcessEvent(int iPollFd, int iEvent)
{
    int iRes, iRead, iData = 0;

    LOG_INFO ("File desc: %d, event : %d", iPollFd, iEvent);

    if (!m_pPipe)
        return false;

    iRead = m_pPipe->iGetPipe(PIPE_END_READ);
    if (iRead != iPollFd)
        return true;

    if ((iEvent & EPOLLERR) || (iEvent & EPOLLHUP)) {
        LOG_ERROR ("Error occured for the fd: %d", iEvent);
        return false;
    }

    iRes = static_cast<int> (read (iPollFd, &iData,sizeof(iData)));
    if (iRes < 0) {
        LOG_ERROR ("Failed to read from the pipe: %s/%d", strerror (errno),
                errno);
    } else {
        LOG_INFO ("Event received from the pipe: %s/%d",
                m_pPipe->sGetPipeEvent ((ePipeCmd)iData).c_str (), iData);
        if (PIPE_CMD_TERMINATE == (ePipeCmd) iData)
            _terminate = true;
    }

    return true;
}

int cPoller::stopEventLoop()
{
    int iWrite, iData, iRet = -ENOTCONN;

    if (!m_pPipe)
        return -ENOTCONN;

    m_oLock.s16Lock();
    iWrite = m_pPipe->iGetPipe(PIPE_END_WRITE);
    if (iWrite != -1) {
        iData = (int) PIPE_CMD_TERMINATE;
        iRet = static_cast<int> (write (iWrite, &iData,sizeof(iData)));
        if (iRet < 0){
            iRet = -errno;
            LOG_ERROR ("Failed to write to the pipe: %s/%d", strerror (-iRet),
                    -iRet);
        }
    }

    m_oLock.vUnlock();
    return iRet;
}

cPoller::cPoller() :
        cIOChannel (POLL_EVENT_READ)
{
    int iRet;

    _terminate = false;
    m_iCount = 0;
    m_iMaxFds = EPOLL_DEF_MAX_FD_COUNTS;
    m_pPipe = new cPipe ();

    m_pEvents = (struct epoll_event*) calloc (m_iMaxFds, sizeof (*m_pEvents));
    if (!m_pEvents)
        LOG_ERROR ("Failed to allocate the epoller events");

    m_iPollFd = epoll_create1 (EPOLL_CLOEXEC);
    if (m_iPollFd < 0)
        LOG_ERROR ("Failed to create the EPOLLER : %s/%d",
                strerror (errno), errno);

    iRet = iSetPollFd (m_pPipe->iGetPipe(PIPE_END_READ));
    if (!iRet) {
        m_lListeners.push_back (this);
    } else {
        LOG_ERROR ("Failed to set the Poll FD : %s/%d", strerror (-iRet), iRet);
    }
}

cPoller::~cPoller()
{
    if (m_iPollFd != -1)
        close (m_iPollFd);
    m_lListeners.clear ();
    free (m_pEvents);
    delete m_pPipe;
}

    }
}

/** @} */
