/**
 * @file NetlinkSocket.cpp
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief Main file for Wifi_Business_Logic.
 *
 *
 * @{
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <errno.h>
#include <string.h>
#include <netlink/socket.h>
#include <netlink/genl/genl.h>
#include <netlink/genl/ctrl.h>
#include <linux/nl80211.h>
#include <linux/if_ether.h>
#include "asf/core/Logger.h"
#include "NetlinkObserver.h"
#include "NetlinkSocket.h"

#define WBL_OBJ_PATH			"org/bosch/wbl/wifi_business_logic"
#define NL_CLASS_NAME			"cNetLink"
//#define NL_CLASS_PATH			WBL_OBJ_PATH "/" NL_CLASS_NAME
#define NL_CLASS_PATH			"wifi_business_logic/Regulation"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cNetLink, Debug);

cNetLink::cNetLink(const eNlSocketType eType)
{
	int iRet;

	m_eSkType = eType;
	m_iRealNlSk = -1;
	m_pNetlinkCb = nullptr;
	m_pNetlinkSk = nullptr;

	iRet = bInitializeNlChannel(m_eSkType);
	if (!iRet) {
		iRet = bInitializeNlCB();
		if (iRet < 0) {
			LOG_ERROR ("Failed to initialize the callback for family : %s",
					sNlSockToString(m_eSkType).c_str());
			if (iRet != -EALREADY) {
				nl_socket_free (m_pNetlinkSk);
				m_pNetlinkSk = nullptr;
			}
		}
	} else {
        LOG_ERROR ("Failed to initialize the socket for family : %s",
        		sNlSockToString(m_eSkType).c_str());
	}
}

cNetLink::~cNetLink()
{
	if (m_pNetlinkCb)
		nl_cb_put (m_pNetlinkCb);
	if (m_pNetlinkSk)
		nl_socket_free (m_pNetlinkSk);
}

int &cNetLink::iGetUnixSocket()
{
	return m_iRealNlSk;
}

struct nl_cb *cNetLink::pGetNlCB()
{
	return m_pNetlinkCb;
}

struct nl_sock *cNetLink::pGetNlSock()
{
	return m_pNetlinkSk;
}

int cNetLink::getSocketType()
{
	return iGetRealNlSockType(m_eSkType);
}

void cNetLink::vNotifyObservers(cNlMessage *msg)
{
	std::vector<cNlObserver *>::iterator it;

	if (!msg)
		return;

	for (it = _nleventobservers.begin(); it < _nleventobservers.end(); it++) {
		if ((*it)->getMessageType() == msg->getNlMessageId()) {
			LOG_INFO ("Notifying the client [%p] on event: %d", *it,
					(*it)->getMessageType());
			(*it)->bNetlinkEvent(msg);
		}
	}
}

int cNetLink::iRegisterForNlEvents(cNlObserver *observer)
{
	std::vector<cNlObserver *>::iterator it;

	LOG_INFO ("Registering the observer: %p", observer);

	if (!observer)
		return -EINVAL;

	for (it = _nleventobservers.begin(); it < _nleventobservers.end(); it++)
		if (observer == *it)
			return -EALREADY;

	_nleventobservers.push_back (observer);
	return 0;
}

int cNetLink::iUnRegisterFromNlEvents(cNlObserver *observer)
{
	std::vector<cNlObserver *>::iterator it;

	LOG_INFO ("UnRegistering the observer: %p", observer);

	if (!observer)
		return -EINVAL;

	for (it = _nleventobservers.begin(); it < _nleventobservers.end(); it++)
		if (observer == *it) {
			_nleventobservers.erase (it);
			return 0;
		}

	return -ENOENT;
}

int cNetLink::iGetRealNlSockType(const eNlSocketType &eType)
{
	switch (eType) {
	case NL_SOCKET_TYPE_ROUTE:
		return NETLINK_ROUTE;
	case NL_SOCKET_TYPE_UNUSED:
		return NETLINK_UNUSED;
	case NL_SOCKET_TYPE_USERSOCK:
		return NETLINK_USERSOCK;
	case NL_SOCKET_TYPE_FIREWALL:
		return NETLINK_FIREWALL;
	case NL_SOCKET_TYPE_SOCK_DIAG:
		return NETLINK_SOCK_DIAG;
	case NL_SOCKET_TYPE_NFLOG:
		return NETLINK_NFLOG;
	case NL_SOCKET_TYPE_XFRM:
		return NETLINK_XFRM;
	case NL_SOCKET_TYPE_SELINUX:
		return NETLINK_SELINUX;
	case NL_SOCKET_TYPE_ISCSI:
		return NETLINK_ISCSI;
	case NL_SOCKET_TYPE_AUDIT:
		return NETLINK_AUDIT;
	case NL_SOCKET_TYPE_FIB_LOOKUP:
		return NETLINK_FIB_LOOKUP;
	case NL_SOCKET_TYPE_CONNECTOR:
		return NETLINK_CONNECTOR;
	case NL_SOCKET_TYPE_NETFILITER:
		return NETLINK_NETFILTER;
	case NL_SOCKET_TYPE_IP6_FW:
		return NETLINK_IP6_FW;
	case NL_SOCKET_TYPE_DNRTMSG:
		return NETLINK_DNRTMSG;
	case NL_SOCKET_TYPE_KOBJ_UEVENT:
		return NETLINK_KOBJECT_UEVENT;
	case NL_SOCKET_TYPE_GENERIC:
		return NETLINK_GENERIC;
	case NL_SOCKET_TYPE_SCSITRANSPORT:
		return NETLINK_SCSITRANSPORT;
	case NL_SOCKET_TYPE_ECRYPTFS:
		return NETLINK_ECRYPTFS;
	case NL_SOCKET_TYPE_RDMA:
		return NETLINK_RDMA;
	case NL_SOCKET_TYPE_CRYPTO:
		return NETLINK_CRYPTO;
	}

	return -1;
}

const std::string cNetLink::sNlSockToString(const eNlSocketType &eType)
{
	switch (eType) {
	case NL_SOCKET_TYPE_ROUTE:
		return "NL_SOCKET_TYPE_ROUTE";
	case NL_SOCKET_TYPE_UNUSED:
		return "NL_SOCKET_TYPE_UNUSED";
	case NL_SOCKET_TYPE_USERSOCK:
		return "NL_SOCKET_TYPE_USERSOCK";
	case NL_SOCKET_TYPE_FIREWALL:
		return "NL_SOCKET_TYPE_FIREWALL";
	case NL_SOCKET_TYPE_SOCK_DIAG:
		return "NL_SOCKET_TYPE_SOCK_DIAG";
	case NL_SOCKET_TYPE_NFLOG:
		return "NL_SOCKET_TYPE_NFLOG";
	case NL_SOCKET_TYPE_XFRM:
		return "NL_SOCKET_TYPE_XFRM";
	case NL_SOCKET_TYPE_SELINUX:
		return "NL_SOCKET_TYPE_SELINUX";
	case NL_SOCKET_TYPE_ISCSI:
		return "NL_SOCKET_TYPE_ISCSI";
	case NL_SOCKET_TYPE_AUDIT:
		return "NL_SOCKET_TYPE_AUDIT";
	case NL_SOCKET_TYPE_FIB_LOOKUP:
		return "NL_SOCKET_TYPE_FIB_LOOKUP";
	case NL_SOCKET_TYPE_CONNECTOR:
		return "NL_SOCKET_TYPE_CONNECTOR";
	case NL_SOCKET_TYPE_NETFILITER:
		return "NL_SOCKET_TYPE_NETFILITER";
	case NL_SOCKET_TYPE_IP6_FW:
		return "NL_SOCKET_TYPE_IP6_FW";
	case NL_SOCKET_TYPE_DNRTMSG:
		return "NL_SOCKET_TYPE_DNRTMSG";
	case NL_SOCKET_TYPE_KOBJ_UEVENT:
		return "NL_SOCKET_TYPE_KOBJ_UEVENT";
	case NL_SOCKET_TYPE_GENERIC:
		return "NL_SOCKET_TYPE_GENERIC";
	case NL_SOCKET_TYPE_SCSITRANSPORT:
		return "NL_SOCKET_TYPE_SCSITRANSPORT";
	case NL_SOCKET_TYPE_ECRYPTFS:
		return "NL_SOCKET_TYPE_ECRYPTFS";
	case NL_SOCKET_TYPE_RDMA:
		return "NL_SOCKET_TYPE_RDMA";
	case NL_SOCKET_TYPE_CRYPTO:
		return "NL_SOCKET_TYPE_CRYPTO";
	}
	return "UNKNOWN";
}


int cNetLink::bInitializeNlChannel(const eNlSocketType &eType)
{
    int iRet = -ENOENT, iRealNlSock;

    if (m_iRealNlSk != -1)
        return -EALREADY;

    iRealNlSock = iGetRealNlSockType (eType);
    if (iRealNlSock < 0)
    	return -EINVAL;

    m_pNetlinkSk = nl_socket_alloc ();
    if (!m_pNetlinkSk)
    	return -ENOMEM;

    iRet = nl_connect (m_pNetlinkSk, iRealNlSock);
    if (iRet < 0) {
        LOG_ERROR ("Failed to connect to the genl socket : %s/%d",
        		nl_geterror (-iRet), -iRet);
    	nl_socket_free (m_pNetlinkSk);
    	m_pNetlinkSk = nullptr;
    	return iRet;
    }

    m_iRealNlSk = nl_socket_get_fd (m_pNetlinkSk);
    iRet = fcntl (m_iRealNlSk, F_SETFL, O_NONBLOCK | O_CLOEXEC);
    if (iRet < 0) {
    	iRet = -errno;
    	LOG_ERROR ("fcntl failed for [sock : %d]: %s/%d",
    			m_iRealNlSk, strerror (-iRet), -iRet);
    	nl_socket_free (m_pNetlinkSk);
    	m_pNetlinkSk = nullptr;
    } else {
    	LOG_INFO ("Created NL socket: %d", m_iRealNlSk);
    }

    return iRet;
}

int cNetLink::iGenericErrorHandler(struct sockaddr_nl *nla, struct nlmsgerr *err,
		void *arg)
{
	(void) nla, (void) arg;
	::asf::core::Logger oLogger (NL_CLASS_PATH,
		::asf::core::Logger::Info);

	if (!err)
		return NL_STOP;

    oLogger.error ("Error received for Nl message with sequence no [%d] : %s/%d "
    		"system error : %s/%d", __LINE__, __FILE__, err->msg.nlmsg_seq,
    		nl_geterror (err->error), err->error, strerror (-err->error), -err->error);
    return NL_STOP;
}

int cNetLink::iGenericFinishHandler(struct nl_msg *msg, void *arg)
{
	(void) arg;
    struct nlmsghdr *nlhdr;
	::asf::core::Logger oLogger (NL_CLASS_PATH,
			::asf::core::Logger::Info);

    if (!msg)
    	return NL_SKIP;

    nlhdr = nlmsg_hdr (msg);
    oLogger.info ("Message with seq no [%d] finished successfully", __LINE__,
    		__FILE__, nlhdr->nlmsg_seq);
	return NL_SKIP;
}

int cNetLink::iGenericAckHandler(struct nl_msg *msg, void *arg)
{
	(void) arg;
	struct nlmsghdr *nlhdr;
	::asf::core::Logger oLogger (NL_CLASS_PATH,
			::asf::core::Logger::Info);

    if (!msg)
    	return NL_STOP;

    nlhdr = nlmsg_hdr (msg);
    oLogger.info ("Message with seq no [%d] acknowledged", __LINE__,
    		__FILE__, nlhdr->nlmsg_seq);
	return NL_STOP;
}

int cNetLink::iGenericSeqCheckHandler(struct nl_msg *msg, void *arg)
{
	(void) arg;
	struct nlmsghdr *nlhdr;
	::asf::core::Logger oLogger (NL_CLASS_PATH,
			::asf::core::Logger::Info);

    if (!msg)
    	return NL_OK;

    nlhdr = nlmsg_hdr (msg);
    oLogger.info ("Sequence check for Message with seq no [%d]", __LINE__,
    		__FILE__, nlhdr->nlmsg_seq);
	return NL_OK;
}

int cNetLink::bInitializeNlCB ()
{
	if (m_pNetlinkCb)
		return -EALREADY;

    m_pNetlinkCb = nl_cb_alloc (NL_CB_DEFAULT);
    if (!m_pNetlinkCb)
        return -ENOMEM;

    nl_cb_err (m_pNetlinkCb, NL_CB_CUSTOM, iGenericErrorHandler, nullptr);
    nl_cb_set (m_pNetlinkCb, NL_CB_FINISH, NL_CB_CUSTOM, iGenericFinishHandler, nullptr);
    nl_cb_set (m_pNetlinkCb, NL_CB_ACK, NL_CB_CUSTOM, iGenericAckHandler, nullptr);
    nl_cb_set (m_pNetlinkCb, NL_CB_SEQ_CHECK, NL_CB_CUSTOM, iGenericSeqCheckHandler, nullptr);
    return 0;
}

	}
}

/** @} */
