/**
 * @file LoopbackClient.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include <errno.h>
#include "LoopbackClient.h"
#include "DBusProxyFactory.h"
#include "ILoopbackProxyIf.h"
#include "WBLPortsDefines.h"
#include "WBLDefines.h"
#include "RegulatoryCore.h"
#include "WifiMlmeUpdates.h"
#include "LoopbackTypes.h"

namespace org {
namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/Regulation", cLoopbackClient, Info);

cLoopbackClient::cLoopbackClient()
{
	int iRet;
	cILoopbackProxyIf *lbproxy;

	__eventLoopStartRequested = false;
	lbproxy = DBusProxyFactory::getInstance()->getLoopbackProxyIf();
	iRet = lbproxy->createProxy(sWBLServicePort, this);
	if (iRet == 0) {
		iRet = lbproxy->setCallbackIf(sWBLServicePort,this);
		if (iRet < 0) {
			LOG_ERROR ("Failed to add the callback to the loopback proxy [portName: %s] %s/%d",
					sWBLServicePort.c_str(), strerror(-iRet), -iRet);
		}
	} else {
		LOG_ERROR ("Failed to create the loopback proxy [portName: %s] %s/%d",
				sWBLServicePort.c_str(), strerror(-iRet), -iRet);
	}
}

cLoopbackClient::~cLoopbackClient()
{
   try
   {
      int iRet;
      cILoopbackProxyIf *lbproxy;

      lbproxy = DBusProxyFactory::getInstance()->getLoopbackProxyIf();
      iRet = lbproxy->destroyProxy(sWBLServicePort);
      if (iRet == 0) {
         iRet = lbproxy->setCallbackIf(sWBLServicePort,nullptr);
         if (iRet < 0) {
            LOG_ERROR ("Failed to remove the callback to the loopback proxy [portName: %s] %s/%d",
                  sWBLServicePort.c_str(), strerror(-iRet), -iRet);
         }
      } else {
         LOG_ERROR ("Failed to destroy the loopback proxy [portName: %s] %s/%d",
               sWBLServicePort.c_str(), strerror(-iRet), -iRet);
      }
   }catch(...){}
}

void cLoopbackClient::onLoopbackServiceAvailable(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::asf::core::ServiceState previousState,
		const ::asf::core::ServiceState currentState)
{
	(void) proxy;
	(void) previousState;
	(void) currentState;

	act_t token;
	cILoopbackProxyIf *lbproxy;

	LOG_INFO("Loopback Proxy is connected");

	lbproxy = DBusProxyFactory::getInstance()->getLoopbackProxyIf();
	token = lbproxy->sendCurrentRegSettingsGet(sWBLServicePort, *this);
	if (token == DEFAULT_ACT)
		LOG_ERROR ("Failed to Get the current Regulatory settings");
}

void cLoopbackClient::onLoopbackServiceUnAvailable(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::asf::core::ServiceState previousState,
		const ::asf::core::ServiceState currentState)
{
	(void) proxy;
	(void) previousState;
	(void) currentState;

	LOG_INFO("Loopback Proxy is disconnected");
}

void cLoopbackClient::onRegulatoryupdateRegisterConfirmation(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy)
{
	(void) proxy;

	LOG_INFO("On Country Update register confirmation");
}

void cLoopbackClient::onStarteventloopResponse(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cStarteventloopResponse >& response)
{
	(void) proxy;
	(void) response;

	LOG_INFO ("onStarteventloopResponse is received");
}

void cLoopbackClient::onStarteventloopError(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cStarteventloopError >& error)
{
	(void) proxy;

	switch(error->getField()) {
	case ::asf::cms::CmsTypes::BaseError::E_SYSTEM_ERROR:
		LOG_INFO ("on%sError: %s", "Starteventloop", "E_SYSTEM_ERROR");
		break;
	default:
		break;
	}
}

void cLoopbackClient::onSetpropertyResponse(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cSetpropertyResponse >& response)
{
	(void) proxy;
	(void) response;

	LOG_INFO ("onSetpropertyResponse received");
}

void cLoopbackClient::onSetpropertyError(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cSetpropertyError >& error)
{
	(void) proxy;

	switch(error->getField()) {
	case ::asf::cms::CmsTypes::BaseError::E_SYSTEM_ERROR:
		LOG_INFO ("on%sError: %s", "Setproperty", "E_SYSTEM_ERROR");
		break;
	default:
		break;
	}
}

void cLoopbackClient::onCurrentRegSettingsGetUpdate(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
			const ::boost::shared_ptr< cCurrentRegSettingsUpdate >& update)
{
	(void) proxy;
	(void) update;

	LOG_INFO("onCurrentRegSettingsGetUpdate Get");
}

void cLoopbackClient::onCurrentRegSettingsUpdate(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
			const ::boost::shared_ptr< cCurrentRegSettingsUpdate >& update)
{
	(void) proxy;

	bool signal = true;
	::std::string country;
	cRegulatoryUpdate regupdate;
	act_t token;
	cILoopbackProxyIf *lbproxy;

	regupdate = update->getData();
	country = regupdate.getCountry();

	LOG_INFO("On CurrentRegSettings Update event: %s", regupdate.getGlobal() ?
			country.c_str() : "Specific to Radio");

	if (regupdate.getGlobal() && country.empty())
		signal = false;

	if (signal)
		cRegulatoryCore::getInstance()->loopbacknotification(&regupdate);

	if (__eventLoopStartRequested)
		return;

	lbproxy = DBusProxyFactory::getInstance()->getLoopbackProxyIf();
	token = lbproxy->sendStarteventloopRequest(sWBLServicePort, *this);
	if (token == DEFAULT_ACT)
		LOG_ERROR ("Failed to Start the event loop of other thread");

	__eventLoopStartRequested = true;
}

void cLoopbackClient::onCurrentRegSettingsRegisterError(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
			const ::boost::shared_ptr< cCurrentRegSettingsRegisterError >& error)
{
	(void) proxy;

	switch(error->getField()) {
	case ::asf::cms::CmsTypes::BaseError::E_SYSTEM_ERROR:
		LOG_ERROR("on%sError (%s) has received", "CurrentRegSettingsRegister");
		break;
	default:
		break;
	}
}

void cLoopbackClient::onCurrentRegSettingsGetError(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
			const ::boost::shared_ptr< cCurrentRegSettingsGetError >& error)
{
	(void) proxy;
	act_t token;
	cILoopbackProxyIf *lbproxy;

	switch(error->getField()) {
	case ::asf::cms::CmsTypes::BaseError::E_SYSTEM_ERROR:
		LOG_ERROR("on%sError (%s) has received", "CurrentRegSettingsGet");
		break;
	default:
		break;
	}

	if (__eventLoopStartRequested)
		return;

	lbproxy = DBusProxyFactory::getInstance()->getLoopbackProxyIf();
	token = lbproxy->sendStarteventloopRequest(sWBLServicePort, *this);
	if (token == DEFAULT_ACT)
		LOG_ERROR ("Failed to Start the event loop of other thread");

	__eventLoopStartRequested = true;
}

void cLoopbackClient::onMediaConnectedUpdate(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cMediaConnectedUpdate >& update)
{
	(void) proxy;
	LOG_DEBUG("Mlme Connection update");
	cWifiMlmeUpdates::getInstance()->mlmeConnected(update->getData());
}

void cLoopbackClient::onMediaConnectedRegisterError(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cMediaConnectedRegisterError >& error)
{
	(void) proxy;

	switch(error->getField()) {
	case ::asf::cms::CmsTypes::BaseError::E_SYSTEM_ERROR:
		LOG_ERROR("on%sError (%s) is not implemented yet", "MediaConnectedRegister", "E_SYSTEM_ERROR");
		break;
	default:
		break;
	}
}

void cLoopbackClient::onMediaConnectedGetError(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cMediaConnectedGetError >& error)
{
	(void) proxy;

	switch(error->getField()) {
	case ::asf::cms::CmsTypes::BaseError::E_SYSTEM_ERROR:
		LOG_ERROR("on%sError (%s) is not implemented yet", "MediaConnectedGet", "E_SYSTEM_ERROR");
		break;
	default:
		break;
	}
}

void cLoopbackClient::onMediaDisconnectedUpdate(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cMediaDisconnectedUpdate >& update)
{
	(void) proxy;
	LOG_DEBUG("Mlme DisConnection update");
	cWifiMlmeUpdates::getInstance()->mlmeDisConnected(update->getData());
}

void cLoopbackClient::onMediaDisconnectedRegisterError(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cMediaDisconnectedRegisterError >& error)
{
	(void) proxy;

	switch(error->getField()) {
	case ::asf::cms::CmsTypes::BaseError::E_SYSTEM_ERROR:
		LOG_ERROR("on%sError (%s) is not implemented yet", "MediaConnectedRegister", "E_SYSTEM_ERROR");
		break;
	default:
		break;
	}
}

void cLoopbackClient::onMediaDisconnectedGetError(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cMediaDisconnectedGetError >& error)
{
	(void) proxy;

	switch(error->getField()) {
	case ::asf::cms::CmsTypes::BaseError::E_SYSTEM_ERROR:
		LOG_ERROR("on%sError (%s) is not implemented yet", "MediaDisconnectedGet", "E_SYSTEM_ERROR");
		break;
	default:
		break;
	}
}

void cLoopbackClient::onMediaConnectedGetUpdate(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cMediaConnectedUpdate >& update)
{
	(void) proxy;
	(void) update;

	LOG_DEBUG("onMediaConnectedGetUpdate:: Mlme Connection update");
}

void cLoopbackClient::onMediaDisconnectedGetUpdate(const ::boost::shared_ptr< cWBLASFServiceProxy >& proxy,
		const ::boost::shared_ptr< cMediaDisconnectedUpdate >& update)
{
	(void) proxy;
	(void) update;

	LOG_DEBUG("onMediaDisconnectedGetUpdate:: Mlme DisConnection update");
}

}
}

/** @} */
