/**
 * @file KDSConfiguration.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "KDSConfiguration.h"
#include "string.h"
#include "DbusHelper.h"
#include "asf/core/Utf8String.h"
#include "sstream"
#include "DataProvider.h"
#include <unistd.h>

#ifndef DP_S_IMPORT_INTERFACE_BASE

#define DP_S_IMPORT_INTERFACE_BASE
#include "dp_if.h"


#endif

#define WBL_KDS_VEHICLE_NAME_CONFIG_KEY         0XA030
#define WBL_KDS_VEHICLE_NAME_CONFIG_LEN         22
#define WBL_VEHICLE_NAME_LEN                    50

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cKDSConfiguration, Debug);

cKDSConfiguration::cKDSConfiguration()
{
    _wifiFCCConfiguration = 0;
   _wifi5GHzPDSupport = (bool)DataProvider::getInstance()->getWBLConfiguration()._5GHzPDSupportDefault;
   _wifi5GHzCDSupport = (bool)DataProvider::getInstance()->getWBLConfiguration()._5GHzCDSupportDefault;
   _supportCPW = (bool)DataProvider::getInstance()->getWBLConfiguration()._CPWSupportDefault;
   _supportAAW = (bool)DataProvider::getInstance()->getWBLConfiguration()._AAWSupportDefault;
   readWiFi5GHzConfiguration();
    readCPWSupportData();
   readAAWSupportData();
   getVehicleFriendlyName();
    getManufacturerConfigurationData();
}

cKDSConfiguration::~cKDSConfiguration()
{
}

void cKDSConfiguration::readWiFi5GHzConfiguration()
{
   LOG_INFO("cKDSConfiguration::%s", __func__);
   uint8_t wifi5GHzValue = 0;
   uint8_t wifi5GhzCDValue = 0;
   int32_t _s32Res = 0;
   if(DataProvider::getInstance()->getWBLConfiguration()._5GhzPDSupportRead)
   {
      _s32Res = DP_s32GetConfigItem("CMVariantCoding", "WIFI5GHz", &wifi5GHzValue, sizeof(uint8_t));
      if (_s32Res == DP_S32_NO_ERR)
      {
         LOG_DEBUG ("cKDSConfiguration : 5GHz PD value from Datapool: %d",wifi5GHzValue);
         _wifi5GHzPDSupport = (bool)wifi5GHzValue;
      }
      else
      {
         LOG_ERROR ("cKDSConfiguration : Reading WiFi 5GHz PD support ERR: Setting Default Value");
      }
   }
   else
   {
      LOG_ERROR ("cKDSConfiguration : Read Value not enabled for 5GHz PD Support : Setting Default Value");
   }
   LOG_DEBUG ("cKDSConfiguration : WiFi 5 GHz PD Support: %d",_wifi5GHzPDSupport);

   if(DataProvider::getInstance()->getWBLConfiguration()._5GhzCDSupportRead)
   {
      _s32Res = DP_s32GetConfigItem("SystemConfiguration1", "Wifi5GHZSupport", &wifi5GhzCDValue, sizeof(uint8_t));
      if (_s32Res == DP_S32_NO_ERR)
      {
         LOG_DEBUG ("cKDSConfiguration : Read KDS Configuration from Datapool, 5GHz CD Support: %d",wifi5GhzCDValue);
         _wifi5GHzCDSupport = (bool)wifi5GhzCDValue;
      }
      else
      {
         LOG_ERROR ("cKDSConfiguration : Reading 5GHz CD support ERR : Setting Default Value");
      }
   }
   else
   {
      LOG_ERROR ("cKDSConfiguration : Read Value not enabled for 5GHz CD Support : Setting Default Value");
   }
   LOG_DEBUG ("cKDSConfiguration : 5GHz CD Support: %d",_wifi5GHzCDSupport);
}

void cKDSConfiguration::readAAWSupportData()
{
    LOG_INFO("cKDSConfiguration::%s", __func__);
   uint8_t u8AAWEnabled = 0;
   int32_t _s32Res = 0;

   if(DataProvider::getInstance()->getWBLConfiguration()._AAWSupportRead)
    {
      _s32Res = DP_s32GetConfigItem("SystemConfiguration1", "WirelessAndroidAutoFunction", &u8AAWEnabled, sizeof(uint8_t));
      if (_s32Res == DP_S32_NO_ERR)
      {
         LOG_DEBUG ("cKDSConfiguration : Read from Datapool, AAW Support: %d",u8AAWEnabled);
         _supportAAW = (bool)u8AAWEnabled;
      }
      else
      {
         LOG_ERROR ("cKDSConfiguration : Reading AAW support ERR : Setting Default Value");
    }
   }
   else
   {
      LOG_ERROR ("cKDSConfiguration : Read Value not enabled for AAWSupport : Setting Default Value");
   }
   LOG_DEBUG ("cKDSConfiguration : AAW Support: %d",_supportAAW);

}

::std::string cKDSConfiguration::readVehicleManufacturerName(const ::std::string cstrManufacturerName)
{
   LOG_INFO("cKDSConfiguration::%s", __func__);
   ::std::string manufacturerName = cstrManufacturerName;
   //If the value read from the KDS is empty, then populating it with the Default value
   if((false == getParsedString(manufacturerName))||(true == manufacturerName.empty()))
   {
      manufacturerName = DataProvider::getInstance()->getWBLConfiguration()._ManfNameDefault;
   }
   LOG_DEBUG("cKDSConfiguration::readVehicleManufacturerName ManufacturerName:%s",manufacturerName.c_str());
   return manufacturerName;
}

::std::string cKDSConfiguration::readVehicleModelName(const ::std::string cstrModelName)
{
   LOG_INFO("cKDSConfiguration::%s", __func__);
   ::std::string modelName = cstrModelName;
   //If the value read from the KDS is empty, then populating it with the Default value
   if((false == getParsedString(modelName))||(true == modelName.empty()))
   {
      modelName = DataProvider::getInstance()->getWBLConfiguration()._ModelNameDefault;
   }
   LOG_DEBUG("cKDSConfiguration::readVehicleManufacturerModelName ManufacturerModelName:%s",modelName.c_str());
   return modelName;
}

bool cKDSConfiguration::getWiFi5GHzConfiguration()
{
   return _wifi5GHzPDSupport && _wifi5GHzCDSupport;
}
uint16_t cKDSConfiguration::getCarplayChannelsFromConfiguration(WBLFrequencyBand& requestedFrequency)
{
    if (WBL_FREQ_BAND_5_GHZ == requestedFrequency)
    {
        return DataProvider::getInstance()->getWBLConfiguration()._CPW5GHzChannel;
    }
    else
    {
        return DataProvider::getInstance()->getWBLConfiguration()._CPW2_4GhzChannel;
    }
}
uint16_t cKDSConfiguration::getAAWChannelsFromConfiguration()
{
    return DataProvider::getInstance()->getWBLConfiguration()._AAW5GHzChannel;
}
bool cKDSConfiguration::isCPWSupport()
{
   return _supportCPW;
}
::std::string cKDSConfiguration::getVehicleName()
{
   LOG_INFO("cKDSConfiguration::%s",__func__);
   return _vehicleDeviceName;
}
void cKDSConfiguration::getVehicleFriendlyName()
{
   LOG_INFO("cKDSConfiguration::%s", __func__);
   uint8 u8DataBuff[WBL_VEHICLE_NAME_LEN];
   memset(u8DataBuff, 0, sizeof(u8DataBuff));
   if(bReadData(WBL_KDS_VEHICLE_NAME_CONFIG_KEY, WBL_KDS_VEHICLE_NAME_CONFIG_LEN, u8DataBuff))
   {
      ::std::string tmpName((char*) u8DataBuff);
      if ((false == getParsedString(tmpName)) || (true == tmpName.empty()))
      {
         _vehicleDeviceName = DataProvider::getInstance()->getWBLConfiguration()._VehicleNameDefault;
      }
      else
      {
         _vehicleDeviceName = tmpName;
      }
   }
   LOG_DEBUG("cKDSConfiguration::getVehicleFriendlyName VehicleName: %s",_vehicleDeviceName.c_str());
}
bool cKDSConfiguration::isAAWSupport()
{
   return _supportAAW;
}

bool cKDSConfiguration::isWiFiFCC()
{
   if (_wifiFCCConfiguration)
      return true;
   return false;
}
::std::string cKDSConfiguration::getManufacturerName()
{
   LOG_INFO("cKDSConfiguration::%s",__func__);
   return _manufacturerName;
}
::std::string cKDSConfiguration::getManufacturerModelName()
{
   LOG_INFO("cKDSConfiguration::%s",__func__);
   return _modelName;
}
void cKDSConfiguration::getManufacturerConfigurationData()
{
   int32_t _s32Res = 0;
   LOG_INFO("cKDSConfiguration::%s",__func__);
    if(DataProvider::getInstance()->getWBLConfiguration()._ManfNameRead)
   // In SMART SOP2, the vehicle name is not read from KDS file
   {
        unsigned char ManufacturerName[DataProvider::getInstance()->getWBLConfiguration()._ManfNameLength]={};
        _s32Res = DP_s32GetConfigItem("SmartphoneIntegration", "ManufacturerName", ManufacturerName,
                (DataProvider::getInstance()->getWBLConfiguration()._ManfNameLength));
      if (_s32Res == DP_S32_NO_ERR)
{
            const ::std::string ManfName(reinterpret_cast<const char*>(ManufacturerName));
            LOG_DEBUG ("cKDSConfiguration : Read from Datapool, ManufacturerName: %s",ManfName.c_str());
            _manufacturerName =
                    readVehicleManufacturerName(ManfName);
      }
        else
            {
            LOG_ERROR ("cKDSConfiguration : Reading ManufacturerName ERR : Setting Default Value");
            _manufacturerName = DataProvider::getInstance()->getWBLConfiguration()._ManfNameDefault;
   }

   }
    else
            {
        LOG_ERROR ("cKDSConfiguration : Read Value not enabled for Manufacturer Name : Setting Default Value");
        _manufacturerName = DataProvider::getInstance()->getWBLConfiguration()._ManfNameDefault;
}
    LOG_DEBUG ("cKDSConfiguration : ManufacturerName: %s",_manufacturerName.c_str());

    if(DataProvider::getInstance()->getWBLConfiguration()._ModelNameRead)
{
        unsigned char modelName[DataProvider::getInstance()->getWBLConfiguration()._ModelNameLength]={};
        _s32Res = DP_s32GetConfigItem("SmartphoneIntegration", "ManufacturerModelName", modelName,
                (DataProvider::getInstance()->getWBLConfiguration()._ModelNameLength));
        if (_s32Res == DP_S32_NO_ERR)
   {
            const ::std::string ManfModName(reinterpret_cast<const char*>(modelName));
            LOG_DEBUG ("cKDSConfiguration : Read from Datapool, ManufacturerModelName: %s",ManfModName.c_str());
            _modelName = readVehicleModelName(ManfModName);
   }
   else
   {
            LOG_ERROR ("cKDSConfiguration : Reading ManufacturerModelName ERR : Setting Default Value");
            _modelName = DataProvider::getInstance()->getWBLConfiguration()._ModelNameDefault;
      }
   }
   else
   {
        LOG_ERROR ("cKDSConfiguration : Read Value not enabled for Model Name : Setting Default Value");
        _modelName = DataProvider::getInstance()->getWBLConfiguration()._ModelNameDefault;
   }
   LOG_DEBUG ("cKDSConfiguration : ModelName: %s",_modelName.c_str());
}

bool cKDSConfiguration::getParsedString(::std::string &strToBeParsed)
{
   bool bRet = false;
   ::std::string szwhitespaces("\t\f\v\n\r");
   auto found = strToBeParsed.find_last_not_of(szwhitespaces);
   if (found!=std::string::npos)
   {
      //removing all whitespaces from the end
      strToBeParsed.erase(found + 1);
      bRet = true;
   }
   //explicitly appending with the null
   return bRet;
}

void cKDSConfiguration::readCPWSupportData()
{
   LOG_INFO("cKDSConfiguration::%s", __func__);
   if(access(WBL_CPWBLOCKED_FILEPATH, F_OK) == 0)


   {
      //! Open KDS file
      LOG_DEBUG("cKDSConfiguration : ON/OFF-CarPlayWireless-feature-through-SW  FILE CPW=blocked");
      _supportCPW = false;
         //! Read required data from file
         }
   else if(access(WBL_CPWOVERRIDE_FILEPATH, F_OK) == 0)
         {
      LOG_DEBUG("cKDSConfiguration : ON/OFF-CarPlayWireless-feature-through-SW  FILE CPW=enabled");
      _supportCPW = true;
         //! Close KDS file

      }//if (OSAL_ERROR != tIoKdsHandle)
      else
      {
      uint8_t u8CPWEnabled = 0;
      int32_t _s32Res = 0;
      if(DataProvider::getInstance()->getWBLConfiguration()._CPWSupportRead)
{
         _s32Res = DP_s32GetConfigItem("SystemConfiguration1", "WirelessCarPlayFunction", &u8CPWEnabled, sizeof(uint8_t));
         if (_s32Res == DP_S32_NO_ERR)
   {
            LOG_DEBUG ("cKDSConfiguration : Read from Datapool, CPW Support: %d",u8CPWEnabled);
            _supportCPW = (bool)u8CPWEnabled;
   }
         else
   {
            LOG_ERROR ("cKDSConfiguration : Reading CPW support ERR : Setting Default Value");
         }
   }
   else
      {
         LOG_ERROR ("cKDSConfiguration : Read Value not enabled for CPWSupport : Setting Default Value");
      }
   }
   LOG_DEBUG("cKDSConfiguration : CPW Support: %d",_supportCPW);
}
tBool cKDSConfiguration::bReadData(tU16 u16KdsKey, tU16 u16DataLen, tU8* pu8DataBuffer)
{
   LOG_INFO("cKDSConfiguration::bReadData() entered: u16KdsKey = 0x%x, u16DataLen = 0x%x \n",
         u16KdsKey, u16DataLen);

   tBool bReadSuccess = FALSE;

   if (OSAL_NULL != pu8DataBuffer)
   {
      //! Open KDS file
      OSAL_tIODescriptor tIoKdsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS, OSAL_EN_READONLY);

      if (OSAL_ERROR != tIoKdsHandle)
      {
         tsKDSEntry rKDSEntryData;
         rKDSEntryData.u16Entry = u16KdsKey;
         rKDSEntryData.u16EntryLength = u16DataLen;
         rKDSEntryData.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
         rKDSEntryData.au8EntryData[0] = 0;

         //! Read required data from file
         tS32 s32OsalReadErr = OSAL_s32IORead(tIoKdsHandle, (tPS8) &rKDSEntryData, (sizeof(rKDSEntryData)));
         if (OSAL_ERROR != s32OsalReadErr)
         {
            (tVoid) OSAL_pvMemoryCopy((tVoid*) (pu8DataBuffer), rKDSEntryData.au8EntryData, u16DataLen);
            bReadSuccess = TRUE;
         }
         else
         {
            (tVoid) OSAL_s32IOClose(tIoKdsHandle);
            LOG_ERROR("cKDSConfiguration::bReadData: Error reading KDS file: %d \n", OSAL_u32ErrorCode());
         }

         //! Close KDS file
         (tVoid) OSAL_s32IOClose(tIoKdsHandle);

      }//if (OSAL_ERROR != tIoKdsHandle)
      else
      {
         LOG_ERROR("cKDSConfiguration::bReadData: Error opening KDS file: %d \n", OSAL_u32ErrorCode());
      }
   }//if (OSAL_NULL != pu8DataBuffer)

   return bReadSuccess;
}
}
}
/** @} */
