/**
 * @file GenericNetlink.h
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief Main file for Wifi_Business_Logic.
 *
 *
 * @{
 */

#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/epoll.h>
#include <errno.h>
#include <string.h>
#include <netlink/socket.h>
#include <netlink/genl/genl.h>
#include <netlink/genl/ctrl.h>
#include <linux/if_ether.h>
#include <linux/version.h>
#include "asf/core/Logger.h"
#include "GenlEvent.h"
#include "GenericNetlink.h"
#include "WifiRadio.h"

#define WBL_OBJ_PATH			"org/bosch/wbl/wifi_business_logic"
#define GENL_CLASS_NAME			"cGenNetLink"
//#define GENL_CLASS_PATH			WBL_OBJ_PATH "/" GENL_CLASS_NAME
#define GENL_CLASS_PATH			"wifi_business_logic/Regulation"
#define GENL_DRIVER_IFACE		"nl80211"
#define GENL_CTRL				"nlctrl"
#define GENL_REG_MULTICAST_GRP	"regulatory"

namespace org {
namespace bosch {

static const ::std::string _mcgroups[] = {
		"regulatory",
		"mlme",
		""
};

#define nlattr_for_each_nested(pos, nla, rem) \
	for (pos = (struct nlattr *) nla_data(nla), rem = nla_len(nla); \
	     nla_ok(pos, rem); \
	     pos = nla_next(pos, &(rem)))

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Regulation", cGenNetLink, Debug);

const std::string &cGenlFamilyResolver::sGetGenlFamily()
{
	return m_sFamily;
}

int cGenlFamilyResolver::iSetGenlFamily(const std::string &sGenlFamily)
{
	if (!m_sFamily.compare(sGenlFamily))
		return -EALREADY;

	m_sFamily = sGenlFamily;
	return 0;
}

const int &cGenlFamilyResolver::iGetGenlResolver()
{
	return m_iIdentity;
}

int cGenlFamilyResolver::iSetGenlResolver(const int &iResolver)
{
	if (m_iIdentity == iResolver)
		return -EALREADY;

	m_iIdentity = iResolver;
	return 0;
}

cGenlFamilyResolver::cGenlFamilyResolver(const std::string &sGenlFamily)
{
	m_sFamily = sGenlFamily;
	m_iIdentity = -ENOENT;
}

cGenlFamilyResolver::cGenlFamilyResolver()
{
	m_iIdentity = -ENOENT;
}

cGenlFamilyResolver::~cGenlFamilyResolver()
{
}

cGenNetLink::cGenNetLink() :
		cNetLink(NL_SOCKET_TYPE_GENERIC), cIOChannel(POLL_EVENT_READ)
{
	int iRet;
	struct nl_sock *pNlSk;
	long sz;

	m_bRegistered = false;
	m_iNl80211id = -1;
	_initiator = 0;
	_type = 0;
	_wiphyDumped = false;

	pNlSk = pGetNlSock();
	if (pNlSk) {

		sz = sysconf(_SC_PAGESIZE);
		if (sz < 0) {
			LOG_ERROR("Failed to get the PAGE_SIZE: %s/%d", strerror(errno), errno);
			sz = 4096;
		}

		/* In certain conditions we use another socket to write and read
		 * few values from kernel and the calling thread is blocked during
		 * that period, therefore make sure that the event socket is buffered
		 * enough so that we dont miss out any message i.e., kernel event
		 *
		 * Kernel by default allocates a msg buffer of page_size and therefore
		 * making room for 32 messages so that we have the maximum chances that
		 * we dont miss any event (at max the Gen3 targets have 4 cpus) */
		sz = sz * 32;
		iRet = nl_socket_set_buffer_size(pNlSk, static_cast<int>(sz), 0);
		if (iRet < 0) {
			/* Not fatal, so continue */
			LOG_ERROR ("Failed to increase the receive buffer to: ld [%s/%d]",
					sz, nl_geterror(-iRet), -iRet);
		}

		iRet = iSetPollFd(iGetUnixSocket());
		if (iRet < 0)
			LOG_ERROR("Failed to set the POll FD : %s/%d", strerror(-iRet),
					-iRet);

		iRet = iStartWatch();
		if (!iRet) {
			iRet = genl_ctrl_resolve(pNlSk, GENL_DRIVER_IFACE);
			if (iRet < 0) {
				LOG_ERROR("Failed to Nl80211id : %s/%d", nl_geterror(-iRet),
						-iRet);
			} else {
				m_iNl80211id = iRet;
				iRet = iSetUpEventHandler();
				if (!iRet) {
					/* Dump the available wiphys */
					iRet = iGenlRequest(true, NL80211_CMD_GET_WIPHY, NLM_F_DUMP, iProcessGenlEvent);
					if (iRet < 0) {
						LOG_ERROR("Failed to request the wireless radios information: %s/%d",
									strerror(-iRet), -iRet);
					} else {
						iRet = iGenlRequest(true, NL80211_CMD_GET_INTERFACE, NLM_F_DUMP, iProcessGenlEvent);
						if (iRet == 0) {

							iRet = getRegulatorySettings();
							if (iRet < 0) {
								LOG_ERROR("Failed to request the regulatory settings: %s/%d",
										strerror(-iRet), -iRet);
							}

							_wiphyDumped = true;
							/* Purely for debugging and information */
							WifiRadioDump();
						} else {
							LOG_ERROR("Failed to request the wireless interfaces dump: %s/%d",
									strerror(-iRet), -iRet);
						}
					}
				} else {
					LOG_ERROR("Failed to setup the Genl Event Handler");
				}
			}
		} else {
			LOG_ERROR("Failed to add to the multicast group");
		}
	}
}

int cGenNetLink::getRadioSettings(cWifiRadio *radio, int cmd, int flags)
{
	int iRet;
	struct nl_msg *msg;
	void *temp;

	if(!radio)
		return -EINVAL;

	msg = nlmsg_alloc();
	if (!msg)
		return -ENOMEM;

	temp = genlmsg_put(msg, 0, 0, m_iNl80211id, 0,
			flags,static_cast<uint8_t>(cmd), 0);
	if (!temp) {
		iRet = -ENOMSG;
		goto error;
	}

	iRet = nla_put_u32(msg, NL80211_ATTR_WIPHY,
			radio->getIndex());
	if (iRet < 0)
		goto error;

	return iGenlRequest(msg, iProcessGenlEvent);
error:
	nlmsg_free(msg);
	return iRet;
}

int cGenNetLink::getRegulatorySettings()
{
	int iRet;
	std::vector<cWifiRadio*>::iterator it;

	/* Initially get all those self managed radio settings */
	for (it = _WifiRadios.begin(); it < _WifiRadios.end(); ++it) {
		if ((*it)->getSelfManaged()) {
			iRet = getRadioSettings(*it, NL80211_CMD_GET_REG, 0);
			if (iRet < 0)
				LOG_ERROR("Failed to get the regulatory settings for the self "
						"managed wifi radio: %s "
						"[index: %d] "
						"error: %s/%d",
						(*it)->getPhyName().c_str(),
						(*it)->getIndex(),
						strerror(-iRet), -iRet);
		}
	}

	/* Now the core */
	return iGenlRequest(true, NL80211_CMD_GET_REG, 0, iProcessGenlEvent);
}

cGenNetLink::~cGenNetLink()
{
	for (std::vector<cWifiRadio*>::iterator it = _WifiRadios.begin();
			it < _WifiRadios.end(); ++it) {
		(*it)->bFlushChannels();
		(*it)->bFlushNetDevs();
		delete *it;
	}
}

void cGenNetLink::notifyRegChange(bool global, cWifiRadio *radio)
{
	cGenlRegEvent msg;
	std::vector<cWifiRadio*>::iterator it;

	if (!global && !radio)
		return;

	LOG_INFO("Triggering Regulatory Change Event for: %s "
			"country: %s "
			"initiator: %s "
			"type: %s",
			global ? "Global" : radio->getPhyName().c_str(),
			global ?  _country.c_str() : radio->getRegDomain().c_str(),
			global ?  sRegInitiatorToString(
							static_cast<enum nl80211_reg_initiator>(_initiator)).c_str()
					: sRegInitiatorToString(
							static_cast<enum nl80211_reg_initiator>(radio->getInitiator())).c_str(),
			global ?  eRegChangeTypeToString(
							static_cast<enum nl80211_reg_type>(_type)).c_str()
					: eRegChangeTypeToString(
							static_cast<enum nl80211_reg_type>(radio->getType())).c_str());

	msg.setInitiator(_initiator);
	msg.setType(_type);
	msg.setCountry(_country);
	msg.setGlobal(global);

	for (it = _WifiRadios.begin(); it < _WifiRadios.end(); ++it) {
		msg.addWifiRadio(*it);
		LOG_INFO("Notifying Regulatory Change for the Radio: %s [%d] [regdomain: %s]",
				(*it)->getPhyName().c_str(), (*it)->getIndex(), (*it)->getRegDomain().c_str());
	}

	vNotifyObservers(&msg);
}

void cGenNetLink::notifyMlmeEvent(cGenlMessage *event)
{
	if (event)
		vNotifyObservers(event);
}

void cGenNetLink::vTriggerUpdate()
{
	notifyRegChange(true, nullptr);
}

std::vector <cWifiRadio*> &cGenNetLink::getRadios()
{
	return _WifiRadios;
}

int cGenNetLink::iSetUpEventHandler()
{
	struct nl_cb *cb;

	cb = pGetNlCB();
	if (!cb) {
		LOG_ERROR("Netlink callback is not created");
		return -EIO;
	}

	nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, iProcessGenlEvent, this);
	return 0;
}

const std::string cGenNetLink::sConvertGenlEventToString(enum nl80211_commands eGeNlEvt)
{
	switch (eGeNlEvt) {
	case NL80211_CMD_REG_CHANGE:
		return "NL80211_CMD_REG_CHANGE";
	case NL80211_CMD_GET_REG:
		return "NL80211_CMD_GET_REG";
	case NL80211_CMD_NEW_WIPHY:
		return "NL80211_CMD_NEW_WIPHY";
	case NL80211_CMD_GET_WIPHY:
		return "NL80211_CMD_GET_WIPHY";
	case NL80211_CMD_NEW_INTERFACE:
		return "NL80211_CMD_NEW_INTERFACE";
	case NL80211_CMD_GET_INTERFACE:
		return "NL80211_CMD_GET_INTERFACE";
	case NL80211_CMD_DISCONNECT:
		return "NL80211_CMD_DISCONNECT";
	case NL80211_CMD_CONNECT:
		return "NL80211_CMD_CONNECT";
	default:
		break;
	}

	return "NL80211_CMD_UNKNOWN";
}

const std::string cGenNetLink::sRegInitiatorToString(enum nl80211_reg_initiator eInitiator)
{
	switch (eInitiator) {
	case NL80211_REGDOM_SET_BY_CORE:
		return "WIRELESS_CORE";
	case NL80211_REGDOM_SET_BY_USER:
		return "SET_BY_USER";
	case NL80211_REGDOM_SET_BY_DRIVER:
		return "SET_BY_DRIVER";
	case NL80211_REGDOM_SET_BY_COUNTRY_IE:
		return "SET_FROM_COUNTRY_IE";
	}

	return "UNKNOWN";
}

const std::string cGenNetLink::eRegChangeTypeToString(enum nl80211_reg_type eType)
{
	switch (eType) {
	case NL80211_REGDOM_TYPE_COUNTRY:
		return "COUNTRY";
	case NL80211_REGDOM_TYPE_WORLD:
		return "WORLD";
	case NL80211_REGDOM_TYPE_CUSTOM_WORLD:
		return "CUSTOM_WORLD";
	case NL80211_REGDOM_TYPE_INTERSECTION:
		return "INTERSECTION";
	}

	return "UNKNOWN";
}

int cGenNetLink::iProcessFamResolveEvent(struct nl_msg *msg, void *data)
{
	cGenlFamilyResolver *pResolver = static_cast<cGenlFamilyResolver *>(data);
	struct nlattr *tb[CTRL_ATTR_MAX + 1];
	struct genlmsghdr *gnlh;
	struct nlattr *mcgrp;
	int index, iResult;
	::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

	if (!pResolver)
		return NL_SKIP;

	if (!msg || nlmsg_get_proto(msg) != NETLINK_GENERIC)
		return NL_SKIP;

	gnlh = static_cast<struct genlmsghdr *>(nlmsg_data(nlmsg_hdr(msg)));
	nla_parse(tb, CTRL_ATTR_MAX, genlmsg_attrdata(gnlh, 0),
			genlmsg_attrlen(gnlh, 0), nullptr);

	if (!tb[CTRL_ATTR_MCAST_GROUPS])
		return NL_SKIP;

	for (mcgrp = static_cast<struct nlattr *>(nla_data(
			tb[CTRL_ATTR_MCAST_GROUPS])), index = nla_len(
			tb[CTRL_ATTR_MCAST_GROUPS]); nla_ok(mcgrp, index);
			mcgrp = nla_next(mcgrp, &(index))) {
		struct nlattr *tb2[CTRL_ATTR_MCAST_GRP_MAX + 1];
		nla_parse(tb2, CTRL_ATTR_MCAST_GRP_MAX,
				static_cast<struct nlattr *>(nla_data(mcgrp)), nla_len(mcgrp),
				nullptr);

		if (!tb2[CTRL_ATTR_MCAST_GRP_NAME])
			continue;
		if (!tb2[CTRL_ATTR_MCAST_GRP_ID])
			continue;

		if (strncmp(
				static_cast<char *>(nla_data(tb2[CTRL_ATTR_MCAST_GRP_NAME])),
				pResolver->sGetGenlFamily().c_str(),
				(size_t) nla_len(tb2[CTRL_ATTR_MCAST_GRP_NAME])) != 0)
			continue;

		iResult = (int) nla_get_u32(tb2[CTRL_ATTR_MCAST_GRP_ID]);
		pResolver->iSetGenlResolver(iResult);
		oLogger.debug("Multicast group \"%s\" id : %d", __LINE__, __FILE__,
				pResolver->sGetGenlFamily().c_str(),
				pResolver->iGetGenlResolver());
		break;
	};

	return NL_SKIP;
}

int cGenNetLink::iGenlErrorHandler(struct sockaddr_nl *nla,
		struct nlmsgerr *err, void *arg)
{
	(void) nla;
	int *iRet = static_cast<int *>(arg);
	::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

	if (!iRet || !err)
		return NL_STOP;

	*iRet = err->error;
	oLogger.error(
			"Error received for Nl message with sequence no [%d] : %s/%d system error : %s/%d",
			__LINE__, __FILE__, err->msg.nlmsg_seq, nl_geterror(err->error),
			err->error, strerror(-err->error), -err->error);
	return NL_STOP;
}

int cGenNetLink::iGenlFinishHandler(struct nl_msg *msg, void *arg)
{
	struct nlmsghdr *nlhdr;
	int *iRet = static_cast<int *>(arg);
	::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

	if (!msg || !iRet)
		return NL_SKIP;

	*iRet = 0;
	nlhdr = nlmsg_hdr(msg);
	oLogger.debug("Message with seq no [%d] finished successfully", __LINE__,
			__FILE__, nlhdr->nlmsg_seq);
	return NL_SKIP;
}

int cGenNetLink::iGenlAckHandler(struct nl_msg *msg, void *arg)
{
	struct nlmsghdr *nlhdr;
	int *iRet = static_cast<int *>(arg);
	::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

	if (!msg || !iRet)
		return NL_STOP;

	*iRet = 0;
	nlhdr = nlmsg_hdr(msg);
	oLogger.debug("Message with seq no [%d] acknowledged", __LINE__,
			__FILE__, nlhdr->nlmsg_seq);
	return NL_STOP;
}

int cGenNetLink::iGenlSeqCheckHandler(struct nl_msg *msg, void *arg)
{
	(void) arg;
	struct nlmsghdr *nlhdr;
	::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

	if (!msg)
		return NL_OK;

	nlhdr = nlmsg_hdr(msg);
	oLogger.debug("Sequence check for Message with seq no [%d]", __LINE__,
			__FILE__, nlhdr->nlmsg_seq);
	return NL_OK;
}

int cGenNetLink::iGenlSendRecv(struct nl_sock *sk, struct nl_msg *msg,
		iEventhandler handler, void *data)
{
	struct nl_cb *cb;
	int err = -ENOMEM, res = 0;

	if (!sk)
		return -EINVAL;

	if (!msg)
		return -ENODATA;

	cb = nl_cb_clone(pGetNlCB());
	if (!cb)
		return -ENOMEM;

	err = nl_send_auto(sk, msg);
	if (err < 0) {
		nl_cb_put(cb);
		return err;
	}

	nl_cb_err(cb, NL_CB_CUSTOM, (nl_recvmsg_err_cb_t) iGenlErrorHandler, &err);
	nl_cb_set(cb, NL_CB_FINISH, NL_CB_CUSTOM, iGenlFinishHandler, &err);
	nl_cb_set(cb, NL_CB_ACK, NL_CB_CUSTOM, iGenlAckHandler, &err);
	nl_cb_set(cb, NL_CB_SEQ_CHECK, NL_CB_CUSTOM, iGenlSeqCheckHandler, nullptr);

	if (handler)
		nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, handler, data);

	while (err > 0) {
		res = nl_recvmsgs(sk, cb);
		if (res < 0)
			LOG_ERROR("Failed to receive nl80211 msgs : %s [%d]",
					nl_geterror(-res), -res);
	}

	nl_cb_put(cb);
	return res;
}

void cGenNetLink::updateRegulatoryData(u_int8_t initiator, u_int8_t type,
		const std::string &country)
{
	cWifiRadio *radio;
	std::vector<cWifiRadio*>::iterator it;

	_initiator = initiator;
	_type = type;
	_country = country;

	for (it = _WifiRadios.begin(); it < _WifiRadios.end(); ++it) {
		radio = *it;
		if (radio->getSelfManaged())
			continue;

		radio->iSetRegDomain(_country);
		radio->setInitiator(_initiator);
		radio->setType(_type);
	}
}

int cGenNetLink::iGetMultiCastGroupId(const std::string sFamily,
		const std::string sGroup)
{
	int iRet = -ENOMEM;
	struct nl_msg *pNlMsg;
	cGenlFamilyResolver *pResolver;
	struct nl_sock *sk;

	if (sFamily.empty() || sGroup.empty())
		return -EINVAL;

	LOG_INFO("Family: \"%s\" Group: \"%s\"", sFamily.c_str(), sGroup.c_str());

	sk = pGetNlSock();
	if (!sk)
		return -ENOTCONN;

	pResolver = new cGenlFamilyResolver(sGroup);
	pNlMsg = nlmsg_alloc();
	if (!pNlMsg)
		goto out;

	iRet = genl_ctrl_resolve(sk, GENL_CTRL);
	if (iRet < 0) {
		LOG_ERROR("Failed to resolve the family name %s to numeric identifier, "
				"error: %s/%d", sFamily.c_str(), nl_geterror(-iRet), -iRet);
		nlmsg_free(pNlMsg);
		goto out;
	}

	LOG_INFO("Genl family identifier : %d", iRet);
	if (!genlmsg_put(pNlMsg, 0, 0, iRet, 0, 0, CTRL_CMD_GETFAMILY, 0)
			|| nla_put_string(pNlMsg, CTRL_ATTR_FAMILY_NAME, sFamily.c_str())) {
		LOG_ERROR("Failed to construct the genl message");
		nlmsg_free(pNlMsg);
		goto out;
	}

	iRet = iGenlSendRecv(sk, pNlMsg, iProcessFamResolveEvent, pResolver);
	if (!iRet)
		iRet = pResolver->iGetGenlResolver();

	nlmsg_free(pNlMsg);
out:
	delete pResolver;
	return iRet;
}

::asf::core::Logger &cGenNetLink::oGetLogger()
{
	return _logger;
}

bool cGenNetLink::isWiphyDumped()
{
	return _wiphyDumped;
}

int cGenNetLink::iProcessRegEvent(struct nlattr **attr, void *arg)
{
	int iRet = 0;
	cGenNetLink *pGeNl;
	const char *alpha2 = NULL;
	u_int8_t u8Initiator = 0, u8Type = 0;
	bool global = true;
	unsigned int index = 0;
	cWifiRadio *radio = NULL;
	::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

	pGeNl = static_cast<cGenNetLink *>(arg);
	if (!pGeNl) {
		oLogger.error("Genl instance is invalid", __LINE__, __FILE__);
		return -EINVAL;
	}

	if (attr[NL80211_ATTR_WIPHY]) {
		index = nla_get_u32(attr[NL80211_ATTR_WIPHY]);
		global = false;
	}

	if (attr[NL80211_ATTR_REG_INITIATOR])
		u8Initiator = nla_get_u8(attr[NL80211_ATTR_REG_INITIATOR]);
	if (attr[NL80211_ATTR_REG_TYPE])
		u8Type = nla_get_u8(attr[NL80211_ATTR_REG_TYPE]);
	if (attr[NL80211_ATTR_REG_ALPHA2]) {
		alpha2 = static_cast<char *>(nla_data(attr[NL80211_ATTR_REG_ALPHA2]));
	} else {
		/* dirty, but will do i.e., nl80211 subsystem does not provide "00"
		 * if the change is WORLD REG Domain */
		if (NL80211_REGDOM_TYPE_WORLD
				== static_cast<enum nl80211_reg_type>(u8Type))
			alpha2 = "00";
	}

	oLogger.info("Regulatory Changed : "
			"Country [%s], "
			"Initiator [%s], "
			"Type [%s] "
			"is global: %s "
			"index: %u",
			__LINE__, __FILE__,
			alpha2 != NULL ? alpha2 : "Unknown",
			sRegInitiatorToString(static_cast<enum nl80211_reg_initiator>(u8Initiator)).c_str(),
			eRegChangeTypeToString(static_cast<enum nl80211_reg_type>(u8Type)).c_str(),
			global ? "TRUE" : "FALSE", index);

	if (!alpha2)
	   return -EINVAL;

	/* Change associated to only one radio */
	if (!global) {
		radio = pGeNl->getWifiRadio(index);
		if (!radio)
			return -ENOMEM;

		if (radio->getSelfManaged()) {
			radio->setInitiator(u8Initiator);
			radio->setType(u8Type);
			radio->iSetRegDomain(alpha2);
		} else  {
			global = true;
			pGeNl->updateRegulatoryData(u8Initiator, u8Type, alpha2);
		}
	} else {
		/* global and therefore update the global values and also
		 * every radios attached */
		pGeNl->updateRegulatoryData(u8Initiator, u8Type, alpha2);
	}

	if (!pGeNl->isWiphyDumped())
		return iRet;

	/* Dump every radio in case if the change is from the core and
	 * only from the changed radio if the change is specific to a
	 * particular wifi radio */
	if (global) {
		iRet = pGeNl->iGenlRequest(false, NL80211_CMD_GET_WIPHY, NLM_F_DUMP,
				iProcessGenlEvent);
	} else {
		iRet = pGeNl->getRadioSettings(radio, NL80211_CMD_GET_WIPHY, NLM_F_DUMP);
	}

	if (iRet == 0) {
		iRet = pGeNl->iGenlRequest(false, NL80211_CMD_GET_INTERFACE, NLM_F_DUMP, iProcessGenlEvent);
		if (iRet == 0) {
			/* Just for debugging, dump everything available */
			pGeNl->WifiRadioDump();
			pGeNl->notifyRegChange(global, global ? nullptr : radio);
		} else {
			oLogger.error ("Failed to retrieve the Interface dump: %s"
					"[%s/%d]",__LINE__, __FILE__, strerror(-iRet), -iRet);
		}
	} else {
		oLogger.error ("Failed to retrieve the changed channel dump for regDom: %s"
				"[%s/%d]",__LINE__, __FILE__, alpha2, strerror(-iRet), -iRet);
	}

	return iRet;
}

cWifiRadio *cGenNetLink::getWifiRadio(int index)
{
	cWifiRadio *radio;
	std::vector<cWifiRadio*>::iterator it;

	if (index < 0)
		return NULL;

	for (it = _WifiRadios.begin(); it < _WifiRadios.end(); it++)
		if ((*it)->getIndex() == index)
			return *it;

	radio = new cWifiRadio(index);
	if (radio)
		_WifiRadios.push_back(radio);

	return radio;
}

cWifiRadio *cGenNetLink::getWifiRadio(int index, const std::string &name)
{
	cWifiRadio *radio;
	std::vector<cWifiRadio*>::iterator it;

	if (index < 0)
		return NULL;

	for (it = _WifiRadios.begin(); it < _WifiRadios.end(); it++) {
		if ((*it)->getIndex() == index) {
			/* Probably device name renamed? One wouldnt expect that */
			if ((*it)->getPhyName().compare(name))
				(*it)->iSetPhyName(name);
			return *it;
		}
	}

	radio = new cWifiRadio(index, name);
	if (radio)
		_WifiRadios.push_back(radio);

	return radio;
}

void cGenNetLink::WifiRadioDump()
{
	cWifiRadio *radio;
	cWifiChannel *channel;
	uint32_t flags = 0;
	std::vector<cWifiRadio*>::iterator it;
	std::vector<cWifiNetworkInterface*>::iterator it2;
	std::vector<cWifiChannel*>::iterator it3;

	LOG_DEBUG("****************Begin: WiFi Radio Dump*******************");
	for (it = _WifiRadios.begin(); it < _WifiRadios.end(); it++) {
		radio = *it;
		if (radio) {

			LOG_INFO("Wiphy Radio: %s "
					"[index: %d] "
					"[self Managed: %s]",
					radio->getPhyName().c_str(),
					radio->getIndex(),
					radio->getSelfManaged() ? "TRUE" : "FALSE");

			for (it2 = radio->getNetDevices().begin(); it2 < radio->getNetDevices().end();
					it2++) {
				LOG_INFO("Exposed Interface: %s "
						"index: %u "
						"MacAddress: %s",
						(*it2)->getIfName().c_str(),
						(*it2)->getIndex(),
						(*it2)->getMacAddress().c_str());
			}

			LOG_DEBUG("****************Begin: Channel Dump*******************");
			for (it3 = radio->getChannels().begin(); it3 < radio->getChannels().end();
					it3++) {
				channel = *it3;
				if (channel) {
					flags = channel->getFlags();
					LOG_INFO("Freq: %d "
							"channel: %03d "
							"Flags: %02x "
							"\"Disabled\": %s "
							"\"NO_IR\": %s "
							"\"Radar\": %s "
							"\"Indoor_only\": %s "
							"\"IR_Concurrent\": %s",
							channel->getFrequency(),
							channel->getChannel(),
							flags,
							flags & IEEE80211_CHANNEL_DISABLED ? "Y" : "N",
							flags & IEEE80211_CHANNEL_NO_IR ? "Y" : "N",
							flags & IEEE80211_CHANNEL_RADAR ? "Y" : "N",
							flags & IEEE80211_CHANNEL_INDOOR ? "Y" : "N",
							flags & IEEE80211_CHANNEL_IR_CONCURRENT ? "Y" : "N");
				}
			}
			LOG_DEBUG("****************End: Channel Dump*********************");
		}
	}
	LOG_DEBUG("****************End: WiFi Radio Dump*******************");
}

int cGenNetLink::convertFreqToChannel(uint32_t freq, uint32_t *channel)
{
    int ret = -EINVAL;

    if (!channel)
    	return ret;

    /* IEEE802.11 B ONLY */
    if (freq == 2484) {
        *channel = 14;
        ret = 0;
    }

    /* IEEE802.11 G & B MODE */
    else if (freq >= 2412 && freq <= 2472) {

        if ((freq - 2407) % 5)
            return -EINVAL;

        *channel = (freq - 2407) / 5;
        ret = 0;
    }
    /* 4.9 GHz (IEEE 802.11 J) */
    else if (freq >= 4900 && freq < 5000) {

        if ((freq - 4000) % 5)
            return -EINVAL;

        *channel = (freq - 4000) / 5;
        ret = 0;
    }
    /* IEEE802.11 A MODE */
    else if (freq >= 5000 && freq <= 5900) {

        if ((freq - 5000) % 5)
            return -EINVAL;

        *channel = (freq - 5000) / 5;
        ret = 0;
    }
    /* IEEE802.11 AD MODE */
    else if (freq >= 56160 + 2160 * 1 && freq <= 56160 + 2160 * 4) {

        if ((freq - 56160) % 2160)
            return -EINVAL;

        *channel = (freq - 56160) / 2160;
        ret = 0;
    }

    return ret;
}

cWifiChannel* cGenNetLink::getChannel(struct nlattr **tb)
{
	int iRet;
    uint32_t freq, chan, flags = 0;
    ::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

    if (!tb || !tb [NL80211_FREQUENCY_ATTR_FREQ])
    	return NULL;

    freq = nla_get_u32 (tb [NL80211_FREQUENCY_ATTR_FREQ]);
    iRet = convertFreqToChannel(freq, &chan);
    if (iRet < 0)
    	return NULL;

    if (tb [NL80211_FREQUENCY_ATTR_DISABLED])
        flags |= IEEE80211_CHANNEL_DISABLED;
    if (tb [NL80211_FREQUENCY_ATTR_NO_IR])
        flags |= IEEE80211_CHANNEL_NO_IR;
    if (tb [NL80211_FREQUENCY_ATTR_RADAR])
        flags |= IEEE80211_CHANNEL_RADAR;

    /* I am not sure if this is enabled from kernel 4.0
     * series or versions lower than that. But the currently
     * used 3 series in Gen3 systems does not have it. Make sure
     * that the right versions are provided here */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
    if (tb [NL80211_FREQUENCY_ATTR_INDOOR_ONLY])
        flags |= IEEE80211_CHANNEL_INDOOR;
    if (tb [NL80211_FREQUENCY_ATTR_GO_CONCURRENT])
        flags |= IEEE80211_CHANNEL_IR_CONCURRENT;
#endif

    if (tb [NL80211_FREQUENCY_ATTR_DFS_STATE]) {
        enum nl80211_dfs_state state = static_cast<enum nl80211_dfs_state>
            (nla_get_u32 (tb [NL80211_FREQUENCY_ATTR_DFS_STATE]));

        switch (state) {
        case NL80211_DFS_USABLE:
            flags |= IEEE80211_CHANNEL_DFS_USABLE;
            break;
        case NL80211_DFS_AVAILABLE:
            flags |= IEEE80211_CHANNEL_DFS_AVAILABLE;
            break;
        case NL80211_DFS_UNAVAILABLE:
            flags |= IEEE80211_CHANNEL_DFS_UNAVAILABLE;
            break;
        }
    }

    return new cWifiChannel(static_cast<uint16_t>(freq),
    		static_cast<uint16_t>(chan), flags);
}

int cGenNetLink::iExtractChannelInfo(struct nlattr *band, cWifiRadio **radio)
{
    int i = 0;
    struct nlattr *tb [NL80211_BAND_ATTR_MAX + 1];
    struct nlattr *tbfreq;
    cWifiChannel *channel;

    if (!band || !radio || !*radio)
    	return -EINVAL;

    nla_parse (tb, NL80211_BAND_ATTR_MAX, static_cast<struct nlattr *>(nla_data (band)),
    		nla_len (band), nullptr);

    nlattr_for_each_nested (tbfreq, tb [NL80211_BAND_ATTR_FREQS], i) {
        struct nlattr *tb2 [NL80211_FREQUENCY_ATTR_MAX + 1];
        nla_parse (tb2, NL80211_FREQUENCY_ATTR_MAX, static_cast<struct nlattr *>(nla_data (tbfreq)),
        		nla_len (tbfreq), nullptr);
        if (tb2 [NL80211_FREQUENCY_ATTR_FREQ]) {
            channel = getChannel (tb2);
            if (channel)
                (*radio)->bAddWiFiChannel(channel);
        }
    }

    return 0;
}

int cGenNetLink::iExtractWiphyInfo(struct nl_msg *msg, void *arg)
{
	int iRet = 0, i = 0;
	cGenNetLink *pGeNl = static_cast<cGenNetLink *>(arg);
    cWifiRadio *radio;
    const char *name;
    struct nlattr *tb [NL80211_ATTR_MAX + 1];
    struct genlmsghdr *gnlh;
    struct nlattr *band;
    unsigned int index, generation = 0;
    ::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

    if (!pGeNl) {
    	oLogger.error("Invalid Generic Netlink instance", __LINE__, __FILE__);
    	return -EINVAL;
    }

    gnlh = static_cast<struct genlmsghdr *>(nlmsg_data (nlmsg_hdr (msg)));
    nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
               genlmsg_attrlen (gnlh, 0), nullptr);

    /* MultiPart messages but shall contain these mandatory
     * informations (i.e., radio name and the index) */
    if (!tb [NL80211_ATTR_WIPHY] ||
    		!tb [NL80211_ATTR_WIPHY_NAME])
        return 0;

    index = nla_get_u32 (tb [NL80211_ATTR_WIPHY]);
    name = nla_get_string (tb [NL80211_ATTR_WIPHY_NAME]);

    if (tb [NL80211_ATTR_GENERATION])
    	generation = nla_get_u32 (tb [NL80211_ATTR_GENERATION]);

    /* <TODO>: Find the proper usage of NL80211_ATTR_GENERATION,
     * could be useful very much in case of multipart messages */
    (void) generation;
    radio = pGeNl->getWifiRadio(index, name);
    if (!radio) {
    	oLogger.error("Failed to add the Radio information: %s[%d]", __LINE__, __FILE__,
    			name, index);
    	return -ENOMEM;
    }

    if (tb [NL80211_ATTR_WIPHY_BANDS]) {
    	radio->bFlushChannels();
    	nlattr_for_each_nested (band, tb [NL80211_ATTR_WIPHY_BANDS], i) {
    		iRet = iExtractChannelInfo(band, &radio);
    		if (iRet < 0) {
    			oLogger.error("Failed to extract the channel information: %s [%s/%d]",
    					__LINE__, __FILE__, name, strerror(-iRet), -iRet);
    		}
    	}
    }

    /* Well, I am not sure which version of Linux kernel introduced it
     * but the currently used 3 series in Gen3 systems definitely does
     * not have it. Make sure it is enabled from correct versions */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
    if (tb [NL80211_ATTR_WIPHY_SELF_MANAGED_REG])
    	radio->setSelfManaged(nla_get_flag(tb [NL80211_ATTR_WIPHY_SELF_MANAGED_REG]));
#endif

    return iRet;
}

int cGenNetLink::addNetDevice(cWifiRadio *radio, uint32_t index, uint8_t *addr,
		const ::std::string &name)
{
	if (!radio)
		return -EINVAL;
	return radio->bAddNetDevice(index, addr, name);
}

int cGenNetLink::iExtractInterfaceinfo(struct nl_msg *msg, void *arg)
{
	int iRet;
    char *name;
    cGenNetLink *pGeNl = static_cast<cGenNetLink *>(arg);
    struct nlattr *tb [NL80211_ATTR_MAX + 1];
    struct genlmsghdr *gnlh;
    unsigned int index = 0, netindex = 0;
    cWifiRadio *radio;
    enum nl80211_iftype type;
    uint8_t mac[ETH_ALEN];
    ::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

    if (!pGeNl)
    	return -EINVAL;

    gnlh = static_cast<struct genlmsghdr *>(nlmsg_data (nlmsg_hdr (msg)));
    nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
               genlmsg_attrlen (gnlh, 0), nullptr);

    /* mandatory guys */
    if (!tb [NL80211_ATTR_IFNAME] || !tb [NL80211_ATTR_WIPHY] ||
    		!tb [NL80211_ATTR_IFTYPE] || !tb [NL80211_ATTR_IFINDEX] ||
			!tb [NL80211_ATTR_MAC])
    	return 0;

    memset(mac, 0, ETH_ALEN);
    memcpy(mac, nla_data(tb[NL80211_ATTR_MAC]), ETH_ALEN);
    index = nla_get_u32 (tb [NL80211_ATTR_WIPHY]);
    netindex = nla_get_u32 (tb [NL80211_ATTR_IFINDEX]);
    type = static_cast<enum nl80211_iftype>(nla_get_u32 (tb [NL80211_ATTR_IFTYPE]));
    name = nla_get_string (tb [NL80211_ATTR_IFNAME]);

    (void) type;
    radio = pGeNl->getWifiRadio(index);
    if (!radio) {
    	oLogger.error("Failed to add the Radio information: %d", __LINE__,
    			__FILE__, index);
    	return -ENOMEM;
    }

    iRet = pGeNl->addNetDevice(radio, netindex, mac, name);
    if (iRet < 0 && iRet != -EALREADY) {
    	oLogger.error("Failed to add the network interface with "
    			"index: %u name: %s to Radio: %s", __LINE__,
    			__FILE__, netindex, name, radio->getPhyName().c_str());
    	return -ENOMEM;
    }
    return 0;
}

int cGenNetLink::iExtractMlmeConnectInfo(struct nlattr **tb, void *arg)
{
	bool timedout = false, connected = false;
    cGenNetLink *pGeNl = static_cast<cGenNetLink *>(arg);
    uint8_t mac[ETH_ALEN];
    char macstring[ETH_ALEN * 3];
    uint16_t status = 0;
    ::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);
    uint32 wiphyid = 0, netindex = 0;
    cWifiRadio *radio = NULL;
    cWifiNetworkInterface *interface = NULL;
    cGenlMlmeConnectEvent *cevent = NULL;
    ::std::string ifname;

    if (!pGeNl || !tb)
    	return -EINVAL;

    if (!tb [NL80211_ATTR_WIPHY] || !tb [NL80211_ATTR_IFINDEX])
    	return 0;

    memset(mac, 0, ETH_ALEN);
    memset(macstring, 0, sizeof(macstring));
    wiphyid = nla_get_u32 (tb [NL80211_ATTR_WIPHY]);
    netindex = nla_get_u32 (tb [NL80211_ATTR_IFINDEX]);

    if (tb[NL80211_ATTR_TIMED_OUT])
    	timedout = true;
    if (tb[NL80211_ATTR_STATUS_CODE]) {
    	status = nla_get_u16(tb[NL80211_ATTR_STATUS_CODE]);
    	connected = status == 0 ? true : false;
    }
    if (tb[NL80211_ATTR_MAC]) {
    	memcpy(mac, nla_data(tb[NL80211_ATTR_MAC]), ETH_ALEN);
    	cUtils::macToString(macstring, mac);
    }

    radio = pGeNl->getWifiRadio(wiphyid);
    if (radio)
    	interface = radio->getNetDevice(netindex);

    oLogger.info("A connect event on the net iface: \"%s\" "
    		"of radio: \"%s\" to \"%s\" is: %s",
			__LINE__, __FILE__,
			interface ? interface->getIfName().c_str() : "unknown",
			radio ? radio->getPhyName().c_str() : "unknown",
			macstring, connected ? "successful" : "timedout");

    ifname = interface ? interface->getIfName() : "";
    cevent = new cGenlMlmeConnectEvent(wiphyid, netindex, status, macstring, timedout, ifname);
    pGeNl->notifyMlmeEvent(cevent);
    delete cevent;
    return 0;
}

int cGenNetLink::iExtractMlmeDisconnectInfo(struct nlattr **tb, void *arg)
{
	bool disconnectedbyap = false;
    cGenNetLink *pGeNl = static_cast<cGenNetLink *>(arg);
    uint16_t reason = 0;
    ::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);
    uint32 wiphyid = 0, netindex = 0;
    cWifiRadio *radio = NULL;
    cWifiNetworkInterface *interface = NULL;
    cGenlMlmeDisConnectEvent *devent = NULL;
    ::std::string ifname;

    if (!pGeNl || !tb)
    	return -EINVAL;

    if (!tb [NL80211_ATTR_WIPHY] || !tb [NL80211_ATTR_IFINDEX])
    	return 0;

    wiphyid = nla_get_u32 (tb [NL80211_ATTR_WIPHY]);
    netindex = nla_get_u32 (tb [NL80211_ATTR_IFINDEX]);

    if (tb[NL80211_ATTR_REASON_CODE])
    	reason = nla_get_u16(tb[NL80211_ATTR_REASON_CODE]);
    if (tb[NL80211_ATTR_DISCONNECTED_BY_AP])
    	disconnectedbyap = nla_get_flag(tb [NL80211_ATTR_DISCONNECTED_BY_AP]) == 1 ?
    			true : false;

    radio = pGeNl->getWifiRadio(wiphyid);
    if (radio)
    	interface = radio->getNetDevice(netindex);

    oLogger.info("A disconnect event on the net iface: \"%s\" "
    		"of radio: \"%s\", reason code: %u disconnectedbyap: %s", __LINE__, __FILE__,
			interface ? interface->getIfName().c_str() : "unknown",
			radio ? radio->getPhyName().c_str() : "unknown",
			reason, disconnectedbyap ? "Y" : "N");

    ifname = interface ? interface->getIfName() : "";
    devent = new cGenlMlmeDisConnectEvent(wiphyid, netindex, reason, disconnectedbyap, ifname);
    pGeNl->notifyMlmeEvent(devent);
    delete devent;
    return 0;
}

int cGenNetLink::iProcessGenlEvent(struct nl_msg *msg, void *arg)
{
	int iRet;
	struct genlmsghdr *gnlh;
	struct nlattr *tb[NL80211_ATTR_MAX + 1];
	::asf::core::Logger oLogger(GENL_CLASS_PATH, ::asf::core::Logger::Info);

	if (nlmsg_get_proto(msg) != NETLINK_GENERIC)
		return NL_SKIP;

	gnlh = static_cast<struct genlmsghdr *>(nlmsg_data(nlmsg_hdr(msg)));
	oLogger.debug("Generic Netlink Event received [event : %s/%d]", __LINE__,
			__FILE__,
			sConvertGenlEventToString(
					static_cast<enum nl80211_commands>(gnlh->cmd)).c_str(),
			static_cast<enum nl80211_commands>(gnlh->cmd));

	switch (gnlh->cmd) {
	case NL80211_CMD_GET_REG:
	case NL80211_CMD_REG_CHANGE: {
		nla_parse(tb, NL80211_ATTR_MAX, genlmsg_attrdata(gnlh, 0),
				genlmsg_attrlen(gnlh, 0), nullptr);
		iRet = iProcessRegEvent(tb, arg);
		if (iRet < 0)
			oLogger.error("Failed to update the Regulatory change : %s/%d",
					__LINE__, __FILE__, strerror(-iRet), -iRet);
	}
	break;

	case NL80211_CMD_NEW_WIPHY: {
		iRet = iExtractWiphyInfo(msg, arg);
		if (iRet < 0)
			oLogger.error("Failed to extract the Wiphy information : %s/%d",
						__LINE__, __FILE__, strerror(-iRet), -iRet);
	}
	break;

	case NL80211_CMD_NEW_INTERFACE: {
		iRet = iExtractInterfaceinfo(msg, arg);
		if (iRet < 0)
			oLogger.error("Failed to extract the network interfaces information : %s/%d",
					__LINE__, __FILE__, strerror(-iRet), -iRet);
	}
	break;

	case NL80211_CMD_CONNECT: {
		nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
				genlmsg_attrlen (gnlh, 0), nullptr);
		iRet = iExtractMlmeConnectInfo(tb, arg);
		if (iRet < 0)
			oLogger.error("Failed to extract the MLME event: %s/%d",
					__LINE__, __FILE__, strerror(-iRet), -iRet);
	}
	break;

	case NL80211_CMD_DISCONNECT: {
		nla_parse (tb, NL80211_ATTR_MAX, genlmsg_attrdata (gnlh, 0),
				genlmsg_attrlen (gnlh, 0), nullptr);
		iRet = iExtractMlmeDisconnectInfo(tb, arg);
		if (iRet < 0)
			oLogger.error("Failed to extract the MLME event: %s/%d",
					__LINE__, __FILE__, strerror(-iRet), -iRet);
	}
	break;

	default:
		oLogger.debug("Not interested with the event : %d", __LINE__,
				__FILE__, gnlh->cmd);
		break;
	}

	return NL_SKIP;
}

int cGenNetLink::iGenlRequest(struct nl_msg *msg, iEventhandler handler)
{
	int iRet;
	struct nl_sock *sk;

	if (!msg)
		return -EINVAL;

	sk = pGetNlSock();
	if (!sk)
		return -ENOTCONN;

	iRet = iGenlSendRecv(sk, msg, handler, this);
	if (iRet < 0)
		LOG_ERROR("Failed to send the netlink message: %p, error: %s", msg,
				nl_geterror(-iRet));

	nlmsg_free(msg);
	return iRet;
}

struct nl_sock *cGenNetLink::createNlSocket()
{
	int iRet, realsk;
	struct nl_sock *sk;

	iRet = getSocketType();
	if (iRet < 0)
		return NULL;

	sk = nl_socket_alloc ();
	if (!sk)
		return NULL;

	iRet = nl_connect(sk, getSocketType());
	if (iRet < 0) {
		LOG_ERROR("Failed to connect to the NL socket: %s/%d",
				nl_geterror (-iRet), -iRet);
		return NULL;
	}

    realsk = nl_socket_get_fd (sk);
    if (realsk != -1)
    	(void) fcntl (realsk, F_SETFL, O_CLOEXEC);

    return sk;
}

int cGenNetLink::iGenlRequest(bool event_socket, enum nl80211_commands cmd,
		int flags, iEventhandler handler)
{
	int iRet;
	struct nl_msg *msg;
	void *temp;
	struct nl_sock *sk;

	LOG_INFO("Requesting NL command: %s, new socket: %s",
			sConvertGenlEventToString(cmd).c_str(),
			event_socket ? "Y" : "N");

	if (event_socket) {
		sk = pGetNlSock();
		if (!sk)
			return -ENOTCONN;
	} else {
		sk = createNlSocket ();
		if (!sk)
			return -ENOMEM;
	}

	msg = nlmsg_alloc();
	if (!msg) {
		if (!event_socket)
			nl_socket_free (sk);
		return -ENOMEM;
	}

	temp = genlmsg_put(msg, 0, 0, m_iNl80211id, 0, flags, cmd, 0);
	if (!temp) {
		if (!event_socket)
			nl_socket_free (sk);
		nlmsg_free(msg);
		return -ENOMSG;
	}

	iRet = iGenlSendRecv(sk, msg, handler, this);
	if (iRet < 0)
		LOG_ERROR("Failed to aquire the %d information, error: %s", cmd,
				nl_geterror(-iRet));

	if (!event_socket)
		nl_socket_free (sk);
	nlmsg_free(msg);
	return iRet;
}

int cGenNetLink::iStartWatch()
{
	int iRet = 0,iMtCastId = 0,index = 0;
	struct nl_sock *sk;

	if (m_bRegistered)
		return -EALREADY;

	sk = pGetNlSock();
	if (!sk)
		return -ENOTCONN;

	while (!_mcgroups[index].empty()) {
		iRet = iGetMultiCastGroupId(GENL_DRIVER_IFACE, _mcgroups[index].c_str());
		if (iRet < 0) {
			LOG_ERROR("Failed to get the id for the multicast group: %s [%s/%d]",
					_mcgroups[index].c_str(), strerror (-iRet), -iRet);
			return iRet;
		}

		iMtCastId = iRet;
		iRet = nl_socket_add_membership(sk, iMtCastId);
		if (iRet < 0)
			LOG_ERROR("Failed to get the membership of family: %s [%s/%d]",
					_mcgroups[index].c_str(), strerror (-iRet), -iRet);

		LOG_INFO("Multicast Id [%d] registration to the kernel \"%s\"", iMtCastId,
				iRet == 0 ? "is Successful" : "Failed");
		index++;
	}

	m_bRegistered = true;
	return iRet;
}

bool cGenNetLink::bProcessEvent(int iPollFd, int iEvent)
{
	int iRes;
	struct nl_sock *sk;
	struct nl_cb *cb;

	LOG_DEBUG("**************BEGIN************************");
	LOG_INFO("File desc: %d, event : %d", iPollFd, iEvent);

	if (iGetUnixSocket() != iPollFd) {
		LOG_INFO("POLL ID does not match");
		return true;
	}

	if ((iEvent & EPOLLERR) || (iEvent & EPOLLHUP)) {
		LOG_ERROR("Error occured for the fd: %d", iEvent);
		return false;
	}

	sk = pGetNlSock();
	cb = pGetNlCB();
	if (!sk || !cb)
		return true;

	iRes = nl_recvmsgs(sk, cb);
	if (iRes < 0) {
		LOG_ERROR("nl80211: global message receive failed : %s [%d]",
				nl_geterror(-iRes), iRes);
	}

	LOG_DEBUG("**************END************************");

	return true;
}

	}
}

/** @} */
