/**
 * @file WifiRadio.h
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 *
 * @{
 */

#ifndef _WIFI_RADIO_H
#define _WIFI_RADIO_H

#include <string.h>
#include "WifiChannel.h"
#include "WifiNetworkInterface.h"

namespace org {
    namespace bosch {

        class cWifiRadio {
        private:
            int _ieee80211index, _self_managed;
            uint8_t _initiator, _type;
            std::string _phyName, _regDomain;
            std::vector<cWifiChannel *> _supportedChannels;
            std::vector<cWifiNetworkInterface *> _exposedNetDevs;
        public:
            cWifiRadio() :
                _ieee80211index (0), _self_managed(0),
                _initiator (0), _type (0) { }

            cWifiRadio(int index, const std::string &name) {
                _ieee80211index = index;
                _phyName = name;
                _self_managed = 0;
                _initiator = 0;
                _type = 0;
            }

            cWifiRadio(int index) {
                _ieee80211index = index;
                _self_managed = 0;
                _initiator = 0;
                _type = 0;
            }

            ~cWifiRadio() {
                bFlushChannels();
            }

            int setType(uint8_t type) {
                if (_type != type) {
                    _type = type;
                    return 0;
                }
                return -EALREADY;
            }

            int getType() {
                return _type;
            }

            int setInitiator(uint8_t initiator) {
                if (_initiator != initiator) {
                    _initiator = initiator;
                    return 0;
                }
                return -EALREADY;
            }

            int getInitiator() {
                return _initiator;
            }

            int setSelfManaged(int selfManaged) {
                if (_self_managed == selfManaged)
                    return -EALREADY;
                _self_managed = selfManaged;
                return 0;
            }

            int &getSelfManaged() {
                return _self_managed;
            }

            int iSetIndex(int index) {
                if (_ieee80211index == index)
                    return -EALREADY;
                _ieee80211index = index;
                return 0;
            }

            int &getIndex() {
                return _ieee80211index;
            }

            int iSetPhyName(const std::string &phyName) {
                if (!_phyName.compare(phyName))
                    return -EALREADY;
                _phyName = phyName;
                return 0;
            }

            std::string &getPhyName() {
                return _phyName;
            }

            std::vector<cWifiNetworkInterface*> &getNetDevices() {
                return _exposedNetDevs;
            }

            std::vector<cWifiChannel*> &getChannels() {
                return _supportedChannels;
            }

            int iSetRegDomain(const std::string &regDomain) {
                if (!_regDomain.compare(regDomain))
                    return -EALREADY;
                _regDomain = regDomain;
                return 0;
            }

            std::string &getRegDomain() {
                return _regDomain;
            }

            bool bFlushChannels() {
                std::vector<cWifiChannel *>::iterator it;
                if (_supportedChannels.empty())
                    return false;
                for (it = _supportedChannels.begin(); it < _supportedChannels.end(); ++it)
                    delete *it;
                _supportedChannels.clear();
                return true;
            }

            bool bFlushNetDevs() {
                ::std::vector<cWifiNetworkInterface*>::iterator it;
                if (_exposedNetDevs.empty())
                    return false;
                for (it = _exposedNetDevs.begin(); it < _exposedNetDevs.end(); ++it)
                    delete *it;
                _exposedNetDevs.clear();
                return true;
            }

            bool bAddNetDevice(uint32_t index, const ::std::string &addr, const ::std::string &name) {
                bool bPresent = false;
                cWifiNetworkInterface *interface = NULL;
                std::vector<cWifiNetworkInterface*>::iterator it;

                if (!_exposedNetDevs.empty()) {
                    for (it = _exposedNetDevs.begin(); it < _exposedNetDevs.end(); ++it) {
                        if ((*it)->getIndex() == index) {
                            bPresent = true;
                            break;
                        }
                    }
                }

                if (!bPresent) {
                    interface = new cWifiNetworkInterface(name, addr, index);
                    if (interface) {
                        _exposedNetDevs.push_back(interface);
                        return true;
                    }
                } else {
                    if ((*it)->getIfName().compare(name))
                        (*it)->setIfname(name);
                    if ((*it)->getMacAddress().compare(addr))
                        (*it)->setMacAddress(addr);
                }

                return false;
            }

            int bAddNetDevice(uint32_t index, uint8_t *addr, const ::std::string &name) {
                bool bPresent = false;
                cWifiNetworkInterface *interface = NULL;
                std::vector<cWifiNetworkInterface*>::iterator it;

                if (!_exposedNetDevs.empty()) {
                    for (it = _exposedNetDevs.begin(); it < _exposedNetDevs.end(); ++it) {
                        if ((*it)->getIndex() == index) {
                            bPresent = true;
                            break;
                        }
                    }
                }

                if (!bPresent) {
                    interface = new cWifiNetworkInterface(name, addr, index);
                    if (interface) {
                        _exposedNetDevs.push_back(interface);
                        return 0;
                    } else return -ENOMEM;
                } else {
                    if ((*it)->getIfName().compare(name))
                        (*it)->setIfname(name);
                    if (memcmp((*it)->getRMacAddress(), addr, ETH_ALEN))
                        (*it)->setMacAddress(addr);
                }

                return -EALREADY;
            }

            cWifiNetworkInterface *getNetDevice(uint32_t index) {
                std::vector<cWifiNetworkInterface*>::iterator it;
                for (it = _exposedNetDevs.begin(); it < _exposedNetDevs.end(); ++it)
                    if ((*it)->getIndex() == index)
                        return *it;
                return NULL;
            }

            bool bRemNetDevice(cWifiNetworkInterface *netDev) {
                bool bPresent = false;
                std::vector<cWifiNetworkInterface*>::iterator it;

                if (!_exposedNetDevs.empty()) {
                    for (it = _exposedNetDevs.begin(); it < _exposedNetDevs.end(); ++it) {
                        if ((*it)->getIndex() == netDev->getIndex()) {
                            bPresent = true;
                            break;
                        }
                    }
                }

                if (bPresent) {
                    delete *it;
                    _exposedNetDevs.erase(it);
                    return true;
                }

                return false;
            }

            bool bAddWiFiChannel(cWifiChannel *channel) {
                bool bPresent = false;
                cWifiChannel *chan;
                std::vector<cWifiChannel*>::iterator it;

                if (!channel)
                    return false;

                if (!_supportedChannels.empty()) {
                    for (it = _supportedChannels.begin(); it < _supportedChannels.end(); ++it) {
                        if ((*it)->getFrequency() == channel->getFrequency()) {
                            bPresent = true;
                            break;
                        }
                    }
                }

                if (!bPresent) {
                    chan = new cWifiChannel();
                    if (chan) {
                        chan->iSetChannel(channel->getChannel());
                        chan->iSetFrequency(channel->getFrequency());
                        chan->iSetFlags(channel->getFlags());
                        _supportedChannels.push_back(chan);
                        return true;
                    }
                }

                return false;
            }

            bool bRemWiFiChannel(cWifiChannel *channel) {
                bool bPresent = false;
                std::vector<cWifiChannel*>::iterator it;

                if (!channel)
                    return false;

                if (!_supportedChannels.empty()) {
                    for (it = _supportedChannels.begin(); it < _supportedChannels.end(); ++it) {
                        if ((*it)->getFrequency() == channel->getFrequency()) {
                            bPresent = true;
                            break;
                        }
                    }
                }

                if (bPresent) {
                    delete *it;
                    _supportedChannels.erase(it);
                    return true;
                }

                return false;
            }
        };

    }
}

#endif //_WIFI_RADIO_H

/** @} */
