/**
 * @file RegulatoryCore.h
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#ifndef _REGULATORY_CORE_METHOD_H
#define _REGULATORY_CORE_METHOD_H

#include "asf/core/Logger.h"
#include "GenericSingleton.h"
#include "RegulatoryMethod.h"
#include "RegulatoryChannelAvailability.h"
#include "LoopbackTypes.h"
#include "WpaSupplicantData.h"
#include "IEEE80211STAConnectionStatus.h"
#include "WBLUtils.h"
#include "WifiMlmeUpdates.h"
#include "WatchDog.h"

#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif

#define REGULATORY_CORE_UPDATE_EXPECTED		BIT(0)
#define REGULATORY_CORE_UPDATE_RECEIVED		BIT(1)
#define REGULATORY_SET_BY_USER_EXPECTED		BIT(2)
#define REGULATORY_SET_BY_USER_RECEIVED		BIT(3)
#define REGULATORY_REGDOM_CHANGE_REQUIRED	BIT(4)
#define REGULATORY_REGDOM_CHANGE_INPROGRESS	BIT(5)

namespace org {
    namespace bosch {

        class cRegulatoryMethod;
        class cBasicServiceSet;
        class cWifiRadio;
        class cGenlRegEvent;

        typedef enum _reg_source {
            eREG_SOURCE_INVAL = 0,
            eREG_SOURCE_DEFAULT,
            eREG_SOURCE_LASTMODE,
            eREG_SOURCE_OTHERS
        } regsource_t;

        typedef enum {
            IEEE80211_NETWORK_DISCONNECT_CHAN_INVAL,
            IEEE80211_NETWORK_DISCONNECT_CHAN_CONNMAN,
            IEEE80211_NETWORK_DISCONNECT_CHAN_WPA_SUPP,
            IEEE80211_NETWORK_DISCONNECT_CHAN_NL80211
        } eIeee80211NetworkDisconnectChannel_t;

        class cCfg80211RegEvents {
        public:
            uint32_t _cfg80211regevents;
            cCfg80211RegEvents() : _cfg80211regevents(0) { }
            ~cCfg80211RegEvents() { }
            void expectedRegUpdates(const bool setbyuser) {
                if (setbyuser)
                    _cfg80211regevents |= REGULATORY_SET_BY_USER_EXPECTED;
                _cfg80211regevents |= REGULATORY_CORE_UPDATE_EXPECTED;
            }
            void setByUserReceived() {
                _cfg80211regevents |= REGULATORY_SET_BY_USER_RECEIVED;
                _cfg80211regevents &= ~REGULATORY_SET_BY_USER_EXPECTED;
            }
            void coreUpdateReceived() {
                _cfg80211regevents |= REGULATORY_CORE_UPDATE_RECEIVED;
                _cfg80211regevents &= ~REGULATORY_CORE_UPDATE_EXPECTED;
            }
            bool isSetByUserExpected() { return !!(_cfg80211regevents & REGULATORY_SET_BY_USER_EXPECTED); }
            bool isCoreUpdateExpected() { return !!(_cfg80211regevents & REGULATORY_CORE_UPDATE_EXPECTED); }
            bool isSetByUserReceived() { return !!(_cfg80211regevents & REGULATORY_SET_BY_USER_RECEIVED); }
            bool isCoreUpdateReceived() { return !!(_cfg80211regevents & REGULATORY_CORE_UPDATE_RECEIVED); }
            void regdomChangeRequired() { _cfg80211regevents |= REGULATORY_REGDOM_CHANGE_REQUIRED; }
            void regdomChangeInProgress() {
                _cfg80211regevents |= REGULATORY_REGDOM_CHANGE_INPROGRESS;
                _cfg80211regevents &= ~REGULATORY_REGDOM_CHANGE_REQUIRED;
            }
            bool isRegDomChangeRequired() { return !!(_cfg80211regevents & REGULATORY_REGDOM_CHANGE_REQUIRED); }
            bool isRegDomChangeInProgress() { return !!(_cfg80211regevents & REGULATORY_REGDOM_CHANGE_INPROGRESS); }
            void reset() { _cfg80211regevents = 0; }
        };

        class cDisconnectionStatus {
        public:
            bool _disconnectrequested;
            eIeee80211NetworkDisconnectChannel_t _disconnectchannel;
            eIeee80211NetworkStatus_t _status;
            ::std::string _disconnectingbss, _regdomainchange, _ifname;
            cCfg80211RegEvents _cfg80211events;
            bool _completed;
            cDisconnectionStatus() :
                _disconnectrequested(false),
                _disconnectchannel(IEEE80211_NETWORK_DISCONNECT_CHAN_INVAL),
                _status(IEEE80211_NETWORK_INVAL),
                _completed(false){}
            void reset() {
                _completed = false;
                _disconnectrequested = false;
                _status = IEEE80211_NETWORK_INVAL;
                _disconnectingbss.clear();
                _regdomainchange.clear();
                _ifname.clear();
                _disconnectchannel = IEEE80211_NETWORK_DISCONNECT_CHAN_INVAL;
                _cfg80211events.reset();
            }
        };

        class cPendingConnection {
        public:
            ::std::string _ifName, _bss;
            cPendingConnection() { }
            ~cPendingConnection() { }
        };

        class cRegulatoryCore final:
        public GenericSingleton<cRegulatoryCore>,
        public cRegulatoryChangeNotifier,
        public cRegChannelAvailability,
        public cWpaSupplicantConnectionNotifier,
        public IEEE80211STAConnectionStatusNotifier,
        public cWifiMlmeConnectionObserver,
        public cWatchDogCallbackIf {

        private:
            bool _readkernelReg, _validRegSources;
            ::std::string _country;
            ::std::string _regchangeinprogress;
            ::std::string _regtobechanged;
            ::std::string _regSources;
            eRegulatoryMethod_t _currentmethod;
            tU8 _regulatorychannel;
            cGenlRegEvent *_currentRegSettings;
            cDisconnectionStatus _disconnectionstatus;
            cCfg80211RegEvents _cfg80211regevents;
            regsource_t _reg_source;
            ::std::vector<cPendingConnection> _pendingvalidations;
            cWatchDog *_watchDog;

            cRegulatoryCore();
            friend class GenericSingleton<cRegulatoryCore>;
            int flushOldRegMethod();
            int eraseRadioSettings(int index);
            cWifiRadio *getRadioFromIfname(const ::std::string &ifname);
            const ::std::string getMacAddressFromIfname(const ::std::string &ifname);
            int computeObjPath(const ::std::string &ifName, const cBasicServiceSet *bss,
                    ::std::string &path);
            int isAssociatedChannelValid(cWifiRadio *radio, uint16_t frequency);
            int getMacAddress(const ::std::string &ifname, ::std::string &macaddress);
            void setDisconnectionStatus(bool requested, eIeee80211NetworkDisconnectChannel_t channel,
                    const ::std::string &bss, const ::std::string &regdomain, const ::std::string &ifname,
                    bool expect);
            void dumpInternals();
            int handleDisconnectionUpdates(cRegulatoryUpdate *regUpdate);
            int handleNormalDisconnectUpdates(cRegulatoryUpdate *regUpdate);
            int disconnectBss(const ::std::string &bss, const ::std::string &ifName);
            cPendingConnection *getPendingConnection(const ::std::string &ifName);
            int removePendingConnection(const ::std::string &ifName);
            void addPendingConnection(const ::std::string &ifName, const ::std::string &bss);
            cPendingConnection *getNextConnection();
            int validateNextConnections();
            int validateRadios();
            void updateDb(const ::std::string &country, ::std::string &group);
            void updateCountryCode(const ::std::string &country);
            eRegulatoryMethod_t &getCurrentRegSource();
            void configureRegulatorySources();
            void destroyRegulatorySource(void *source, bool configured = false);
            void destroyRegulatorySources();
            int validateRegulatorySource(const char *source);
            int validateRegulatorySources();
            void createRegulatorySources(void *sources, bool configured = false);
            void createRegulatorySource(const eRegulatoryMethod_t &method);
            ::std::string regSourceToString(const regsource_t source);
            cc_source_t regMethodToCCSource(const eRegulatoryMethod_t method);
            void startWatchDog();
            void stopWatchDog();
            DECLARE_CLASS_LOGGER();
        public:
            virtual ~cRegulatoryCore();
            cRegulatoryCore(const cRegulatoryCore &r) = delete;
            const cRegulatoryCore &operator=(const cRegulatoryCore &r) = delete;
            void loopbacknotification(cRegulatoryUpdate *regUpdate);
            int changeRegulatory(const ::std::string &country);
            virtual int notifyRegChange(const std::string &country) override;
            virtual void notifyRegChannelAvailabilty(const eRegChannelType_t type,
                    const bool availability) override;
            virtual void serviceAvailability(const eRegulatoryMethod_t method,
                    const bool available) override;
            virtual void wpaConnectionNotification(eWpaSuppConnectionState_t state,
                    const ::std::string &ifName, const ::std::string &associatedbss) override;
            virtual void wpaDisconnectionStatus(int status, const ::std::string &ifName) override;
            virtual void connectionStatusChange (const IEEE80211Network &network,
                    const eIeee80211NetworkStatus_t status) override;
            virtual void MlmeConnected(const ::std::string &ifName, const ::std::string &associatedBss,
                    const unsigned int &ifindex,const ::std::string &macaddr) override;
            virtual void MlmeDisConnected(const ::std::string &ifName, const ::std::string &disconnectedBss) override;
            virtual void watchDogExpired(void *data) override;
        };

    }
}

#endif //_REGULATORY_CORE_METHOD_H

/** @} */
