/**
 * @file GenlEvent.h
 * @author RBEI/ECO32 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief Main file for Wifi_Business_Logic.
 *
 *
 * @{
 */

#ifndef _GENL_EVENT_H
#define _GENL_EVENT_H

#include <errno.h>
#include <vector>
#include "NetlinkEvent.h"
#include "WifiRadio.h"
#include "LoopbackTypes.h"
#include "WBLUtils.h"

namespace org {
	namespace bosch {

typedef enum {
	GENL_EVENT_INVAL,
	GENL_EVENT_REGULATORY_UPDATE,
	GENL_EVENT_MLME_CONNECT,
	GENL_EVENT_MLME_DISCONNECT
} eGenlEvent_t;

class cWifiRadio;

class cGenlMessage : public cNlMessage {
	eGenlEvent_t genlMsgId;
public:
	eGenlEvent_t getGenlMessageId() {
		return genlMsgId;
	}
	virtual ~cGenlMessage() {}
	explicit cGenlMessage(const eGenlEvent_t event) :
	cNlMessage(NL_MESSAGE_TYPE_GENERIC) {
		genlMsgId = event;
	}
};

class cGenlRegEvent final: public cGenlMessage {
private:
	u_int8_t _initiator;
	u_int8_t _type;
	bool _global;
	::std::string _country;
	::std::vector<cWifiRadio*> _radios;
public:
	cGenlRegEvent() : cGenlMessage(GENL_EVENT_REGULATORY_UPDATE) {
		_initiator = 0;
		_type = 0;
		_global = false;
	};

	cGenlRegEvent(const u_int8_t initiator, const u_int8_t type,
			const ::std::string &country) :
				cGenlMessage(GENL_EVENT_REGULATORY_UPDATE) {
		_initiator = initiator;
		_type = type;
		_country = country;
		_global = false;
	}

	int setGlobal(bool global) {
		if (_global == global)
			return -EALREADY;
		_global = global;
		return 0;
	}

	bool &getGlobal() {
		return _global;
	}

	int setInitiator(const u_int8_t init) {
		if (_initiator == init)
			return -EALREADY;
		_initiator = init;
		return 0;
	}
	int setType(const u_int8_t type) {
		if (_type == type)
			return -EALREADY;
		_type = type;
		return 0;
	}
	int setCountry(const ::std::string &country) {
		if (!_country.compare(country))
			return -EALREADY;
		_country = country;
		return 0;
	}

	u_int8_t &getInitiator() {
		return _initiator;
	}
	u_int8_t &getType() {
		return _type;
	}
	const std::string &getCountry() {
		return _country;
	}

	std::vector<cWifiRadio*> &getWifiRadios() {
		return _radios;
	}

	int addWifiRadio (cWifiRadio *radio) {
		int added = 0;

		if (!radio)
			return -EINVAL;

		for (std::vector<cWifiRadio*>::iterator it = _radios.begin();
				it < _radios.end(); ++it) {
			if ((*it)->getIndex() == radio->getIndex()) {
				added = 1;
				break;
			}
		}

		if (!added) {
			_radios.push_back(radio);
			return 0;
		}

		return -EALREADY;
	}

	virtual ~cGenlRegEvent() {}
};

class cGenlMlmeConnectEvent final: public cGenlMessage {
	uint32_t _ieee80211index, _ifindex;
	::std::string _bssid, _ifname;
	uint16_t _statuscode;
	bool _timedout;
public:
	explicit cGenlMlmeConnectEvent(uint32_t wiphyid, uint32_t ifindex,
			uint16_t code, const char *mac, bool timedout, const ::std::string &ifname) :
			cGenlMessage(GENL_EVENT_MLME_CONNECT) {
		_ifindex = ifindex;
		_ieee80211index = wiphyid;
		_timedout = timedout;
		_ifname = ifname;
		if (mac)
			_bssid = mac;
		_statuscode = code;
	};
	const ::std::string &getBssid() {
		return _bssid;
	}
	const uint16_t &getStatusCode() {
		return _statuscode;
	}
	const uint32_t &getWiphyId() {
		return _ieee80211index;
	}
	const uint32_t &getIfIndex() {
		return _ifindex;
	}
	bool &getTimedout() {
		return _timedout;
	}
	const ::std::string &getIfname() {
		return _ifname;
	}
};

class cGenlMlmeDisConnectEvent final: public cGenlMessage {
	uint32_t _ieee80211index, _ifindex;
	bool _disconnectedbyap;
	uint16_t _reasoncode;
	::std::string _ifname;
public:
	explicit cGenlMlmeDisConnectEvent(uint32_t wiphyid, uint32_t ifindex,
			uint16_t code, bool byap, const ::std::string &ifname) :
			cGenlMessage(GENL_EVENT_MLME_DISCONNECT) {
		_ifindex = ifindex;
		_ieee80211index = wiphyid;
		_reasoncode = code;
		_disconnectedbyap = byap;
		_ifname = ifname;
	};
	bool &isDisconnectedByAp() {
		return _disconnectedbyap;
	}
	const uint16_t &getReasonCode() {
		return _reasoncode;
	}
	const uint32_t &getWiphyId() {
		return _ieee80211index;
	}
	const uint32_t &getIfIndex() {
		return _ifindex;
	}
	const ::std::string &getIfname() {
		return _ifname;
	}
};

cRegulatoryUpdate *GenlEventToRegulatoryUpdate(cGenlRegEvent *event);
cGenlRegEvent *RegulatoryUpdateToGenlEvent(cRegulatoryUpdate *event);
void genlEventCleanup(cGenlRegEvent **event);
cMlmeEventConnect *GenlEventToMlmeConnectEvent(cGenlMlmeConnectEvent *cevent);
cMlmeEventDisconnect *GenlEventToMlmeDisConnectEvent(cGenlMlmeDisConnectEvent *devent);

	}
}

#endif //_GENL_EVENT_H

/** @} */
