/**
 * @file NavigationClient.cpp
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string.h>
#include "NavigationClient.h"
#include "RegulatoryGNSSMethod.h"
#include "ISO3166CountryInfo.h"

namespace org {
	namespace bosch {

DEFINE_CLASS_LOGGER_AND_LEVEL ("wifi_business_logic/Regulation", cNavigationClient, Debug);

cNavigationClient::cNavigationClient() :
		cNavigationUpdateNotifier(NAV_METHOD_PRIO_HIGH)
{

}

cNavigationClient::~cNavigationClient()
{

}

void cNavigationClient::updateCountryCodeInfo()
{
	LOG_INFO("Navigation Client's service availability: %s",
			_serviceAvailability == true ? "Y" : "F");

	if (_serviceAvailability)
		notifyCountryChange(_country);
}

void cNavigationClient::setCountryCodeInfo(const ::std::string &cc)
{
	LOG_INFO ("Navigation country code: %s country code size:%d ", cc.c_str(),cc.size());
	bool propagate = true;
	::std::string alpha2;

	if (!cc.empty() &&
			0 == cISO3166CountryInfo::getInstance()->isValidAlpha2(cc.c_str()))
	{
		alpha2 = cc;
		LOG_INFO ("Valid Aplha2 country code reported by the Navigation module: %s",
				alpha2.c_str());
	}
	else if(cc.size() == 3)
	{
		alpha2 = convertAlpha3ToAlpha2(cc);
		LOG_INFO ("Valid Alpha3 country code reported by the Navigation module: %s",
				alpha2.c_str());
	}
	else
	{
		LOG_INFO ("Invalid Alpha2/Alpha3 from Navigation module: %s",cc.c_str());
		propagate = false;
	}

	if (_country.compare(alpha2))
		_country = alpha2;
	else {
		/* There is an update from Navigation module
		 * 	1) Either WBL cannot convert it to alpha2 as the conversion
		 * 		entry is missing
		 * 	2) Or It is an empty string update means that there is no GPS signal
		 *
		 * In either case we clear the _country variable due to below reasons
		 * 	1) Even if the alpha3 code is valid but if thats missing in WBL's
		 * 	   internal db (it is equal to CRDA) which means that we dont the country
		 * 	   based settings for this particular alpha3
		 * 	2) GPS signal lost --> switch to safe regulatory dom */
		 _country.clear();
		 }

	setServiceAvailability(alpha2.empty() ? false : true);
	if(propagate)
		updateCountryChange(alpha2);
}

	}
}

/** @} */
