/**
 * @file IEEE80211STAConnectionStatus.h
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */
#ifndef _IEEE_80211_STA_CONNECTION_STATUS_H
#define _IEEE_80211_STA_CONNECTION_STATUS_H

#include <stdio.h>
#include <errno.h>
#include <string>
#include "WBLUtils.h"

namespace org {
	namespace bosch {

#define NETWORK_SSID_SIZE_INITIALIZER       65

typedef enum {
	IEEE80211_NETWORK_INVAL,
	IEEE80211_NETWORK_CONNECTING,
	IEEE80211_NETWORK_CONNECTED,
	IEEE80211_NETWORK_CONNECT_FAILED,
	IEEE80211_NETWORK_DISCONNECTING,
	IEEE80211_NETWORK_DISCONNECTED,
	IEEE80211_NETWORK_DISCONNECT_FAILED
} eIeee80211NetworkStatus_t;

class IEEE80211Network {
	char _networkSSID[33];
	::std::string _objPath;
	bool _hidden;

	int setNetworkSsid(const ::std::string &objPath) {
	    const char *raw, *realobj;
	    char ssid [NETWORK_SSID_SIZE_INITIALIZER];
	    int delimiterCnt = 0, index = 0;
	    size_t length= 0;

	    if (objPath.empty())
	        return -ENODATA;

	    raw = objPath.c_str();
	    realobj = strrchr(raw, '/');
	    if (!realobj)
	        return -EINVAL;

	    memset(ssid, 0, sizeof(ssid));
	    for (realobj++; realobj[index]; index++) {
	        if (realobj[index] == '_')
	            ++delimiterCnt;
	        if (2 == delimiterCnt)
	            break;
	    }

	    if (realobj[++index] == '\0')
	        return -EINVAL;

	    realobj = &realobj[index];
	    for (index= 0; realobj[index]; index++) {
	        if (realobj[index] == '_')
	            break;
	        if (length > 64)
	            return -EINVAL;
	        ssid[length++] = realobj[index];
	    }

	    if (!length || (length & 0x01))
	        return -EINVAL;

	    memset(_networkSSID, 0, sizeof(_networkSSID));
	    if (!strncmp(ssid, "hidden", strlen("hidden"))) {
	    	_hidden = true;
	        return 0;
	    }

	    return cUtils::convertHexToBin(ssid, _networkSSID,
	    		length/2);
	}

public:
	IEEE80211Network() {
		memset(_networkSSID, 0, sizeof(_networkSSID));
		_hidden = false;
	}
	IEEE80211Network(const ::std::string &objpath) {
		memset(_networkSSID, 0, sizeof(_networkSSID));
		_hidden = false;
		setObjectPath(objpath);
	}
	virtual ~IEEE80211Network() { }

	int setObjectPath(const ::std::string &path) {
		if (_objPath.compare(path)) {
			_objPath = path;
			setNetworkSsid(_objPath);
			return 0;
		}
		return -EALREADY;
	}

	const char *getNetworkSsid() const {
		return _networkSSID;
	}

	const ::std::string &getObjectpath() const {
		return _objPath;
	}

	const bool &getHidden() const {
		return _hidden;
	}
};

class IEEE80211STAConnectionStatusNotifier  {
public:
	IEEE80211STAConnectionStatusNotifier() { }
	virtual ~IEEE80211STAConnectionStatusNotifier() { }
	static ::std::string IEEE80211StatusToString (const eIeee80211NetworkStatus_t status) {
		switch (status) {
		case IEEE80211_NETWORK_INVAL:
			return "Unknown";
		case IEEE80211_NETWORK_CONNECTING:
			return "Connecting";
		case IEEE80211_NETWORK_CONNECTED:
			return "Connected";
		case IEEE80211_NETWORK_CONNECT_FAILED:
			return "Connection Failed";
		case IEEE80211_NETWORK_DISCONNECTING:
			return "Disconnecting";
		case IEEE80211_NETWORK_DISCONNECTED:
			return "Disconnected";
		case IEEE80211_NETWORK_DISCONNECT_FAILED:
			return "Disconnection Failed";
		}
		return "unknown";
	}
	virtual void connectionStatusChange (const IEEE80211Network &network,
			const eIeee80211NetworkStatus_t status) = 0;
};

	}
}

#endif //_IEEE_80211_STA_CONNECTION_STATUS_H

/** @} */
