/**
 * @file IEEE80211APClient.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

//#include <algorithm>
#include <map>
#include <cstdlib>
#include <boost/algorithm/string.hpp>
#include <boost/algorithm/string/replace.hpp>

#include "IEEE80211APClient.h"
#include "DBManagerFactory.h"
#include "KDSConfiguration.h"

namespace org
{
namespace bosch 
{

static DbusParser<WapdmgrDbusInterfaces> wapdmgrDbusParser;
#define POWERSTATE_PREFIX             "WIFI_STATE_"

ByteArray OUIelements {0x00,0xA0,0X40};
ByteArray BoschOUIElements {0xA8,0x5B,0X6C};

#define WBL_VENDOR_ELEMENT_VALUE_ZERO           0X00
#define WBL_VENDOR_ELEMENT_ID                   0XDD
#define WBL_VENDOR_SUB_TYPE_ELEMENT             0X00

#define WBL_VENDOR_INFO_ELEMENT_ID_FLAG         0X00
#define WBL_VENDOR_INFO_ELEMENT_ID_NAME         0X01
#define WBL_VENDOR_INFO_ELEMENT_ID_MANUFACTURER 0X02
#define WBL_VENDOR_INFO_ELEMENT_ID_MODEL        0X03
#define WBL_VENDOR_INFO_ELEMENT_ID_OUI          0X04
#define WBL_VENDOR_INFO_ELEMENT_ID_BT_MAC       0X06
#define WBL_VENDOR_INFO_ELEMENT_ID_DEV_ID       0X07

#define WBL_VENDOR_ELEMENT_OUI_LEN              0X03
#define WBL_VENDOR_ELEMENT_SUB_TYPE_LEN         0X01
#define WBL_VENDOR_INFO_ELEMENT_LEN             0X02

#define WBL_VENDOR_CODE_SUPPORT_CARPLAY         0X0020
#define WBL_VENDOR_CODE_SUPPORT_2_4_GHZ         0X0002
#define WBL_VENDOR_CODE_SUPPORT_5_GHZ           0X0001


DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/IEEE80211Clients", IEEE80211APClient, Info);

IEEE80211APClient::IEEE80211APClient()
{
   LOG_INFO("IEEE80211APClient() entered ");

   _wapdman_ap_service_availability = false;
   _restoringAP = false;

   //! Create Wapdmgr Manager Proxy
   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IWapdmgrManagerProxyIf* wapdmgrManagerProxyIf = (dbusFactory)?
         (dbusFactory->getWapdmgrManagerProxyIf()) : (nullptr);
   if (wapdmgrManagerProxyIf)
   {
      wapdmgrManagerProxyIf->setCallbackIf(e8PROXY_USER_AP_CLIENT, this);
      wapdmgrManagerProxyIf->createProxy();
   }
   //! Create Bluez Adapter Proxy
   IBluezAdapterProxyIf* bluezAdapterProxyIf = (dbusFactory)?
         (dbusFactory->getBluezAdapterProxyIf()) : (nullptr);
   if (bluezAdapterProxyIf)
   {
      bluezAdapterProxyIf->setCallbackIf(e8PROXY_USER_BLUEZ_CLIENT, this);
      bluezAdapterProxyIf->createProxy();
   }
   wapdmgrDbusParser.setTraces(true);
} // IEEE80211APClient::IEEE80211APClient

IEEE80211APClient::~IEEE80211APClient()
{
   try
   {
      LOG_INFO("~IEEE80211APClient() entered ");

      //! Destroy Wapdmgr Manager Proxy
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrManagerProxyIf* wapdmgrManagerProxyIf = (dbusFactory)?
            (dbusFactory->getWapdmgrManagerProxyIf()) : (nullptr);
      if (wapdmgrManagerProxyIf)
      {
         wapdmgrManagerProxyIf->destroyProxy();
      }
      //! Destroy Bluez Adapter Proxy
      IBluezAdapterProxyIf* bluezAdapterProxyIf = (dbusFactory)?
            (dbusFactory->getBluezAdapterProxyIf()) : (nullptr);
      if (bluezAdapterProxyIf)
      {
         bluezAdapterProxyIf->destroyProxy();
      }
   }catch(...){}
}// IEEE80211APClient::~IEEE80211APClient

bool IEEE80211APClient::setPowered(const ::std::string& objPath, bool powered)
{
   LOG_INFO("IEEE80211APClient::setPowered() entered: Powered = %d, ObjPath = %s ", powered, objPath.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory)?
               (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendPoweredSet(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, powered);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("setPowered: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
}

void IEEE80211APClient::getAccessPoints(AccessPointList& apList) const
{
   LOG_INFO("IEEE80211APClient::getAccessPoints() entered ");
   apList = _apList;
} // void IEEE80211APClient::getAccessPoints

void IEEE80211APClient::getAssociatedStations(AssociatedStationList& asList) const
{
   LOG_INFO("IEEE80211APClient::getAssociatedStations() entered ");
   asList = _asList;
} // void IEEE80211APClient::getAssociatedStations

bool IEEE80211APClient::setSSID(const ::std::string& objPath, const ::std::vector< uint8 >& ssid, bool utf_ssid)
{
   LOG_INFO("IEEE80211APClient::setSSID() entered: ObjPath = %s", objPath.c_str());

   for (auto it = ssid.begin(); it != ssid.end(); ++it)
   {
      LOG_DEBUG("setSSID: SSID byte = %d", *it);
   }

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory) ?
                (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendSetSSIDRequest(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, ssid, utf_ssid);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("setSSID: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::setSSID

bool IEEE80211APClient::setPassphrase(const ::std::string& objPath, const ::std::string& passphrase)
{
   LOG_INFO("IEEE80211APClient::setPassphrase(): Passphrase = %s, ObjPath = %s", passphrase.c_str(), objPath.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory) ?
                (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendPassphraseSet(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, passphrase);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("setPassphrase: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::setPassphrase

bool IEEE80211APClient::setPassphrase(const ::std::string& objPath,WBLAPType apType,WBLActionType actionType)
{
   LOG_INFO("IEEE80211APClient::%s ObjPath = %s Action:%d", __func__,objPath.c_str(),actionType);
   ::std::string passphrase;
   bool isRequestSent = false;
   if(WIFI_ACTION_RANDOMIZE ==actionType)
   {
       passphrase = APConfigurationManager::getInstance()->generateRandomPassphrase();
       LOG_DEBUG("IEEE80211APClient::%s RandomizePassphrase:%s", __func__,passphrase.c_str());
   }
   else if(apType != WBL_AP_TYPE_UNKNOWN)
   {
       passphrase = APConfigurationManager::getInstance()->getAPTypePassphrase(apType);
       LOG_DEBUG("IEEE80211APClient::%s APTypePassphrase:%s", __func__,passphrase.c_str());
   }
   if (isAccessPoint(objPath)&& (!passphrase.empty()))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory) ?
                (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendPassphraseSet(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, passphrase);
      }
   } // if (isAccessPoint(objPath))
   LOG_DEBUG("setPassphrase: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::setPassphrase
bool IEEE80211APClient::setHidden(const ::std::string& objPath, bool hidden)
{
   LOG_INFO("IEEE80211APClient::setHidden() entered: Hidden = %d, ObjPath = %s", hidden, objPath.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory) ?
                (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendHiddenSet(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, hidden);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("setHidden: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::setHidden

bool IEEE80211APClient::setTethering(const ::std::string& objPath, bool tethering)
{
   LOG_INFO("IEEE80211APClient::setTethering() entered: Tethering = %d, ObjPath = %s", tethering, objPath.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf =
            DBusProxyFactory::getInstance()->getWapdmgrAccessPointProxyIf();
      if ((NULL != wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendTetheringSet(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, tethering);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("setTethering: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::setHidden

bool IEEE80211APClient::setNetworkType(const ::std::string& objPath, WifiAccessNetworkType& networkType)
{
   LOG_INFO("IEEE80211APClient::setNetworkType() entered: NetworkType = %d, ObjPath = %s", networkType, objPath.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf =
            DBusProxyFactory::getInstance()->getWapdmgrAccessPointProxyIf();
      if ((NULL != wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendAccessNetworkTypeSet(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, networkType);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("setNetworkType: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::setHidden

bool IEEE80211APClient::setMaxStationsAllowed(const ::std::string& objPath, uint16 noOfStations)
{
   LOG_INFO("IEEE80211APClient::setMaxStationsAllowed() entered: noOfStations = %d, ObjPath = %s", noOfStations, objPath.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf =
            DBusProxyFactory::getInstance()->getWapdmgrAccessPointProxyIf();
      if ((NULL != wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
            isRequestSent = wapdmgrAccessPointProxyIf->sendMaxStationsAllowedSet(sWapdmgrBusName, objPath,
                  DBUS_BUS_SYSTEM, *this, noOfStations);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("noOfStations: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
}
bool IEEE80211APClient::setSecurity(const ::std::string& objPath, const ::std::string& security)
{
   LOG_INFO("IEEE80211APClient::setSecurity() entered: Security = %s, ObjPath = %s",
         objPath.c_str(), security.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory) ?
                (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendSecuritySet(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, security);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("setSecurity: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::setSecurity

bool IEEE80211APClient::deAuthenticateStation(const ::std::string& objPath, const ::std::string& macAddress)
{
   LOG_INFO("IEEE80211APClient::deAuthenticateStation() entered: MacAddress = %s, ObjPath = %s",
         macAddress.c_str(), objPath.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory) ?
                (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendDeAuthenticateStationRequest(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, macAddress);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("deAuthenticateStation: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::deAuthenticateStation

bool IEEE80211APClient::setOperatingChannels(const ::std::string& objPath, const ::std::vector< uint16 >& channels)
{
   LOG_INFO("IEEE80211APClient::setOperatingChannels() entered: list size = %lu, ObjPath = %s",
         channels.size(), objPath.c_str());

   bool isRequestSent = false;

   if (isAccessPoint(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory) ?
                (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendSetOperatingChannelsRequest(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this, channels);
      }
   } // if (isAccessPoint(objPath))

   LOG_DEBUG("setOperatingChannels: isRequestSent = %s ", BOOL_TO_STR(isRequestSent));
   return isRequestSent;
} // void IEEE80211APClient::setOperatingChannels

bool IEEE80211APClient::restoreAPSettings(const ::std::string& objPath)
{
   bool isRequestSent = false;
   if (isAccessPoint(objPath))
   {
      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory) ?
                (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
      if ((wapdmgrAccessPointProxyIf) &&
            (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
      {
         isRequestSent = wapdmgrAccessPointProxyIf->sendRestoreDefaultAPSettingsRequest(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM, *this);
      }
   } // if (isAccessPoint(objPath))

   return isRequestSent;
}

void IEEE80211APClient::setAPType(const ::std::string& objPath,WBLAPType userAPType)
{
   LOG_INFO("IEEE80211APClient::setAPType() Objpath :%s, APtype:%s ", objPath.c_str(),
         wblAPType.getAPType2String(userAPType).c_str());
   AccessPointProperty property;
   if (isAccessPoint(objPath))
   {
         property.type = userAPType;
         LOG_INFO("IEEE80211APClient::setAPType() Objpath :%s, Updated APtype:%s ", objPath.c_str(),
         wblAPType.getAPType2String(property.type).c_str());
         updateAccessPoint(objPath,property,AP_TYPE);
   }
   //IEEE80211_MSG_AP_MODE_AP_TYPE_IND
}

void IEEE80211APClient::onWapdmgrManagerServiceAvailable(const ::std::string& busName, const ::std::string& objPath,
      const ::DBusBusType busType, const ServiceState previousState, const ServiceState currentState)
{
   LOG_INFO("IEEE80211APClient::onWapdmgrManagerServiceAvailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IWapdmgrManagerProxyIf* wapdmgrManagerProxyIf =
         (dbusFactory) ? (dbusFactory->getWapdmgrManagerProxyIf()) : (nullptr);
   if (wapdmgrManagerProxyIf)
   {
      wapdmgrManagerProxyIf->sendGetAccessPointsRequest(*this);
   }
} // void IEEE80211APClient::onWapdmgrManagerServiceAvailable

void IEEE80211APClient::onWapdmgrManagerServiceUnavailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      const ServiceState previousState, const ServiceState currentState)
{
   LOG_INFO("IEEE80211APClient::onWapdmgrManagerServiceUnavailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);
   //! Add code
} // void IEEE80211APClient::onWapdmgrManagerServiceUnavailable

void IEEE80211APClient::onAccessPointAddedError(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAccessPointAddedError >& error)
{
   LOG_INFO("IEEE80211APClient::onAccessPointAddedError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onAccessPointAddedError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onAccessPointAddedError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_APADDED_IND);

} // void IEEE80211APClient::onAccessPointAddedError

void IEEE80211APClient::onAccessPointAddedSignal(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAccessPointAddedSignal >& signal)
{
   LOG_INFO("IEEE80211APClient::onAccessPointAddedSignal() entered ");
   WBL_ASSERT_AND_EXIT(!(signal && proxy));

   std::string apObjPath = signal->getAccesspoint();
   AccessPointProperty apProperty;
   addAccessPoint(apObjPath, apProperty);

} // void IEEE80211APClient::onAccessPointAddedSignal

void IEEE80211APClient::onAccessPointRemovedError(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAccessPointRemovedError >& error)
{
   LOG_INFO("IEEE80211APClient::onAccessPointRemovedError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onAccessPointRemovedError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onAccessPointRemovedError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_APREMOVED_IND);

} // void IEEE80211APClient::onAccessPointRemovedError(

void IEEE80211APClient::onAccessPointRemovedSignal(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAccessPointRemovedSignal >& signal)
{
   LOG_INFO("IEEE80211APClient::onAccessPointRemovedSignal() entered ");
   WBL_ASSERT_AND_EXIT(!(signal && proxy));

   removeAccessPoint(signal->getAccesspoint());

} // void IEEE80211APClient::onAccessPointRemovedSignal

void IEEE80211APClient::onGetAccessPointsError(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrGetAccessPointsError >& error)
{
   LOG_INFO("IEEE80211APClient::onGetAccessPointsError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onGetAccessPointsError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onGetAccessPointsError: Error message - %s ", error->getMessage().c_str());
   //! Add code

} // void IEEE80211APClient::onGetAccessPointsError

void IEEE80211APClient::onGetAccessPointsResponse(
      const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrGetAccessPointsResponse >& response)
{
   LOG_INFO("IEEE80211APClient::onGetAccessPointsResponse() entered ");
   WBL_ASSERT_AND_EXIT(!(proxy && response));

   typedef ::std::vector< WapdmgrGetAPResponseAPStruct > APStructList;
   APStructList apStructList = response->getAccesspoints();

   for (APStructList::iterator it = apStructList.begin(); it != apStructList.end(); ++it)
   {
      ::std::string apObjPath = (*it).getElem1();
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > apData = (*it).getElem2();

      AccessPointProperty apProperty;
      parseAccessPointInfo(proxy->getDBusBusName(), proxy->getDBusObjectPath(),
            proxy->getBusType(), proxy->getInterfaceName(), apData, apProperty);
      addAccessPoint(apObjPath, apProperty);
   } // for (APStructList::iterator it =...)
} // void IEEE80211APClient::onGetAccessPointsResponse

void IEEE80211APClient::onWapdmgrAccessPointServiceAvailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      const ServiceState previousState, const ServiceState currentState)
{
   LOG_INFO("IEEE80211APClient::onWapdmgrAccessPointServiceAvailable() entered ");
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf =
         (dbusFactory) ? (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
   if (wapdmgrAccessPointProxyIf)
   {
      wapdmgrAccessPointProxyIf->sendGetAssociatedStationsRequest(busName, objPath, busType, *this);
      wapdmgrAccessPointProxyIf->sendGetAllProperties(busName, objPath, busType,*this);
      wapdmgrAccessPointProxyIf->createDHCPDNSConfigurationProxy(busName, objPath, busType);
      wapdmgrAccessPointProxyIf->createTetheringSettingsProxy(busName, objPath, busType);
   }
   cWifiConnectionStatus::getInstance()->subscribeWifiConnectionStatus(this);
   if(currentState == ServiceState__Available)
       notifyAPServiceAvailability((bool)currentState);
} // void IEEE80211APClient::onWapdmgrAccessPointServiceAvailable

void IEEE80211APClient::onWapdmgrAccessPointServiceUnavailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      const ServiceState previousState, const ServiceState currentState)
{
   LOG_INFO("IEEE80211APClient::onWapdmgrAccessPointServiceUnavailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);
   cWifiConnectionStatus::getInstance()->subscribeWifiConnectionStatus(this);
   if(currentState == ServiceState__Disconnected)
       notifyAPServiceAvailability((bool)currentState);
} // void IEEE80211APClient::onWapdmgrAccessPointServiceUnavailable

void IEEE80211APClient::onAssociatedStationsChangedError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAssociatedStationsChangedError >& error)
{
   LOG_INFO("IEEE80211APClient::onAssociatedStationsChangedError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onAssociatedStationsChangedError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onAssociatedStationsChangedError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAssociatedStationError(objPath, IEEE80211_MSG_AP_MODE_ASADDED_IND);
   //TODO - how to determine if it is AS added or removed error?

} // void IEEE80211APClient::onAssociatedStationsChangedError

void IEEE80211APClient::onAssociatedStationsChangedSignal(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAssociatedStationsChangedSignal >& signal)
{
   LOG_INFO("IEEE80211APClient::onAssociatedStationsChangedSignal() entered ");
   WBL_ASSERT_AND_EXIT(!(signal && proxy));

   typedef ::std::vector< WapdmgrASChangdSignalChangedStationsStruct > ASChangedStationsStructList;
   ASChangedStationsStructList changedStationsList = signal->getChangedstations();
   ::std::vector< ::std::string > removedStationsList = signal->getRemovedstations();
   AssociatedStationItem item;
   bool isValidAS = false;

   for (ASChangedStationsStructList::iterator it = changedStationsList.begin(); it != changedStationsList.end(); ++it)
   {
      ::std::string asObjPath = (*it).getElem1();
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > asData = (*it).getElem2();

      isValidAS = getAssociatedStationInfo(asObjPath, item);
      LOG_DEBUG("IEEE80211APClient::onAssociatedStationsChangedSignal() %s Station Exist :%s ", asObjPath.c_str(),
            BOOL_TO_STR(isValidAS));
      //! This signal is used only to detect new associated stations.
      //! Changes to already associated stations will be notified via org.freedesktop.DBus.Properties interface
      if (!isValidAS || !(item.updated))
      {
         AssociatedStationProperty asProperty;
         parseAssociatedStationInfo(proxy->getDBusBusName(), proxy->getDBusObjectPath(),
               proxy->getBusType(), proxy->getInterfaceName(), asData, asProperty);
         addAssociatedStation(asObjPath, asProperty);
      }
   } // for (ASChangedStationsStructList::iterator it =...)

   for (::std::vector< ::std::string >::iterator it = removedStationsList.begin(); it != removedStationsList.end(); ++it)
   {
      removeAssociatedStation(*it);
   }
} // void IEEE80211APClient::onAssociatedStationsChangedSignal

void IEEE80211APClient::onMacAddressError(const ::boost::shared_ptr< WapdmgrAccesspointProxy > & proxy,
      const ::boost::shared_ptr< WapdmgrAPMacAddressError > & error)
{
   //TODO: Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onMacAddressError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onMacAddressError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onMacAddressError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_MACADDRESS_IND);

} // void IEEE80211APClient::onMacAddressError

void IEEE80211APClient::onMacAddressUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPMacAddressUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onMacAddressUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   LOG_DEBUG("onMacAddressUpdate: %s ", update->getMacAddress().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      apProperty.macAddress = update->getMacAddress();
      updateAccessPoint(objPath, apProperty, AP_MAC_ADDRESS);
   }
} // void IEEE80211APClient::onMacAddressUpdate
void IEEE80211APClient::onPowerStateError(const ::boost::shared_ptr< WapdmgrAccesspointProxy > & proxy,
      const ::boost::shared_ptr< WapdmgrAPPowerStateError > & error)
{
   //TODO: Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onPowerStateError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onPowerStateError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onPowerStateError: Error message - %s ", error->getMessage().c_str());

   //::std::string objPath = proxy->getDBusObjectPath();
   //onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_MACADDRESS_IND);

} // void IEEE80211APClient::onPowerStateError

void IEEE80211APClient::onPowerStateUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPPowerStateUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onPowerStateUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   LOG_DEBUG("onPowerStateUpdate: %s ", update->getPowerState().c_str());

   ::std::string strPowerState = POWERSTATE_PREFIX + update->getPowerState();
   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      WBLPowerStateType powerState = wblPowerState.getPowerState2Enum(strPowerState);
      WBL_ASSERT(!IS_ENUM_VALID(powerState));
      apProperty.powerState = powerState;
      if(_restoringAP)
         updateLastModeSettings(apProperty);
      updateAccessPoint(objPath, apProperty, AP_POWERSTATE);
   }
} // void IEEE80211APClient::onPowerStateUpdate

void IEEE80211APClient::onPowerFailureReasonError(const ::boost::shared_ptr< WapdmgrAccesspointProxy > & proxy,
      const ::boost::shared_ptr< WapdmgrAPPowerFailureReasonError > & error)
{
   //TODO: Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onPowerFailureReasonError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onPowerFailureReasonError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onPowerFailureReasonError: Error message - %s ", error->getMessage().c_str());

   //::std::string objPath = proxy->getDBusObjectPath();
   //onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_MACADDRESS_IND);

} // void IEEE80211APClient::onPowerStateError

void IEEE80211APClient::onPowerFailureReasonUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPPowerFailureReasonUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onPowerFailureReasonUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   if(update->getPowerFailureReason().empty())
      return;

   LOG_DEBUG("onPowerFailureReasonUpdate: %s ", update->getPowerFailureReason().c_str());
   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      WBLPowerFailureType powerFailure = wblPowerFailure.getPowerFailure2Enum(update->getPowerFailureReason());
      WBL_ASSERT(!IS_ENUM_VALID(powerFailure));
      apProperty.powerFailure = powerFailure;
      apProperty.powerState = WIFI_STATE_POWER_FAILURE;
      updateAccessPoint(objPath, apProperty, AP_POWERSTATE);
      updateAccessPoint(objPath, apProperty, AP_POWERFAILURE);
   }
}

void IEEE80211APClient::onSSIDError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPSSIDError >& error)
{
   //TODO:Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onSSIDError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onSSIDError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onSSIDError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_SSID_IND);

} // void IEEE80211APClient::onSSIDError

void IEEE80211APClient::onSSIDUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPSSIDUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onSSIDUpdate() entered");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      apProperty.ssid = update->getSSID();
      updateAccessPoint(objPath, apProperty, AP_SSID);
   }
} // void IEEE80211APClient::onSSIDUpdate

void IEEE80211APClient::onInterfaceError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPInterfaceError >& error)
{
   //TODO:Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onInterfaceError() entered");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onInterfaceError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onInterfaceError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_INTERFACE_IND);

} // void IEEE80211APClient::onInterfaceError

void IEEE80211APClient::onInterfaceUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPInterfaceUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onInterfaceUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   LOG_DEBUG("onInterfaceUpdate: %s ", update->getInterface().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      apProperty.interface = update->getInterface();
      updateAccessPoint(objPath, apProperty, AP_INTERFACE);
   }
} // void IEEE80211APClient::onInterfaceUpdate

void IEEE80211APClient::onPoweredError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPPoweredError >& error)
{
   LOG_INFO("IEEE80211APClient::onPoweredError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onPoweredError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onPoweredError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   AccessPointProperty apProperty;
   if(error->getName().compare("org.bosch.wapdman.AlreadyEnabled") == 0)
   {
      apProperty.isPowered = true;
      updateAccessPoint(objPath, apProperty, AP_POWERED);
   }
   else if(error->getName().compare("org.bosch.wapdman.AlreadyDisabled") == 0)
   {
      apProperty.isPowered = false;
      updateAccessPoint(objPath, apProperty, AP_POWERED);
   }
   else
   {
      onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_POWERED_IND);
   }
   //At the time of Error it is necessary to trigger the PowerFailureReason
   proxy->sendPowerFailureReasonGet(*this);
} // void IEEE80211APClient::onPoweredError

void IEEE80211APClient::onPoweredUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPPoweredUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onPoweredUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   LOG_DEBUG("onPoweredUpdate: %d ", update->getPowered());

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      apProperty.isPowered = update->getPowered();
      if(_restoringAP)
         updateLastModeSettings(apProperty);
      updateAccessPoint(objPath, apProperty, AP_POWERED);
   }
} // void IEEE80211APClient::onPoweredUpdate

void IEEE80211APClient::onHiddenError(const ::boost::shared_ptr< WapdmgrAccesspointProxy > & proxy,
      const ::boost::shared_ptr< WapdmgrAPHiddenError > & error)
{
   //TODO:Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onHiddenError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onHiddenError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onHiddenError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_HIDDEN_IND);

} // void IEEE80211APClient::

void IEEE80211APClient::onHiddenUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy > & proxy,
      const ::boost::shared_ptr< WapdmgrAPHiddenUpdate > & update)
{
   LOG_INFO("IEEE80211APClient::onHiddenUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   LOG_DEBUG("onHiddenUpdate: %d ", update->getHidden());

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      apProperty.isHidden = update->getHidden();
      updateAccessPoint(objPath, apProperty, AP_HIDDEN);
   }
} // void IEEE80211APClient::onHiddenUpdate

void IEEE80211APClient::onSecurityError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPSecurityError >& error)
{
   //TODO:Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onSecurityError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onSecurityError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onSecurityError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_SECURITY_IND);

} // void IEEE80211APClient::onSecurityError

void IEEE80211APClient::onSecurityUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPSecurityUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onSecurityUpdate() entered");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   
   LOG_DEBUG("onSecurityUpdate: %s ", update->getSecurity().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      WBLSecurityType security = wblSecurity.getSecurity2Enum(update->getSecurity());
      WBL_ASSERT(!IS_ENUM_VALID(security));
      apProperty.security = security;
      updateAccessPoint(objPath, apProperty, AP_SECURITY);
   }
} // void IEEE80211APClient::onSecurityUpdate

void IEEE80211APClient::onPassphraseError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPPassphraseError >& error)
{
   //TODO:Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onPassphraseError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onPassphraseError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onPassphraseError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   AccessPointItem item;
   AccessPointChangedMsg msg;
   if(boost::algorithm::contains(error->getName(),"Already"))
   {
      if (isAccessPoint(objPath))
      {
         if (getAccessPointInfo(objPath, item))
         {
            msg.setAccessPointItem(item);
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_PASSPHRASE_IND);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
      }
   }
   else
   {
       ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_PASSPHRASE_IND);
   }

} // void IEEE80211APClient::onPassphraseError

void IEEE80211APClient::onPassphraseUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPPassphraseUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onPassphraseUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   LOG_DEBUG("onPassphraseUpdate: %s ", update->getPassphrase().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      apProperty.passphrase = update->getPassphrase();
      updateAccessPoint(objPath, apProperty, AP_PASSPHRASE);
   }
} // void IEEE80211APClient::onPassphraseUpdate

void IEEE80211APClient::onCurrentOperatingChannelError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPCurrChannelError >& error)
{
   //TODO:Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onCurrentOperatingChannelError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onCurrentOperatingChannelError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onCurrentOperatingChannelError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_CURRENTCHANNEL_IND);

} // void IEEE80211APClient::onCurrentOperatingChannelError

void IEEE80211APClient::onCurrentOperatingChannelUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPCurrChannelUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onCurrentOperatingChannelUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   LOG_DEBUG("onCurrentOperatingChannelUpdate: %d ", update->getCurrentOperatingChannel());

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      apProperty.currChannel = update->getCurrentOperatingChannel();
      updateAccessPoint(objPath, apProperty, AP_CURRENT_CHANNEL);
   }
} // void IEEE80211APClient::onCurrentOperatingChannelUpdate

void IEEE80211APClient::onSupportedChannelsError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPSupportedChannelsError >& error)
{
   LOG_INFO("IEEE80211APClient::onSupportedChannelsError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onSupportedChannelsError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onSupportedChannelsError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_SUPPCHANNELS_IND);

} // void IEEE80211APClient::onSupportedChannelsError

void IEEE80211APClient::onSupportedChannelsUpdate(
      const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPSupportedChannelsUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onSupportedChannelsUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   AccessPointProperty apProperty;
   SupportedChannelList channelList;
   std::map<::std::string, ::asf::dbus::DBusVariant> suppChannelsData = update->getSupportedChannels();
   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      parseSupportedChannels(suppChannelsData, channelList);
      apProperty.chanList = channelList;
      updateAccessPoint(objPath, apProperty, AP_SUPPORTED_CHANNELS);
   }
} // void IEEE80211APClient::onSupportedChannelsUpdate

void IEEE80211APClient::onMaximumStationsAllowedError(
      const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPMaxStationsAllowedError >& error)
{
   //TODO:Check when this might be triggered
   LOG_INFO("IEEE80211APClient::onMaximumStationsAllowedError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onMaximumStationsAllowedError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onMaximumStationsAllowedError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_MAXSTA_IND);

} // void IEEE80211APClient::onMaximumStationsAllowedError

void IEEE80211APClient::onMaximumStationsAllowedUpdate(
      const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrAPMaxStationsAllowedUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onMaximumStationsAllowedUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));

   LOG_DEBUG("onMaximumStationsAllowedUpdate: %d ", update->getMaximumStationsAllowed());

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      AccessPointProperty apProperty;
      apProperty.maxStations = static_cast<uint16>(update->getMaximumStationsAllowed());
      updateAccessPoint(objPath, apProperty, AP_MAX_STA_ALLOWED);
   }
} // void IEEE80211APClient::onMaximumStationsAllowedUpdate

void IEEE80211APClient::onSetSSIDError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrSetSSIDError >& error)
{
   LOG_INFO("IEEE80211APClient::onSetSSIDError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onSetSSIDError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onSetSSIDError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_SSID_IND);

} // void IEEE80211APClient::onSetSSIDError

void IEEE80211APClient::onSetSSIDResponse(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrSetSSIDResponse >& response)
{
   LOG_INFO("IEEE80211APClient::onSetSSIDResponse() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(response);
   //@Note: SetSSID response message not sent to observers since SSID property is also notified.
} // void IEEE80211APClient::onSetSSIDResponse

void IEEE80211APClient::onDeAuthenticateStationError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrDeAuthenticateStationError >& error)
{
   LOG_INFO("IEEE80211APClient::onDeAuthenticateStationError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onDeAuthenticateStationError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onDeAuthenticateStationError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_DEAUTHENTICATESTATION);

} // void IEEE80211APClient::onDeAuthenticateStationError

void IEEE80211APClient::onDeAuthenticateStationResponse(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrDeAuthenticateStationResponse >& response)
{
   LOG_INFO("IEEE80211APClient::onDeAuthenticateStationResponse() entered ");
   WBL_ASSERT_AND_EXIT(!(response && proxy));

   ::std::string objPath = proxy->getDBusObjectPath();
   AccessPointItem item;
   if (getAccessPointInfo(objPath, item))
   {
      AccessPointChangedMsg msg;
      msg.setAccessPointItem(item);
      msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_DEAUTHENTICATESTATION);
      notifyObservers(IEEE80211_EVENT_AP, &msg);
   }
} // void IEEE80211APClient::onDeAuthenticateStationResponse

void IEEE80211APClient::onSetOperatingChannelsError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrSetOperatingChannelsError >& error)
{
   LOG_INFO("IEEE80211APClient::onSetOperatingChannelsError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onSetOperatingChannelsError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onSetOperatingChannelsError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_SETOPERATINGCHANNEL);

} // void IEEE80211APClient::onSetOperatingChannelsError

void IEEE80211APClient::onSetOperatingChannelsResponse(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrSetOperatingChannelsResponse >& response)
{
   LOG_INFO("IEEE80211APClient::onSetOperatingChannelsResponse() entered ");
   WBL_ASSERT_AND_EXIT(!(response && proxy));

   ::std::string objPath = proxy->getDBusObjectPath();
   AccessPointItem item;
   if (getAccessPointInfo(objPath, item))
   {
      AccessPointChangedMsg msg;
      msg.setAccessPointItem(item);
      msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_SETOPERATINGCHANNEL);
      notifyObservers(IEEE80211_EVENT_AP, &msg);
   }
} // void IEEE80211APClient::onSetOperatingChannelsResponse

void IEEE80211APClient::onGetAssociatedStationsError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrGetAssociatedStationsError >& error)
{
   LOG_INFO("IEEE80211APClient::onGetAssociatedStationsError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onGetAssociatedStationsError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onGetAssociatedStationsError: Error message - %s ", error->getMessage().c_str());
   //!Add code

} // void IEEE80211APClient::onGetAssociatedStationsError

void IEEE80211APClient::onGetAssociatedStationsResponse(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrGetAssociatedStationsResponse >& response)
{
   LOG_INFO("IEEE80211APClient::onGetAssociatedStationsResponse() entered ");
   WBL_ASSERT_AND_EXIT(!(response && proxy));

   typedef ::std::vector< WapdmgrGetASRespASStruct > ASStructList;
   ASStructList asResponseList = response->getAssociatedstations();

   for (ASStructList::iterator it = asResponseList.begin(); it != asResponseList.end(); ++it)
   {
      ::std::string asObjPath = (*it).getElem1();
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > asData = (*it).getElem2();

      AssociatedStationProperty asProperty;
      parseAssociatedStationInfo(proxy->getDBusBusName(), proxy->getDBusObjectPath(),
            proxy->getBusType(), proxy->getInterfaceName(), asData, asProperty);
      addAssociatedStation(asObjPath, asProperty);
   } // for (ASStructList::iterator it =...)
} // void IEEE80211APClient::onGetAssociatedStationsResponse

void IEEE80211APClient::onRegisterVendorServiceError(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr < WapdmgrRegisterVendorServiceError >& error)
{
   LOG_INFO("IEEE80211APClient::onRegisterVendorServiceError() entered ");
   LOG_DEBUG("onRegisterVendorServiceError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onRegisterVendorServiceError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_REG_VENDOR_IND);
}

void IEEE80211APClient::onRegisterVendorServiceResponse(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr < WapdmgrRegisterVendorServiceResponse >& response)
{
   LOG_INFO("IEEE80211APClient::onRegisterVendorServiceResponse() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(response);
   ::std::string objPath = proxy->getDBusObjectPath();
   onVendorServiceUpdate(objPath,true);
}

void IEEE80211APClient::onUnregisterVendorServiceError(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr < WapdmgrUnregisterVendorServiceError >& error)
{
   LOG_INFO("IEEE80211APClient::onUnregisterVendorServiceError() entered ");
   LOG_DEBUG("onUnregisterVendorServiceError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onUnregisterVendorServiceError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_UNREG_VENDOR_IND);
}

void IEEE80211APClient::onUnregisterVendorServiceResponse(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
      const ::boost::shared_ptr < WapdmgrUnregisterVendorServiceResponse >& response)
{
   LOG_INFO("IEEE80211APClient::onUnregisterVendorServiceResponse() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(response);
   ::std::string objPath = proxy->getDBusObjectPath();
   onVendorServiceUpdate(objPath,false);
}

void IEEE80211APClient::onGetAllError(const ::boost::shared_ptr<WapdmgrAPPropertiesProxy>& proxy,
      const ::boost::shared_ptr<WapdmgrAPGetAllError>& error)
{
   LOG_INFO("IEEE80211APClient::onGetAllError() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(error);
}

void IEEE80211APClient::onGetAllResponse(const ::boost::shared_ptr<WapdmgrAPPropertiesProxy>& proxy,
      const ::boost::shared_ptr<WapdmgrAPGetAllResponse>& response)
{
   LOG_INFO("IEEE80211APClient::onGetAllResponse() entered ");
   WBL_ASSERT_AND_EXIT(!(proxy && response));
   AccessPointProperty apProperty;
   ::std::map< ::std::string, ::asf::dbus::DBusVariant > apData;
   ::std::string apObjPath;
   apObjPath = proxy->getDBusObjectPath();
   apData = response->getProperties();
   parseAccessPointInfo(proxy->getDBusBusName(), proxy->getDBusObjectPath(),
            proxy->getBusType(), proxy->getInterfaceName(), apData, apProperty);
   addAccessPoint(apObjPath, apProperty);
}

void IEEE80211APClient::onIPv4PoolError(const ::boost::shared_ptr<WapdmgrDHCPDNSConfigurationProxy>& proxy,
      const ::boost::shared_ptr<WapdmgrIPv4PoolError>& error)
{
   LOG_INFO("IEEE80211APClient::onIPv4PoolError() entered ");
   (void) proxy;
   (void) error;
}

void IEEE80211APClient::onIPv4PoolUpdate(const ::boost::shared_ptr<WapdmgrDHCPDNSConfigurationProxy>& proxy,
      const ::boost::shared_ptr<WapdmgrIPv4PoolUpdate>& update)
{
   LOG_INFO("IEEE80211APClient::onIPv4PoolUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   (void) proxy;
   (void) update;
   IPv4PoolList ipv4PoolList;
   IPv4ConfigurationList ipv4ConfigurationList;
   AccessPointProperty apProperty;

   ipv4PoolList = update->getIPv4Pool();
   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      parseIPv4PoolUpdate(ipv4PoolList,ipv4ConfigurationList);
      apProperty.ipv4ConfigurationList = ipv4ConfigurationList;
      printIPV4Configuration(ipv4ConfigurationList);
      updateAccessPoint(objPath, apProperty, AP_IPV4_CONFIGURATION);
   }

}

void IEEE80211APClient::onTetheringError(const ::boost::shared_ptr< WapdmgrTetheringSettingsProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrTetheringError >& error)
{
   LOG_INFO("IEEE80211APClient::onTetheringError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_TETHERING_IND);
}

void IEEE80211APClient::onTetheringUpdate(const ::boost::shared_ptr< WapdmgrTetheringSettingsProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrTetheringUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onTetheringUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   AccessPointProperty apProperty;

   ::std::string objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath))
   {
      apProperty.isTethering = update->getTethering();
      LOG_DEBUG("IEEE80211APClient::updateAccessPoint() entered: isTethering = %s ", BOOL_TO_STR(apProperty.isTethering));
      updateAccessPoint(objPath, apProperty, AP_TETHERING);
   }
}

void IEEE80211APClient::onAccessNetworkTypeError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
               const ::boost::shared_ptr<WapdmgrAccessNetworkTypeError >& error)
{
    LOG_INFO("IEEE80211APClient::onAccessNetworkTypeError() entered ");
    WBL_ASSERT_AND_EXIT(!(error && proxy));
    ::std::string objPath = proxy->getDBusObjectPath();
    onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_AP_ACCESS_NETWORK_TYPE_IND);
}

void IEEE80211APClient::onAccessNetworkTypeUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
               const ::boost::shared_ptr<WapdmgrAccessNetworkTypeUpdate >& update)
{
    LOG_INFO("IEEE80211APClient::onAccessNetworkTypeUpdate() entered ");
    WBL_ASSERT_AND_EXIT(!(update && proxy));
    AccessPointProperty apProperty;

    ::std::string objPath = proxy->getDBusObjectPath();
    if (isAccessPoint(objPath))
    {
     apProperty.wifiAccessNetworkType = (WifiAccessNetworkType) update->getAccessNetworkType();
     LOG_DEBUG("IEEE80211APClient::updateAccessPoint() entered: wifiAccessNetworkType = %d ", apProperty.wifiAccessNetworkType);
     updateAccessPoint(objPath, apProperty, AP_ACCESS_NETWORK_TYPE);
    }
}
void IEEE80211APClient::parseIPv4PoolUpdate(IPv4PoolList ipv4PoolList, IPv4ConfigurationList &ipv4ConfigList)
{
   LOG_INFO("IEEE80211APClient::parseIPv4PoolUpdate() entered ");
   DBusMessageIter* valIter;
   IPv4Configuration ipv4Config;
   for(auto it=ipv4PoolList.begin();it != ipv4PoolList.end();it++)
   {
      IPv4Pool ipv4Pool = (::std::map< ::std::string, ::asf::dbus::DBusVariant > ) *it;
      for(auto innerIt=ipv4Pool.begin();innerIt != ipv4Pool.end();innerIt++)
      {
         ::asf::dbus::DBusVariant valVariant = innerIt->second;
         valIter= valVariant.getReadIterator();
         if(valIter)
         {
            if (innerIt->first.compare("router") == 0)
                ipv4Config.router= ::std::string((char*)getDbusIterValue(valIter));
            else if (innerIt->first.compare("start") == 0)
               ipv4Config.startAddress = ::std::string((char*)getDbusIterValue(valIter));
            else if (innerIt->first.compare("end") == 0)
               ipv4Config.endAddress= ::std::string((char*)getDbusIterValue(valIter));
            else if (innerIt->first.compare("netmask") == 0)
               ipv4Config.netMask  = ::std::string((char*)getDbusIterValue(valIter));
            else if (innerIt->first.compare("network") == 0)
               ipv4Config.network = ::std::string((char*)getDbusIterValue(valIter));
            else if (innerIt->first.compare("broadcast") == 0)
               ipv4Config.broadcast = ::std::string((char*)getDbusIterValue(valIter));
            else if (innerIt->first.compare("leasetime") == 0)
               ipv4Config.leasetime = *((uint32*)getDbusIterValue(valIter));
            else
               LOG_ERROR("IEEE80211APClient::parseIPv4PoolUpdate() Invalid IPv4Configuration");
         }
      }
      ipv4ConfigList.push_back(ipv4Config);
   }
}

void IEEE80211APClient::printIPV4Configuration(const IPv4ConfigurationList &ipv4ConfigList)
{
   IPv4Configuration ipv4Config;
   for(auto it = ipv4ConfigList.begin(); it != ipv4ConfigList.end();it++)
      ipv4Config = *it;
   if(ipv4Config.isEmpty())
       return;
   LOG_DEBUG("IEEE80211APClient::printIPV4Configuration() IPv4Configuration "
         "\n Router:%s \n StartAddress:%s \n NetMask:%s \n EndAddress:%s \n network:%s \n brodacast:%s \n leastime:%d ",
         ipv4Config.router.c_str(),ipv4Config.startAddress.c_str(),ipv4Config.netMask.c_str(),ipv4Config.endAddress.c_str(),
         ipv4Config.network.c_str(),ipv4Config.broadcast.c_str(),ipv4Config.leasetime);
}

void IEEE80211APClient::onWapdmgrAssociatedStnServiceAvailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      const ServiceState previousState, const ServiceState currentState)
{
   LOG_INFO("IEEE80211APClient::onWapdmgrAssociatedStnServiceAvailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IWapdmgrAssociatedStationProxyIf* wapdmgrAssociatedStnProxyIf = (dbusFactory)?
         (dbusFactory->getWapdmgrAssociatedStationProxyIf()) : (nullptr);
   if (wapdmgrAssociatedStnProxyIf)
   {
      wapdmgrAssociatedStnProxyIf->getAllProperties(sWapdmgrBusName,
            objPath, DBUS_BUS_SYSTEM, this);
   }
} // void IEEE80211APClient::onWapdmgrAssociatedStnServiceAvailable

void IEEE80211APClient::onWapdmgrAssociatedStnServiceUnavailable(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType,
      const ServiceState previousState, const ServiceState currentState)
{
   LOG_INFO("IEEE80211APClient::onWapdmgrAssociatedStnServiceUnavailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);
   //! Add code
} // void IEEE80211APClient::onWapdmgrAssociatedStnServiceUnavailable

void IEEE80211APClient::onMacAddressError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASMacAddressError >& error)
{
   LOG_INFO("IEEE80211APClient::onMacAddressError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onMacAddressError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onMacAddressError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAssociatedStationError(objPath, IEEE80211_MSG_AP_MODE_AS_MACADDRESS_IND);

} // void IEEE80211APClient::onMacAddressError

void IEEE80211APClient::onMacAddressUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASMacAddressUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onMacAddressUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   LOG_DEBUG("onMacAddressUpdate: %s ", update->getMacAddress().c_str());
   ::std::string objPath = proxy->getDBusObjectPath();
   bool isValidAS = false;
   AssociatedStationItem item;
   isValidAS = getAssociatedStationInfo(objPath, item);
   if (isValidAS)
   {
      AssociatedStationProperty asProperty;
      asProperty = item.property;
      asProperty.macAddress = update->getMacAddress();
      updateAssociatedStation(objPath, asProperty, AS_MAC_ADDRESS);
   }
} // void IEEE80211APClient::onMacAddressUpdate

void IEEE80211APClient::onAccesspointError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASAccesspointError >& error)
{
   LOG_INFO("IEEE80211APClient::onAccesspointError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onAccesspointError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onAccesspointError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAssociatedStationError(objPath, IEEE80211_MSG_AP_MODE_AS_APPATH_IND);

} // void IEEE80211APClient::onAccesspointError

void IEEE80211APClient::onAccesspointUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASAccesspointUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onAccesspointUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   LOG_DEBUG("onAccesspointUpdate: %s ", update->getAccesspoint().c_str());
   ::std::string objPath = proxy->getDBusObjectPath();
   bool isValidAS = false;
   AssociatedStationItem item;
   isValidAS = getAssociatedStationInfo(objPath, item);
   if (isValidAS)
   {
      AssociatedStationProperty asProperty;
      asProperty = item.property;
      asProperty.apPath = update->getAccesspoint();
      updateAssociatedStation(objPath, asProperty, AS_ACCESSPOINT_PATH);
   }
} // void IEEE80211APClient::onAccesspointUpdate

void IEEE80211APClient::onInterfaceError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASInterfaceError >& error)
{
   LOG_INFO("IEEE80211APClient::onInterfaceError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onInterfaceError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onInterfaceError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAssociatedStationError(objPath, IEEE80211_MSG_AP_MODE_AS_INTERFACE_IND);

} // void IEEE80211APClient::onInterfaceError

void IEEE80211APClient::onInterfaceUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASInterfaceUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onInterfaceUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   LOG_DEBUG("onInterfaceUpdate: %s ", update->getInterface().c_str());
   ::std::string objPath = proxy->getDBusObjectPath();
   bool isValidAS = false;
   AssociatedStationItem item;
   isValidAS = getAssociatedStationInfo(objPath, item);
   if (isValidAS)
   {
      AssociatedStationProperty asProperty;
      asProperty = item.property;
      asProperty.interface = update->getInterface();
      updateAssociatedStation(objPath, asProperty, AS_INTERFACE);
   }
} // void IEEE80211APClient::onInterfaceUpdate

void IEEE80211APClient::onIPAddressError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASIPAddressError >& error)
{
   LOG_INFO("IEEE80211APClient::onIPAddressError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onIPAddressError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onIPAddressError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAssociatedStationError(objPath, IEEE80211_MSG_AP_MODE_AS_IPADDRESS_IND);

} // void IEEE80211APClient::onIPAddressError

void IEEE80211APClient::onIPAddressUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASIPAddressUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onIPAddressUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   LOG_DEBUG("onIPAddressUpdate: %s ", update->getIPAddress().c_str());
   ::std::string objPath = proxy->getDBusObjectPath();
   bool isValidAS = false;
   AssociatedStationItem item;
   isValidAS = getAssociatedStationInfo(objPath, item);
   if (isValidAS)
   {
      AssociatedStationProperty asProperty;
      asProperty = item.property;
      asProperty.ipAddress = update->getIPAddress();
      updateAssociatedStation(objPath, asProperty, AS_IP_ADDRESS);
   }
} // void IEEE80211APClient::onIPAddressUpdate

void IEEE80211APClient::onHostnameError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASHostnameError >& error)
{
   LOG_INFO("IEEE80211APClient::onHostnameError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));

   LOG_DEBUG("onHostnameError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onHostnameError: Error message - %s ", error->getMessage().c_str());

   ::std::string objPath = proxy->getDBusObjectPath();
   onAssociatedStationError(objPath, IEEE80211_MSG_AP_MODE_AS_HOSTNAME_IND);

} // void IEEE80211APClient::onHostnameError

void IEEE80211APClient::onHostnameUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
      const ::boost::shared_ptr< WapdmgrASHostnameUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onHostnameUpdate() entered ");
   WBL_ASSERT_AND_EXIT(!(update && proxy));
   LOG_DEBUG("onHostnameUpdate: %s ", update->getHostname().c_str());
   ::std::string objPath = proxy->getDBusObjectPath();
   bool isValidAS = false;
   AssociatedStationItem item;
   isValidAS = getAssociatedStationInfo(objPath, item);
   if (isValidAS)
   {
      AssociatedStationProperty asProperty;
      asProperty = item.property;
      asProperty.hostName = update->getHostname();
      updateAssociatedStation(objPath, asProperty, AS_HOST_NAME);
   }
} // void IEEE80211APClient::onHostnameUpdate

bool IEEE80211APClient::isAccessPoint(const ::std::string& objPath) const
{
   bool isValidAP = false;
   for (AccessPointList::const_iterator it = _apList.begin(); it != _apList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         isValidAP = true;
         break;
      }
   }
   LOG_DEBUG("IEEE80211APClient::isAccessPoint: %s (ObjPath = %s)",
         BOOL_TO_STR(isValidAP), objPath.c_str());
   return isValidAP;
} // bool IEEE80211APClient::isAccessPoint

bool IEEE80211APClient::getAccessPointInfo(const ::std::string& objPath, AccessPointItem& item)
{
   bool isValidAP = false;
   for (AccessPointList::iterator it = _apList.begin(); it != _apList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         isValidAP = true;
         item = (*it);
         break;
      }
   }
   LOG_DEBUG("IEEE80211APClient::getAccessPointInfo: isValidAP = %s (ObjPath = %s)",
         BOOL_TO_STR(isValidAP), objPath.c_str());
   return isValidAP;
} // bool IEEE80211APClient::getAccessPointInfo

void IEEE80211APClient::addAccessPoint(const ::std::string& objPath, const AccessPointProperty& property)
{
   LOG_INFO("IEEE80211APClient::addAccessPoint() entered: ObjPath = %s ", objPath.c_str());

   //printAccessPointInfo(property);

   if (property.interface.empty())
   {
      LOG_INFO("IEEE80211APClient::addAccessPoint() Interface Unavailable. Create Accesspoint Proxy");
      addAccessPointProxy(objPath);

      DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
      IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf =
            (dbusFactory) ? (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);

      if (wapdmgrAccessPointProxyIf)
      {
         wapdmgrAccessPointProxyIf->sendGetAllProperties(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM, *this);
      }
   }
   else
   {
      bool isNewAccessPoint = true;

      AccessPointItem newAPItem;
      newAPItem.objPath = objPath;
      newAPItem.property = property;

      for (AccessPointList::iterator it = _apList.begin(); it != _apList.end(); ++it)
      {
         AccessPointItem curAPItem(*it);
         if (objPath == curAPItem.objPath)
         {
            isNewAccessPoint = false;
            break;
         } // if (objPath == curAPItem.objPath)
      } // for (AccessPointList::iterator it =...)

      if (true == isNewAccessPoint)
      {
         initLastModeSettings(objPath,property);
         addAccessPointProxy(objPath);
         _apList.push_back(newAPItem);
         AccessPointAddedMsg msg;
         msg.setAccessPointItem(newAPItem);
         notifyObservers(IEEE80211_EVENT_AP, &msg);
         //APService Should be notified upon the Proxy Service Availability/Unavailability
         //notifyAPServiceAvailability();
      } // if (true == isNewAccessPoint)
      else
      {
         LOG_INFO("IEEE80211APClient::addAccessPoint() Already Exist: ObjPath = %s ", objPath.c_str());
         restoreLastModeSettings(objPath,property);
         //Update the Existing accesspoint with GetAll
         updateChangedAccessPointProperties(newAPItem.objPath, newAPItem.property);
      }
   }
} // void IEEE80211APClient::addAccessPoint


void IEEE80211APClient::initLastModeSettings(::std::string objPath,const AccessPointProperty& property)
{
   LOG_INFO("IEEE80211APClient::%s ", __func__);
   bool isPoweredSetting = false;

   IDBManager* dbmgrIf = DBManagerFactory::getInstance()->getDBManagerIf();
   if (dbmgrIf)
   {
      isPoweredSetting = dbmgrIf->readPoweredSetting(WIFI_MODE_AP1, objPath);
      if ((isPoweredSetting) && (!property.isPowered) && (LCM_WIFISTATE_NORMAL == getWifiState()))
      {
         RestoreAccessPointMsg msg;
         msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_RESTORE_START);
         msg.setRestoreType(WBL_AP_RESTORE_POWER);
         notifyObservers(IEEE80211_EVENT_AP, &msg);
         _restoringAP = true;
      }
   }
}

void IEEE80211APClient::restoreLastModeSettings(const ::std::string& objPath,const AccessPointProperty& property)
{
   LOG_INFO("IEEE80211APClient::%s ", __func__);
   std::string objectPath;
   bool isRestricted = false;
   bool isPoweredSetting = false;
   APSetup apSetup;

   IDBManager* dbmgrIf = DBManagerFactory::getInstance()->getDBManagerIf();
   if (dbmgrIf)
   {
       if(dbmgrIf->readAPConfigsFromDB(apSetup,isRestricted))
       {
          if(isRestricted)
          {
             LOG_DEBUG("IEEE80211APClient:::restoreLastModeSettings Proceed to Restore Restricted AP");
             restoreOnRestrictedAP(objPath,apSetup);
             dbmgrIf->writeAPConfigsToDB(apSetup,false);
          }
          else
          {
             LOG_DEBUG("IEEE80211APClient::restoreOnRestrictedAP Proceed to Restore Power");
             isPoweredSetting = dbmgrIf->readPoweredSetting(WIFI_MODE_AP1, objectPath);
             if ((isPoweredSetting) && (!property.isPowered) && (LCM_WIFISTATE_NORMAL == getWifiState()))
             {
                setPowered(objPath, isPoweredSetting);
                if(apSetup.type == WBL_AP_TYPE_CARPLAY) //If CPW APType Register Vendor Elements
                {
                   updateRegisterVendorService(objPath); 
                }
             }
             updateLastModeSettings(property);
          }
       }
   }
}

void IEEE80211APClient::updateLastModeSettings(const AccessPointProperty& property)
{
   LOG_INFO("IEEE80211APClient::%s ", __func__);
   if(property.isPowered ||
         property.powerState == WIFI_STATE_POWERING_ON ||
         property.powerState == WIFI_STATE_POWER_FAILURE)
   {
      RestoreAccessPointMsg msg;
      msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_RESTORE_COMPLETE);
      msg.setRestoreType(WBL_AP_RESTORE_POWER);
      notifyObservers(IEEE80211_EVENT_AP, &msg);
      _restoringAP = false;
   }
}

void IEEE80211APClient::restoreOnRestrictedAP(const ::std::string &objectPath,const APSetup &apSetup)
{
   LOG_INFO("IEEE80211APClient::%s ", __func__);
   setAPType(objectPath,apSetup.type);
   setSSID(objectPath,apSetup.SSID,false);
   setPassphrase(objectPath,apSetup.passphrase);
   setPowered(objectPath,false);
}

void IEEE80211APClient::updateChangedAccessPointProperties(const ::std::string& objPath,
         const AccessPointProperty& property)
{
   LOG_INFO("IEEE80211APClient::updateChangedAccessPointProperties() entered: ObjPath = %s ", objPath.c_str());
   for ( int apInt = AP_MAC_ADDRESS; apInt <= AP_HIDDEN; apInt++ )
   {
      AccessPointPropertyName propertyName = static_cast<AccessPointPropertyName>(apInt);
      updateAccessPoint(objPath,property,propertyName);
   }
} //IEEE80211APClient::updateChangedAccessPointProperties

void IEEE80211APClient::updateAccessPoint(const ::std::string& objPath, const AccessPointProperty& property,
      const AccessPointPropertyName& changedPropertyName)
{
   bool notify = true;
   auto it =_apList.begin();
   for (; it != _apList.end(); ++it)
   {
      if (objPath == (*it).objPath)
      {
         break;
      } // if (objPath == curAPItem.objPath)
   } // for (AccessPointList::iterator it =...)

   if (_apList.end() != it)
   {
      AccessPointChangedMsg msg;
      AccessPointProperty curAPProperty = it->property;

      switch (changedPropertyName)
      {
         case AP_REG_DOM:
         {
            it->property._regDom = property._regDom;
            notify = false;
         }
         break;
         case AP_MAC_ADDRESS:
         {
            it->property.macAddress = property.macAddress;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_MACADDRESS_IND);
         }
         break;
         case AP_TYPE:
         {
           LOG_INFO("IEEE80211APClient::%s AP Type:%s", __func__,
                wblAPType.getAPType2String(property.type).c_str());
           it->property.type = property.type;
           msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_TYPE_IND);
         }
         break;
         case AP_SSID:
         {
            it->property.ssid = property.ssid;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_SSID_IND);
         }
         break;
         case AP_INTERFACE:
         {
            it->property.interface = property.interface;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_INTERFACE_IND);
         }
         break;
         case AP_POWERSTATE:
         {
            it->property.powerState = property.powerState;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_POWERSTATE_IND);
         }
         break;
         case AP_POWERED:
         {
            it->property.isPowered = property.isPowered;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_POWERED_IND);
         }
         break;
         case AP_POWERFAILURE:
         {
            it->property.powerFailure = property.powerFailure;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_POWERFAILURE_IND);
         }
         break;
         case AP_SECURITY:
         {
            it->property.security = property.security;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_SECURITY_IND);
         }
         break;
         case AP_PASSPHRASE:
         {
            it->property.passphrase = property.passphrase;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_PASSPHRASE_IND);
         }
         break;
         case AP_SUPPORTED_CHANNELS:
         {
            it->property.chanList = property.chanList;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_SUPPCHANNELS_IND);
         }
         break;
         case AP_CURRENT_CHANNEL:
         {
            it->property.currChannel = property.currChannel;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_CURRENTCHANNEL_IND);
         }
         break;
         case AP_MAX_STA_ALLOWED:
         {
            it->property.maxStations = property.maxStations;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_MAXSTA_IND);
         }
         break;
         case AP_VENDOR_IE:
         {
            it->property.vendorIEs = property.vendorIEs;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_VENDOR_IES_IND);
         }
         break;
         case AP_HIDDEN:
         {
            it->property.isHidden = property.isHidden;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_HIDDEN_IND);
         }
         break;
         case AP_TETHERING:
         {
            it->property.isTethering = property.isTethering;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_TETHERING_IND);
         }
         break;
         case AP_IPV4_CONFIGURATION:
         {
            it->property.ipv4ConfigurationList = property.ipv4ConfigurationList;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_IPV4_CONFIGURATION_IND);
            msg.setAccessPointItem(*it);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
            notify = false;
         }
         break;
         case AP_HEALTHINDICATOR:
         {
            it->property.connHealth = property.connHealth;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_HEALTH_IND);
         }
         break;
         case AP_ACCESS_NETWORK_TYPE:
         {
             it->property.wifiAccessNetworkType = property.wifiAccessNetworkType;
             msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_ACCESS_NETWORK_TYPE_IND);
         }
         break;
         default:
            LOG_ERROR("updateAccessPoint: Invalid changed property name %d ", changedPropertyName);
         break;
      } //switch (propertyName)

      AccessPointProperty newAPProperty = it->property;
      bool notifyASChange = !(curAPProperty == newAPProperty);

      if (notifyASChange && notify)
      {
         LOG_DEBUG("IEEE80211APClient::updateAccessPoint() entered: ObjPath = %s ", objPath.c_str());
         //printAccessPointInfo(newAPProperty);
         msg.setAccessPointItem(*it);
         notifyObservers(IEEE80211_EVENT_AP, &msg);
      }
   }
} // void IEEE80211APClient::updateAccessPoint

void IEEE80211APClient::removeAccessPoint(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211APClient::removeAccessPoint() entered: ObjPath = %s ", objPath.c_str());

   for (AccessPointList::iterator it = _apList.begin(); it != _apList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         removeAccessPointProxy(objPath);

         _apList.erase(it);

         AccessPointRemovedMsg msg;
         msg.setobjPath(objPath);
         notifyObservers(IEEE80211_EVENT_AP, &msg);
         //APService Should be notified upon the Proxy Service Availability/Unavailability
         //notifyAPServiceAvailability();
         break;
      } // if ((*it).objPath == objPath)
   } // for (AccessPointList::iterator it =...)
} // void IEEE80211APClient::removeAccessPoint

void IEEE80211APClient::printAccessPointInfo(const AccessPointProperty& property) const
{
   LOG_DEBUG("IEEE80211APClient:%s",__func__);
   ::std::string ssid = convertByteArray2String(property.ssid);
   LOG_DEBUG("Interface: \"%s\" "
             "SSID: \"%s\" "
             "MacAddress: \"%s\" "
             "Passphrase: \"%s\" "
             "Security: \"%s\" ",
             property.interface.c_str(),ssid.c_str(),property.macAddress.c_str(),
             property.passphrase.c_str(),wblSecurity.getSecurity2String(property.security).c_str());
   LOG_DEBUG("APType: %s ",wblAPType.getAPType2String(property.type).c_str());
   LOG_DEBUG("CurrChannel: \"%d\" "
             "MaxStations: \"%d\" "
             "isHidden: \"%s\" "
             "Reg Domain: \"%s\" ",
             property.currChannel,property.maxStations,
             BOOL_TO_STR(property.isHidden),property._regDom.c_str());
   LOG_DEBUG("Tethering = \"%s\" "
             "IsVendorElementRegistered = \"%s\""
             "NetworkType = \"%d\" ",
             BOOL_TO_STR(property.isTethering),BOOL_TO_STR(property.isVendorElementRegistered),
             property.wifiAccessNetworkType);
   LOG_DEBUG("IsPowered: %s PowerState = %s ",BOOL_TO_STR(property.isPowered),POWER_STATE_TO_STR(property.powerState));
} // void IEEE80211APClient::printAccessPointInfo

bool IEEE80211APClient::isAssociatedStation(const ::std::string& objPath) const
{
   bool isValidAS = false;
   for (AssociatedStationList::const_iterator it = _asList.begin(); it != _asList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         isValidAS = true;
         break;
      }
   }
   LOG_DEBUG("IEEE80211APClient::isAssociatedStation: %s (ObjPath = %s)", BOOL_TO_STR(isValidAS),
         objPath.c_str());
   return isValidAS;
} // bool IEEE80211APClient::isAssociatedStation

bool IEEE80211APClient::getAssociatedStationInfo(const ::std::string& objPath, AssociatedStationItem& item)
{
   bool isValidAS = false;
   for (AssociatedStationList::iterator it = _asList.begin(); it != _asList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         isValidAS = true;
         item = (*it);
         break;
      }
   }
   LOG_DEBUG("IEEE80211APClient::getAssociatedStationInfo: isValidAS = %s (ObjPath = %s)",
         BOOL_TO_STR(isValidAS), objPath.c_str());
   return isValidAS;
} // bool IEEE80211APClient::getAssociatedStationInfo

void IEEE80211APClient::addAssociatedStation(const ::std::string& objPath, const AssociatedStationProperty& property)
{
   LOG_INFO("IEEE80211APClient::addAssociatedStation() entered: ObjPath = %s ", objPath.c_str());

   bool isNewAssociatedStation = true;
   AssociatedStationChangedMsg msgChangedAS;
   AssociatedStationAddedMsg msgNewAS;
   AssociatedStationItem newASItem;
   newASItem.objPath = objPath;
   newASItem.property = property;

   for (auto it = _asList.begin(); it != _asList.end(); ++it)
   {
      AssociatedStationItem curASItem(*it);
      if (!curASItem.updated && objPath == curASItem.objPath)
      {
         if (!(curASItem.property == newASItem.property) && !(newASItem.property.isEmpty())) //If Property Changed Update new property
         {
            *it = newASItem;
            msgChangedAS.setAssociatedStationItem(newASItem);
            notifyObservers(IEEE80211_EVENT_AP, &msgChangedAS);
            isNewAssociatedStation = false;
         }
         break;
      } // if (objPath == curASItem.objPath)
   } // for (AssociatedStationList::iterator it =...)

   if (true == isNewAssociatedStation)
   {
      addAssociatedStnProxy(objPath);
      if (!newASItem.property.isEmpty()) {
         newASItem.updated = true;
         _asList.push_back(newASItem);
         msgNewAS.setAssociatedStationItem(newASItem);
         notifyObservers(IEEE80211_EVENT_AP, &msgNewAS);
      }
   } // if (true == isNewAssociatedStation)
} // void IEEE80211APClient::addAssociatedStation

void IEEE80211APClient::updateAssociatedStation(const ::std::string& objPath, const AssociatedStationProperty& property,
      const AssociatedStationPropertyName& changedPropertyName)
{
   LOG_INFO("IEEE80211APClient::updateAssociatedStation() entered: ObjPath = %s ", objPath.c_str());

   AssociatedStationList::iterator it = _asList.begin();
   for (; it != _asList.end(); ++it)
   {
      if (objPath == (*it).objPath)
      {
         break;
      } // if (objPath == curAPItem.objPath)
   } // for (AssociatedStationList::iterator it =...)

   if (_asList.end() != it)
   {
      AssociatedStationChangedMsg msg;
      AssociatedStationProperty curASProperty = (*it).property;

      switch (changedPropertyName)
      {
         case AS_MAC_ADDRESS:
         {
            (*it).property.macAddress = property.macAddress;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AS_MACADDRESS_IND);
         }
            break;
         case AS_ACCESSPOINT_PATH:
         {
            (*it).property.apPath = property.apPath;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AS_APPATH_IND);
         }
            break;
         case AS_INTERFACE:
         {
            (*it).property.interface = property.interface;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AS_INTERFACE_IND);
         }
            break;
         case AS_IP_ADDRESS:
         {
            (*it).property.ipAddress = property.ipAddress;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AS_IPADDRESS_IND);
         }
            break;
         case AS_HOST_NAME:
         {
            (*it).property.hostName = property.hostName;
            msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AS_HOSTNAME_IND);
         }
            break;
         default:
            LOG_ERROR("updateAssociatedStation: Invalid changed property name %d ", changedPropertyName);
            break;
      } //switch (propertyName)

      AssociatedStationProperty newASProperty = (*it).property;
      bool notifyASChange = !(curASProperty == newASProperty);
      if (notifyASChange)
      {
         msg.setAssociatedStationItem(*it);
         notifyObservers(IEEE80211_EVENT_AP, &msg);
      }
   } // if (_asList.end() != it)
} // void IEEE80211APClient::updateAssociatedStation

void IEEE80211APClient::removeAssociatedStation(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211APClient::removeAssociatedStation() entered: ObjPath = %s ", objPath.c_str());

   for (AssociatedStationList::iterator it = _asList.begin(); it != _asList.end(); ++it)
   {
      if ((*it).objPath == objPath)
      {
         removeAssociatedStnProxy(objPath);

         _asList.erase(it);

         AssociatedStationRemovedMsg msg;
         msg.setobjPath(objPath);
         notifyObservers(IEEE80211_EVENT_AP, &msg);
         break;
      } // if ((*it).objPath == objPath)
   } // for (AssociatedStationList::iterator it =...)
} // void IEEE80211APClient::removeAssociatedStation

void IEEE80211APClient::printAssociatedStationInfo(const AssociatedStationProperty& property) const
{
   LOG_DEBUG("printAssociatedStationInfo: apPath = %s ", property.apPath.c_str());
   LOG_DEBUG("printAssociatedStationInfo: interface = %s ", property.interface.c_str());
   LOG_DEBUG("printAssociatedStationInfo: macAddress = %s ", property.macAddress.c_str());
   LOG_DEBUG("printAssociatedStationInfo: ipAddress = %s ", property.ipAddress.c_str());
   LOG_DEBUG("printAssociatedStationInfo: hostName = %s ", property.hostName.c_str());
} // void IEEE80211APClient::printAssociatedStationInfo

void IEEE80211APClient::addAccessPointProxy(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211APClient::addAccessPointProxy() entered: ObjPath = %s ", objPath.c_str());

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory)?
         (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
   if (wapdmgrAccessPointProxyIf)
   {
      wapdmgrAccessPointProxyIf->setCallbackIf(sWapdmgrBusName,
            objPath, DBUS_BUS_SYSTEM, e8PROXY_USER_AP_CLIENT, this);
      wapdmgrAccessPointProxyIf->createProxy(sWapdmgrBusName,
            objPath, DBUS_BUS_SYSTEM);
   }
} // void IEEE80211APClient::addAccessPointProxy

void IEEE80211APClient::removeAccessPointProxy(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211APClient::removeAccessPointProxy() entered: ObjPath = %s ", objPath.c_str());

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf = (dbusFactory)?
         (dbusFactory->getWapdmgrAccessPointProxyIf()) : (nullptr);
   if (wapdmgrAccessPointProxyIf)
   {
      wapdmgrAccessPointProxyIf->setCallbackIf(sWapdmgrBusName,
            objPath, DBUS_BUS_SYSTEM, e8PROXY_USER_AP_CLIENT, nullptr);
   }
} // void IEEE80211APClient::removeAccessPointProxy

void IEEE80211APClient::addAssociatedStnProxy(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211APClient::addAssociatedStnProxy() entered: ObjPath = %s ", objPath.c_str());

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IWapdmgrAssociatedStationProxyIf* wapdmgrAssociatedStnProxyIf = (dbusFactory)?
         (dbusFactory->getWapdmgrAssociatedStationProxyIf()) : (nullptr);
   if (wapdmgrAssociatedStnProxyIf)
   {
      wapdmgrAssociatedStnProxyIf->setCallbackIf(sWapdmgrBusName,
            objPath, DBUS_BUS_SYSTEM, e8PROXY_USER_AP_CLIENT, this);
      wapdmgrAssociatedStnProxyIf->createProxy(sWapdmgrBusName,
               objPath, DBUS_BUS_SYSTEM);
   }
} // void IEEE80211APClient::addAssociatedStnProxy

void IEEE80211APClient::removeAssociatedStnProxy(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211APClient::removeAssociatedStnProxy() entered: ObjPath = %s ", objPath.c_str());

   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IWapdmgrAssociatedStationProxyIf* wapdmgrAssociatedStnProxyIf = (dbusFactory)?
         (dbusFactory->getWapdmgrAssociatedStationProxyIf()) : (nullptr);
   if (wapdmgrAssociatedStnProxyIf)
   {
      wapdmgrAssociatedStnProxyIf->setCallbackIf(sWapdmgrBusName,
            objPath, DBUS_BUS_SYSTEM, e8PROXY_USER_AP_CLIENT, nullptr);
   }
} // void IEEE80211APClient::removeAssociatedStnProxy

void IEEE80211APClient::parseAccessPointInfo(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType, const ::std::string& interfaceName,
      ::std::map< ::std::string, ::asf::dbus::DBusVariant >& data, AccessPointProperty& property) const
{
   LOG_INFO("IEEE80211APClient::parseAccessPointInfo() entered ");

   ::std::vector<int> matchingInterfaces;
   matchingInterfaces.push_back((int)WAPDMGR_IF_ACCESS_POINT);

   ::std::vector< DbusVariantProperty > ifProperties;
   wapdmgrDbusParser.parseProperties(ifProperties, matchingInterfaces, data,
         busType, busName, objPath, interfaceName, false);

   for(size_t i = 0; i < ifProperties.size(); ++i)
   {
      AccessPointPropertyName propertyName = static_cast<AccessPointPropertyName>(ifProperties[i].propEnum);
      switch(propertyName)
      {
       case AP_REG_DOM:
       {
          if (VARIANT_STRING == ifProperties[i].propData.getType()) {
             property._regDom = ifProperties[i].propData.getString();
             LOG_DEBUG("Country Code = %s ", property._regDom.c_str());
          }
       }
       break;
         case AP_MAC_ADDRESS:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.macAddress = ifProperties[i].propData.getString();
            LOG_DEBUG("parseAccessPointInfo: MacAddress = %s ", property.macAddress.c_str());
            break;
         }
         case AP_SSID:
         {
            WBL_ASSERT(VARIANT_BYTE_ARRAY != ifProperties[i].propData.getType());
            const std::vector<uint8>& byteArray = ifProperties[i].propData.getByteArray();
            property.ssid = byteArray;
            ::std::string SSID = convertByteArray2String(byteArray);
            LOG_DEBUG("parseAccessPointInfo: SSID in string format = %s ", SSID.c_str());
            break;
         }
         case AP_INTERFACE:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.interface = ifProperties[i].propData.getString();
            LOG_DEBUG("parseAccessPointInfo: Interface = %s ", property.interface.c_str());
            break;
         }
         case AP_POWERSTATE:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            ::std::string powerstate = POWERSTATE_PREFIX + ifProperties[i].propData.getString();
            property.powerState = wblPowerState.getPowerState2Enum(powerstate);
            LOG_DEBUG("parseAccessPointInfo: powerstate = %s ", powerstate.c_str());
            break;
         }
         case AP_POWERED:
         {
            WBL_ASSERT(VARIANT_BOOL != ifProperties[i].propData.getType());
            property.isPowered = ifProperties[i].propData.getBool();
            LOG_DEBUG("parseAccessPointInfo: isPowered = %d ", property.isPowered);
            break;
         }
         case AP_POWERFAILURE:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            ::std::string powerfailure = ifProperties[i].propData.getString();
            property.powerFailure = wblPowerFailure.getPowerFailure2Enum(powerfailure);
            LOG_DEBUG("parseAccessPointInfo: powerfailure = %s ", powerfailure.c_str());
            break;
         }
         case AP_SECURITY:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            ::std::string security = ifProperties[i].propData.getString();
            property.security = wblSecurity.getSecurity2Enum(security);
            LOG_DEBUG("parseAccessPointInfo: Security = %s ", security.c_str());
            break;
         }
         case AP_PASSPHRASE:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.passphrase = ifProperties[i].propData.getString();
            LOG_DEBUG("parseAccessPointInfo: Passphrase = %s ", property.passphrase.c_str());
            break;
         }
         case AP_SUPPORTED_CHANNELS:
         {
            WBL_ASSERT(VARIANT_DICT != ifProperties[i].propData.getType());
            ::std::map < ::std::string, uint16_t > channelsDict =
                  ifProperties[i].propData.getString2UInt16Dict();
            //LOG_DEBUG("parseAccessPointInfo: Supported channels list size = %d ", channelsDict.size());
            for (auto itr = channelsDict.begin(); itr != channelsDict.end(); ++itr)
            {
               //LOG_DEBUG("parseAccessPointInfo: Received Frequency = %d, Channel = %s ",
                     //itr->second, (itr->first).c_str());
               Channel channel = 0;
               if (parseChannelNumber(itr->first, channel))
               {
                  property.chanList[channel] = itr->second;
                  LOG_DEBUG("parseAccessPointInfo: Stored Channel = %d, Frequency = %d ",
                        channel, property.chanList[channel]);
               }
            }
            break;
         }
         case AP_CURRENT_CHANNEL:
         {
            WBL_ASSERT(VARIANT_UINT16 != ifProperties[i].propData.getType());
            property.currChannel = ifProperties[i].propData.getUInt16();
            LOG_DEBUG("parseAccessPointInfo: currChannel = %d ", property.currChannel);
            break;
         }
         case AP_MAX_STA_ALLOWED:
         {
            WBL_ASSERT(VARIANT_UINT16 != ifProperties[i].propData.getType());
            property.maxStations = ifProperties[i].propData.getUInt16();
            LOG_DEBUG("parseAccessPointInfo: maxStations = %d ", property.maxStations);
            break;
         }
         case AP_VENDOR_IE:
         {
            //WBL_ASSERT(VARIANT_UINT16 != ifProperties[i].propData.getType());
            //property.vendorIEs = ifProperties[i].propData.getString(); //TODO
            break;
         }
         case AP_HIDDEN:
         {
            WBL_ASSERT(VARIANT_BOOL != ifProperties[i].propData.getType());
            property.isHidden = ifProperties[i].propData.getBool();
            LOG_DEBUG("parseAccessPointInfo: isHidden = %d ", property.isHidden);
            break;
         }
         case AP_ACCESS_NETWORK_TYPE:
         {
             WBL_ASSERT(VARIANT_INT16 != ifProperties[i].propData.getType());
             property.wifiAccessNetworkType = (WifiAccessNetworkType) ifProperties[i].propData.getUInt16();
             LOG_DEBUG("parseAccessPointInfo: wifiAccessNetworkType = %d ", property.wifiAccessNetworkType);
             break;
         }
         default:
         {
            LOG_ERROR("parseAccessPointInfo: Invalid access point property %d ", propertyName);
            break;
         }
      } // switch(ifProperties[i].propEnum)
   } // for(size_t i = 0; i < ifProperties.size(); i++)
} // void IEEE80211APClient::parseAccessPointInfo

void IEEE80211APClient::parseAssociatedStationInfo(const ::std::string& busName,
      const ::std::string& objPath, const ::DBusBusType busType, const ::std::string& interfaceName,
      ::std::map< ::std::string, ::asf::dbus::DBusVariant >& data, AssociatedStationProperty& property) const
{
   LOG_INFO("IEEE80211APClient::parseAssociatedStationInfo() entered ");

   ::std::vector<int> matchingInterfaces;
   matchingInterfaces.push_back((int)WAPDMGR_IF_ASSOCIATED_STATIONS);

   ::std::vector< DbusVariantProperty > ifProperties;
   wapdmgrDbusParser.parseProperties(ifProperties, matchingInterfaces, data,
         busType, busName, objPath, interfaceName);

   for(size_t i = 0; i < ifProperties.size(); ++i)
   {
      AssociatedStationPropertyName propertyName = static_cast<AssociatedStationPropertyName>(ifProperties[i].propEnum);
      switch(propertyName)
      {
         case AS_MAC_ADDRESS:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.macAddress = ifProperties[i].propData.getString();
            LOG_DEBUG("parseAssociatedStationInfo: MacAddress = %s ", property.macAddress.c_str());
            break;
         }
         case AS_ACCESSPOINT_PATH:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.apPath = ifProperties[i].propData.getString();
            LOG_DEBUG("parseAssociatedStationInfo: apPath = %s ", property.apPath.c_str());
            break;
         }
         case AS_INTERFACE:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.interface = ifProperties[i].propData.getString();
            LOG_DEBUG("parseAssociatedStationInfo: interface = %s ", property.interface.c_str());
            break;
         }
         case AS_IP_ADDRESS:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.ipAddress = ifProperties[i].propData.getString();
            LOG_DEBUG("parseAssociatedStationInfo: ipAddress = %s ", property.ipAddress.c_str());
            break;
         }
         case AS_HOST_NAME:
         {
            WBL_ASSERT(VARIANT_STRING != ifProperties[i].propData.getType());
            property.hostName = ifProperties[i].propData.getString();
            LOG_DEBUG("parseAssociatedStationInfo: hostName = %s ", property.hostName.c_str());
            break;
         }
         default:
         {
            LOG_ERROR("parseAssociatedStationInfo: Invalid associated station property %d ", propertyName);
            break;
         }
      } // switch(ifProperties[i].propEnum)
   } // for(size_t i = 0; i < ifProperties.size(); i++)
} // void IEEE80211APClient::parseAssociatedStationInfo

void IEEE80211APClient::parseSupportedChannels(::std::map< ::std::string, ::asf::dbus::DBusVariant >& suppChannelsData, SupportedChannelList& channelList) const
{
   LOG_INFO("IEEE80211APClient::parseSupportedChannels() entered ");
   ::asf::dbus::DBusVariant valVariant;
   DBusMessageIter* valIter;
   void *vpVal = NULL;
   Channel channel;
   Frequency *frequency;
   LOG_DEBUG("parseAccessPointInfo: Supported channels list size = %d ", suppChannelsData.size());
   for (auto itr = suppChannelsData.begin(); itr != suppChannelsData.end(); ++itr)
   {
      channel = 0;
      if (parseChannelNumber(itr->first, channel))
      {
         valVariant = itr->second;
         valIter = valVariant.getReadIterator();
         if(valIter)
         {
            vpVal = getDbusIterValue(valIter);
            frequency = (uint16*)vpVal;
            channelList[channel] = *frequency;
            LOG_DEBUG("parseAccessPointInfo: Stored Channel = %d, Frequency = %d ",
                  channel, channelList[channel]);
         }
      }
   }
   LOG_DEBUG("IEEE80211APClient::parseSupportedChannels: supportedChannels size = %d", channelList.size());
} // void IEEE80211APClient::parseSupportedChannels

bool IEEE80211APClient::parseChannelNumber(const ::std::string& channelNumber, Channel& channel) const
{
   bool isParsingSuccessful = false;
   if (!channelNumber.empty())
   {
      std::vector< ::std::string > splitChannelData;
      boost::split(splitChannelData, channelNumber, boost::is_any_of(" "));
      if (2 == splitChannelData.size())
      {
         channel = static_cast<uint16>(atoi(splitChannelData[1].c_str()));
         isParsingSuccessful = true;
      }
   }
   //LOG_DEBUG("IEEE80211APClient::parseChannelNumber: isParsingSuccessful = %s (Channel = %d, ChannelNumber = %s)",
         //BOOL_TO_STR(isParsingSuccessful), channel, channelNumber.c_str());
   return isParsingSuccessful;
} // bool IEEE80211APClient::parseChannelNumber

void IEEE80211APClient::onAccessPointError(const ::std::string& apObjPath, IEEE80211EventID event,
      IEEE80211ErrorCode error)
{
   AccessPointItem item;
   if (getAccessPointInfo(apObjPath, item))
   {
      switch (event)
      {
         case IEEE80211_MSG_AP_MODE_APADDED_IND:
         {
            AccessPointAddedMsg msg;
            msg.setAccessPointItem(item);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
         case IEEE80211_MSG_AP_MODE_APREMOVED_IND:
         {
            AccessPointRemovedMsg msg;
            msg.setobjPath(apObjPath);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
         default:
         {
            AccessPointChangedMsg msg;
            msg.setAccessPointItem(item);
            msg.setIEEE80211EventID(event);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
      }
   }
} // void IEEE80211APClient::onAccessPointError

void IEEE80211APClient::onAssociatedStationError(const ::std::string& asObjPath, IEEE80211EventID event,
      IEEE80211ErrorCode error)
{
   AssociatedStationItem item;
   if (getAssociatedStationInfo(asObjPath, item))
   {
      switch (event)
      {
         case IEEE80211_MSG_AP_MODE_ASADDED_IND:
         {
            AssociatedStationAddedMsg msg;
            msg.setAssociatedStationItem(item);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
         case IEEE80211_MSG_AP_MODE_ASREMOVED_IND:
         {
            AssociatedStationRemovedMsg msg;
            msg.setobjPath(asObjPath);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
         default:
         {
            AssociatedStationChangedMsg msg;
            msg.setAssociatedStationItem(item);
            msg.setIEEE80211EventID(event);
            msg.setIEEE80211ErrCode(error);
            notifyObservers(IEEE80211_EVENT_AP, &msg);
         }
         break;
      }
   }
} // void IEEE80211APClient::onAssociatedStationError

void IEEE80211APClient::onBluezAdapterServiceAvailable(const ::std::string& busName, const ::std::string& objPath,
      const ::DBusBusType busType, const ::asf::core::ServiceState previousState,
      const ::asf::core::ServiceState currentState)
{
   LOG_INFO("IEEE80211APClient::onBluezAdapterServiceAvailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);
   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();

   IBluezAdapterProxyIf* bluezAdpaterProxyIf = (dbusFactory)?
        (dbusFactory->getBluezAdapterProxyIf()) : (nullptr);
   if (bluezAdpaterProxyIf)
   {
      bluezAdpaterProxyIf->sendGetAddressRequest(*this);
      bluezAdpaterProxyIf->sendGetAliasRequest(*this);
   }

}
void IEEE80211APClient::onBluezAdapterServiceUnAvailable(const ::std::string& busName, const ::std::string& objPath,
      const ::DBusBusType busType, const ::asf::core::ServiceState previousState,
      const ::asf::core::ServiceState currentState)
{
   LOG_INFO("IEEE80211APClient::onBluezAdapterServiceUnAvailable() entered ");
   WBL_INTENTIONALLY_UNUSED(busName);
   WBL_INTENTIONALLY_UNUSED(objPath);
   WBL_INTENTIONALLY_UNUSED(busType);
   WBL_INTENTIONALLY_UNUSED(previousState);
   WBL_INTENTIONALLY_UNUSED(currentState);
}
void IEEE80211APClient::onAdapterInitializationError(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
      const ::boost::shared_ptr < BluezAdapterInitializationError >& error)
{
   LOG_INFO("IEEE80211APClient::onAdapterInitializationError() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(error);
}
void IEEE80211APClient::onAdapterInitializationSignal(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
      const ::boost::shared_ptr < BluezAdapterInitializationSignal >& signal)
{
   LOG_INFO("IEEE80211APClient::onAdapterInitializationSignal() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(signal);
   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IBluezAdapterProxyIf* bluezAdapterProxyIf =
        (dbusFactory) ? (dbusFactory->getBluezAdapterProxyIf()) : (nullptr);
   if (bluezAdapterProxyIf)
   {
       ::std::string address;
       if(!bluezAdapterProxyIf->getBluezAdapterAddress(address))
       {
           LOG_DEBUG("IEEE80211APClient::%s BTAddress Still Empty.Retry getting Address",__func__);
           bluezAdapterProxyIf->sendGetAddressRequest(*this);
       }
       else
       {
           LOG_DEBUG("IEEE80211APClient::onAdapterInitializationSignal() Valid BT Address Available");
       }
   }
}
void IEEE80211APClient::onAddressError(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
      const ::boost::shared_ptr < BluezAddressError >& error)
{
   LOG_INFO("IEEE80211APClient::onAddressError() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   WBL_INTENTIONALLY_UNUSED(error);
}
void IEEE80211APClient::onAddressUpdate(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
      const ::boost::shared_ptr < BluezAddressUpdate >& update)
{
   LOG_INFO("IEEE80211APClient::onAddressUpdate() entered ");
   WBL_INTENTIONALLY_UNUSED(proxy);
   ::std::string address = update->getAddress();

   LOG_DEBUG("IEEE80211APClient::onAddressUpdate() %s ", address.c_str());
   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IBluezAdapterProxyIf* bluezAdapterProxyIf = (dbusFactory) ? (dbusFactory->getBluezAdapterProxyIf()) : (nullptr);
   if (bluezAdapterProxyIf)
   {
      bluezAdapterProxyIf->setBluezAdapterAddress(address);
   }
}


void IEEE80211APClient::onAliasError(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
        const ::boost::shared_ptr < BluezAliasError >& error)
{
    LOG_INFO("IEEE80211APClient::onAliasError() entered ");
    WBL_INTENTIONALLY_UNUSED(proxy);
    WBL_INTENTIONALLY_UNUSED(error);
}
void IEEE80211APClient::onAliasUpdate(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
        const ::boost::shared_ptr < BluezAliasUpdate >& update)
{
    LOG_INFO("IEEE80211APClient::onAliasUpdate() entered ");
    WBL_INTENTIONALLY_UNUSED(proxy);

    DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
    IBluezAdapterProxyIf* bluezAdapterProxyIf = (dbusFactory) ? (dbusFactory->getBluezAdapterProxyIf()) : (nullptr);
    if (bluezAdapterProxyIf)
    {
        ::std::string oldAlias;
        (void) bluezAdapterProxyIf->getBluezAdapterAlias(oldAlias);
        ::std::string alias = update->getAlias();
        LOG_DEBUG("IEEE80211APClient::onAliasUpdate() %s ", alias.c_str());
        LOG_DEBUG("IEEE80211APClient::onAliasUpdate()oldAlias %s ", oldAlias.c_str());
        if((!alias.empty()) && alias.compare(oldAlias)!= 0)
        {
           bluezAdapterProxyIf->setBluezAdapterAlias(alias);
           //aliasUpdateToVendor();
        }
    }
}
void IEEE80211APClient::aliasUpdateToVendor()
{
    LOG_INFO("IEEE80211APClient::%s",__func__);
    ::std::string objPath;
    AccessPointItem item;
    getAPObjectPath(objPath);
    bool bRet = false;
    //Check if AP is enabled and CP is on going then
    //Alias Update on Vendor Element is Performed
    //Unregister and Register vendor would update the beacons in AP
    if (isAccessPoint(objPath))
    {
        if (getAccessPointInfo(objPath, item))
        {
            LOG_DEBUG("IEEE80211APClient::%s APType:%s Powered:%s", __func__,
                    wblAPType.getAPType2String(item.property.type).c_str(),
                    BOOL_TO_STR(item.property.isPowered));
            if (item.property.type == WBL_AP_TYPE_CARPLAY
                    && item.property.isPowered
                    && item.property.isVendorElementRegistered)
            {
                printVendorElements(item.property.vendorIEs);
                bRet = updateUnregisterVendorService(objPath);
                LOG_DEBUG("IEEE80211APClient::%s Unregister vendor Status:%s", __func__,BOOL_TO_STR(bRet));
                if(bRet)
                {
                    onVendorServiceUpdate(objPath,false);
                    bRet = updateRegisterVendorService(objPath);
                    LOG_DEBUG("IEEE80211APClient::%s register vendor Status:%s", __func__,BOOL_TO_STR(bRet));
                }
            }
        }
     }
}

bool IEEE80211APClient::getAPObjectPath(::std::string &objPath)
{
   for (auto it = _apList.begin(); it != _apList.end(); ++it) {
      if (!it->objPath.empty()) {
         objPath =it->objPath;
         return true;
      }
   }
   return false;
}

bool IEEE80211APClient::updateUnregisterVendorService(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211APClient::updateUnregisterVendorService() entered ");
   bool isRequestSent = false;
   ::std::vector<ByteArray> iEs;
   AccessPointItem item;
   AccessPointChangedMsg msg;
   if (isAccessPoint(objPath))
   {
     if (getAccessPointInfo(objPath, item))
     {
        if(item.property.isVendorElementRegistered == false)
        {
           isRequestSent = true;
           msg.setAccessPointItem(item);
           msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_UNREG_VENDOR_IND);
           notifyObservers(IEEE80211_EVENT_AP, &msg);
        }
        else
        {
           IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf =
                   DBusProxyFactory::getInstance()->getWapdmgrAccessPointProxyIf();
           printVendorElements(item.property.vendorIEs);
           if ((wapdmgrAccessPointProxyIf) &&
              (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
           {
              if(!item.property.vendorIEs.empty())
              {
                 iEs.push_back(item.property.vendorIEs);
                 isRequestSent = wapdmgrAccessPointProxyIf->sendUnregisterVendorServiceRequest(sWapdmgrBusName, objPath,
                       DBUS_BUS_SYSTEM, *this, iEs);
              }
              else
              {
                 LOG_DEBUG("IEEE80211APClient::%s Invalid IE. Unregistration is not possible",__func__);
              }
           }
        }
     }
   }
   return isRequestSent;
}

bool IEEE80211APClient::updateRegisterVendorService(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211APClient::updateRegisterVendorService() entered ");
   bool isRequestSent = false;
   ::std::vector<ByteArray> iEs;
   bool isCPWSupport=false;
   AccessPointItem item;
   AccessPointChangedMsg msg;
   if (isAccessPoint(objPath))
   {
     if (getAccessPointInfo(objPath, item))
     {
        if(item.property.isVendorElementRegistered == true)
        {
           isRequestSent = true;
           msg.setAccessPointItem(item);
           msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_AP_REG_VENDOR_IND);
           notifyObservers(IEEE80211_EVENT_AP, &msg);
        }
        else
        {
           isCPWSupport = cKDSConfiguration::getInstance()->isCPWSupport();
           LOG_DEBUG("IEEE80211APClient::updateRegisterVendorService() isCPWSupport: %s",BOOL_TO_STR(isCPWSupport));
           if(isCPWSupport)
           {
              IWapdmgrAccessPointProxyIf* wapdmgrAccessPointProxyIf =
                    DBusProxyFactory::getInstance()->getWapdmgrAccessPointProxyIf();

              if ((wapdmgrAccessPointProxyIf) &&
                    (wapdmgrAccessPointProxyIf->isProxyServiceAvailable(sWapdmgrBusName, objPath, DBUS_BUS_SYSTEM)))
              {
                 if(prepareAccesspointVendorElements(item.property.vendorIEs))
                 {
                    iEs.push_back(item.property.vendorIEs);
                    isRequestSent = wapdmgrAccessPointProxyIf->sendRegisterVendorServiceRequest(sWapdmgrBusName, objPath,
                          DBUS_BUS_SYSTEM, *this, iEs);
                    updateAccessPoint(item.objPath,item.property,AP_VENDOR_IE);
                 }
                 else
                 {
                    LOG_DEBUG("IEEE80211APClient::%s Vendor Element Preparation Failed",__func__);
                 }
              }
           }
        }
     }
   }
   return isRequestSent;
}

void IEEE80211APClient::printVendorElements(const ByteArray vendorList)
{
    LOG_DEBUG("IEEE80211APClient::%s",__func__);
    if(vendorList.empty())
    {
        LOG_DEBUG("No Elements in the list");
    }
    else
    {
        for(auto it = vendorList.begin();it != vendorList.end();it++)
                LOG_DEBUG("%0x ",*it);
    }
}
void IEEE80211APClient::getHardwareLimitations(ByteArray &payload)
{
   LOG_INFO("IEEE80211APClient::getHardwareLimitations() entered ");
   HWLimitationInfo limitInfo;
   uint8 configVal = WBL_VENDOR_ELEMENT_VALUE_ZERO;
   int iRet;
   ::std::string group = "CPWSettings";
   iRet = DBManagerFactory::getInstance()->getDBManagerIf()->getHWLimitationInfo(group,limitInfo);
   bool is5GHzSupport = cKDSConfiguration::getInstance()->getWiFi5GHzConfiguration();

   if(iRet < 0)
   {
      LOG_ERROR("IEEE80211APClient::getHardwareLimitations Failed to Get Limitation Info");
   }
   if(limitInfo.isCPWSupport)
      configVal |=WBL_VENDOR_CODE_SUPPORT_CARPLAY;
   if(limitInfo.is2_4GHz_Support)
      configVal |=WBL_VENDOR_CODE_SUPPORT_2_4_GHZ;
   if(limitInfo.is5_GHz_Support && is5GHzSupport)
      configVal |=WBL_VENDOR_CODE_SUPPORT_5_GHZ;

   payload.push_back(WBL_VENDOR_ELEMENT_VALUE_ZERO);
   payload.push_back(configVal);
   payload.push_back(WBL_VENDOR_ELEMENT_VALUE_ZERO);
}

void IEEE80211APClient::convertAddresstoPayload(::std::string &address,ByteArray &payload)
{
    LOG_INFO("IEEE80211APClient::convertAddresstoPayload() entered ");
    ::std::string strAddress = address;
    ByteArray list;
    //::boost::replace_all(strAddress, ":", " ");
    uint16 count = static_cast<uint16> ((address.size() + 1) / 3);
    for(int i= 0;i<count;i++)
    {
        ::std::string numStr = strAddress.substr(i * 3, strAddress.find(":",i+3));
        int num = 0;
        ::std::sscanf(numStr.c_str(), "%x", &num);
        list.push_back((uint8)num);
    }
    payload = list;
}

void IEEE80211APClient::getWifiDeviceAddress(::std::string &address)
{
   LOG_INFO("IEEE80211APClient::getWifiDeviceAddress() entered ");
   for(auto it = _apList.begin(); it != _apList.end(); ++it)
   {
      if (!it->objPath.empty())
      {
         address = it->property.macAddress;
         LOG_DEBUG("IEEE80211APClient::getWifiDeviceAddress() Address:%s",address.c_str());
      }
   }
}
void IEEE80211APClient::getBluetoothMacAddress(::std::string &address)
{
   LOG_INFO("IEEE80211APClient::getBluetoothMacAddress() entered ");
   DBusProxyFactory* dbusFactory = DBusProxyFactory::getInstance();
   IBluezAdapterProxyIf* bluezAdapterProxyIf = (dbusFactory) ? (dbusFactory->getBluezAdapterProxyIf()) : (nullptr);
   if (bluezAdapterProxyIf)
   {
      if(bluezAdapterProxyIf->getBluezAdapterAddress(address))
      {
          LOG_DEBUG("IEEE80211APClient::getBluetoothMacAddress() Address:%s",address.c_str());
      }
   }
}
void IEEE80211APClient::getVehicleDeviceName(ByteArray &payload)
{
    LOG_INFO("IEEE80211APClient::getVehicleDeviceName() entered ");
    ::std::string aliasName;
    ByteArray vehicleName;
    aliasName = cKDSConfiguration::getInstance()->getVehicleName();
    LOG_DEBUG("IEEE80211APClient::getVehicleDeviceName() Name:%s",aliasName.c_str());
    if(!aliasName.empty())
    {
        for(auto it = aliasName.begin(); it != aliasName.end();++it)
              vehicleName.push_back((uint8)*it);
        payload =vehicleName;

    }
}
void IEEE80211APClient::getManufacturerModelName(ByteArray &payload)
{
   LOG_INFO("IEEE80211APClient::getManufacturerModelName() entered ");
   ByteArray manufacturerModelName;
   ::std::string buffer = cKDSConfiguration::getInstance()->getManufacturerModelName();
   for(auto it = buffer.begin(); it != buffer.end();++it)
      manufacturerModelName.push_back((uint8)*it);
   payload = manufacturerModelName;
}
void IEEE80211APClient::getManufacturerName(ByteArray &payload)
{
   LOG_INFO("IEEE80211APClient::getManufacturerName() entered ");
   ByteArray manufacturerName;
   ::std::string buffer = cKDSConfiguration::getInstance()->getManufacturerName();
   for(auto it = buffer.begin(); it != buffer.end();++it)
      manufacturerName.push_back((uint8)*it);
   payload = manufacturerName;
}

void IEEE80211APClient::initializeVendorElement(VendorElement &element)
{
   LOG_INFO("IEEE80211APClient::initializeVendorElement() entered ");
   element.elementID = WBL_VENDOR_ELEMENT_ID;
   element.oui = OUIelements;
   element.subtype = WBL_VENDOR_SUB_TYPE_ELEMENT;
}
void IEEE80211APClient::getVendorElement(InformationElement &element,ByteArray &list)
{
   LOG_INFO("IEEE80211APClient::getVendorElement() entered ");
   list.push_back(element.elementID);
   list.push_back(element.length);
   list.insert(list.end(),element.payload.begin(),element.payload.end());
}
void IEEE80211APClient::insertInfoElements(InformationElement &element)
{
   LOG_INFO("IEEE80211APClient::insertInfoElements() entered ");

   switch(element.type)
   {
      case WBL_VENDOR_ELEMENT_TYPE_FLAG:
      {
         LOG_DEBUG("IEEE80211APClient::insertInfoElements() WBL_VENDOR_ELEMENT_TYPE_FLAG ");
         element.elementID = WBL_VENDOR_INFO_ELEMENT_ID_FLAG;
         getHardwareLimitations(element.payload);
      }
      break;
      case WBL_VENDOR_ELEMENT_TYPE_VECHICLE_NAME:
      {
         LOG_DEBUG("IEEE80211APClient::insertInfoElements() WBL_VENDOR_ELEMENT_TYPE_VECHICLE_NAME ");
         element.elementID = WBL_VENDOR_INFO_ELEMENT_ID_NAME;
         getVehicleDeviceName(element.payload);
      }
      break;
      case WBL_VENDOR_ELEMENT_TYPE_MANUFACTURER_NAME:
      {
         LOG_DEBUG("IEEE80211APClient::insertInfoElements() WBL_VENDOR_ELEMENT_TYPE_MANUFACTURER_NAME ");
         element.elementID = WBL_VENDOR_INFO_ELEMENT_ID_MANUFACTURER;
         getManufacturerName(element.payload);
      }
      break;
      case WBL_VENDOR_ELEMENT_TYPE_MODEL_NAME:
      {
         LOG_DEBUG("IEEE80211APClient::insertInfoElements() WBL_VENDOR_ELEMENT_TYPE_MODEL_NAME ");
         element.elementID = WBL_VENDOR_INFO_ELEMENT_ID_MODEL;
         getManufacturerModelName(element.payload);
      }
      break;
      case WBL_VENDOR_ELEMENT_TYPE_DEVICE_OUI:
      {
         LOG_DEBUG("IEEE80211APClient::insertInfoElements() WBL_VENDOR_ELEMENT_TYPE_DEVICE_OUI ");
         element.elementID = WBL_VENDOR_INFO_ELEMENT_ID_OUI;
         element.payload = BoschOUIElements;
      }
      break;
      case WBL_VENDOR_ELEMENT_TYPE_BLUETOOTH_MAC:
      {
         LOG_DEBUG("IEEE80211APClient::insertInfoElements() WBL_VENDOR_ELEMENT_TYPE_BLUETOOTH_MAC ");
         element.elementID = WBL_VENDOR_INFO_ELEMENT_ID_BT_MAC;
         ::std::string address;
         getBluetoothMacAddress(address);
         if(!address.empty())
            convertAddresstoPayload(address,element.payload);
      }
      break;
      case WBL_VENDOR_ELEMENT_TYPE_DEVICE_MAC:
      {
         LOG_DEBUG("IEEE80211APClient::insertInfoElements() WBL_VENDOR_ELEMENT_TYPE_DEVICE_MAC ");
         element.elementID = WBL_VENDOR_INFO_ELEMENT_ID_DEV_ID;
         ::std::string address;
         //getWifiDeviceAddress(address);
         getBluetoothMacAddress(address);
         if(!address.empty())
            convertAddresstoPayload(address,element.payload);
      }
      break;
      case WBL_VENDOR_ELEMENT_TYPE_UNKNOWN:
      default:
      {
         LOG_DEBUG("IEEE80211APClient::insertInfoElements() WBL_VENDOR_ELEMENT_TYPE_UNKNOWN|DEFAULT ");
      }
      break;
   }
}
bool IEEE80211APClient::prepareAccesspointVendorElements(ByteArray &vendorList)
{
    LOG_INFO("IEEE80211APClient::prepareAccesspointVendorElements() entered ");
    bool bRet = true;
    vendorList.clear();
    VendorElement element;
    initializeVendorElement(element);
    vendorList.insert(vendorList.end(),element.oui.begin(),element.oui.end());
    vendorList.push_back(element.subtype);
    for (uint8 indexType = WBL_VENDOR_ELEMENT_TYPE_FLAG;
            indexType != WBL_VENDOR_ELEMENT_TYPE_UNKNOWN; ++indexType)
    {
        InformationElement subElement;
        subElement.type = (VendorElementType) indexType;
        insertInfoElements(subElement);
        subElement.length = (uint8) (subElement.payload.size());
        if(subElement.length > 0)
        {
           getVendorElement(subElement,vendorList);
        }
        else
        {
           bRet = false;
           LOG_DEBUG("IEEE80211APClient::%s Invalid Element Type:%d",__func__,subElement.type);
           break;
        }
    }
    element.length =static_cast<uint8>(vendorList.size());
    vendorList.insert(vendorList.begin(),1,element.length);
    vendorList.insert(vendorList.begin(),1,element.elementID);
    LOG_DEBUG("IEEE80211APClient::%s Element Items:%s",__func__,BOOL_TO_STR(bRet));
    return bRet;
}

void IEEE80211APClient::onRestoreDefaultAPSettingsError(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
      const ::boost::shared_ptr<WapdmgrRestoreDefaultAPSettingsError>& error)
{
   LOG_INFO("IEEE80211APClient::onRestoreDefaultAPSettingsError() entered ");
   WBL_ASSERT_AND_EXIT(!(error && proxy));
   LOG_DEBUG("onRestoreFactorySettingsError: Error name - %s ", error->getName().c_str());
   LOG_DEBUG("onRestoreFactorySettingsError: Error message - %s ", error->getMessage().c_str());
   ::std::string objPath = proxy->getDBusObjectPath();
   onAccessPointError(objPath, IEEE80211_MSG_AP_MODE_CLEAR_DATA_RESP);
}

void IEEE80211APClient::onRestoreDefaultAPSettingsResponse(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
      const ::boost::shared_ptr<WapdmgrRestoreDefaultAPSettingsResponse>& response)
{
   LOG_INFO("IEEE80211APClient::onRestoreDefaultAPSettingsResponse() entered ");
   WBL_ASSERT_AND_EXIT(!(response && proxy));

   ::std::string objPath = proxy->getDBusObjectPath();
   AccessPointItem item;
   if (getAccessPointInfo(objPath, item))
   {
      AccessPointChangedMsg msg;
      //msg.setAccessPointItem(item);
      msg.setIEEE80211EventID(IEEE80211_MSG_AP_MODE_CLEAR_DATA_RESP);
      notifyObservers(IEEE80211_EVENT_AP, &msg);
   }
}

void IEEE80211APClient::onCountryCodeError(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
      const ::boost::shared_ptr<WapdmgrAPCountryCodeError>& error)
{
   ::std::string objPath;

   objPath = proxy->getDBusObjectPath();
   LOG_ERROR("onMacAddressError: Object: %s Error name: %s, Error message: %s",
         objPath.c_str(), error->getName().c_str(), error->getMessage().c_str());
}

void IEEE80211APClient::onCountryCodeUpdate(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
      const ::boost::shared_ptr<WapdmgrAPCountryCodeUpdate>& update)
{
   ::std::string objPath;
   AccessPointProperty apProperty;

   LOG_DEBUG("Country Code update: %s ", update->getCountryCode().c_str());

   objPath = proxy->getDBusObjectPath();
   if (isAccessPoint(objPath)) {
      apProperty._regDom = update->getCountryCode();
      updateAccessPoint(objPath, apProperty, AP_REG_DOM);
   }
}

void IEEE80211APClient::notifyAPServiceAvailability(const bool serviceState)
{
   //AP Service Notification for Regulatory
   notifyRegChanStatus(serviceState);

   //AP Service Notification for LCM

   ILCMClient *lcmClientIf = LCMFactory::getInstance()->getLCMClientIf();
   if(lcmClientIf)
   {
     lcmClientIf->notifyAPServiceAvailability(serviceState);
   }
}

int IEEE80211APClient::changeRegulatory(const ::std::string &country)
{
   bool channel = false, sent = false;
   IWapdmgrAccessPointProxyIf *proxyIf;
   AccessPointList::const_iterator it;

   if (country.empty())
      return -EINVAL;

   LOG_INFO ("Changing the Regulatory information to: %s", country.c_str());

   for (it = _apList.begin(); it != _apList.end(); ++it) {
      if (!(*it).objPath.empty()) {
         channel = true;
         break;
      }
   }

   if (channel) {
      LOG_INFO ("Using the AP interface \"%s\" objpath: %s", (*it).property.interface.c_str(),
            (*it).objPath.c_str());
      proxyIf = DBusProxyFactory::getInstance()->getWapdmgrAccessPointProxyIf();
      if (proxyIf) {
         sent = proxyIf->sendCountryCodeSet(sWapdmgrBusName, (*it).objPath, DBUS_BUS_SYSTEM, *this, country);
         if (sent)
            return 0;
      }
   }

   return -ENOTCONN;
}

void IEEE80211APClient::wifiConnectionStatusChanged(const WifiMode mode,const uint32 healthValue)
{
   LOG_INFO("IEEE80211APClient::%s ", __func__);
   bool isValidAP = false;
   if(mode != WIFI_MODE_AP1)
      return;
   AccessPointList::const_iterator it;
   AccessPointProperty apProperty;
   for (it = _apList.begin(); it != _apList.end(); ++it)
   {
     if (it->property.isPowered)
     {
       isValidAP = true;
        break;
     } // if (objPath == curTechItem.objPath)
   } // for (TechnologyList::iterator it =...)
   LOG_DEBUG("IEEE80211APClient::%s AP:%s mode:%s healthvalue:%d ", __func__,
         BOOL_TO_STR(isValidAP), WIFI_MODE_TO_STR(mode),
         healthValue);
   if (isValidAP)
   {
      apProperty.connHealth = healthValue;
      updateAccessPoint(it->objPath, apProperty, AP_HEALTHINDICATOR);
   }
}

void IEEE80211APClient::onVendorServiceUpdate(const ::std::string &objectPath,const bool isRegistered)
{
   LOG_INFO("IEEE80211APClient::%s ", __func__);
   auto apListIt = std::find_if(_apList.begin(), _apList.end(),
            [&objectPath](AccessPointItem const &item)
            {   return item.objPath == objectPath;});
   if (_apList.end() != apListIt)
   {
      AccessPointChangedMsg msg;
      apListIt->property.isVendorElementRegistered = isRegistered;
      msg.setAccessPointItem(*apListIt);
      msg.setIEEE80211EventID(isRegistered ?
            IEEE80211_MSG_AP_MODE_AP_REG_VENDOR_IND :
            IEEE80211_MSG_AP_MODE_AP_UNREG_VENDOR_IND);
      notifyObservers(IEEE80211_EVENT_AP, &msg);
   }
}

}//namespace bosch
}//namespace org

/** @} */
