/**
 * @file IEEE80211APClient.h
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief 
 *
 * @{
 */

#ifndef _IEEE_80211_AP_CLIENT_H
#define _IEEE_80211_AP_CLIENT_H

#include <string>

#include "asf/core/Logger.h"

#include "IWapdmgrManagerProxyIf.h"
#include "IWapdmgrAccessPointProxyIf.h"
#include "IWapdmgrAssociatedStationsProxyIf.h"
#include "IEEE80211APClientIf.h"
#include "IBluezAdapterProxyIf.h"
#include "APModeClient_Types.h"
#include "WapdmgrDbusTypes.h"
#include "WBLDefines.h"
#include "DbusIfTypes.h"
#include "DbusParser.h"
#include "WapdmgrDbusInterfaces.h"
#include "WapdmgrDefines.h"
#include "WBLTypes.h"
#include "WBLTypeProperties.h"
#include "WBLHelper.h"
#include "DbusHelper.h"
#include "DBusProxyFactory.h"
#include "APModeClient_Msgs.h"
#include "DBManagerFactory.h"
#include "WifiConnectionStatus.h"
#include "APConfigurationManager.h"

namespace org
{
namespace bosch
{

typedef enum WBLVendorElementType
{
  WBL_VENDOR_ELEMENT_TYPE_FLAG = 0,
  WBL_VENDOR_ELEMENT_TYPE_VECHICLE_NAME,
  WBL_VENDOR_ELEMENT_TYPE_MANUFACTURER_NAME,
  WBL_VENDOR_ELEMENT_TYPE_MODEL_NAME,
  WBL_VENDOR_ELEMENT_TYPE_DEVICE_OUI,
  WBL_VENDOR_ELEMENT_TYPE_BLUETOOTH_MAC,
  WBL_VENDOR_ELEMENT_TYPE_DEVICE_MAC,
  WBL_VENDOR_ELEMENT_TYPE_UNKNOWN
}VendorElementType;

typedef struct WBLInformationElement
{
  uint8 elementID;
  uint8 length;
  ByteArray payload;
  VendorElementType type;
}InformationElement;

typedef struct WBLVendorElement
{
  uint8 elementID;
  uint8 length;
  ByteArray oui;
  uint8 subtype;
  InformationElement ie;
}VendorElement;

using namespace ::asf::core;


class IEEE80211APClient final: public IEEE80211APClientIf,
      public IWapdmgrManagerNotifCallbackIf,
      public IWapdmgrAccessPointNotifCallbackIf,
      public IWapdmgrAssociatedStationNotifCallbackIf,
      public IBluezAdapterNotifCallbackIf,
      public WapdmgrGetAccessPointsCbIf,
      public WapdmgrSetSSIDCbIf,
      public WapdmgrDeAuthenticateStationCbIf,
      public WapdmgrSetOperatingChannelsCbIf,
      public WapdmgrGetAssociatedStationsCbIf,
      public WapdmgrRegisterVendorServiceCbIf,
      public WapdmgrUnregisterVendorServiceCbIf,
      public cWifiConnectionStatusObserver
{
   public:

      IEEE80211APClient();

      virtual ~IEEE80211APClient();

      //! IEEE80211APClientIf interfaces
      virtual bool setPowered(const ::std::string& objPath, bool powered) override;
      virtual void getAccessPoints(AccessPointList& apList) const override;
      virtual void getAssociatedStations(AssociatedStationList& asList) const override;
      virtual bool setSSID(const ::std::string& objPath, const ::std::vector< uint8 >& ssid, bool utf_ssid) override;
      virtual bool setPassphrase(const ::std::string& objPath, const ::std::string& passphrase) override;
      virtual bool setPassphrase(const ::std::string& objPath, WBLAPType apType,WBLActionType actionType) override;
      virtual bool setHidden(const ::std::string& objPath, bool hidden) override;
      virtual bool setTethering(const ::std::string& objPath, bool tethering) override;
      virtual bool setSecurity(const ::std::string& objPath, const ::std::string& security) override;
      virtual bool deAuthenticateStation(const ::std::string& objPath, const ::std::string& macAddress) override;
      virtual bool setOperatingChannels(const ::std::string& objPath, const ::std::vector< uint16 >& channels) override;
      virtual bool restoreAPSettings(const ::std::string& objPath) override;
      virtual void setAPType(const ::std::string& objPath,WBLAPType userAPType) override;
      virtual bool setMaxStationsAllowed(const ::std::string& objPath, uint16 noOfStations) override;
      virtual bool updateRegisterVendorService(const ::std::string& objPath) override;
      virtual bool updateUnregisterVendorService(const ::std::string& objPath)override;
      virtual bool setNetworkType(const ::std::string& objPath, WifiAccessNetworkType& networkType)override;

      //! IWapdmgrManagerNotifCallbackIf interfaces
      virtual void onWapdmgrManagerServiceAvailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ServiceState previousState, const ServiceState currentState) override;
      virtual void onWapdmgrManagerServiceUnavailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ServiceState previousState, const ServiceState currentState) override;

      virtual void onAccessPointAddedError(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAccessPointAddedError >& error) override;
      virtual void onAccessPointAddedSignal(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAccessPointAddedSignal >& signal) override;
      virtual void onAccessPointRemovedError(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAccessPointRemovedError >& error) override;
      virtual void onAccessPointRemovedSignal(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAccessPointRemovedSignal >& signal) override;

      //! Wapdmgr Manager response callbacks
      virtual void onGetAccessPointsError(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrGetAccessPointsError >& error) override;
      virtual void onGetAccessPointsResponse(const ::boost::shared_ptr< WapdmgrManagerProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrGetAccessPointsResponse >& response) override;

      //! IWapdmgrAccessPointNotifCallbackIf interfaces
      virtual void onWapdmgrAccessPointServiceAvailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ServiceState previousState, const ServiceState currentState) override;
      virtual void onWapdmgrAccessPointServiceUnavailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ServiceState previousState, const ServiceState currentState) override;

      virtual void onAssociatedStationsChangedError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAssociatedStationsChangedError >& error) override;
      virtual void onAssociatedStationsChangedSignal(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAssociatedStationsChangedSignal >& signal) override;

      virtual void onMacAddressError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPMacAddressError >& error) override;
      virtual void onMacAddressUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPMacAddressUpdate >& update) override;

      virtual void onPowerStateError(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr < WapdmgrAPPowerStateError >& error) override;
      virtual void onPowerStateUpdate(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr < WapdmgrAPPowerStateUpdate >& update) override;

      virtual void onPowerFailureReasonError(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr < WapdmgrAPPowerFailureReasonError >& error) override;
      virtual void onPowerFailureReasonUpdate(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr < WapdmgrAPPowerFailureReasonUpdate >& update) override;

      virtual void onSSIDError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPSSIDError >& error) override;
      virtual void onSSIDUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPSSIDUpdate >& update) override;

      virtual void onInterfaceError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPInterfaceError >& error) override;
      virtual void onInterfaceUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPInterfaceUpdate >& update) override;

      virtual void onPoweredError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPPoweredError >& error) override;
      virtual void onPoweredUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPPoweredUpdate >& update) override;

      virtual void onHiddenError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPHiddenError >& error) override;
      virtual void onHiddenUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPHiddenUpdate >& error) override;

      virtual void onSecurityError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPSecurityError >& error) override;
      virtual void onSecurityUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPSecurityUpdate >& update) override;

      virtual void onPassphraseError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPPassphraseError >& error) override;
      virtual void onPassphraseUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPPassphraseUpdate >& update) override;

      virtual void onCurrentOperatingChannelError(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr< WapdmgrAPCurrChannelError >& error) override;
      virtual void onCurrentOperatingChannelUpdate(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr< WapdmgrAPCurrChannelUpdate >& update) override;

      virtual void onSupportedChannelsError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPSupportedChannelsError >& error) override;
      virtual void onSupportedChannelsUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPSupportedChannelsUpdate >& update) override;

      virtual void onMaximumStationsAllowedError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPMaxStationsAllowedError >& error) override;
      virtual void onMaximumStationsAllowedUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrAPMaxStationsAllowedUpdate >& update) override;

      virtual void onRegisterVendorServiceError(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr < WapdmgrRegisterVendorServiceError >& error) override;

      virtual void onRegisterVendorServiceResponse(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr < WapdmgrRegisterVendorServiceResponse >& response) override;

      virtual void onUnregisterVendorServiceError(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr < WapdmgrUnregisterVendorServiceError >& error) override;

      virtual void onUnregisterVendorServiceResponse(const ::boost::shared_ptr < WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr < WapdmgrUnregisterVendorServiceResponse >& response) override;

         virtual void onAssociationRequestRejectedError(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAssociationRequestRejectedError>& error) override
      {
         WBL_INTENTIONALLY_UNUSED(proxy);
         WBL_INTENTIONALLY_UNUSED(error);
      }
      virtual void onAssociationRequestRejectedSignal(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAssociationRequestRejectedSignal>& signal) override
      {
         WBL_INTENTIONALLY_UNUSED(proxy);
         WBL_INTENTIONALLY_UNUSED(signal);
      }
      virtual void onBlackListedStationsError(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAPBlackListedStationsError>& error) override
      {
         WBL_INTENTIONALLY_UNUSED(proxy);
         WBL_INTENTIONALLY_UNUSED(error);
      }
      virtual void onBlackListedStationsUpdate(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAPBlackListedStationsUpdate>& update) override
      {
         WBL_INTENTIONALLY_UNUSED(proxy);
         WBL_INTENTIONALLY_UNUSED(update);
      }
      virtual void onHWFeaturesError(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAPHWFeaturesError>& error) override
      {
         WBL_INTENTIONALLY_UNUSED(proxy);
         WBL_INTENTIONALLY_UNUSED(error);
      }
      virtual void onHWFeaturesUpdate(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAPHWFeaturesUpdate>& update) override
      {
         WBL_INTENTIONALLY_UNUSED(proxy);
         WBL_INTENTIONALLY_UNUSED(update);
      }

      virtual void onCountryCodeError(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAPCountryCodeError>& error) override;
      virtual void onCountryCodeUpdate(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAPCountryCodeUpdate>& update) override;


      //! Wapdmgr AccessPoint response callbacks
      virtual void onSetSSIDError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrSetSSIDError >& error) override;
      virtual void onSetSSIDResponse(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrSetSSIDResponse >& response) override;
      virtual void onDeAuthenticateStationError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrDeAuthenticateStationError >& error) override;
      virtual void onDeAuthenticateStationResponse(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrDeAuthenticateStationResponse >& response) override;
      virtual void onSetOperatingChannelsError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrSetOperatingChannelsError >& error) override;
      virtual void onSetOperatingChannelsResponse(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrSetOperatingChannelsResponse >& response) override;
      virtual void onGetAssociatedStationsError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrGetAssociatedStationsError >& error) override;
      virtual void onGetAssociatedStationsResponse(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrGetAssociatedStationsResponse >& response) override;
      virtual void onRestoreDefaultAPSettingsError(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
         const ::boost::shared_ptr<WapdmgrRestoreDefaultAPSettingsError>& error) override;
      virtual void onRestoreDefaultAPSettingsResponse(const ::boost::shared_ptr<WapdmgrAccesspointProxy>& proxy,
         const ::boost::shared_ptr<WapdmgrRestoreDefaultAPSettingsResponse>& response) override;
      virtual void onGetAllError(const ::boost::shared_ptr<WapdmgrAPPropertiesProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAPGetAllError>& error) override;
      virtual void onGetAllResponse(const ::boost::shared_ptr<WapdmgrAPPropertiesProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrAPGetAllResponse>& response) override;
      virtual void onIPv4PoolError(const ::boost::shared_ptr<WapdmgrDHCPDNSConfigurationProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrIPv4PoolError>& error) override;
      virtual void onIPv4PoolUpdate(const ::boost::shared_ptr<WapdmgrDHCPDNSConfigurationProxy>& proxy,
            const ::boost::shared_ptr<WapdmgrIPv4PoolUpdate>& update) override;
      virtual void onTetheringError(const ::boost::shared_ptr< WapdmgrTetheringSettingsProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrTetheringError >& error) override;
      virtual void onTetheringUpdate(const ::boost::shared_ptr< WapdmgrTetheringSettingsProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrTetheringUpdate >& update) override;
      virtual void onAccessNetworkTypeError(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr<WapdmgrAccessNetworkTypeError >& error) override;
      virtual void onAccessNetworkTypeUpdate(const ::boost::shared_ptr< WapdmgrAccesspointProxy >& proxy,
            const ::boost::shared_ptr<WapdmgrAccessNetworkTypeUpdate >& update) override;


      //! IWapdmgrAssociatedStationNotifCallbackIf interfaces
      virtual void onWapdmgrAssociatedStnServiceAvailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState) override;
      virtual void onWapdmgrAssociatedStnServiceUnavailable(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType,
            const ::asf::core::ServiceState previousState, const ::asf::core::ServiceState currentState) override;

      virtual void onMacAddressError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASMacAddressError >& error) override;
      virtual void onMacAddressUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASMacAddressUpdate >& update) override;
      virtual void onAccesspointError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASAccesspointError >& error) override;
      virtual void onAccesspointUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASAccesspointUpdate >& update) override;
      virtual void onInterfaceError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASInterfaceError >& error) override;
      virtual void onInterfaceUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASInterfaceUpdate >& update) override;
      virtual void onIPAddressError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASIPAddressError >& error) override;
      virtual void onIPAddressUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASIPAddressUpdate >& update) override;
      virtual void onHostnameError(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASHostnameError >& error) override;
      virtual void onHostnameUpdate(const ::boost::shared_ptr< WapdmgrAssociatedStationProxy >& proxy,
            const ::boost::shared_ptr< WapdmgrASHostnameUpdate >& update) override;

      //! IBluezAdapterNotifCallbackIf interfaces

      virtual void onBluezAdapterServiceAvailable(const ::std::string& busName, const ::std::string& objPath,
            const ::DBusBusType busType, const ::asf::core::ServiceState previousState,
            const ::asf::core::ServiceState currentState) override;
      virtual void onBluezAdapterServiceUnAvailable(const ::std::string& busName, const ::std::string& objPath,
            const ::DBusBusType busType, const ::asf::core::ServiceState previousState,
            const ::asf::core::ServiceState currentState) override;

      virtual void onAddressError(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
            const ::boost::shared_ptr < BluezAddressError >& error) override;
      virtual void onAddressUpdate(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
            const ::boost::shared_ptr < BluezAddressUpdate >& update) override;
    virtual void onAliasError(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
            const ::boost::shared_ptr < BluezAliasError >& error) override;
    virtual void onAliasUpdate(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
            const ::boost::shared_ptr < BluezAliasUpdate >& update) override;

      virtual void onAdapterInitializationError(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
            const ::boost::shared_ptr < BluezAdapterInitializationError >& error) override;
      virtual void onAdapterInitializationSignal(const ::boost::shared_ptr < BluezAdpaterProxy >& proxy,
            const ::boost::shared_ptr < BluezAdapterInitializationSignal >& signal) override;

      int changeRegulatory(const ::std::string &country) override;

      //cWifiConnectionStatusObserver updates the ConnectionStatus
      virtual void wifiConnectionStatusChanged(const WifiMode mode,const uint32 healthValue) override;

   protected:

   private:

      bool isAccessPoint(const ::std::string& objPath) const;
      bool getAccessPointInfo(const ::std::string& objPath, AccessPointItem& item);
      void addAccessPoint(const ::std::string& objPath, const AccessPointProperty& property);
      void updateAccessPoint(const ::std::string& objPath, const AccessPointProperty& property,
            const AccessPointPropertyName& changedPropertyName);
      void updateChangedAccessPointProperties(const ::std::string& objPath,const AccessPointProperty& property);
      void removeAccessPoint(const ::std::string& objPath);
      void printAccessPointInfo(const AccessPointProperty& property) const;

      bool isAssociatedStation(const ::std::string& objPath) const;
      bool getAssociatedStationInfo(const ::std::string& objPath, AssociatedStationItem& item);
      void addAssociatedStation(const ::std::string& objPath, const AssociatedStationProperty& property);
      void updateAssociatedStation(const ::std::string& objPath, const AssociatedStationProperty& property,
            const AssociatedStationPropertyName& changedPropertyName);
      void removeAssociatedStation(const ::std::string& objPath);
      void printAssociatedStationInfo(const AssociatedStationProperty& property) const;

      void addAccessPointProxy(const ::std::string& objPath);
      void removeAccessPointProxy(const ::std::string& objPath);

      void addAssociatedStnProxy(const ::std::string& objPath);
      void removeAssociatedStnProxy(const ::std::string& objPath);

      void parseAccessPointInfo(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType, const ::std::string& interfaceName,
            ::std::map< ::std::string, ::asf::dbus::DBusVariant >& data, AccessPointProperty& property) const;
      void parseAssociatedStationInfo(const ::std::string& busName,
            const ::std::string& objPath, const ::DBusBusType busType, const ::std::string& interfaceName,
            ::std::map< ::std::string, ::asf::dbus::DBusVariant >& data, AssociatedStationProperty& property) const;
      void parseSupportedChannels(::std::map< ::std::string, ::asf::dbus::DBusVariant >& data, SupportedChannelList& channelList) const;

      bool parseChannelNumber(const ::std::string& channelNumber, Channel& channel) const;

      void onAccessPointError(const ::std::string& apObjPath, IEEE80211EventID event,
            IEEE80211ErrorCode error = IEEE80211_ERROR_UNKNOWN); //TODO error code
      void onAssociatedStationError(const ::std::string& asObjPath, IEEE80211EventID event,
            IEEE80211ErrorCode error = IEEE80211_ERROR_UNKNOWN); //TODO error code

      void printVendorElements(const ByteArray vendorList);
      bool prepareAccesspointVendorElements(ByteArray &vendorList);
      void initializeVendorElement(VendorElement &vendorElement);
      void insertInfoElements(InformationElement &infoElement);
      void getVendorElement(InformationElement &element,ByteArray &list);
      void getHardwareLimitations(ByteArray &payload);
      void getVehicleDeviceName(ByteArray &payload);
      void getManufacturerName(ByteArray &payload);
      void getManufacturerModelName(ByteArray &payload);
      void getBluetoothMacAddress(::std::string &address);
      void getWifiDeviceAddress(::std::string &address);
      void convertAddresstoPayload(::std::string &address,ByteArray &payload);
      bool getAPObjectPath(::std::string &objPath);

      void notifyAPServiceAvailability(const bool serviceState);

      void parseIPv4PoolUpdate(::std::vector<::std::map < ::std::string, ::asf::dbus::DBusVariant> > ipv4Pool,
               IPv4ConfigurationList &list);
      void printIPV4Configuration(const IPv4ConfigurationList &ipv4ConfigList);

      void initLastModeSettings(::std::string objPath,const AccessPointProperty& property);
      void restoreLastModeSettings(const ::std::string& objPath,const AccessPointProperty& property);
      void restoreOnRestrictedAP(const ::std::string &objectPath,const APSetup &apSetup);
      void aliasUpdateToVendor();
      void updateLastModeSettings(const AccessPointProperty& property);
      void onVendorServiceUpdate(const ::std::string &objectPath,const bool isRegistered);

      AccessPointList         _apList;
      AssociatedStationList   _asList;

      bool _wapdman_ap_service_availability;
      bool _restoringAP;


      DECLARE_CLASS_LOGGER();
};

} // namespace bosch
} // namespace org

#endif //_IEEE_80211_AP_CLIENT_H

/** @} */
