/**
 * @file DBManager.cpp
 * @author RBEI/ECO32 Jiji Anna Jacob
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief Reads/writes persistent data for WBL .
 * Currently used to handle LastIntendedMode and
 * Mode based priorities.
 * @{
 */

#include "DBManager.h"
#include "DBDefines.h"
#include "WBLHelper.h"
#include "WBLTypeProperties.h"
#include <string>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Database", DBManager, Debug);

DBManager::DBManager()
{
   LOG_INFO("DBManager:: %s", __func__);
   /*To Remove Old files- LastIntendedMode/WBLPriorityManagement latter*/
   int ret = 0;
   bool failed = true;
   struct stat info;
   LIMkeyFile = NULL;
   ret = unlink (WBL_DB_PATH"/LastIntendedMode.txt");
   if (ret < 0) {
      LOG_ERROR ("DBManager:: %s Failed to delete the file %s/LastIntendedMode.txt "
              "error : %s",__func__,WBL_DB_PATH,strerror (errno));
   }
   ret = unlink (WBL_DB_PATH"/WBLPriorityManagement.txt");
   if (ret < 0) {
      LOG_ERROR ("DBManager:: %s Failed to delete the file %s/WBLPriorityManagement.txt "
              "error : %s",__func__,WBL_DB_PATH,strerror (errno));
   }
   poCryptoUtil = CryptoUtilFactory::getInstance()->getCryptoUtil();
    if (poCryptoUtil)
    {
        failed = poCryptoUtil->cryptoUtilRegister();
        LOG_INFO("DBManager:: %s SDC registration %s",
                __func__, failed ? "failed" : "success");
        IF_NOT_FAILED {
           LIMkeyFile = decryptAndLoadFile(WBL_LIM_FILE_PATH);
            if(!LIMkeyFile) {
                LOG_DEBUG("DBManager:: %s File %s is empty or corrupted",
                        __func__,WBL_LIM_FILE_PATH);
                ret = unlink(WBL_LIM_FILE_PATH);
                return;
            }
            failed = encryptAndSaveFile(WBL_LIM_FILE_PATH);
            IF_FAILED {
                LOG_ERROR ("DBManager:: %s Either already encrypted or Encryption "
                 "failed for : [%s]",__func__, WBL_LIM_FILE_PATH);
            }
        }
    }
} //DBManager::DBManager

DBManager::~DBManager()
{

    LOG_DEBUG("DBManager:: %s", __func__);
    bool failed = false;
    if (poCryptoUtil) {
        failed = poCryptoUtil->cryptoUtilUnregister();
        (void)failed;
    }
    clearLIMData();
    poCryptoUtil = NULL;
} //DBManager::~DBManager

GKeyFile* DBManager::loadConfFile(std::string filePath)
{
   LOG_INFO("DBManager:: %s %s" , __func__, filePath.c_str());
   return_val_if_fail (filePath.c_str(), NULL);
   GKeyFile *keyFile = NULL;
   GError *error = NULL;
   keyFile = g_key_file_new();
   if(keyFile)
   {
      if (!g_key_file_load_from_file(keyFile, filePath.c_str(),
              G_KEY_FILE_KEEP_COMMENTS, &error))
      {
         LOG_DEBUG("DBManager:: %s Error is %s", __func__,error->message);
         g_error_free(error);
         g_key_file_free(keyFile);
         keyFile = nullptr;
      }
   }
   else
   {
      LOG_ERROR ("DBManager:: %s loadConfFile:: Failed to create a new keyfile",
              __func__);
   }
   return keyFile;
} //GKeyFile* DBManager::loadConfFile

int
DBManager::destroyTempFile (const char *pathname)
{
    int ret = 0;
    return_val_if_fail (pathname, -EINVAL);
    LOG_DEBUG ("%s Removing file: %s", __func__,pathname);
    ret = unlink (pathname);
    if (ret < 0) {
        LOG_ERROR ("%s Failed to delete the file \"%s\", error : %s",
                __func__,pathname, strerror (errno));
        ret = -errno;
    }
    return ret;
}
void
DBManager::dbDumpFile (const char *file)
{
    LOG_INFO("DBManager::%s File - [%s]",__func__,file);
    FILE *fp = NULL;
    char line [512];
    return_if_fail (file);
    fp = fopen (file, "r");
    if (!fp) {
        LOG_ERROR ("Failed to open the file [%s] : %s", file, strerror (errno));
        return;
    }
    memset (line, 0, sizeof (line));
    LOG_DEBUG("DBManager::%s *****************START*****************",__func__);
    while (fgets (line, sizeof (line), fp)) {
        line [strlen (line) - 1] = '\0';
        LOG_DEBUG("DBManager::%s \t %s", __func__,line);
        memset (line, 0, sizeof (line));
    }
    LOG_DEBUG("DBManager::%s *****************END*******************",__func__);
    if (fclose (fp) < 0)
        LOG_ERROR ("Failed to close the file [%s] : %s", file, strerror (errno));
}
void DBManager::clearLIMData()
{
    LOG_DEBUG("DBManager::%s", __func__);
    if(LIMkeyFile) {
        g_key_file_free(LIMkeyFile);
        LIMkeyFile = NULL;
    }
}
bool DBManager::checkFileIsEncrypted(std::string filePath)
{
    LOG_INFO("DBManager::%s File [%s]", __func__,filePath.c_str());
    bool isEnc = false;
    GKeyFile *keyFile = NULL;
    keyFile = loadConfFile(filePath);
    if(keyFile) {
        gsize grpCount = 0;
        (void*)g_key_file_get_groups(keyFile, &grpCount);
        if ((grpCount <= 0)) {
            LOG_INFO("DBManager::%s Group count = [%d]", __func__,grpCount);
            isEnc = true;
        }
        g_key_file_free(keyFile);
        keyFile = nullptr;
    }
    else {
        isEnc = true;
    }
    return isEnc;
}
GKeyFile* DBManager::decryptAndLoadFile(std::string filePath)
{
    LOG_INFO("DBManager::%s - File - [%s]",__func__,filePath.c_str());
    bool failed = false;
    int ret = 0;
    GKeyFile *keyFile = NULL;

    bool isEnc = false;
    return_val_if_fail (filePath.c_str(), NULL);
    isEnc = checkFileIsEncrypted(filePath);
    if (isEnc) {
       if(poCryptoUtil)
       {
            char l_tempfile [PATH_MAX];
            memset (l_tempfile, 0, sizeof (l_tempfile));
            sprintf (l_tempfile, "%s/%s", WBL_TEMP_PTH, WBL_LIM_FILE);
            LOG_INFO("DBManager::%s - tempfile: %s", __func__, l_tempfile);

           LOG_INFO("DBManager::%s - Temp file created",__func__,l_tempfile);
           failed = poCryptoUtil->cryptoUtilDecrypt(filePath.c_str(),
                   l_tempfile);
           LOG_INFO("DBManager::%s - File [%s] decryption %s",
                   __func__,filePath.c_str(), failed ? "failed" : "success");
           IF_NOT_FAILED {
               ::std::string l_strTempfile(l_tempfile);
               keyFile = loadConfFile(l_strTempfile);
               dbDumpFile(l_tempfile);
           }
           (void)destroyTempFile(l_tempfile);
       }
    }
    else {
        LOG_INFO("DBManager::%s - %s file is not encrypted",
                __func__,filePath.c_str());
        keyFile = loadConfFile(filePath);
        dbDumpFile(filePath.c_str());
    }

    return keyFile;
}
bool DBManager::storeAndEncrypt(GKeyFile *keyFile, std::string sFilePath)
{
    LOG_INFO("DBManager::%s Filepath [%s]",__func__,sFilePath.c_str());
    bool failed = true;
    size_t length = 0;
    char* data = NULL;
    if(keyFile) {
        data = g_key_file_to_data(keyFile, &length, nullptr);
        if(poCryptoUtil && data)
        {
            LOG_INFO("DBManager::%s Encrypt data [%s] to file [%s]",
                    __func__,data, sFilePath.c_str());
            failed = poCryptoUtil->cryptoUtilEncrypt(sFilePath.c_str(),data,length);
            LOG_INFO("DBManager::%s - File [%s] Encryption %s",
                    __func__,sFilePath.c_str(), failed ? "failed" : "success");
        }
        if(data)
            g_free(data);
    }
    return failed;
}
bool
DBManager::encryptAndSaveFile(std::string sFilePath)
{
    LOG_INFO("DBManager::%s [%s]",__func__,sFilePath.c_str());
    bool failed = false,isEnc = false;
    GKeyFile *keyFile = NULL;
    return_val_if_fail(sFilePath.c_str(), !failed);
    isEnc = checkFileIsEncrypted(sFilePath/*, &keyFile*/);
    if (!isEnc) {
        keyFile = loadConfFile(sFilePath);
        if(keyFile) {
            failed = storeAndEncrypt(keyFile,sFilePath.c_str());
            LOG_INFO("DBManager::%s - File [%s] Encryption %s",
                    __func__,sFilePath.c_str(), failed ? "failed" : "success");
            g_key_file_free (keyFile);
            keyFile = NULL;
        }
    }
    return failed;
} //GKeyFile* DBManager::loadConfFile

::std::string DBManager::getRegulatoryInfo(const ::std::string &group)
{
   GKeyFile *keyFile;
   char *alpha = NULL;
   ::std::string country;

   if (group.empty())
      return country;

   keyFile = loadConfFile(WBL_REGULATORY_FILE_PATH);
   if (keyFile) {
      alpha = g_key_file_get_string(keyFile, group.c_str(), "Alpha2", nullptr);
      if (alpha) {
         country.assign(alpha);
         g_free(alpha);
      }
      g_key_file_free (keyFile);
   }

   return country;
}

int DBManager::setRegulatoryInfo(const ::std::string &country,
      ::std::string &group)
{
   GKeyFile *keyFile;
   bool success = false;
   GError *error = NULL;

   if (country.empty())
      return -ENODATA;

   if (group.empty())
      group = "Global";

   keyFile = loadConfFile(WBL_REGULATORY_FILE_PATH);
   if (!keyFile)
      keyFile = g_key_file_new();
   if (keyFile)
   {
      g_key_file_set_string(keyFile, group.c_str(), "Alpha2", country.c_str());
      success = g_key_file_save_to_file(keyFile, WBL_REGULATORY_FILE_PATH, &error);
      if (!success) {
         LOG_ERROR ("Failed to set the Regulatory information: %s", error->message);
         g_key_file_free (keyFile);
         g_error_free (error);
         return -EIO;
      }
   }
   else
   {
      LOG_ERROR ("Failed to create the KeyFile setRegulatoryInfo");
   }
   g_key_file_free (keyFile);
   return 0;
}

int DBManager::getRegulatoryWatchDogInfo(const ::std::string &group, const ::std::string &key,
      unsigned int *watchDogVal)
{
   int iRet = -ENOENT;
   GKeyFile *keyFile;
   GError *error = NULL;
   gboolean result = FALSE;

   if (group.empty() || key.empty() || !watchDogVal)
      return -EINVAL;

   keyFile = loadConfFile(WBL_REGULATORY_FILE_PATH);
   if (keyFile) {
      result = g_key_file_has_key(keyFile, group.c_str(), key.c_str(), &error);
      if (result) {
         *watchDogVal = (unsigned int) g_key_file_get_integer(keyFile, group.c_str(), key.c_str(), nullptr);
         iRet = 0;
      }
      else if (!result && error) {
         LOG_ERROR ("Failed to get the value for key \"%s\": %s/%d", key.c_str(),
               error->message, error->code);
         g_error_free (error);
      }
      g_key_file_free (keyFile);
   }

   return iRet;
}

int DBManager::setRegulatoryWatchDogInfo(::std::string &group, const ::std::string &key,
      unsigned int watchDogVal)
{
   GKeyFile *keyFile;
   bool success = false;
   GError *error = NULL;

   if (group.empty() || key.empty())
      return -EINVAL;

   keyFile = loadConfFile(WBL_REGULATORY_FILE_PATH);
   if (!keyFile)
      keyFile = g_key_file_new();
   if(keyFile)
   {
      g_key_file_set_integer(keyFile, group.c_str(), key.c_str(), (int) watchDogVal);
      success = g_key_file_save_to_file(keyFile, WBL_REGULATORY_FILE_PATH, &error);
      if (!success) {
         LOG_ERROR ("Failed to set the Regulatory information WatchDogInfo: %s", error->message);
         g_key_file_free (keyFile);
         g_error_free (error);
         return -EIO;
      }
   }
   else
   {
      LOG_ERROR ("Failed to create the KeyFile setRegulatoryWatchDogInfo");
   }
   g_key_file_free (keyFile);
   return 0;
}

int DBManager::removeKey(::std::string &group, const ::std::string &key)
{
   GKeyFile *keyFile;
   bool success = false;
   GError *error = nullptr;

   if (group.empty() || key.empty())
      return -EINVAL;

   LOG_DEBUG("Removing the key \"%s\" from the group \"%s\"",
         key.c_str(), group.c_str());

   keyFile = loadConfFile(WBL_REGULATORY_FILE_PATH);
   if (!keyFile)
      return 0;

   if (g_key_file_has_group(keyFile, group.c_str()) &&
         g_key_file_has_key(keyFile, group.c_str(), key.c_str(), nullptr)) {
      success = g_key_file_remove_key(keyFile, group.c_str(), key.c_str(), &error);
      if (!success) {
         LOG_ERROR ("Failed to remove the key \"%s\" from the group \"%s\": %s",
               key.c_str(), group.c_str(), error->message);
         g_key_file_free (keyFile);
         g_error_free (error);
         return -EIO;
      } else {
         error = nullptr;
         success = g_key_file_save_to_file(keyFile, WBL_REGULATORY_FILE_PATH, &error);
         if (!success) {
            LOG_ERROR ("Failed to set the Regulatory information removeKey: %s", error->message);
            g_key_file_free (keyFile);
            g_error_free (error);
            return -EIO;
         }
      }
   }

   g_key_file_free (keyFile);
   return 0;
}

int DBManager::getRegulatorySources(::std::string &group, const ::std::string &key,
      ::std::string &sources)
{
   int iRet = -ENOENT;
   GKeyFile *keyFile;
   char *src = nullptr;
   GError *error = NULL;
   gboolean result = FALSE;

   if (group.empty() || key.empty())
      return -EINVAL;

   keyFile = loadConfFile(WBL_REGULATORY_FILE_PATH);
   if (keyFile) {
      result = g_key_file_has_key(keyFile, group.c_str(), key.c_str(), &error);
      if (result) {
         src = g_key_file_get_string(keyFile, group.c_str(), key.c_str(), nullptr);
         if (src) {
            sources = src;
            g_free(src);
            iRet = 0;
         }
      }
      else if (!result && error) {
         LOG_ERROR ("Failed to get the value for key \"%s\": %s/%d", key.c_str(),
               error->message, error->code);
         g_error_free (error);
      }
      g_key_file_free (keyFile);
   }

   return iRet;

}

int DBManager::getHWLimitationInfo(const ::std::string &group, HWLimitationInfo &info)
{
   int iRet = -ENOENT;
   GKeyFile *keyFile;

   if (group.empty())
      return -EINVAL;

   keyFile = loadConfFile(WBL_HWLIMITATION_FILE_PATH);
   if (keyFile)
   {
      info.isCPWSupport = g_key_file_get_boolean(keyFile, group.c_str(), "CPWSupport",nullptr);
      info.is2_4GHz_Support = g_key_file_get_boolean(keyFile, group.c_str(), "Frequency_2_4Ghz",nullptr);
      info.is5_GHz_Support = g_key_file_get_boolean(keyFile, group.c_str(), "Frequency_5Ghz",nullptr);
      iRet = 0;
      g_key_file_free(keyFile);
   }
   return iRet;
}

bool DBManager::writeLIMToDB(const LIMItem& item)
{
    LOG_INFO("DBManager::%s ", __func__);
    bool failed = false;

    if (!LIMkeyFile)
        LIMkeyFile = g_key_file_new();
    if (LIMkeyFile)
   {
        LOG_INFO("DBManager::%s - Update LIMkeyFile ",__func__);
      ::std::string wifiMode = wblWifiMode.getWifiMode2String(item.property.mode);
      bool isPowered = item.property.poweredSetting;

      if ((item.property.mode == WIFI_MODE_AP1) || (item.property.mode == WIFI_MODE_AP2))
      {
         ::std::string frequency = wblFrequency.getFrequency2String(item.property.limAPConfig.frequency);
         ::std::string apType = wblAPType.getAPType2String(item.property.limAPConfig.apType);

            g_key_file_set_string(LIMkeyFile, "AP", "ObjectPath", item.objPath.c_str());
            g_key_file_set_string(LIMkeyFile, "AP", "Mode", wifiMode.c_str());
            g_key_file_set_boolean(LIMkeyFile, "AP", "PoweredSetting",isPowered);
            g_key_file_set_string(LIMkeyFile, "AP", "Frequency", frequency.c_str());
            g_key_file_set_string(LIMkeyFile, "AP", "Type", apType.c_str());
      }

      else if (item.property.mode == WIFI_MODE_STA)
      {
            g_key_file_set_string(LIMkeyFile, "STA", "ObjectPath", item.objPath.c_str());
            g_key_file_set_string(LIMkeyFile, "STA", "Mode", wifiMode.c_str());
            g_key_file_set_boolean(LIMkeyFile,"STA", "PoweredSetting",isPowered);
      }

        failed = storeAndEncrypt(LIMkeyFile,WBL_LIM_FILE_PATH);
        LOG_INFO("DBManager::%s - File [%s] Encryption %s",
                __func__,WBL_LIM_FILE_PATH, failed ? "failed" : "success");
      }

    LOG_INFO("DBManager::%s:isWrite %s", __func__,BOOL_TO_STR(!failed));
    return !failed;

} //bool DBManager::writeLIMToDB

bool DBManager::readLIMFromDB(LIMList& list)
{
   LOG_INFO("DBManager::%s ", __func__);
   bool ap = false, station = false;


   if (LIMkeyFile)
   {
      gsize grpCount = 0;
        char **groups = g_key_file_get_groups(LIMkeyFile, &grpCount);
      for (uint i = 0; i < grpCount; i++)
      {
         const ::std::string groupName = groups[i];
         if (groupName.compare("AP") == 0)
         {
            LIMItem apItem;
                ap = readAPLIMItem(LIMkeyFile,apItem) ? true:false;
            if(ap)
            {
               list.push_back(apItem);
            }
         }
         else if (groupName.compare("STA") == 0)
         {
            LIMItem staItem;
                station = readSTALIMItem(LIMkeyFile,staItem) ? true:false;
            if(station)
            {
               list.push_back(staItem);
            }
         }
      }
      if(groups)
      {
         g_strfreev(groups);
      }
   }
   LOG_INFO("DBManager::%s:isRead %s", __func__,BOOL_TO_STR(ap || station));
   return (ap || station);
} //bool DBManager::readLIMFromDB

bool DBManager::readAPLIMItem(GKeyFile* keyFile,LIMItem& item)const
{
   bool isRead = false;

   if(keyFile)
   {
      char* objPath = g_key_file_get_string(keyFile, "AP" ,"ObjectPath", nullptr);
      if(objPath)
      {
         isRead = true;
         char *mode = g_key_file_get_string(keyFile, "AP", "Mode", nullptr);
         bool poweredSetting = g_key_file_get_boolean(keyFile, "AP","PoweredSetting", nullptr);
         char *frequency = g_key_file_get_string(keyFile, "AP","Frequency", nullptr);
         char *apType = g_key_file_get_string(keyFile, "AP", "Type", nullptr);

         item.objPath.assign(objPath);
         g_free(objPath);

         if(mode)
         {
            item.property.mode = wblWifiMode.getWifiMode2Enum(mode);
            g_free(mode);
         }
         if(frequency)
         {
            item.property.limAPConfig.frequency = wblFrequency.getFrequency2Enum(frequency);
            g_free(frequency);
         }
         if(apType)
         {
            item.property.limAPConfig.apType = wblAPType.getAPType2Enum(apType);
            g_free(apType);
         }
         item.property.poweredSetting = poweredSetting;
      }
   }
   LOG_INFO("DBManager::%s:isRead %s", __func__,BOOL_TO_STR(isRead));
   return isRead;

}//bool DBManager::readAPLIMItem

bool DBManager::readSTALIMItem(GKeyFile* keyFile, LIMItem& item) const
{
   bool isRead = false;
   if (keyFile)
   {

      char* objPath = g_key_file_get_string(keyFile, "STA","ObjectPath", nullptr);
      if (objPath)
      {
         isRead = true;
         char *mode = g_key_file_get_string(keyFile, "STA", "Mode",nullptr);
         bool poweredSetting = g_key_file_get_boolean(keyFile, "STA","PoweredSetting", nullptr);

         item.objPath.assign(objPath);
         g_free(objPath);
         if(mode)
         {
            item.property.mode = wblWifiMode.getWifiMode2Enum(mode);
            g_free(mode);
         }
         item.property.poweredSetting = poweredSetting;
      }
   }
   LOG_INFO("DBManager::%s:isRead %s", __func__,BOOL_TO_STR(isRead));
   return isRead;

} //bool DBManager::readSTALIMItem

void DBManager::removeLIMItem(const WifiMode& mode)
{
   LOG_INFO("DBManager:: %s", __func__);
   GError *error = NULL;
   bool failed = false;

   if (LIMkeyFile)
   {
      if((mode == WIFI_MODE_AP1)||(mode == WIFI_MODE_AP2))
      {
         if(!g_key_file_remove_key(LIMkeyFile, "AP", "ObjectPath", &error))
         {
            LOG_DEBUG("Failed to write ObjectPath with error %s", error->message);
            g_error_free(error);
         }

         if(!g_key_file_remove_key(LIMkeyFile, "AP", "Mode", &error))
         {
            LOG_DEBUG("Failed to write Mode with error %s", error->message);
            g_error_free(error);
         }
         if(!g_key_file_remove_key(LIMkeyFile, "AP", "PoweredSetting", &error))
         {
            LOG_DEBUG("Failed to write PoweredSetting with error %s", error->message);
            g_error_free(error);

         }
         if(!g_key_file_remove_key(LIMkeyFile, "AP", "Frequency", &error))
         {
            LOG_DEBUG("Failed to write Frequency with error %s", error->message);
            g_error_free(error);
         }
         if(!g_key_file_remove_key(LIMkeyFile, "AP", "Type", &error))
         {
            LOG_DEBUG("Failed to write Type with error %s", error->message);
            g_error_free(error);
         }

      }
      else if (mode == WIFI_MODE_STA)
      {
         if(!g_key_file_remove_key(LIMkeyFile, "STA", "ObjectPath", &error))
         {
            LOG_DEBUG("Failed to write ObjectPath with error %s", error->message);
            g_error_free(error);
         }
         if(!g_key_file_remove_key(LIMkeyFile, "STA", "Mode", &error))
         {
            LOG_DEBUG("Failed to write Mode with error %s", error->message);
            g_error_free(error);
         }
         if(!g_key_file_remove_key(LIMkeyFile, "STA", "PoweredSetting", &error))
         {
            LOG_DEBUG("Failed to write PoweredSetting with error %s", error->message);
            g_error_free(error);
         }
      }

      failed = storeAndEncrypt(LIMkeyFile,WBL_LIM_FILE_PATH);
      LOG_INFO("DBManager::%s - File [%s] Encryption %s",
              __func__,WBL_LIM_FILE_PATH, failed ? "failed" : "success");
   }
} //void DBManager::removeLIMItem

bool DBManager::readPMFromDB(PriorityList& list,uint &timer)
{
   LOG_INFO("DBManager:: %s", __func__);
   GKeyFile *keyFile = NULL;
   gsize mlen;
   bool bRead = FALSE;
   keyFile = loadConfFile(WBL_PM_FILE_PATH);

   if (keyFile)
   {
      gsize grpCount = 0;
      char **groups = g_key_file_get_groups(keyFile, &grpCount);
      if (grpCount > 0)
      {
         for (uint i = 0; i < grpCount; i++)
         {
            std::string groupName = groups[i];
            LOG_DEBUG("DBManager::%s Group found %s", __func__,groupName.c_str());
            {
               if (groupName.compare("PRIORITIES") == 0)
               {
                  int *modes = g_key_file_get_integer_list(keyFile, "PRIORITIES", "MODE", &mlen, nullptr);
                  if(modes)
                  {
                     //Mode Based Priority
                     list.clear();
                     for (uint mcount = 0; mcount < mlen; mcount++)
                     {
                        LOG_INFO("DBManager::readPMFromDB mode:%d",modes[mcount]);
                        list.push_back((WifiMode)modes[mcount]);
                     }
                     g_free(modes);
                  }
               }
               else if (groupName.compare("CONFLICT_TIMER") == 0)
               {
                  timer = g_key_file_get_integer(keyFile, "CONFLICT_TIMER", "TIMER", nullptr);
                  LOG_INFO("DBManager::readPMFromDB timer:%d",timer);
               }
               else
               {
                  LOG_DEBUG("readPMFromDB::readPMFromDB Invalid Group Name");
               }
            }
         }
         bRead = TRUE;
      }
      if(groups)
      {
         g_strfreev(groups);
      }
      g_key_file_free (keyFile);
   }
   return bRead;
}

bool DBManager::writeModesToDB(const PriorityList& list)
{
   LOG_INFO("DBManager:: %s", __func__);
   GKeyFile *keyFile = NULL;
   GError * error = NULL;
   bool bWrite = FALSE;

   keyFile = loadConfFile(WBL_PM_FILE_PATH);

   if (!keyFile)
      keyFile = g_key_file_new();

   if (keyFile)
   {
      int mcount = static_cast<int>(list.size());
      if (mcount > 0 )
      {
         int modes[mcount];
         for (int i = 0; i < mcount; i++)
         {
            modes[i] = list.at(i);
            LOG_INFO("DBManager::writeModesToDB mode:%d",modes[i]);
         }
         g_key_file_set_integer_list(keyFile, "PRIORITIES", "MODE", modes,mcount);
      }
      else
      {
         LOG_DEBUG("DBManager::writeModesToDB  No Valid WBLPriority write operation");
      }
      gsize length = 0;
      gchar* data = g_key_file_to_data(keyFile, &length, nullptr);
      bWrite = g_file_set_contents(WBL_PM_FILE_PATH, data, length, &error);
      if (!bWrite)
      {
         LOG_DEBUG("DBManager::writeModesToDB Failed to write contents with error %s", error->message);
         g_error_free(error);
      }
      g_key_file_free (keyFile);
      if(data)
          g_free(data);
   }
   else
   {
      LOG_ERROR("DBManager::writeModesToDB  Failed to create KeyFile");
   }
   return bWrite;
}

bool DBManager::writeTimerToDB(const uint &timer)
{
   LOG_INFO("DBManager:: %s", __func__);
   GKeyFile *keyFile;
   bool success = false;
   GError *error = NULL;

   keyFile = loadConfFile(WBL_PM_FILE_PATH);

   if (!keyFile)
      keyFile = g_key_file_new();
   if(keyFile)
   {
      g_key_file_set_integer(keyFile, "CONFLICT_TIMER", "TIMER", timer);
      success = g_key_file_save_to_file(keyFile, WBL_PM_FILE_PATH, &error);
      if (!success)
      {
         LOG_ERROR("Failed to set the timer: %s", error->message);
         g_key_file_free(keyFile);
         g_error_free(error);
         return -EIO;
      }
   }
   else
   {
      LOG_ERROR("DBManager::writeTimer Failed to create KeyFile");
   }
   g_key_file_free(keyFile);
   return success;
}

bool DBManager::readPoweredSetting(const WifiMode mode, std::string& objPath)
{
   bool isPowered = false;
   ::std::string key = "";
   if ((WIFI_MODE_AP1 == mode) || (WIFI_MODE_AP2 == mode))
   {
      key = "AP";
   }
   else if (WIFI_MODE_STA == mode)
   {
      key = "STA";
   }

   if (!key.empty())
   {
      if (LIMkeyFile)
      {
         gsize grpCount = 0;
         char **groups = g_key_file_get_groups(LIMkeyFile, &grpCount);
         for (uint i = 0; i < grpCount; i++)
         {
            ::std::string groupName = groups[i];
            LOG_DEBUG("DBManager::%s Group found %s", __func__,groupName.c_str());

            if (groupName.compare(key) == 0)
            {
               isPowered = g_key_file_get_boolean(LIMkeyFile, key.c_str(),
                       "PoweredSetting", nullptr);
               char *objectPath = g_key_file_get_string(LIMkeyFile, key.c_str(),
                       "ObjectPath", nullptr);

               if (objectPath)
               {
                  objPath = objectPath;
                  g_free(objectPath);
                  break;
               }
            }
         }
         if (groups)
         {
            g_strfreev(groups);
         }
      }
   }
   LOG_DEBUG("DBManager::%s Power value for Mode: %s is: %s",
           __func__,WIFI_MODE_TO_STR(mode), BOOL_TO_STR(isPowered));
   return isPowered;
}

bool DBManager::writeAPConfigsToDB(const APSetup &configAPSetup,const bool isRestricted)
{
   LOG_INFO("DBManager:: %s", __func__);
   ::std::string group = "LAST_MODE_AP";
   ::std::string ssid = convertByteArray2String(configAPSetup.SSID);
    bool failed = false;


    if(!LIMkeyFile)
        LIMkeyFile = g_key_file_new();
    if(LIMkeyFile)
   {
        g_key_file_set_boolean(LIMkeyFile, group.c_str(),"Restricted",
                isRestricted);
        g_key_file_set_string(LIMkeyFile,group.c_str(),"SSID",ssid.c_str());
        g_key_file_set_string(LIMkeyFile, group.c_str(),"Passphrase",
                configAPSetup.passphrase.c_str());
        g_key_file_set_string(LIMkeyFile, group.c_str(), "Type",
                wblAPType.getAPType2String(configAPSetup.type).c_str());

        failed = storeAndEncrypt(LIMkeyFile,WBL_LIM_FILE_PATH);
        LOG_INFO("DBManager::%s - File [%s] Encryption %s",
              __func__,WBL_LIM_FILE_PATH, failed ? "failed" : "success");
      }
   else
   {
        LOG_ERROR ("DBManager::%s Failed to create KeyFile",__func__);
   }

    return !failed;
}

bool DBManager::readAPConfigsFromDB(APSetup &configAPSetup,bool &isRestricted)
{
   LOG_INFO("DBManager::%s", __func__);
   bool bRead = FALSE;
   char *strBuffer = nullptr;
   ::std::string ssid,passphrase,type;

   if (!LIMkeyFile)
      return bRead;

   gsize grpCount = 0;
   char **groups = g_key_file_get_groups(LIMkeyFile, &grpCount);
   if (grpCount <= 0)
      return bRead;

   for (uint i = 0; i < grpCount; i++)
   {
      std::string groupName = groups[i];
      if (groupName.compare("LAST_MODE_AP") == 0)
      {
         LOG_DEBUG("DBManager::readAPConfigsFromDB Reading from %s", groupName.c_str());

         isRestricted = g_key_file_get_boolean(LIMkeyFile, groupName.c_str(), "Restricted", NULL);
         strBuffer = g_key_file_get_string(LIMkeyFile, groupName.c_str(), "SSID", NULL);
         if (strBuffer)
         {
            ssid.assign(strBuffer);
            g_free(strBuffer);
         }
         strBuffer = g_key_file_get_string(LIMkeyFile, groupName.c_str(),"Passphrase", NULL);
         if (strBuffer)
         {
            passphrase.assign(strBuffer);
            g_free(strBuffer);
         }
         strBuffer = g_key_file_get_string(LIMkeyFile, groupName.c_str(), "Type", NULL);
         if (strBuffer)
         {
            type.assign(strBuffer);
            g_free(strBuffer);
         }
         LOG_DEBUG("Restricted :%s SSID:%s Passphrase:%s Type:%s",BOOL_TO_STR(isRestricted),ssid.c_str(),passphrase.c_str(),type.c_str());
         convertString2ByteArray(ssid,configAPSetup.SSID);
         configAPSetup.passphrase = passphrase;
         configAPSetup.type = wblAPType.getAPType2Enum(type);
         bRead = TRUE;
         break;
      }
   }
   if(groups)
      g_strfreev(groups);
   return bRead;
}

void DBManager::convertString2ByteArray(const ::std::string stringData, std::vector < uint8 >& byteArray)
{
   if (!stringData.empty())
   {
      for (uint i = 0; i < stringData.length(); ++i){
         byteArray.push_back(stringData[i]);
      }
   }
} //convertByteArray2String

::std::string DBManager::getAPTypePassphrase(const WBLAPType apType)
{
    LOG_INFO("DBManager::%s", __func__);
    ::std::string passphrase;

    if (LIMkeyFile)
    {
        gsize grpCount = 0;
        char **groups = g_key_file_get_groups(LIMkeyFile, &grpCount);
        if (grpCount > 0)
        {
            for (uint i = 0; i < grpCount; i++)
            {
                std::string groupName = groups[i];
                LOG_DEBUG("DBManager::%s Group found %s", __func__,groupName.c_str());
                if (groupName.compare("PP_MGMT")==0)
                {
                    char *passWd = g_key_file_get_string(LIMkeyFile,
                            groupName.c_str(),
                            wblAPType.getAPType2String(apType).c_str(), nullptr);
                    if (passWd)
                    {
                        passphrase = passWd;
                        g_free(passWd);
                    }
                    break;
                }
            }
        }
        if (groups)
            g_strfreev (groups);
    }
    LOG_DEBUG("DBManager::%s passphrase is %s", __func__,passphrase.c_str());
    return passphrase;
}
bool DBManager::setAPTypePassphrase(const WBLAPType apType,const ::std::string &passphrase)
{
    LOG_INFO("DBManager::%s", __func__);
    ::std::string group = "PP_MGMT";
    bool failed = false;

    if(!LIMkeyFile)
        LIMkeyFile = g_key_file_new();
    if(LIMkeyFile)
    {
        g_key_file_set_string(LIMkeyFile, group.c_str(),
                wblAPType.getAPType2String(apType).c_str(), passphrase.c_str());
        failed = storeAndEncrypt(LIMkeyFile,WBL_LIM_FILE_PATH);
        LOG_INFO("DBManager::%s - File [%s] Encryption %s",
                __func__,WBL_LIM_FILE_PATH, failed ? "failed" : "success");
    }
    return !failed;
}
} //namespace bosch
} //namespace org

/** @} */
