/**
 * @file StateMachineManager.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <boost/pointer_cast.hpp>

#include "WBLTypes.h"
#include "WBLDefines.h"
#include "WBLMessages.h"
#include "StateMachineProperties.h"
#include "IRespStateMachineManager.h"
#include "WifiSetupControllerSM.h"
#include "StateMachineManager.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", StateMachineManager, Info);

using boost::static_pointer_cast;

StateMachineManager::StateMachineManager():
      _respSMManagerIf(nullptr)
{
   LOG_INFO("StateMachineManager::Constructor entered");
   createStateMachines();
}

StateMachineManager::~StateMachineManager()
{
   try
   {
      LOG_INFO("StateMachineManager::Destructor entered");
      destroyStateMachines();
   }
   catch(...){}
}

void StateMachineManager::createStateMachines()
{
   LOG_INFO("StateMachineManager::createStateMachines() entered");

   ::boost::shared_ptr < WifiSetupControllerSM > staModeSM(new WifiSetupControllerSM(WIFI_MODE_STA, this));
   _stateMachinesMap[WIFI_MODE_STA] = staModeSM;
   ::boost::shared_ptr < WifiSetupControllerSM > apModeSM(new WifiSetupControllerSM(WIFI_MODE_AP1, this));
   _stateMachinesMap[WIFI_MODE_AP1] = apModeSM;
}

void StateMachineManager::destroyStateMachines()
{
   LOG_INFO("StateMachineManager::destroyStateMachines() entered");

   _stateMachinesMap.clear();
}

void StateMachineManager::registerResponseIf(IRespStateMachineManager* responseIf)
{
   _respSMManagerIf = responseIf;
   WBL_ASSERT(nullptr == _respSMManagerIf);
}

void StateMachineManager::processRequest(::boost::shared_ptr< WBLBaseMsg > msg)
{
   LOG_INFO("StateMachineManager::processRequest(WBLBaseMsg) entered");
   WBL_ASSERT_AND_EXIT(!msg);

   bool isMsgProcessed = false;
   switch (msg->getMsgID())
   {
      case WBL_MSG_PREPARE_SETUP_REQ:
      {
         ::boost::shared_ptr< PrepareSetupMsg > prepareSetupMsg =
               static_pointer_cast< PrepareSetupMsg >(msg);
         WifiMode requestedWifiMode = prepareSetupMsg->getWifiMode();
         if (isSMAvailable(requestedWifiMode))
         {
            isMsgProcessed = true;
            _stateMachinesMap[requestedWifiMode]->onRequest(prepareSetupMsg);
         }
      }
      break;
      case WBL_MSG_DEACTIVATE_SETUP_REQ:
      {
         ::boost::shared_ptr< DeActivateSetupMsg > deactSetupMsg =
               static_pointer_cast< DeActivateSetupMsg >(msg);
         WifiMode requestedWifiMode = deactSetupMsg->getWifiMode();
         if (isSMAvailable(requestedWifiMode))
         {
            isMsgProcessed = true;
            _stateMachinesMap[requestedWifiMode]->onRequest(deactSetupMsg);
         }
      }
      break;
      case WBL_MSG_INT_DISABLE_SETUP_REQ:
      {
         ::boost::shared_ptr< DisableSetupMsg > disableSetupMsg =
               static_pointer_cast< DisableSetupMsg >(msg);
         WifiMode requestedWifiMode = disableSetupMsg->getWifiMode();
         if (requestedWifiMode)
         {
            isMsgProcessed = true;
            _stateMachinesMap[requestedWifiMode]->onRequest(disableSetupMsg);
         }
      }
      break;
      case WBL_MSG_RESTORE_SETTINGS_REQ:
      {
         ::boost::shared_ptr< RestoreFactorySettingsMsg > resetSetupMsg =
               static_pointer_cast< RestoreFactorySettingsMsg >(msg);
         WifiMode requestedWifiMode = resetSetupMsg->getWifiMode();
         LOG_DEBUG("StateMachineManager::processRequest:%s",WIFI_MODE_TO_STR(requestedWifiMode));
         if(isSMAvailable(requestedWifiMode))
         {
            isMsgProcessed = true;
            _stateMachinesMap[requestedWifiMode]->onRequest(resetSetupMsg);
         }
      }
      break;
      default:
         LOG_ERROR("processRequest: Invalid message ID received: %d", msg->getMsgID());
      break;
   } // switch (msg->getMsgID())

   if (!isMsgProcessed)
   {
      LOG_DEBUG("processRequest: Rejecting message %d", msg->getMsgID());
      sendMsgResponse(msg, WBL_ERR_REJECTED);
   }
}

void StateMachineManager::processEvent(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("StateMachineManager::processEvent() entered: EventID = %s, isError = %d",
         SM_EVENT_TO_STR(eventID), isError);

   //! Send event to all SM instances
   for (auto itr = _stateMachinesMap.begin(); itr != _stateMachinesMap.end(); ++itr)
   {
      if (itr->second)
      {
         itr->second->onEvent(eventID, setup, isError);
      }
   }
}

void StateMachineManager::notifyActiveWiFiSetups(WifiSetUpList wifiSetups)
{
   _activeWifiSetups = wifiSetups;
}

bool StateMachineManager::isSMBusy(WifiMode wifiMode) const
{
   bool isBusy = true;
   if (isSMAvailable(wifiMode))
   {
      auto itr = _stateMachinesMap.find(wifiMode);
      isBusy = itr->second->isSMBusy();
   }
   return isBusy;
}

void StateMachineManager::sendMsgResponse(::boost::shared_ptr< WBLBaseMsg > msg, WBLErrorCode errorCode)
{
   LOG_INFO("StateMachineManager::sendMsgResponse() entered ");
   WBL_ASSERT_AND_EXIT(!((nullptr != _respSMManagerIf) && (msg)));

   LOG_DEBUG("sendMsgResponse: Sending response for EventID = %d, ErrorCode = %d ",
         msg->getMsgID(), errorCode);
   msg->setErrorCode(errorCode);
   _respSMManagerIf->sendMsgResponse(msg);
   //Note- Need to send DisableSetup response when it is requested externally
}

void StateMachineManager::sendRequest(::boost::shared_ptr< WBLBaseMsg > msg)
{
   LOG_INFO("StateMachineManager::sendRequest() entered ");
   WBL_ASSERT_AND_EXIT(!msg);

   switch (msg->getMsgID())
   {
      case WBL_MSG_DEACTIVATE_SETUP_REQ:
      {
         ::boost::shared_ptr< DeActivateSetupMsg > deactivateSetupMsg  = static_pointer_cast< DeActivateSetupMsg > (msg);
         WifiMode requestedWifiMode = deactivateSetupMsg->getWifiMode();
         if (isSMAvailable(requestedWifiMode))
         {
            _stateMachinesMap[requestedWifiMode]->onRequest(deactivateSetupMsg);
         }
      }
      break;
      default:
         LOG_ERROR("sendRequest: Request not supported");
      break;
   }
}

const WifiSetUpList& StateMachineManager::getAllWifiSetups() const
{
   return _activeWifiSetups;
}

bool StateMachineManager::isSMAvailable(WifiMode wifiMode) const
{
   auto itr = _stateMachinesMap.find(wifiMode);
   bool isAvailable = (_stateMachinesMap.end() != itr) && (itr->second);
   LOG_DEBUG("StateMachineManager::isSMAvailable: %s (WifiMode = %s)",
         BOOL_TO_STR(isAvailable), WIFI_MODE_TO_STR(wifiMode));
   return isAvailable;
}

} // namespace bosch
} // namespace org

/** @} */
