/**
 * @file WifiSetupState.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "WBLDefines.h"
#include "StateMachineProperties.h"
#include "IWifiSetupSM.h"
#include "WifiSetupState.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", WifiSetupState, Info);

WifiSetupState::WifiSetupState(WifiSetupStateType stateType, IWifiSetupSM* wifiSetupSMIf):
      _wifiSetupSMIf(wifiSetupSMIf),
      _wifiMode(WIFI_MODE_UNKNOWN),
      _stateType(stateType)
{
   WBL_ASSERT_AND_EXIT(nullptr == _wifiSetupSMIf);

   _wifiMode = _wifiSetupSMIf->getWifiMode();
   WBL_ASSERT(WIFI_MODE_UNKNOWN == _wifiMode);

   LOG_INFO("[WifiMode=%d] WifiSetupState::Constructor entered for State %s",
         _wifiMode, SM_STATE_TO_STR(_stateType));
}

WifiSetupState::~WifiSetupState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   /*LOG_INFO("[WifiMode=%d] WifiSetupState::Destructor entered for State %s",
         _wifiMode, SM_STATE_TO_STR(_stateType));*/
}

WifiSetupStateType WifiSetupState::getStateType() const
{
   return _stateType;
}

IEEE80211APClientIf* WifiSetupState::getAPClientIf() const
{
   IEEE80211ClientFactory* clientFactory = IEEE80211ClientFactory::getInstance();
   return ((clientFactory) ? (clientFactory->getAPClientIf()) : (nullptr));
}

IEEE80211STAClientIf* WifiSetupState::getSTAClientIf() const
{
   IEEE80211ClientFactory* clientFactory = IEEE80211ClientFactory::getInstance();
   return ((clientFactory) ? (clientFactory->getSTAClientIf()) : (nullptr));
}

void WifiSetupState::onEntry(::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupState::onEntry() for State %s: Not handled",
         _wifiMode, SM_STATE_TO_STR(_stateType));
   WBL_INTENTIONALLY_UNUSED(msg);
   //! Add code
}

void WifiSetupState::onRequest(::boost::shared_ptr< PrepareSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupState::onRequest(PrepareSetup) entered for State %s: "
         "Rejecting unhandled request", _wifiMode, SM_STATE_TO_STR(_stateType));
   WBL_ASSERT_AND_EXIT(!((nullptr != _wifiSetupSMIf) && (msg)));

   ::boost::shared_ptr< PrepareSetupMsg > curPrepareSetupMsg = _wifiSetupSMIf->getPrepareSetupMsg();
   bool isSameRequestOngoing = (ST_PREPARING_SETUP == _stateType) && (curPrepareSetupMsg) &&
         (curPrepareSetupMsg->getAPConfigItemParams() == msg->getAPConfigItemParams());

   (isSameRequestOngoing)?
         (_wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_INPROGRESS)) :
         (_wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_REJECTED)); //TODO - check error code
}

void WifiSetupState::onRequest(::boost::shared_ptr< DeActivateSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupState::onRequest(DeactivateSetup) entered for State %s: "
         "Rejecting unhandled request", _wifiMode, SM_STATE_TO_STR(_stateType));
   WBL_ASSERT_AND_EXIT(!((nullptr != _wifiSetupSMIf) && (msg)));

   (ST_DEACTIVATING_SETUP == _stateType) ?
         (_wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_INPROGRESS)) :
         (_wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_REJECTED)); //TODO - check error code
}

void WifiSetupState::onRequest(::boost::shared_ptr< DisableSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupState::onRequest(DisableSetup) entered for State %s: "
         "Rejecting unhandled request", _wifiMode, SM_STATE_TO_STR(_stateType));
   WBL_ASSERT_AND_EXIT(!((nullptr != _wifiSetupSMIf) && (msg)));

   (ST_DISABLING_SETUP == _stateType) ?
         (_wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_INPROGRESS)) :
         (_wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_REJECTED)); //TODO - check error code
}

void WifiSetupState::onRequest(::boost::shared_ptr< RestoreFactorySettingsMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupState::onRequest(FactoryResetMsg) entered for State %s: "
         "Rejecting unhandled request", _wifiMode, SM_STATE_TO_STR(_stateType));
   WBL_ASSERT_AND_EXIT(!((nullptr != _wifiSetupSMIf) && (msg)));

   (ST_RESETTING_SETUP == _stateType)?
         (_wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_INPROGRESS)) :
         (_wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_REJECTED)); //TODO - check error code
}

void WifiSetupState::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode=%d] WifiSetupState::onEvent() for State %s: Event = %s, isError = %d",
         _wifiMode, SM_STATE_TO_STR(_stateType), SM_EVENT_TO_STR(eventID), isError);

   if (IS_CURRENT_MODE(setup.property.mode))
   {
      switch (eventID)
      {
         case WIFI_SETUP_EVENT_STA_TECH_REMOVED:
         case WIFI_SETUP_EVENT_AP_REMOVED:
         {
            //! If interface is removed, switch to Error state.
            if (!isError)
            {
               switchToErrorState();
            }
         }
         break;
         default:
            LOG_DEBUG("onEvent: Event not handled");
         break;
      }
   } // if (IS_CURRENT_MODE(setup.property.mode))
}

void WifiSetupState::switchToErrorState()
{
   WBL_ASSERT_AND_EXIT(nullptr == _wifiSetupSMIf);

   SMEventID event;
   switch (_stateType)
   {
      case ST_ENABLING_SETUP:
      case ST_PREPARING_SETUP:
         event = SM_EVENT_PREPARE_SETUP_ERROR;
      break;
      case ST_DEACTIVATING_SETUP:
         event = SM_EVENT_DEACTIVATE_SETUP_ERROR;
      break;
      case ST_DISABLING_SETUP:
         event = SM_EVENT_DISABLE_SETUP_ERROR;
      break;
      case ST_RESETTING_SETUP:
         event = SM_EVENT_RESET_SETUP_ERROR;
      break;
      default:
         event = SM_EVENT_UNKNOWN;
      break;
   }

   if (SM_EVENT_UNKNOWN != event)
   {
      LOG_DEBUG("switchToErrorState: Switching to error state");
      ::boost::shared_ptr< StateMachineMsg> deactErrMsg(new StateMachineMsg);
      deactErrMsg->setEventID(event);
      deactErrMsg->setErrorCode(SM_ERR_FAILED);
      _wifiSetupSMIf->setState(ST_SETUP_ERROR, deactErrMsg);
   }
   else
   {
      LOG_ERROR("switchToErrorState: Error triggered for unknown event. State not changed.");
   }
}

} // namespace bosch
} // namespace org

/** @} */
