/**
 * @file SetupErrorState.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "WBLDefines.h"
#include "IWifiSetupSM.h"
#include "SetupErrorState.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", SetupErrorState, Info);

SetupErrorState::SetupErrorState(IWifiSetupSM* wifiSetupSMIf):
      WifiSetupState(ST_SETUP_ERROR, wifiSetupSMIf)
{
   LOG_INFO("[WifiMode=%d] SetupErrorState::Constructor entered", _wifiMode);
}

SetupErrorState::~SetupErrorState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   //LOG_INFO("[WifiMode=%d] SetupErrorState::Destructor entered", _wifiMode);
}

void SetupErrorState::onEntry(::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] SetupErrorState::onEntry()", _wifiMode);
   WBL_ASSERT_AND_EXIT(!(_wifiSetupSMIf && msg));

   switch(msg->getEventID())
   {
      case SM_EVENT_PREPARE_SETUP_ERROR:
      {
         LOG_DEBUG("onEntry: Sending PrepareSetup error");
         WBLErrorCode errorCode = convertToWBLErrorCode(msg->getErrorCode());
         _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getPrepareSetupMsg(), errorCode);
         _wifiSetupSMIf->setPrepareSetupMsg(nullptr);
      }
      break;
      case SM_EVENT_DEACTIVATE_SETUP_ERROR:
      {
         LOG_DEBUG("onEntry: Sending DeactivateSetup error");
         WBLErrorCode errorCode = convertToWBLErrorCode(msg->getErrorCode());
         if(WIFI_POWER_OFF_USER_REQUEST == _wifiSetupSMIf->getDeactivateSetupMsg()->getRequestType())
         {
            _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getDeactivateSetupMsg(), errorCode);
         }
         else
         {
            ILCMClient *lcmClientIf = LCMFactory::getInstance()->getLCMClientIf();
            if(lcmClientIf)
            {
              lcmClientIf->onDeactivateError(_wifiSetupSMIf->getDeactivateSetupMsg());
            }
         }
         _wifiSetupSMIf->setDeactivateSetupMsg(nullptr);
      }
      break;
      case SM_EVENT_DISABLE_SETUP_ERROR:
      {
         LOG_DEBUG("onEntry: Sending DisableSetup error");
         WBLErrorCode errorCode = convertToWBLErrorCode(msg->getErrorCode());
         _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getDisableSetupMsg(), errorCode);
         _wifiSetupSMIf->setDisableSetupMsg(nullptr);
      }
      break;
      case SM_EVENT_RESET_SETUP_ERROR:
      {
         LOG_DEBUG("onEntry: Sending ResetSetup error");
         WBLErrorCode errorCode = convertToWBLErrorCode(msg->getErrorCode());
         _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getRestoreFactorySettingsMsg(), errorCode);
         _wifiSetupSMIf->setRestoreFactorySettingsMsg(nullptr);
      }
      break;
      default:
         LOG_ERROR("onEntry: Event not handled - %d", msg->getEventID());
      break;
   }

   _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
}

WBLErrorCode SetupErrorState::convertToWBLErrorCode(SMErrorCode smErrorCode) const
{
   WBLErrorCode wblErrorCode;
   switch (smErrorCode)
   {
      case SM_ERR_NOERROR:
         wblErrorCode = WBL_ERR_NOERROR;
      break;
      case SM_ERR_PERM_DENIED:
         wblErrorCode = WBL_ERR_PERM_DENIED;
      break;
      case SM_ERR_NOTSUPPORTED:
         wblErrorCode = WBL_ERR_NOTSUPPORTED;
      break;
      case SM_ERR_REJECTED:
         wblErrorCode = WBL_ERR_REJECTED;
      break;
      case SM_ERR_INPROGRESS:
         wblErrorCode = WBL_ERR_INPROGRESS;
      break;
      case SM_ERR_CANCELLED:
      case SM_ERR_FAILED:
         wblErrorCode = WBL_ERR_CANCELLED;
      break;
      case SM_ERR_INVALID_ARGS:
         wblErrorCode = WBL_ERR_INVALID_ARGS;
      break;
      case SM_ERR_ALREADY_EXIST:
         wblErrorCode = WBL_ERR_ALREADY_EXIST;
      break;
      case SM_ERR_BUSY: //TODO - map to WBL error code
      case SM_ERR_UNKNOWN:
      default:
         wblErrorCode = WBL_ERR_UNKNOWN;
      break;
   }
   return wblErrorCode;
}

} // namespace bosch
} // namespace org

/** @} */
