/**
 * @file ResettingSetupState.cpp
 * @author RBEI/ECO32 Jiji Anna Jacob
 * @copyright (c) 2017 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <wifi_rootdaemon_client.h>
#include "ResettingSetupState.h"
#include "WBLDefines.h"
#include "DBDefines.h"
#include <unistd.h>
#include <sys/stat.h>
#include "StateMachineProperties.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", ResettingSetupState, Info);

//! Defines
#define HANDLE_EVENT_IN_SUBSTATE(SUBSTATE_TYPE, SUBSTATE_EVENT_HANDLER, DATA_EVENTID, DATA_WIFISETUP, DATA_ISERROR)    \
   case SUBSTATE_TYPE:                                            \
   {                                                              \
         SUBSTATE_EVENT_HANDLER(DATA_EVENTID, DATA_WIFISETUP, DATA_ISERROR);    \
         break;                                                   \
   }

#define SWITCH_TO_SUBSTATE(SUBSTATE_TYPE, SUBSTATE_ENTRY_HANDLER)    \
   case SUBSTATE_TYPE:                                         \
   {                                                           \
         SUBSTATE_ENTRY_HANDLER();                             \
         break;                                                \
   }


ResettingSetupState::ResettingSetupState(IWifiSetupSM* wifiSetupSMIf) :
      WifiSetupState(ST_RESETTING_SETUP, wifiSetupSMIf),
      _subState(RESET_SETUP_ST_IDLE)
{
   LOG_INFO("[WifiMode:%s]ResettingSetupState::%s", WIFI_MODE_TO_STR(_wifiMode), __func__);

}

ResettingSetupState::~ResettingSetupState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   // LOG_INFO("[WifiMode:%s]ResettingSetupState::%s", WIFI_MODE_TO_STR(_wifiMode), __func__);
}

void ResettingSetupState::onEntryIdle()
{
   LOG_INFO("[WifiMode:%s]ResettingSetupState::%s", WIFI_MODE_TO_STR(_wifiMode), __func__);
}

void ResettingSetupState::setSubState(const ResetSetupSubstate subState)
{
   _subState = subState;

   switch (_subState)
   {
      SWITCH_TO_SUBSTATE(RESET_SETUP_ST_IDLE, onEntryIdle);
      SWITCH_TO_SUBSTATE(RESET_SETUP_ST_POWERING_OFF, onEntryPoweringOff);
      SWITCH_TO_SUBSTATE(RESET_SETUP_ST_CLEAR_DATA, onEntryClearData);
      default:
         LOG_ERROR("setSubState: Invalid substate encountered: %d", _subState);
      break;
   }
}

ResetSetupSubstate ResettingSetupState::getSubState() const
{
   return _subState;
}

void ResettingSetupState::onEntry(::boost::shared_ptr<StateMachineMsg> msg)
{
   LOG_INFO("[WifiMode:%s]ResettingSetupState::%s",WIFI_MODE_TO_STR(_wifiMode),__func__);
   WBL_INTENTIONALLY_UNUSED(msg);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   _activeWifiSetup = _wifiSetupSMIf->getWifiSetupInfo();

   ::boost::shared_ptr< RestoreFactorySettingsMsg > resetMsg = _wifiSetupSMIf->getRestoreFactorySettingsMsg();
   if (RESET_SETUP_ST_IDLE != getSubState())
   {
      sendRestoreFactorySettingsResult(SM_ERR_BUSY);
   }
   else
   {
      setSubState(RESET_SETUP_ST_POWERING_OFF);
   }
}

void ResettingSetupState::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode:%s]ResettingSetupState::%s:EventID = %s, isError = %d", WIFI_MODE_TO_STR(_wifiMode), __func__,
         SM_EVENT_TO_STR(eventID), isError);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   if (IS_CURRENT_MODE(setup.property.mode))
   {
      _activeWifiSetup = _wifiSetupSMIf->getWifiSetupInfo();
   }
   //! Forward the event to current substate handler
   switch (getSubState())
   {
      HANDLE_EVENT_IN_SUBSTATE(RESET_SETUP_ST_POWERING_OFF, handlePoweringOffState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(RESET_SETUP_ST_CLEAR_DATA, handleClearDataState, eventID, setup, isError);
      case RESET_SETUP_ST_IDLE:
      default:
         LOG_ERROR("onEvent:Unhandled Event %d", getSubState());
      break;
   }
}

void ResettingSetupState::sendRestoreFactorySettingsResult(const SMErrorCode errCode)
{
   LOG_INFO("[WifiMode:%s] ResettingSetupState::%s: errorCode = %d",WIFI_MODE_TO_STR(_wifiMode),__func__, errCode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   if (errCode == SM_ERR_NOERROR)
   {
      _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getRestoreFactorySettingsMsg(), WBL_ERR_NOERROR);
      _wifiSetupSMIf->setRestoreFactorySettingsMsg(nullptr);
      _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
   }
   else
   {
      ::boost::shared_ptr<StateMachineMsg> prepareErrMsg(new StateMachineMsg);
      prepareErrMsg->setEventID(SM_EVENT_RESET_SETUP_ERROR);
      prepareErrMsg->setErrorCode(errCode);
      _wifiSetupSMIf->setState(ST_SETUP_ERROR, prepareErrMsg);
   }
   setSubState(RESET_SETUP_ST_IDLE);
}

void ResettingSetupState::onEntryPoweringOff()
{
   LOG_INFO("[WifiMode:%s]ResettingSetupState::%s", WIFI_MODE_TO_STR(_wifiMode), __func__);
   SMErrorCode error = SM_ERR_UNKNOWN;

   bool isPoweredOn = _activeWifiSetup.property.isPowered;

   LOG_DEBUG("onEntryPoweringOff:isPoweredOn %s", BOOL_TO_STR(isPoweredOn));

   //! Proceed with powering OFF interface if it is ON, else proceed to next state
   if (isPoweredOn)
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_STA:
         {
            IEEE80211STAClientIf* staClientIf = getSTAClientIf();
            bool isRequestSent = ((staClientIf)
                  && (staClientIf->setPowered(_activeWifiSetup.nativeObjPath, false)));
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         break;

         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            IEEE80211APClientIf* apClientIf = getAPClientIf();
            bool isRequestSent = ((apClientIf)
                  && (apClientIf->setPowered(_activeWifiSetup.nativeObjPath, false)));
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         break;

         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
      if (SM_ERR_NOERROR != error)
      {
         sendRestoreFactorySettingsResult(error);
      }
   }
   else
   {  //! Switch to next substate
      setSubState(RESET_SETUP_ST_CLEAR_DATA);
   }
}

void ResettingSetupState::onEntryClearData()
{
   LOG_INFO("[WifiMode:%s]ResettingSetupState::%s", WIFI_MODE_TO_STR(_wifiMode), __func__);
   SMErrorCode error = SM_ERR_UNKNOWN;

   IDBManager* dbmgrIf = DBManagerFactory::getInstance()->getDBManagerIf();
   if (dbmgrIf)
   {
       dbmgrIf->clearLIMData();
   }
   switch (_wifiMode)
   {
      case WIFI_MODE_STA:
      {
         bool isSuccess = clearConnmanContent();
         error = isSuccess ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
      }
      break;
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         IEEE80211APClientIf* apClientIf = getAPClientIf();
         bool isRequestSent = ((apClientIf) && (apClientIf->restoreAPSettings(_activeWifiSetup.nativeObjPath)));
         if(isRequestSent)
         {
            bool isDatabaseDeleted = clearDbFiles();
            error = (isDatabaseDeleted) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         else
         {
            error = SM_ERR_FAILED;
         }

      }
      break;
      default:
         error = SM_ERR_NOTSUPPORTED;
      break;
   }
   sendRestoreFactorySettingsResult(error);

}

void ResettingSetupState::handlePoweringOffState(const WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode:%s]ResettingSetupState::%s", WIFI_MODE_TO_STR(_wifiMode), __func__);
   switch (eventID)
   {
      case WIFI_SETUP_EVENT_STA_POWERED:
      case WIFI_SETUP_EVENT_AP_POWERED:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            bool isPoweredOff = (isError) ? (false) : (!setup.property.isPowered);
            (isPoweredOff) ? (setSubState(RESET_SETUP_ST_CLEAR_DATA)) : (sendRestoreFactorySettingsResult(SM_ERR_FAILED));
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handlePoweringOffState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

void ResettingSetupState::handleClearDataState(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode:%s]ResettingSetupState::%s", WIFI_MODE_TO_STR(_wifiMode), __func__);
   SMErrorCode error = SM_ERR_UNKNOWN;
   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_CLEAR_DATA:
      {
          //TO DO : Add check for WiFi mode , currently setup is empty
          error = isError ? (SM_ERR_FAILED) : (SM_ERR_NOERROR);
          sendRestoreFactorySettingsResult(error);
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleClearDataState:Event unhandled.Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

bool ResettingSetupState::clearConnmanContent()
{
   bool isSuccess = false;
   CmdData result;

   //NCG3D-133518: Required Parameters can be sent in arguments
   char *cmd = g_strdup(" ");
   result = execRootCommand (WIFI_CLIENT_NAME, DEL_CONNMAN_CONTENT, cmd);
   isSuccess = (ERR_NONE ==  result.errorNo) ? true:false;
   if(!isSuccess)
   {
      LOG_DEBUG("clearConnmanContent:%s",strerror(errno));
   }
   LOG_DEBUG("ResettingSetupState::clearConnmanContent:%s",BOOL_TO_STR(isSuccess));
   g_free(cmd);
   return isSuccess;
}

bool ResettingSetupState::clearDbFiles()
{
   bool isSuccess = false;
   int ret = 0;

       ret = unlink(WBL_LIM_FILE_PATH);
       if (ret < 0)
       {
          LOG_ERROR ("Failed to delete the file- %s due to Error: %s", WBL_LIM_FILE_PATH, strerror (errno));
          if (ENOENT != errno)
             return isSuccess;
       }

       ret = unlink(WBL_PM_FILE_PATH);
       if (ret < 0)
       {
          LOG_ERROR ("Failed to delete the file- %s due to Error: %s", WBL_PM_FILE_PATH, strerror (errno));
          if (ENOENT != errno)
             return isSuccess;
       }

       ret = unlink(WBL_REGULATORY_FILE_PATH);
       if (ret < 0)
       {
          LOG_ERROR ("Failed to delete the file- %s due to Error: %s", WBL_REGULATORY_FILE_PATH, strerror (errno));
          if (ENOENT != errno)
             return isSuccess;
       }

       isSuccess = true;
       return isSuccess;
    }

} //namespace bosch
} //namespace org

/** @} */

