/**
 * @file PreparingSetupState.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <algorithm>
#include <string>

#include "WBLDefines.h"
#include "StateMachineProperties.h"
#include "IWifiSetupSM.h"
#include "WBLASFComponent.h"
#include "PreparingSetupState.h"
#include "WBLUtils.h"
#include "DataProvider.h"
#include "KDSConfiguration.h"

namespace org
{
namespace bosch
{
#define CARPLAY_SSID "CarPlay_"//SSID Prefix for Carplay
#define AAW_SSID "AndroidAuto_"//SSID Prefix for AAW
#define MAC_STR_LEN     17
#define MAC_LAST_BYTE_LEN 4

#define DEFAULT_MAX_STATION_ALLOWED 7
#define DEFAULT_SPI_STATION_ALLOWED 10
#define DEFAULT_RESTRCITED_STATION_ALLOWED 1

#define RAND_SSID_TOKEN_START_RANGE 10
#define RAND_SSID_TOKEN_END_RANGE 999

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", PreparingSetupState, Info);

static APConfig defaultAPConfig;

static ChannelList carPlayChannels_2_4GHz = {1, 6, 11};
static ChannelList carPlayChannels_5GHz = {36, 40, 44, 48, 149, 153, 157, 161};

static const ::std::vector < PreparingSetupSubstate > apModeSubStatesSequence = {
      PREP_SETUP_ST_RESOLVING_LIMITATIONS,
      PREP_SETUP_ST_INITIATE_CONFIGURATION,
      PREP_SETUP_ST_CONFIGURING_SSID,
      PREP_SETUP_ST_CONFIGURING_PASSPHRASE,
      PREP_SETUP_ST_CONFIGURING_SECURITY,
      PREP_SETUP_ST_CONFIGURING_VISIBILITY,
      PREP_SETUP_ST_CONFIGURING_FREQUENCY,
      PREP_SETUP_ST_CONFIGURING_CHANNEL,
      PREP_SETUP_ST_CONFIGURING_TETHERING,
      PREP_SETUP_ST_CONFIGURING_NETWORKTYPE,
      PREP_SETUP_ST_CONFIGURING_MAXSTA,
      PREP_SETUP_ST_POWERING_ON,
      PREP_SETUP_ST_IDLE //! Should be last state
};

static const ::std::vector < PreparingSetupSubstate > staModeSubStatesSequence = {
      PREP_SETUP_ST_RESOLVING_LIMITATIONS,
      PREP_SETUP_ST_POWERING_ON,
      PREP_SETUP_ST_IDLE //! Should be last state
};

//! Defines
#define HANDLE_EVENT_IN_SUBSTATE(SUBSTATE_TYPE, SUBSTATE_EVENT_HANDLER, DATA_EVENTID, DATA_WIFISETUP, DATA_ISERROR)    \
   case SUBSTATE_TYPE:                                            \
   {                                                              \
         SUBSTATE_EVENT_HANDLER(DATA_EVENTID, DATA_WIFISETUP, DATA_ISERROR);    \
         break;                                                   \
   }

#define SWITCH_TO_SUBSTATE(SUBSTATE_TYPE, SUBSTATE_ENTRY_HANDLER)    \
   case SUBSTATE_TYPE:                                         \
   {                                                           \
         SUBSTATE_ENTRY_HANDLER();                             \
         break;                                                \
   }

#define CONVERT_FREQ_TO_FREQBAND(FREQ) ( ((FREQ) < WBL_FREQ_BAND_5_GHZ) ? (WBL_FREQ_BAND_2_4_GHZ) : (WBL_FREQ_BAND_5_GHZ) );


PreparingSetupState::PreparingSetupState(IWifiSetupSM* wifiSetupSMIf):
      WifiSetupState(ST_PREPARING_SETUP, wifiSetupSMIf),
      _subState(PREP_SETUP_ST_IDLE),
      _requestedAction(WIFI_ACTION_UNKNOWN),
      _isChannelChangeOngoing(false),
      _abortSetup(false)
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::Constructor entered", _wifiMode);
   /*To generate random numbers:1000 to 9999*/
}

PreparingSetupState::~PreparingSetupState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   //LOG_INFO("[WifiMode=%d] PreparingSetupState::Destructor entered", _wifiMode);
}

void PreparingSetupState::onEntry(::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntry()", _wifiMode);
   WBL_INTENTIONALLY_UNUSED(msg);

   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   ::boost::shared_ptr< PrepareSetupMsg > prepareSetupMsg = _wifiSetupSMIf->getPrepareSetupMsg();
   if (PREP_SETUP_ST_IDLE != getSubState())
   {
      sendPrepareSetupResult(SM_ERR_BUSY);
   }
   else if (!isRequestValid(prepareSetupMsg))
   {
      sendPrepareSetupResult(SM_ERR_NOTSUPPORTED);
   }
   else
   {
      const WifiSetUpItem& wifiSetupItem = _wifiSetupSMIf->getWifiSetupInfo();
      _activeWifiSetup = wifiSetupItem;
      _requestedAPConfig = prepareSetupMsg->getAPConfigItemParams();
      _requestedAction = prepareSetupMsg->getWifiAction();
      if(isCarplayRequest())
      {
          getUniqueSSID(CARPLAY_SSID);
      }
      else if(isAAWRequest())
      {
          getUniqueSSID(AAW_SSID);
      }
      setSubState(PREP_SETUP_ST_RESOLVING_LIMITATIONS);
   }
}

void PreparingSetupState::onRequest(::boost::shared_ptr< DeActivateSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onRequest(DeactivateSetup) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   if (PREP_SETUP_ST_IDLE == getSubState())
   {
      LOG_ERROR("onRequest: Request cannot be processed in Idle substate. Sending error. ");
      _wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_NOTSUPPORTED); //TODO - error code
      _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
   }
   else
   {
      _abortSetup = true;
      _wifiSetupSMIf->setDeactivateSetupMsg(msg);
   }
}

void PreparingSetupState::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEvent() entered: EventID = %s, isError = %d",
         _wifiMode, SM_EVENT_TO_STR(eventID), isError);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   if (IS_CURRENT_MODE(setup.property.mode))
   {
      _activeWifiSetup = _wifiSetupSMIf->getWifiSetupInfo();

      if (eventID == WIFI_SETUP_EVENT_AP_SETOPERATINGCHANNEL)
      {
         _isChannelChangeOngoing = false;
      }
   }

   //! Forward the event to current substate handler
   switch (getSubState())
   {
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_RESOLVING_LIMITATIONS, handleResolvingLimState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_INITIATE_CONFIGURATION, handleIntitateConfigurationState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_FREQUENCY, handleConfigFrequencyState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_CHANNEL, handleConfigChannelsState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_SSID, handleConfigSSIDState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_PASSPHRASE, handleConfigPassphraseState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_SECURITY, handleConfigSecurityState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_VISIBILITY, handleConfigVisibilityState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_TETHERING, handleConfigTetheringState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_NETWORKTYPE, handleConfigNetworkTypeState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_CONFIGURING_MAXSTA, handleConfigMaxStaState, eventID, setup, isError);
      HANDLE_EVENT_IN_SUBSTATE(PREP_SETUP_ST_POWERING_ON, handlePoweringOnState, eventID, setup, isError);
      case PREP_SETUP_ST_IDLE:
      default:
         LOG_ERROR("onEvent: Event unhandled in substate: %d", getSubState());
      break;
   }
}


PreparingSetupSubstate PreparingSetupState::getSubState() const
{
   return _subState;
}

void PreparingSetupState::setSubState(PreparingSetupSubstate subState)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::setSubState() entered: SubState = %s", _wifiMode, PREP_SUBSTATE_TO_STR(subState));

   //! Store new substate type
   _subState = subState;

   //! Trigger new substate handler
   switch (_subState)
   {
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_RESOLVING_LIMITATIONS, onEntryResolvingLimState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_INITIATE_CONFIGURATION, onEntryInitiateConfigs);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_FREQUENCY, onEntryConfigFrequencyState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_CHANNEL, onEntryConfigChannelsState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_SSID, onEntryConfigSSIDState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_PASSPHRASE, onEntryConfigPassphraseState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_SECURITY, onEntryConfigSecurityState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_VISIBILITY, onEntryConfigVisibilityState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_TETHERING, onEntryConfigTetheringState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_NETWORKTYPE, onEntryConfigNetworkTypeState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_CONFIGURING_MAXSTA, onEntryConfigMaxStaState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_POWERING_ON, onEntryPoweringOnState);
      SWITCH_TO_SUBSTATE(PREP_SETUP_ST_IDLE, onEntryIdleState);
      default:
         LOG_ERROR("setSubState: Invalid substate encountered: %d", _subState);
      break;
   }
}

bool PreparingSetupState::isSubstateEntryRequired(PreparingSetupSubstate subState)
{
   switch (subState)
   {
      case PREP_SETUP_ST_RESOLVING_LIMITATIONS:
         return true;
      case PREP_SETUP_ST_INITIATE_CONFIGURATION:
         return true;
      case PREP_SETUP_ST_CONFIGURING_FREQUENCY:
         return (isFrequencyChangeRequired());
      case PREP_SETUP_ST_CONFIGURING_CHANNEL:
         return (isChannelChangeRequired());
      case PREP_SETUP_ST_CONFIGURING_SSID:
         return (isSSIDChangeRequired());
      case PREP_SETUP_ST_CONFIGURING_PASSPHRASE:
         return (isPassphraseChangeRequired());
      case PREP_SETUP_ST_CONFIGURING_SECURITY:
         return (isSecurityChangeRequired());
      case PREP_SETUP_ST_CONFIGURING_VISIBILITY:
         return (isVisibilityChangeRequired());
      case PREP_SETUP_ST_CONFIGURING_TETHERING:
         return (isTetheringChangeRequired());
      case PREP_SETUP_ST_CONFIGURING_NETWORKTYPE:
         return (isNetworkTypeChangeRequired());
      case PREP_SETUP_ST_CONFIGURING_MAXSTA:
         return (isMaxStationChangeRequired());
      case PREP_SETUP_ST_POWERING_ON:
         return (isInterfacePoweredOff() && isActionActivate());
      case PREP_SETUP_ST_IDLE:
      default:
         return false;
   }
}

void PreparingSetupState::switchToNextSubState()
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::switchToNextSubState() entered", _wifiMode);

   //! If abort flag is set, abort setup preparation. Else proceed with setup.
   if (_abortSetup)
   {
      LOG_DEBUG("switchToNextSubState: Abort flag set. Hence canceling PrepareSetup");
      //_abortSetup = false;
      sendPrepareSetupResult(SM_ERR_CANCELLED); //TODO - error code
   }
   else
   {
      PreparingSetupSubstate curSubState = getSubState();
      const ::std::vector < PreparingSetupSubstate >& subStatesList =
            (IS_STA_MODE()) ? (staModeSubStatesSequence) : (apModeSubStatesSequence);
      if (curSubState == subStatesList.back())
      {
         LOG_ERROR("switchToNextSubState: substate %d unchanged", curSubState);
      }
      else
      {
         bool isSeqComplete = true;
         auto curSubStateItr = std::find(subStatesList.begin(), subStatesList.end(), curSubState);
         for (auto itr = curSubStateItr + 1; itr != subStatesList.end(); ++itr)
         {
            PreparingSetupSubstate subState = *itr;
            if (isSubstateEntryRequired(subState))
            {
               isSeqComplete = false;
               setSubState(subState);
               break;
            }
         }
         if (isSeqComplete)
         {
            LOG_DEBUG("switchToNextSubState: No more substates to execute. Sending PrepareSetup result");
            sendPrepareSetupResult(SM_ERR_NOERROR);
         }
      }
   }
}

void PreparingSetupState::onEntryIdleState()
{
   //! Clear stored data
   _requestedAPConfig = defaultAPConfig;
   _requestedAction = WIFI_ACTION_UNKNOWN;
   _isChannelChangeOngoing = false;
   _abortSetup = false;
}

void PreparingSetupState::onEntryResolvingLimState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryResolvingLimState() entered", _wifiMode);

   resolveLimitations();
}

void PreparingSetupState::onEntryInitiateConfigs()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryInitiateConfigs() entered", _wifiMode);
   SMErrorCode error = SM_ERR_UNKNOWN;
   bool isRequestSent  = false;
   if(IS_AP_MODE())
   {
      IEEE80211APClientIf* apClientIf = getAPClientIf();
      if(apClientIf)
      {
         apClientIf->setAPType(_activeWifiSetup.nativeObjPath,_requestedAPConfig.type);
         if(isCarplayRequest())
         {
           isRequestSent = apClientIf->updateRegisterVendorService(_activeWifiSetup.nativeObjPath);
         }
         else if(isAAWRequest())
         {
           isRequestSent = apClientIf->updateUnregisterVendorService(_activeWifiSetup.nativeObjPath);
         }
         else
         {
           //If Normal or AAW request unregister vendor service if registered.
           //If required vendor element info needs to persistent
           isRequestSent = apClientIf->updateUnregisterVendorService(_activeWifiSetup.nativeObjPath);
         }
         error = (isRequestSent)? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
      }
   }
   else
   {
      switchToNextSubState();
   }
   if (SM_ERR_NOERROR != error)
   {
     sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::onEntryConfigFrequencyState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigFrequencyState() entered", _wifiMode);

   SMErrorCode error = SM_ERR_UNKNOWN;

   //! Proceed with configuring frequency if the change is still required, else proceed to next state
   if (isFrequencyChangeRequired())
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            ChannelList channels;
            getRequestedChannels(channels);
            if(!isChannelUpdateRequired(channels))
            {
               error =  SM_ERR_NOERROR;
               switchToNextSubState();
            }
            else
            {
            IEEE80211APClientIf* apClientIf = getAPClientIf();

               _isChannelChangeOngoing = ((apClientIf) && (IS_LIST_VALID(channels)) &&
                                  (apClientIf->setOperatingChannels(_activeWifiSetup.nativeObjPath, channels)));
            error = (_isChannelChangeOngoing) ? (SM_ERR_NOERROR) : (SM_ERR_NOTSUPPORTED);
            }
         }
         break;
         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
   }
   else
   {
      error = SM_ERR_NOERROR;
      switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::onEntryConfigChannelsState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigChannelsState() entered", _wifiMode);

   SMErrorCode error = SM_ERR_UNKNOWN;

   //! Proceed with configuring channels if the change is still required, else proceed to next state
   if (isChannelChangeRequired())
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            //! Validate if all requested channels are in the requested/current frequency band
            ChannelList channels;
            getRequestedChannels(channels);
            if(!isChannelUpdateRequired(channels))
            {
               error =  SM_ERR_NOERROR;
               switchToNextSubState();
            }
            else
            {
            IEEE80211APClientIf* apClientIf = getAPClientIf();

            _isChannelChangeOngoing = ((apClientIf) && (IS_LIST_VALID(channels)) &&
                  (apClientIf->setOperatingChannels(_activeWifiSetup.nativeObjPath, channels)));
            error = (_isChannelChangeOngoing) ? (SM_ERR_NOERROR) : (SM_ERR_NOTSUPPORTED);
            }
         }
         break;
         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
   }
   else
   {
      error = SM_ERR_NOERROR;
      switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::onEntryConfigSSIDState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigSSIDState() entered", _wifiMode);

   SMErrorCode error = SM_ERR_UNKNOWN;

   //! Proceed with configuring SSID if the change is still required, else proceed to next state
   if (isSSIDChangeRequired())
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            IEEE80211APClientIf* apClientIf = getAPClientIf();
            ::std::vector< uint8 > ssid = _requestedAPConfig.ssid.SSID;
            bool isUTF8 = _requestedAPConfig.ssid.isUTF8;

            bool isRequestSent = ((apClientIf) && (IS_LIST_VALID(ssid)) &&
                 (apClientIf->setSSID(_activeWifiSetup.nativeObjPath, ssid, isUTF8)));
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         break;
         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
   }
   else
   {
      error = SM_ERR_NOERROR;
      switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::onEntryConfigPassphraseState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigPassphraseState() entered", _wifiMode);

   SMErrorCode error = SM_ERR_UNKNOWN;

   //! Proceed with configuring Passphrase if the change is still required, else proceed to next state
   if (isPassphraseChangeRequired())
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            bool isRequestSent = false;
            IEEE80211APClientIf* apClientIf = getAPClientIf();
            if(apClientIf)
            {
                if(_requestedAPConfig.isPassphraseValid)
                {
                    isRequestSent = ((IS_STRING_VALID(_requestedAPConfig.passphrase)) &&
                  (apClientIf->setPassphrase(_activeWifiSetup.nativeObjPath, _requestedAPConfig.passphrase)));
                }
                else
                {
                    isRequestSent = apClientIf->setPassphrase(_activeWifiSetup.nativeObjPath,_requestedAPConfig.type,_requestedAction);
                }
            }
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         break;
         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
   }
   else
   {
      error = SM_ERR_NOERROR;
      switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::onEntryConfigSecurityState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigSecurityState() entered", _wifiMode);

   SMErrorCode error = SM_ERR_UNKNOWN;

   //! Proceed with configuring Security if the change is still required, else proceed to next state
   if (isSecurityChangeRequired())
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            IEEE80211APClientIf* apClientIf = getAPClientIf();
            ::std::string security =
                  wblSecurity.getSecurity2String(_requestedAPConfig.security);

            bool isRequestSent = ((apClientIf) && (IS_STRING_VALID(security)) &&
                 (apClientIf->setSecurity(_activeWifiSetup.nativeObjPath, security)));
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         break;
         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
   }
   else
   {
      error = SM_ERR_NOERROR;
      switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::onEntryConfigVisibilityState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigVisibilityState() entered", _wifiMode);

   SMErrorCode error = SM_ERR_UNKNOWN;

   //! Proceed with configuring Visibility if the change is still required, else proceed to next state
   if (isVisibilityChangeRequired())
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            IEEE80211APClientIf* apClientIf = getAPClientIf();
            bool hidden = !(_requestedAPConfig.isVisible);

            bool isRequestSent = ((apClientIf) &&
                 (apClientIf->setHidden(_activeWifiSetup.nativeObjPath, hidden)));
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         break;
         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
   }
   else
   {
      error = SM_ERR_NOERROR;
      switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
}
void PreparingSetupState::onEntryConfigMaxStaState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigMaxStaState() entered", _wifiMode);

   SMErrorCode error = SM_ERR_UNKNOWN;
   uint16 noOfStations = 0;

   //! Proceed with configuring MaxStationAllowed if the change is still required, else proceed to next state
   if (isMaxStationChangeRequired())
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            if (_requestedAPConfig.type == WBL_AP_TYPE_CARPLAY||
                    _requestedAPConfig.type == WBL_AP_TYPE_AAW)
            {
               noOfStations = DEFAULT_SPI_STATION_ALLOWED;
            }
            else if(_requestedAPConfig.type == WBL_AP_TYPE_RESTRICTED)
            {
               noOfStations = DEFAULT_RESTRCITED_STATION_ALLOWED;
            }
            else
            {
               //For Normal
               noOfStations = DEFAULT_MAX_STATION_ALLOWED;
            }
            IEEE80211APClientIf* apClientIf = getAPClientIf();
            bool isRequestSent = ((apClientIf) &&
                 (apClientIf->setMaxStationsAllowed(_activeWifiSetup.nativeObjPath,noOfStations)));
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         break;
         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
   }
   else
   {
      error = SM_ERR_NOERROR;
      switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::onEntryConfigTetheringState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigTetheringState() entered", _wifiMode);
   SMErrorCode error = SM_ERR_UNKNOWN;

   //! Proceed with configuring Tethering if the change is still required, else proceed to next state
   if (isTetheringChangeRequired())
   {
     switch (_wifiMode)
     {
        case WIFI_MODE_AP1:
        case WIFI_MODE_AP2:
        {
           IEEE80211APClientIf* apClientIf = getAPClientIf();
           bool tethering = !(_activeWifiSetup.property.apSetup.isTethering);
           bool isRequestSent = ((NULL != apClientIf) &&
                (apClientIf->setTethering(_activeWifiSetup.nativeObjPath, tethering)));
           error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
        }
        break;
        default:
           error = SM_ERR_NOTSUPPORTED;
        break;
     }
   }
   else
   {
     error = SM_ERR_NOERROR;
     switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
     sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::onEntryConfigNetworkTypeState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryConfigNetworkTypeState() entered", _wifiMode);
   SMErrorCode error = SM_ERR_UNKNOWN;
   WifiAccessNetworkType networkType = WIFI_PRIVATE_NETWORK;
   //! Proceed with network type change is still required, else proceed to next state
   if (isNetworkTypeChangeRequired())
   {
     switch (_wifiMode)
     {
        case WIFI_MODE_AP1:
        case WIFI_MODE_AP2:
        {
           if (WBL_AP_TYPE_CARPLAY == _requestedAPConfig.type)
           {
               networkType = WIFI_PERSONAL_DEVICE_NETWORK;
           }
           IEEE80211APClientIf* apClientIf = getAPClientIf();
           bool isRequestSent = ((NULL != apClientIf) &&
                          (apClientIf->setNetworkType(_activeWifiSetup.nativeObjPath, networkType)));
           error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
        }
        break;
        default:
           error = SM_ERR_NOTSUPPORTED;
        break;
     }
   }
   else
   {
     error = SM_ERR_NOERROR;
     switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
     sendPrepareSetupResult(error);
   }
}
void PreparingSetupState::onEntryPoweringOnState()
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::onEntryPoweringOnState() entered", _wifiMode);

   SMErrorCode error = SM_ERR_UNKNOWN;

   //! Proceed with powering ON interface if it is OFF, else proceed to next state
   if (isInterfacePoweredOff())
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_STA:
         {
            IEEE80211STAClientIf* staClientIf = getSTAClientIf();

            bool isRequestSent = ((staClientIf) &&
                 (staClientIf->setPowered(_activeWifiSetup.nativeObjPath, true)));
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
         break;
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            IEEE80211APClientIf* apClientIf = getAPClientIf();

            bool isRequestSent = ((apClientIf) &&
                 (apClientIf->setPowered(_activeWifiSetup.nativeObjPath, true)));
            error = (isRequestSent) ? (SM_ERR_NOERROR) : (SM_ERR_FAILED);
         }
            break;
         default:
            error = SM_ERR_NOTSUPPORTED;
         break;
      }
   }
   else
   {
      error = SM_ERR_NOERROR;
      switchToNextSubState();
   }

   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
}

void PreparingSetupState::handleResolvingLimState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleResolvingLimState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));

   //! If operation has failed, cancel setup preparation.
   //! Else again check for next limitation until all limitations are resolved.
   switch (eventID)
   {
      case WIFI_SETUP_EVENT_STA_TECH_REMOVED:
      {
         if (IS_AP_MODE())
         {
            (isError) ? (sendPrepareSetupResult(SM_ERR_FAILED)) : (resolveLimitations());
         }
         else if (!isError)
         {
            onInterfaceRemoved(setup.property.mode);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_REMOVED:
      {
         if (IS_STA_MODE())
         {
            (isError) ? (sendPrepareSetupResult(SM_ERR_FAILED)) : (resolveLimitations());
         }
         else if (!isError)
         {
            onInterfaceRemoved(setup.property.mode);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_POWERED:
      {
         if ((IS_STA_MODE()) && (!setup.property.isPowered))
         {
            (isError) ? (sendPrepareSetupResult(SM_ERR_FAILED)) : (resolveLimitations());
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_DEAUTHSTATION:
      {
         bool isDeauthError = (IS_CURRENT_MODE(setup.property.mode)) && (isError) ;
         if (isDeauthError)
         {
            sendPrepareSetupResult(SM_ERR_FAILED);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AS_REMOVED:
      {
         if (IS_AP_MODE())
         {
            (isError) ? (sendPrepareSetupResult(SM_ERR_FAILED)) : (resolveLimitations());
         }
      }
      break;
      case WIFI_SETUP_EVENT_AS_ADDED:
      {
         //TODO - cancel setup if new station is added?
      }
      break;
      case WIFI_SETUP_EVENT_STA_POWERED:
      {
         if (!setup.property.isPowered)
         {
            resolveLimitations();
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleResolvingLimState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

void PreparingSetupState::handleConfigFrequencyState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigFrequencyState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_SETOPERATINGCHANNEL:
      {
         bool isChannelChangeError = (IS_CURRENT_MODE(setup.property.mode)) && (isError) ;
         LOG_DEBUG("handleConfigFrequencyState: SetOperatingChannel:%d",isChannelChangeError);
         (isChannelChangeError) ? (sendPrepareSetupResult(SM_ERR_FAILED)) : (switchToNextSubState());
      }
      break;
      case WIFI_SETUP_EVENT_AP_CURRENTCHANNEL:
      {
         bool isChannelChanged = (IS_CURRENT_MODE(setup.property.mode)) && (!_isChannelChangeOngoing);
         if (isChannelChanged)
         {
            ChannelList channels;
            getChannelsInRequestedFreq(_requestedAPConfig.frequency,channels);
            bool isCurrentChannelValid = (channels.end() !=
                  std::find(channels.begin(), channels.end(), setup.property.currChannel));
            bool isChnChangeSuccessful = (isError) ? (false) : (isCurrentChannelValid);
            (isChnChangeSuccessful) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      case WIFI_SETUP_EVENT_STA_TECH_ADDED:
      case WIFI_SETUP_EVENT_AP_ADDED:
      {
         //TODO - read HW limitations & cancel PrepareSetup if any conflict
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleConfigFrequencyState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

void PreparingSetupState::handleConfigChannelsState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigChannelsState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_SETOPERATINGCHANNEL:
      {
         bool isChannelChangeError = (IS_CURRENT_MODE(setup.property.mode)) && (isError) ;
         LOG_DEBUG("handleConfigChannelsState: SetOperatingChannel:%d",isChannelChangeError);
         (isChannelChangeError) ? (sendPrepareSetupResult(SM_ERR_FAILED)) : (switchToNextSubState());
      }
      break;
      case WIFI_SETUP_EVENT_AP_CURRENTCHANNEL:
      {
         bool isChannelChanged = (IS_CURRENT_MODE(setup.property.mode)) && (!_isChannelChangeOngoing);
         if (isChannelChanged)
         {
            ChannelList channels;
            getRequestedChannels(channels);
            bool isCurrentChannelValid = (channels.end() !=
                  std::find(channels.begin(), channels.end(), setup.property.currChannel));
            bool isChnChangeSuccessful = (isError) ? (false) : (isCurrentChannelValid);
            (isChnChangeSuccessful) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      case WIFI_SETUP_EVENT_STA_TECH_ADDED:
      case WIFI_SETUP_EVENT_AP_ADDED:
      {
         //TODO - read HW limitations & cancel PrepareSetup if any conflict
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleConfigChannelsState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

void PreparingSetupState::handleConfigSSIDState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigSSIDState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_SSID:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            bool isSSIDChangeSuccessful = (isError) ? (false) : (_requestedAPConfig.ssid.SSID
                  == setup.property.apSetup.SSID);
            (isSSIDChangeSuccessful) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      case WIFI_SETUP_EVENT_STA_TECH_ADDED:
      case WIFI_SETUP_EVENT_AP_ADDED:
      {
         //TODO - read HW limitations & cancel PrepareSetup if any conflict
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleConfigSSIDState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

void PreparingSetupState::handleConfigPassphraseState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigPassphraseState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_PASSPHRASE:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            (!isError) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      case WIFI_SETUP_EVENT_STA_TECH_ADDED:
      case WIFI_SETUP_EVENT_AP_ADDED:
      {
         //TODO - read HW limitations & cancel PrepareSetup if any conflict
      }
      break;
      case WIFI_SETUP_EVENT_AP_SSID:
      {
         if (isSSIDChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleConfigPassphraseState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

void PreparingSetupState::handleConfigSecurityState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigSecurityState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_SECURITY:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            bool isSecurityChangeSuccessful = (isError) ? (false) : (_requestedAPConfig.security
                  == setup.property.apSetup.security);
            (isSecurityChangeSuccessful) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      case WIFI_SETUP_EVENT_STA_TECH_ADDED:
      case WIFI_SETUP_EVENT_AP_ADDED:
      {
         //TODO - read HW limitations & cancel PrepareSetup if any conflict
      }
      break;
      case WIFI_SETUP_EVENT_AP_SSID:
      {
         if (isSSIDChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_PASSPHRASE:
      {
         if (isPassphraseChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleConfigSecurityState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

void PreparingSetupState::handleConfigVisibilityState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigVisibilityState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_HIDDEN:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            bool isVisibilityChangeSuccessful = (isError) ? (false) : (_requestedAPConfig.isVisible
                  == setup.property.apSetup.isVisible);
            (isVisibilityChangeSuccessful) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      case WIFI_SETUP_EVENT_STA_TECH_ADDED:
      case WIFI_SETUP_EVENT_AP_ADDED:
      {
         //TODO - read HW limitations & cancel PrepareSetup if any conflict
      }
      break;
      case WIFI_SETUP_EVENT_AP_SSID:
      {
         if (isSSIDChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_PASSPHRASE:
      {
         if (isPassphraseChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_SECURITY:
      {
         if (isSecurityChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleConfigVisibilityState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

void PreparingSetupState::handleConfigTetheringState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigTetheringState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_TETHERING:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            bool isTetheringChangeSuccessful = (isError) ? (false) : !(isTetheringChangeRequired());
            (isTetheringChangeSuccessful) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleConfigTetheringState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}


void PreparingSetupState::handleConfigNetworkTypeState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigNetworkTypeState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_NETWORK_TYPE:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            bool isNetworkChangeSuccessful = (isError) ? (false) : !(isNetworkTypeChangeRequired());
            (isNetworkChangeSuccessful) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleConfigNetworkTypeState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}
void PreparingSetupState::handleConfigMaxStaState(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleConfigMaxStaState() entered: EventID = %s",
            _wifiMode, SM_EVENT_TO_STR(eventID));
      WBL_ASSERT(!IS_AP_MODE());

      switch (eventID)
      {
         case WIFI_SETUP_EVENT_AP_MAXSTA:
         {
            if (IS_CURRENT_MODE(setup.property.mode))
            {
               bool isMaxStaSuccessful = (isError) ? (false) : !(isMaxStationChangeRequired());
               (isMaxStaSuccessful) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
            }
         }
         break;
         default:
         {
            //! Forward unhandled events to base class event handler
            LOG_DEBUG("handleConfigMaxStaState: Event unhandled. Forwarding to default event handler");
            WifiSetupState::onEvent(eventID, setup, isError);
         }
         break;
      }
}

void PreparingSetupState::handleIntitateConfigurationState(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handleIntitateConfigurationState() entered: EventID = %s",
            _wifiMode, SM_EVENT_TO_STR(eventID));
   WBL_ASSERT(!IS_AP_MODE());

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_AP_REG_VENDOR:
      case WIFI_SETUP_EVENT_AP_UNREG_VENDOR:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            if(!isError)
            {
               switchToNextSubState();
            }
            else
            {
               sendPrepareSetupResult(SM_ERR_FAILED);
            }
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handleIntitateConfigurationState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}
void PreparingSetupState::handlePoweringOnState(WifiSetupEventID eventID,
      const WifiSetUpItem& setup, bool isError)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::handlePoweringOnState() entered: EventID = %s",
         _wifiMode, SM_EVENT_TO_STR(eventID));

   switch (eventID)
   {
      case WIFI_SETUP_EVENT_STA_POWERED:
      case WIFI_SETUP_EVENT_AP_POWERED:
      {
         if (IS_CURRENT_MODE(setup.property.mode))
         {
            bool isPoweredOn = (isError) ? (false) : (setup.property.isPowered);
            (isPoweredOn) ? (switchToNextSubState()) : (sendPrepareSetupResult(SM_ERR_FAILED));
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_ADDED:
      case WIFI_SETUP_EVENT_STA_TECH_ADDED:
      {
         //TODO - read HW limitations & cancel PrepareSetup if any conflict
      }
      break;
      case WIFI_SETUP_EVENT_AP_SSID:
      {
         if (isSSIDChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_PASSPHRASE:
      {
         if (isPassphraseChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_SECURITY:
      {
         if (isSecurityChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      case WIFI_SETUP_EVENT_AP_HIDDEN:
      {
         if (isVisibilityChangeRequired())
         {
            sendPrepareSetupResult(SM_ERR_CANCELLED);
         }
      }
      break;
      default:
      {
         //! Forward unhandled events to base class event handler
         LOG_DEBUG("handlePoweringOnState: Event unhandled. Forwarding to default event handler");
         WifiSetupState::onEvent(eventID, setup, isError);
      }
      break;
   }
}

bool PreparingSetupState::isFrequencyChangeRequired() const
{
   bool isChangeRequired = false;
   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         isChangeRequired = (!isChannelChangeRequired()) && (_requestedAPConfig.isFrequencyValid);
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isFrequencyChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}

bool PreparingSetupState::isChannelChangeRequired() const
{
   bool isChangeRequired = false;

   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         if(!_requestedAPConfig.isFrequencyValid)
            isChangeRequired = true;
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isChannelChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}

bool PreparingSetupState::isChannelUpdateRequired(ChannelList& channels) const
{
   bool isChangeRequired = true;
   if((channels.size() == 1) && (_activeWifiSetup.property.isPowered)
                && (channels.front() == _activeWifiSetup.property.currChannel))
   {
      isChangeRequired = false;
   }
   LOG_INFO("PreparingSetupState::isChannelUpdateRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}
bool PreparingSetupState::isSSIDChangeRequired() const
{
   bool isChangeRequired = false;
   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         isChangeRequired = ((_requestedAPConfig.isSSIDValid) &&
               (_requestedAPConfig.ssid.SSID != _activeWifiSetup.property.apSetup.SSID));
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isSSIDChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}

bool PreparingSetupState::isPassphraseChangeRequired() const
{
    LOG_INFO("PreparingSetupState::%s",__func__);
   bool isChangeRequired = false;
   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
        LOG_DEBUG("PreparingSetupState::%s PassphraseValid:%s APType:%d Action:%d",
                __func__,
                BOOL_TO_STR(_requestedAPConfig.isPassphraseValid),_requestedAPConfig.type,_requestedAction);

         if((_requestedAPConfig.isPassphraseValid) ||
                 (WIFI_ACTION_RANDOMIZE == _requestedAction) ||
                 (WBL_AP_TYPE_AAW == _requestedAPConfig.type) ||
                 (WBL_AP_TYPE_CARPLAY == _requestedAPConfig.type) ||
                 (WBL_AP_TYPE_NORMAL == _requestedAPConfig.type))
             isChangeRequired = true;
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isPassphraseChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}

bool PreparingSetupState::isSecurityChangeRequired() const
{
   bool isChangeRequired = false;
   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         isChangeRequired = ((_requestedAPConfig.isSecurityValid) &&
               (_requestedAPConfig.security != _activeWifiSetup.property.apSetup.security));
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isSecurityChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}

bool PreparingSetupState::isMaxStationChangeRequired() const
{
   bool isChangeRequired = false;
   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         if (_requestedAPConfig.type == WBL_AP_TYPE_CARPLAY)
         {
            isChangeRequired = (_activeWifiSetup.property.apSetup.maxStations != DEFAULT_SPI_STATION_ALLOWED);
         }
         else if (_requestedAPConfig.type == WBL_AP_TYPE_RESTRICTED)
         {
            isChangeRequired = (_activeWifiSetup.property.apSetup.maxStations != DEFAULT_RESTRCITED_STATION_ALLOWED);
         }
         else
         {
            isChangeRequired = (DEFAULT_MAX_STATION_ALLOWED != _activeWifiSetup.property.apSetup.maxStations);
         }
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isMaxStationChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}


bool PreparingSetupState::isTetheringChangeRequired() const
{
   bool isChangeRequired = false;
   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         //Tethering Disabled for CPW/AAW/RESTRICTED AP
         //Tethering Enabled for Normal AP
         if(_requestedAPConfig.type == WBL_AP_TYPE_RESTRICTED ||
                 _requestedAPConfig.type == WBL_AP_TYPE_CARPLAY ||
                 _requestedAPConfig.type == WBL_AP_TYPE_AAW)
         {
            if(_activeWifiSetup.property.apSetup.isTethering)
               isChangeRequired = true;
         }
         else
         {
            if(!_activeWifiSetup.property.apSetup.isTethering)
               isChangeRequired = true;
         }
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isTetheringChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}

bool PreparingSetupState::isNetworkTypeChangeRequired() const
{
   bool isChangeRequired = false;
   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         if(_requestedAPConfig.type == WBL_AP_TYPE_CARPLAY)
         {
             if(WIFI_PERSONAL_DEVICE_NETWORK != _activeWifiSetup.property.apSetup.networkType)
               isChangeRequired = true;
         }
         else
         {
             if(WIFI_PRIVATE_NETWORK != _activeWifiSetup.property.apSetup.networkType)
               isChangeRequired = true;
         }
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isNetworkTypeChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}

bool PreparingSetupState::isVisibilityChangeRequired() const
{
   bool isChangeRequired = false;
   switch (_wifiMode)
   {
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         isChangeRequired = ((_requestedAPConfig.isVisibilityValid) &&
               (_requestedAPConfig.isVisible != _activeWifiSetup.property.apSetup.isVisible));
      }
      break;
      default:
         //! Add code
      break;
   }
   LOG_INFO("PreparingSetupState::isVisibilityChangeRequired: %s", BOOL_TO_STR(isChangeRequired));
   return isChangeRequired;
}

bool PreparingSetupState::isInterfacePoweredOff() const
{
   bool isPoweredOff = !(_activeWifiSetup.property.isPowered);
   LOG_INFO("PreparingSetupState::isInterfacePoweredOff: %s", BOOL_TO_STR(isPoweredOff));
   return isPoweredOff;
}

bool PreparingSetupState::isActionActivate() const
{
   bool isActivate = (WIFI_ACTION_ACTIVATE == _requestedAction) ? true:false;
   LOG_INFO("PreparingSetupState::isActionActivate:%s", BOOL_TO_STR(isActivate));
   return isActivate;
}

void PreparingSetupState::updatePoweredSetting(WifiSetUpItem& setup)
{
   LOG_DEBUG("[WifiMode=%d] PreparingSetupState::updatePoweredSetting() entered", setup.property.mode);
   //267335: [A-IVI][CMD]: WiFi activation in Partial ON
   if(setup.property.powerOffType == WIFI_POWER_OFF_WIFISTATE_REQUEST)
   {
      setup.property.powerOffType = WIFI_POWER_OFF_USER_REQUEST;
      WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
      ILastIntendedModeService* limServiceIf =
           (serviceFactory) ? (serviceFactory->getLastIntendedModeServiceIf()) : (nullptr);
      if (limServiceIf)
      {
        limServiceIf->onSetupUpdated(setup);
      }
   }
}

bool PreparingSetupState::isSTAPoweredON()
{
   bool bRet = false;
   WifiSetUpList wifiSetups =_wifiSetupSMIf->getAllWifiSetups();
   for(auto it = wifiSetups.begin(); it != wifiSetups.end(); ++it)
   {
      if(WIFI_MODE_STA == it->property.mode)
      {
         bRet = it->property.isPowered;
         updatePoweredSetting(*it);
      }
   }
   return bRet;
}
bool PreparingSetupState::isAPPoweredON()
{
   bool bRet = false;
   WifiSetUpList wifiSetups =_wifiSetupSMIf->getAllWifiSetups();
   for(auto it = wifiSetups.begin(); it != wifiSetups.end(); ++it)
   {
      if(WIFI_MODE_AP1 == it->property.mode)
      {
         bRet = it->property.isPowered;
         updatePoweredSetting(*it);
      }
   }
   return bRet;
}
bool PreparingSetupState::isStationSlotFull()
{
   bool bRet = false;
   uint16 numStationSlotsAvailable =  static_cast<uint16>(DEFAULT_MAX_STATION_ALLOWED
         - _activeWifiSetup.property.apSetup.stationList.size());
   if(_requestedAPConfig.stationsToReserve > 0)
      bRet = (_requestedAPConfig.isStationsToReserveValid) &&
            (numStationSlotsAvailable < _requestedAPConfig.stationsToReserve);

   return bRet;
}
bool PreparingSetupState::resolveByStationDeauthentication()
{
   bool bRet = false;
   //De-authenticate last connected station
   IEEE80211APClientIf* apClientIf = getAPClientIf();
   if(_activeWifiSetup.property.apSetup.stationList.empty())
      return bRet;
   const AssociatedStationList& asList = _activeWifiSetup.property.apSetup.stationList;
   ::std::string apObjPath = _activeWifiSetup.nativeObjPath;
   ::std::string asMacAddress = asList.back().property.macAddress;
   if(apClientIf)
   {
      bRet =  apClientIf->deAuthenticateStation(apObjPath, asMacAddress);
   }
   return bRet;
}
bool PreparingSetupState::isMultiRoleLimitation()
{
    bool bRet = false;
    if (IS_AP_MODE())
    {
        if (isSTAPoweredON() && _requestedAction == WIFI_ACTION_ACTIVATE)
        {
            resolveByDeactivation(WIFI_MODE_STA);
            bRet = true;
        }
    }
    else if (IS_STA_MODE())
    {
        if (isAPPoweredON())
        {
            resolveByDeactivation(WIFI_MODE_AP1);
            bRet = true;
        }
    }
    return bRet;
}
void PreparingSetupState::resolveByDeactivation(WifiMode mode)
{
    const ::boost::shared_ptr<DeActivateSetupMsg> deactivatesetup(new DeActivateSetupMsg);
    deactivatesetup->setWifiMode(mode);
    deactivatesetup->setMsgID(WBL_MSG_DEACTIVATE_SETUP_REQ);
    deactivatesetup->setRequestType(WIFI_POWER_OFF_USER_REQUEST);
    _wifiSetupSMIf->sendRequest(deactivatesetup);
}
void PreparingSetupState::resolveLimitations()
{
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   bool limitationsExist = false;
   SMErrorCode error = SM_ERR_NOERROR;
   if (isMultiRoleLimitation())
   {
      limitationsExist = true;
   }
   else if (IS_AP_MODE())
   {
      if (isStationSlotFull() && _requestedAPConfig.type == WBL_AP_TYPE_NORMAL)
      {
         LOG_DEBUG("resolveLimitations: Resolving Station slots limitation ");
         error = resolveByStationDeauthentication()?(SM_ERR_NOERROR):(SM_ERR_FAILED);
         limitationsExist = true;
      }
   }
   //! On any error, cancel PrepareSetup.
   if (SM_ERR_NOERROR != error)
   {
      sendPrepareSetupResult(error);
   }
   //! If no limitations exist, switch to next substate.
   else if (!limitationsExist)
   {
      switchToNextSubState();
   }
}

bool PreparingSetupState::isRequestValid(::boost::shared_ptr< PrepareSetupMsg > msg) const
{
   //TODO - move logic into service class?
   bool isValid = false;
   if (msg)
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
             if(_requestedAPConfig.type == WBL_AP_TYPE_CARPLAY)
             {
                 isValid = cKDSConfiguration::getInstance()->isCPWSupport();
             }
             else //Normal || Restrictive
             {
                 isValid = true;
             }
         }
         break;
         case WIFI_MODE_STA:
            isValid = true;
            break;
         default:
            LOG_ERROR("isRequestValid: PrepareSetup request not supported for mode %d", _wifiMode);
         break;
      }
   }
   return isValid;
}


      //! Requested channel list is valid if:
      //! (1) list only contains channels of the new frequency band of the AP, and
      //! (2) atleast one channel in that list is a supported channel in that band




int PreparingSetupState::isValidChannel(uint16 channel) const
{
    const SupportedChannelList& supportedChannels =_activeWifiSetup.property.apSetup.chanList;
    for (auto itr = supportedChannels.begin(); itr != supportedChannels.end();++itr)
    {
        if (channel == itr->first)
            return 0;
    }
    return -1;
}

int PreparingSetupState::isCPWCertifiedChannel(uint16 channel,WBLFrequencyBand& requestedFrequency) const
{
   if(WBL_FREQ_BAND_5_GHZ == requestedFrequency)
   {
      for (auto itr = carPlayChannels_5GHz.begin(); itr != carPlayChannels_5GHz.end();++itr)
      {
         if (channel == *itr)
            return 0;
      }
   }
   else
   {
      for (auto itr = carPlayChannels_2_4GHz.begin(); itr != carPlayChannels_2_4GHz.end();++itr)
      {
         if (channel == *itr)
            return 0;
      }
   }
   return -1;
}
void PreparingSetupState::getCarPlayChannels(ChannelList& channels,WBLFrequencyBand& requestedFrequency) const
{
   LOG_DEBUG("PreparingSetupState::%s",__func__);
   uint16_t CPWChannel=cKDSConfiguration::getInstance()->getCarplayChannelsFromConfiguration(requestedFrequency);
   if (WBL_FREQ_BAND_5_GHZ == requestedFrequency)
   {
      if((0 == isCPWCertifiedChannel(CPWChannel,requestedFrequency))
            &&( 0 == isValidChannel(CPWChannel)))
      {
         channels.push_back(CPWChannel);
         LOG_DEBUG("PreparingSetupState::%s: ChannelList element = %d",__func__,CPWChannel);
      }
   }
   else
   {
      if((0 == isCPWCertifiedChannel(CPWChannel,requestedFrequency))
            && (0 == isValidChannel(CPWChannel)))
      {
         channels.push_back(CPWChannel);
         LOG_DEBUG("PreparingSetupState::%s: ChannelList element = %d",__func__,CPWChannel);
      }
   }
   LOG_DEBUG("PreparingSetupState::%s: ChannelList size = %d",__func__,channels.size());

}

void PreparingSetupState::getRequestedChannels(ChannelList& channels) const
{
   LOG_DEBUG("PreparingSetupState::%s",__func__);
   WBLFrequencyBand frequency = WBL_FREQ_BAND_UNKNOWN;
   if(_requestedAPConfig.isFrequencyValid)
   {
       frequency = _requestedAPConfig.frequency;
   }
   else
   {
       if (_activeWifiSetup.property.apSetup.supportedFrequencies.find(WBL_FREQ_BAND_5_GHZ)
                != _activeWifiSetup.property.apSetup.supportedFrequencies.end())
       {
           frequency = WBL_FREQ_BAND_5_GHZ;
       }
       else
       {
           frequency = WBL_FREQ_BAND_2_4_GHZ;
       }
   }
   if(WBL_AP_TYPE_CARPLAY == _requestedAPConfig.type)
   {
      getCarPlayChannels(channels,frequency);
   }
   else // For Normal AP || Restricted AP
   {
      getChannelsInRequestedFreq(frequency,channels);
   }
}

void PreparingSetupState::getChannelsInRequestedFreq(WBLFrequencyBand reqFrequency,ChannelList& channels) const
{
   const SupportedChannelList& supportedChannels = _activeWifiSetup.property.apSetup.chanList;

   for (auto itr = supportedChannels.begin(); itr != supportedChannels.end(); ++itr)
   {
      WBLFrequencyBand frequency = CONVERT_FREQ_TO_FREQBAND(itr->second);
      if (reqFrequency == frequency)
      {
         channels.push_back(itr->first);
      }
   }
   LOG_DEBUG("PreparingSetupState::getChannelsInRequestedFreq: ChannelList size = %d", channels.size());
}

void PreparingSetupState::onInterfaceRemoved(WifiMode removedWifiMode)
{
   if (_wifiMode == removedWifiMode)
   {
      sendPrepareSetupResult(SM_ERR_CANCELLED);
   }
}

void PreparingSetupState::sendPrepareSetupResult(SMErrorCode errorCode)
{
   LOG_INFO("[WifiMode=%d] PreparingSetupState::sendPrepareSetupResult() entered: errorCode = %d",
         _wifiMode, errorCode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   if (errorCode == SM_ERR_NOERROR)
   {
      _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getPrepareSetupMsg(), WBL_ERR_NOERROR);
      _wifiSetupSMIf->setPrepareSetupMsg(nullptr);
      _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
   }
   else
   {
      ::boost::shared_ptr< StateMachineMsg > prepareErrMsg(new StateMachineMsg);
      prepareErrMsg->setEventID(SM_EVENT_PREPARE_SETUP_ERROR);
      prepareErrMsg->setErrorCode(errorCode);
      _wifiSetupSMIf->setState(ST_SETUP_ERROR, prepareErrMsg);
   }

   setSubState(PREP_SETUP_ST_IDLE);
}

bool PreparingSetupState::isCarplayRequest()
{
   bool isCarPlayAP = (_requestedAPConfig.isTypeValid) && (WBL_AP_TYPE_CARPLAY == _requestedAPConfig.type);
   return isCarPlayAP;
}

bool PreparingSetupState::isAAWRequest()
{
   bool isAAWAP = (_requestedAPConfig.isTypeValid) && (WBL_AP_TYPE_AAW == _requestedAPConfig.type);
   return isAAWAP;
}
void PreparingSetupState::getUniqueSSID(const ::std::string ssid)
{
   LOG_INFO("PreparingSetupState::%s SSID: %s",__func__,ssid.c_str());

   /*Unique SSID generated by appending last two bytes of AP Mac with AP type*/
   ::std::string uniqueSSID = ssid;
   ::std::string macAddress = _activeWifiSetup.property.apSetup.macAddress;

   if(!macAddress.empty() && MAC_STR_LEN == macAddress.size())
   {
      macAddress.replace(macAddress.size()-3,1,"");
      uniqueSSID.append(macAddress.substr(macAddress.size()-MAC_LAST_BYTE_LEN));
   }
   else
   {
      int randNo = cUtils::getRandomNo(RAND_SSID_TOKEN_START_RANGE,RAND_SSID_TOKEN_END_RANGE);
      uniqueSSID.append(std::to_string(randNo));
   }
   LOG_INFO("PreparingSetupState::%s Unique SSID: %s",__func__,uniqueSSID.c_str());
   ByteArray byteSSID(uniqueSSID.begin(),uniqueSSID.end());
   _requestedAPConfig.ssid.SSID = byteSSID;
   _requestedAPConfig.ssid.isUTF8 = false;
   _requestedAPConfig.isSSIDValid = true;
}
} // namespace bosch
} // namespace org

/** @} */
