/**
 * @file DisablingSetupState.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <string>

#include "WBLDefines.h"
#include "StateMachineProperties.h"
#include "IWifiSetupSM.h"
#include "WBLASFComponent.h"
#include "DisablingSetupState.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", DisablingSetupState, Info);

DisablingSetupState::DisablingSetupState(IWifiSetupSM* wifiSetupSMIf):
      WifiSetupState(ST_DISABLING_SETUP, wifiSetupSMIf)
{
   LOG_INFO("[WifiMode=%d] DisablingSetupState::Constructor entered", _wifiMode);
}

DisablingSetupState::~DisablingSetupState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   // LOG_INFO("[WifiMode=%d] DisablingSetupState::Destructor entered", _wifiMode);
}

void DisablingSetupState::onEntry(::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] DisablingSetupState::onEntry()", _wifiMode);
   WBL_INTENTIONALLY_UNUSED(msg);

   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   if (_wifiSetupSMIf->getDisableSetupMsg())
   {
      if (!sendDisableRequest())
      {
         LOG_ERROR("onEntry: Could not disable Wifi interface. Switching to SetupError state.");
         switchToErrorState();
      }
   } 
   else
   {
      LOG_ERROR("onEntry: No DisableSetup request available. Defaulting to SetupErrorIdle state");
      _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
   }
}

void DisablingSetupState::onRequest(::boost::shared_ptr< DisableSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] DisablingSetupState::onRequest(DisableSetup) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   //! Send success response since setup is disabled
   _wifiSetupSMIf->sendMsgResponse(msg, WBL_ERR_INPROGRESS);
}

void DisablingSetupState::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode=%d] DisablingSetupState::onEvent() entered: EventID = %s, isError = %d",
         _wifiMode, SM_EVENT_TO_STR(eventID), isError);

   if (IS_CURRENT_MODE(setup.property.mode))
   {
      switch (eventID)
      {
         case WIFI_SETUP_EVENT_STA_TECH_REMOVED:
         {
            //! If STA interface is removed, switch to SetupDisabled state
            (isError) ? (switchToErrorState()) : (onDisableSetupSuccess());
         }
         break;
         case WIFI_SETUP_EVENT_AP_POWERED:
         {
            //! If AP setup is powered OFF, switch to Idle state. Else switch to Error state.
            ((setup.property.isPowered) || (isError)) ? (switchToErrorState()) : (onDisableSetupSuccess());
         }
         break;
         case WIFI_SETUP_EVENT_STA_POWERED:
         {
            //! If STA interface is Powrered OFF, switch to SetupDisabled state
            (isError) ? (switchToErrorState()) : (onDisableSetupSuccess());
         }
         break;
         default:
         {
            //! Forward unhandled events to base class event handler
            WifiSetupState::onEvent(eventID, setup, isError);
         }
         break;
      }
   } // if (IS_CURRENT_MODE(setup.property.mode))
}

bool DisablingSetupState::isSTAPoweredON()
{
   bool bRet = false;
   WifiSetUpList wifiSetups =_wifiSetupSMIf->getAllWifiSetups();
   for(auto it = wifiSetups.begin(); it != wifiSetups.end(); ++it)
   {
      if(WIFI_MODE_STA == it->property.mode)
         bRet = it->property.isPowered;
   }
   return bRet;
}

bool DisablingSetupState::sendDisableRequest()
{
   bool isRequestSent = false;
   WBL_ASSERT_AND_RETURN(!_wifiSetupSMIf, isRequestSent);

   IEEE80211ClientFactory* factory = IEEE80211ClientFactory::getInstance();

   switch (_wifiMode)
   {
      //@Note: STA will only be powered OFF.
      case WIFI_MODE_STA:
      {
         bool isInterfaceAvailable = isSTAPoweredON();
         LOG_INFO("[WifiMode=%d] DisablingSetupState::sendDisableRequest: isInterfaceAvailable = %s",_wifiMode, BOOL_TO_STR(isInterfaceAvailable));
         if (isInterfaceAvailable)
         {
            IEEE80211STAClientIf* staClientIf = (factory) ? (factory->getSTAClientIf()) : (nullptr);
            if (staClientIf)
            {
               const ::std::string& staObjPath = (_wifiSetupSMIf->getWifiSetupInfo()).nativeObjPath;
               isRequestSent = staClientIf->setPowered(staObjPath,false);

               LOG_INFO("[WifiMode=%d] DisablingSetupState::sendDisableRequest: isRequestSent = %s",
                     _wifiMode, BOOL_TO_STR(isRequestSent));
            }
         }
         else
         {
            isRequestSent = true;
            onDisableSetupSuccess();
         }
      }
      break;
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         //@Note: AP will only be powered OFF on DisableSetup request.
         bool isInterfacePoweredOn = ((_wifiSetupSMIf->getWifiSetupInfo()).property.isPowered);
         if (isInterfacePoweredOn)
         {
            IEEE80211APClientIf* apClientIf = (factory) ? (factory->getAPClientIf()) : (nullptr);
            if (apClientIf)
            {
               const ::std::string& apObjPath = (_wifiSetupSMIf->getWifiSetupInfo()).nativeObjPath;
               isRequestSent = apClientIf->setPowered(apObjPath, false);

               LOG_INFO("[WifiMode=%d] DisablingSetupState::sendDisableRequest: isRequestSent = %s",
                     _wifiMode, BOOL_TO_STR(isRequestSent));
            }
         }
         else
         {
            isRequestSent = true;
            onDisableSetupSuccess();
         }
      }
      break;
      default:
      {
         LOG_ERROR("sendDisableRequest: Disabling mode %s not supported", WIFI_MODE_TO_STR(_wifiMode));
         switchToErrorState();
      }
      break;
   }

   return isRequestSent;
}

void DisablingSetupState::onDisableSetupSuccess()
{
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   switch (_wifiMode)
   {
      case WIFI_MODE_STA:
      {
         //LOG_DEBUG("onDisableSetupSuccess: STA interface is removed. Switching to SetupDisabled state.");
         LOG_DEBUG("onDisableSetupSuccess: STA interface is PoweredOff . Switching to SetupIdle state.");
         _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getDisableSetupMsg(), WBL_ERR_NOERROR);
         _wifiSetupSMIf->setDisableSetupMsg(nullptr);
         _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);

      }
      break;
      case WIFI_MODE_AP1:
      case WIFI_MODE_AP2:
      {
         LOG_DEBUG("onDisableSetupSuccess: AP is powered OFF. Switching to SetupIdle state.");
         _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getDisableSetupMsg(), WBL_ERR_NOERROR);
         _wifiSetupSMIf->setDisableSetupMsg(nullptr);
         _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
      }
      break;
      default:
         LOG_ERROR("onDisableSetupSuccess: Invalid Wifi mode %s", WIFI_MODE_TO_STR(_wifiMode));
      break;
   }
}

} // namespace bosch
} // namespace org

/** @} */
