/**
 * @file DeactivatingSetupState.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "WBLDefines.h"
#include "StateMachineProperties.h"
#include "IWifiSetupSM.h"
#include "DeactivatingSetupState.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", DeactivatingSetupState, Info);

DeactivatingSetupState::DeactivatingSetupState(IWifiSetupSM* wifiSetupSMIf):
      WifiSetupState(ST_DEACTIVATING_SETUP, wifiSetupSMIf)
{
   LOG_INFO("[WifiMode=%d] DeactivatingSetupState::Constructor entered", _wifiMode);
}

DeactivatingSetupState::~DeactivatingSetupState()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   // LOG_INFO("[WifiMode=%d] DeactivatingSetupState::Destructor entered", _wifiMode);
}

void DeactivatingSetupState::onEntry(::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] DeactivatingSetupState::onEntry()", _wifiMode);
   WBL_INTENTIONALLY_UNUSED(msg);

   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   if (_wifiSetupSMIf->getDeactivateSetupMsg())
   {
      LOG_DEBUG("DeactivatingSetupState:onEntry PowerOFFType:%d",_wifiSetupSMIf->getDeactivateSetupMsg()->getRequestType());
      if (!sendPowerOffRequest())
      {
         LOG_ERROR("onEntry: Could not Power OFF Wifi mode. Switching to SetupError state.");
         switchToErrorState();
      }
   } 
   else
   {
      LOG_ERROR("onEntry: No DeactivateSetup request available. Defaulting to SetupIdle state");
      _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
   }
}

void DeactivatingSetupState::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   LOG_INFO("[WifiMode=%d] DeactivatingSetupState::onEvent() entered: EventID = %s, isError = %d",
         _wifiMode, SM_EVENT_TO_STR(eventID), isError);

   if (IS_CURRENT_MODE(setup.property.mode))
   {
      switch (eventID)
      {
         case WIFI_SETUP_EVENT_STA_POWERED:
         case WIFI_SETUP_EVENT_AP_POWERED:
         {
            //! If wifi setup is powered ON, switch to error state. Else switch to Idle state.
            ((setup.property.isPowered) || (isError)) ? (switchToErrorState()) : (onDeactivateSetupSuccess(setup));
        }
        break;
        case WIFI_SETUP_EVENT_AP_UNREG_VENDOR:
        {
            bool isRequestSent = false;
            const WifiSetUpItem& wifiSetupItem = _wifiSetupSMIf->getWifiSetupInfo();
            IEEE80211APClientIf* apClientIf = getAPClientIf();
            if (apClientIf)
            {
                isRequestSent = apClientIf->setPowered(wifiSetupItem.nativeObjPath, false);
            }
            if (!isRequestSent)
            {
                 LOG_ERROR("onEntry: Could not Power OFF Wifi mode. Switching to SetupError state.");
                 switchToErrorState();
            }
        }
        break;
        default:
        {
            //! Forward unhandled events to base class event handler
            WifiSetupState::onEvent(eventID, setup, isError);
         }
         break;
      }
   } // if (IS_CURRENT_MODE(setup.property.mode))
}

bool DeactivatingSetupState::sendPowerOffRequest()
{
   bool isRequestSent = false;
   WBL_ASSERT_AND_RETURN(!_wifiSetupSMIf, isRequestSent);

   const WifiSetUpItem& wifiSetupItem = _wifiSetupSMIf->getWifiSetupInfo();

   bool isInterfacePoweredOff = !((_wifiSetupSMIf->getWifiSetupInfo()).property.isPowered);
   if (isInterfacePoweredOff)
   {
      isRequestSent = true;
      onDeactivateSetupSuccess(wifiSetupItem);
   }
   else
   {
      switch (_wifiMode)
      {
         case WIFI_MODE_STA:
         {
            IEEE80211STAClientIf* staClientIf = getSTAClientIf();
            if (staClientIf)
            {
               isRequestSent = staClientIf->setPowered(wifiSetupItem.nativeObjPath, false);
            }
         }
         break;
         case WIFI_MODE_AP1:
         case WIFI_MODE_AP2:
         {
            IEEE80211APClientIf* apClientIf = getAPClientIf();
            if (apClientIf)
            {
               /*De-register Vendor Element if registered */
               isRequestSent = apClientIf->updateUnregisterVendorService(wifiSetupItem.nativeObjPath);
               /*isRequestSent = apClientIf->setPowered(wifiSetupItem.nativeObjPath, false);*/
            }
         }
         break;
         default:
         {
            LOG_ERROR("sendPowerOffRequest: Invalid Wifi mode %s", WIFI_MODE_TO_STR(_wifiMode));
            switchToErrorState();
         }
         break;
      } // switch (_wifiMode)

      LOG_INFO("[WifiMode=%d] DeactivatingSetupState::sendPowerOffRequest: isRequestSent = %s",
            _wifiMode, BOOL_TO_STR(isRequestSent));
   }
   return isRequestSent;
}

void DeactivatingSetupState::onDeactivateSetupSuccess(const WifiSetUpItem& setup)
{
   LOG_DEBUG("onDeactivateSetupSuccess: Wifi mode powered OFF. Switching to SetupIdle state.");
   WBL_ASSERT_AND_EXIT(!_wifiSetupSMIf);

   LOG_DEBUG("onDeactivateSetupSuccess: Update LIM PowerOFFReason:%d",
         _wifiSetupSMIf->getDeactivateSetupMsg()->getRequestType());

   if(WIFI_POWER_OFF_USER_REQUEST == _wifiSetupSMIf->getDeactivateSetupMsg()->getRequestType())
   {
       WifiSetUpItem wifiSetup = setup;
       wifiSetup.property.powerOffType = WIFI_POWER_OFF_USER_REQUEST;
       WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
       ILastIntendedModeService* limServiceIf =
            (serviceFactory) ? (serviceFactory->getLastIntendedModeServiceIf()) : (nullptr);
       if (limServiceIf)
       {
         limServiceIf->onSetupUpdated(wifiSetup);
       }
      _wifiSetupSMIf->sendMsgResponse(_wifiSetupSMIf->getDeactivateSetupMsg(), WBL_ERR_NOERROR);
   }
   else
   {
      ILCMClient *lcmClientIf = LCMFactory::getInstance()->getLCMClientIf();
      if(lcmClientIf)
      {
        lcmClientIf->onDeactivateResponse(_wifiSetupSMIf->getDeactivateSetupMsg());
      }
   }
   _wifiSetupSMIf->setDeactivateSetupMsg(nullptr);
   _wifiSetupSMIf->setState(ST_SETUP_IDLE, nullptr);
}

} // namespace bosch
} // namespace org

/** @} */
