/**
 * @file WifiSetupControllerSM.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include "WBLDefines.h"
#include "IRespWifiSetupControllerSM.h"
#include "SetupDisabledState.h"
#include "SetupIdleState.h"
#include "PreparingSetupState.h"
#include "DeactivatingSetupState.h"
#include "DisablingSetupState.h"
#include "EnablingSetupState.h"
#include "SetupErrorState.h"
#include "ResettingSetupState.h"
#include "StateMachineProperties.h"
#include "WifiSetupControllerSM.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", WifiSetupControllerSM, Info);

WifiSetupControllerSM::WifiSetupControllerSM(WifiMode wifiMode, IRespWifiSetupControllerSM* respWifiSetupSM):
      _wifiMode(wifiMode),
      _respWifiSetupSM(respWifiSetupSM)
{
   LOG_INFO("[WifiMode=%d] WifiSetupControllerSM::Constructor entered ", _wifiMode);
   WBL_ASSERT_AND_EXIT(nullptr == _respWifiSetupSM);
   WBL_ASSERT(WIFI_MODE_UNKNOWN == _wifiMode);

   ::boost::shared_ptr< SetupDisabledState> setupDisabledState(new SetupDisabledState(this));
   _setupDisabledState = setupDisabledState;

   ::boost::shared_ptr< SetupIdleState> setupIdleState(new SetupIdleState(this));
   _idleState = setupIdleState;

   ::boost::shared_ptr< PreparingSetupState> preparingSetupState(new PreparingSetupState(this));
   _preparingSetupState = preparingSetupState;

   ::boost::shared_ptr< DeactivatingSetupState> deactivatingSetupState(new DeactivatingSetupState(this));
   _deactivatingSetupState = deactivatingSetupState;

   ::boost::shared_ptr< DisablingSetupState> disablingSetupState(new DisablingSetupState(this));
   _disablingSetupState = disablingSetupState;

   ::boost::shared_ptr< EnablingSetupState> enablingSetupState(new EnablingSetupState(this));
   _enablingSetupState = enablingSetupState;

   ::boost::shared_ptr< SetupErrorState> setupErrorState(new SetupErrorState(this));
   _errorState = setupErrorState;

   ::boost::shared_ptr< ResettingSetupState > setupResetState(new ResettingSetupState(this));
   _resetSetupState = setupResetState;

   //! Set default state for SM
   _currentState = _setupDisabledState;
   _currentState->onEntry(nullptr);
}

WifiSetupControllerSM::~WifiSetupControllerSM()
{
   // Commented the Log since it produces the Uncaught exception from Coverity
   //LOG_INFO("[WifiMode=%d] WifiSetupControllerSM::Destructor entered", _wifiMode);
}

void WifiSetupControllerSM::onRequest(::boost::shared_ptr< PrepareSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupControllerSM::onRequest(PrepareSetupMsg) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!msg);

   if (_wifiMode == msg->getWifiMode())
   {
      _currentState->onRequest(msg);
   }
   else
   {
      LOG_ERROR("onRequest(PrepareSetupMsg): Request sent to incorrect SM ");
      sendMsgResponse(msg, WBL_ERR_REJECTED);
   }
}

void WifiSetupControllerSM::onRequest(::boost::shared_ptr< DeActivateSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupControllerSM::onRequest(DeActivateSetupMsg) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!msg);

   if (_wifiMode == msg->getWifiMode())
   {
      _currentState->onRequest(msg);
   }
   else
   {
      LOG_ERROR("onRequest(DeActivateSetupMsg): Request sent to incorrect SM ");
      sendMsgResponse(msg, WBL_ERR_REJECTED);
   }
}

void WifiSetupControllerSM::onRequest(::boost::shared_ptr< DisableSetupMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupControllerSM::onRequest(DisableSetupMsg) entered", _wifiMode);
   WBL_ASSERT_AND_EXIT(!(msg && _currentState));

   if (_wifiMode == msg->getWifiMode())
   {
      _currentState->onRequest(msg);
   }
   else
   {
      LOG_ERROR("onRequest(DisableSetupMsg): Request sent to incorrect SM ");
      sendMsgResponse(msg, WBL_ERR_REJECTED);
   }
}

void WifiSetupControllerSM::onRequest(::boost::shared_ptr< RestoreFactorySettingsMsg > msg)
{
   LOG_INFO("WifiSetupControllerSM::onRequest(FactoryResetMsg) entered");
   WBL_ASSERT_AND_EXIT(!(msg && _currentState));

   if (_wifiMode == msg->getWifiMode())
   {
      _currentState->onRequest(msg);
   }
   else
   {
      LOG_ERROR("onRequest(FactoryResetMsg): Request sent to incorrect SM ");
      sendMsgResponse(msg, WBL_ERR_REJECTED);
   }
}

void WifiSetupControllerSM::onEvent(WifiSetupEventID eventID, const WifiSetUpItem& setup, bool isError)
{
   //LOG_INFO("WifiSetupControllerSM::onEvent() entered");
   WBL_ASSERT_AND_EXIT(!_currentState);

   if (setup.property.mode == _wifiMode)
   {
      _wifiSetupItem = setup;
   }
   _currentState->onEvent(eventID, setup, isError);
}

bool WifiSetupControllerSM::isSMBusy() const
{
   bool isBusy = true;
   switch (getStateType())
   {
      case ST_SETUP_DISABLED:
      case ST_SETUP_IDLE:
         isBusy = false;
         break;
      default:
         //! Add code
         break;
   }
   return isBusy;
}

WifiMode WifiSetupControllerSM::getWifiMode() const
{
   return _wifiMode;
}

WifiSetupStateType WifiSetupControllerSM::getStateType() const
{
   WifiSetupStateType stateType = (_currentState) ? (_currentState->getStateType()) : (ST_SETUP_DISABLED);
   return stateType;
}

void WifiSetupControllerSM::setState(WifiSetupStateType state, ::boost::shared_ptr< StateMachineMsg > msg)
{
   LOG_INFO("[WifiMode=%d] WifiSetupControllerSM::setState() entered: %s", _wifiMode, SM_STATE_TO_STR(state));
   WBL_ASSERT_AND_EXIT(!_currentState);

   bool isStateChanged = true;
   WifiSetupStateType prevState = _currentState->getStateType();

   switch (state)
   {
      case ST_SETUP_DISABLED:
         _currentState = _setupDisabledState;
         break;
      case ST_SETUP_IDLE:
         _currentState = _idleState;
         break;
      case ST_PREPARING_SETUP:
         _currentState = _preparingSetupState;
         break;
      case ST_DEACTIVATING_SETUP:
         _currentState = _deactivatingSetupState;
         break;
      case ST_DISABLING_SETUP:
         _currentState = _disablingSetupState;
         break;
      case ST_ENABLING_SETUP:
         _currentState = _enablingSetupState;
         break;
      case ST_RESETTING_SETUP:
         _currentState = _resetSetupState;
         break;
      case ST_SETUP_ERROR:
         _currentState = _errorState;
         break;
      default:
      {
         isStateChanged = false;
         LOG_ERROR("setState: Invalid state received. Current state unchanged.");
      }
         break;
   }//switch (state)

   if (isStateChanged)
   {
      LOG_DEBUG("[WifiMode=%d] setState: Switching from state \'%s\' -> \'%s\'",
            _wifiMode, SM_STATE_TO_STR(prevState), SM_STATE_TO_STR(state));
      _currentState->onEntry(msg);
   }
}

const WifiSetUpItem& WifiSetupControllerSM::getWifiSetupInfo() const
{
   return _wifiSetupItem;
}

const WifiSetUpList& WifiSetupControllerSM::getAllWifiSetups() const
{
   return (_respWifiSetupSM->getAllWifiSetups());
}

::boost::shared_ptr< PrepareSetupMsg > WifiSetupControllerSM::getPrepareSetupMsg()
{
   return _prepareSetupMsg;
}

::boost::shared_ptr< DeActivateSetupMsg > WifiSetupControllerSM::getDeactivateSetupMsg()
{
   return _deactSetupMsg;
}

::boost::shared_ptr< DisableSetupMsg > WifiSetupControllerSM::getDisableSetupMsg()
{
   return _disableSetupMsg;
}

::boost::shared_ptr< RestoreFactorySettingsMsg > WifiSetupControllerSM::getRestoreFactorySettingsMsg()
{
   return _resetSetupMsg;
}

void WifiSetupControllerSM::setPrepareSetupMsg(::boost::shared_ptr< PrepareSetupMsg > msg)
{
   _prepareSetupMsg = msg;
}

void WifiSetupControllerSM::setDeactivateSetupMsg(::boost::shared_ptr< DeActivateSetupMsg > msg)
{
   _deactSetupMsg = msg;
}

void WifiSetupControllerSM::setDisableSetupMsg(::boost::shared_ptr< DisableSetupMsg > msg)
{
   _disableSetupMsg = msg;
}

void WifiSetupControllerSM::setRestoreFactorySettingsMsg(::boost::shared_ptr< RestoreFactorySettingsMsg > msg)
{
   _resetSetupMsg = msg;
}

void WifiSetupControllerSM::sendMsgResponse(::boost::shared_ptr< WBLBaseMsg > msg, WBLErrorCode errorCode)
{
   //LOG_INFO("WifiSetupControllerSM::sendMsgResponse() entered: error code %d", errorCode);
   WBL_ASSERT_AND_EXIT(!((nullptr != _respWifiSetupSM) && (msg)));

   _respWifiSetupSM->sendMsgResponse(msg, errorCode);
}

void WifiSetupControllerSM::sendRequest(::boost::shared_ptr< WBLBaseMsg > msg)
{
   WBL_ASSERT_AND_EXIT(!((nullptr != _respWifiSetupSM) && (msg)));

   _respWifiSetupSM->sendRequest(msg);
}

} // namespace bosch
} // namespace org

/** @} */
