/**
 * @file IEEE80211ConflictManager.cpp
 * @author RBEI/ECO21 Ramya Murthy
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief Core logic for WBL ,Listens for AP and STA
 * related events and updates active setups.
 * Receives requests from WBL service and forwards it
 * for further handling.
 *
 * @{
 */

#include <algorithm>
#include <vector>
#include <map>
#include <boost/shared_ptr.hpp>
#include <boost/pointer_cast.hpp>

#include "WBLTypes.h"
#include "WBLHelper.h"
#include "StateMachineTypes.h"
#include "IEEE80211EventMsg.h"
#include "IEEE80211ClientProperties.h"
#include "WBLASFComponent.h"
#include "ConflictResolver.h"
#include "IEEE80211ConflictManager.h"
#include "WifiConnectionStatus.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", IEEE80211ConflictManager, Debug);

#define CONVERT_FREQ_TO_FREQBAND(FREQ) \
      ( ((FREQ) < WBL_FREQ_BAND_5_GHZ) ? (WBL_FREQ_BAND_2_4_GHZ) : (WBL_FREQ_BAND_5_GHZ) );

using boost::static_pointer_cast;

typedef std::map < ::std::string, WifiSetUpItem >::iterator WifiSetupsListItr;
typedef std::map < ::std::string, WifiSetUpItem >::const_iterator WifiSetupsListConstItr;

/*TO DO:
Move to a class when conf file to read and write
Cplay channels is available */

ChannelList cpChannels_2_4 { 1, 6, 11 };
ChannelList cpChannels_5 { 36, 40, 44, 48, 149, 153, 157, 161 };

IEEE80211ConflictManager::IEEE80211ConflictManager()
{
   LOG_INFO("IEEE80211ConflictManager:Constructor");

   IEEE80211ClientFactory* clientfactory = IEEE80211ClientFactory::getInstance();

   //Register for AP Events
   IEEE80211APClientIf* apClientIf = (clientfactory) ? (clientfactory->getAPClientIf()) : (nullptr);
   if (apClientIf)
   {
      apClientIf->registerObserver(IEEE80211_EVENT_AP, this);
   }

   //Register for STA Events
   IEEE80211STAClientIf* staClientIf = (clientfactory) ? (clientfactory->getSTAClientIf()) : (nullptr);
   if (staClientIf)
   {
      staClientIf->registerObserver(IEEE80211_EVENT_STA, this);
   }

   SMManagerFactory* smFactory = SMManagerFactory::getInstance();
   tICmdStateMachineManagerSptr smManagerIf = (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
   if (smManagerIf)
   {
      smManagerIf->registerResponseIf(this);
   }
   _restoringLastMode = false;
} // IEEE80211ConflictManager::IEEE80211ConflictManager

IEEE80211ConflictManager::~IEEE80211ConflictManager()
{
   try
   {
      LOG_INFO("IEEE80211ConflictManager:Destructor");
      IEEE80211ClientFactory* clientfactory = IEEE80211ClientFactory::getInstance();

      //Deregister from AP Events
      IEEE80211APClientIf* apClientIf = (clientfactory) ? (clientfactory->getAPClientIf()) : (nullptr);
      if (apClientIf)
      {
         apClientIf->deregisterObserver(IEEE80211_EVENT_AP, this);
      }

      //Deregister from STA Events
      IEEE80211STAClientIf* staClientIf = (clientfactory) ? (clientfactory->getSTAClientIf()) : (nullptr);
      if (staClientIf)
      {
         staClientIf->deregisterObserver(IEEE80211_EVENT_STA, this);
      }
   }catch (...){}
} // IEEE80211ConflictManager::~IEEE80211ConflictManager

bool IEEE80211ConflictManager::isSMBusy(const WifiMode& mode)
{
   LOG_INFO("IEEE80211ConflictManager::isSMBusy");

   bool isSMBusy = true;
   SMManagerFactory* smFactory = SMManagerFactory::getInstance();
   tICmdStateMachineManagerSptr smManagerIf = (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
   std::string wifiMode = wblWifiMode.getWifiMode2String(mode);

   if (smManagerIf)
   {
      isSMBusy = smManagerIf->isSMBusy(mode);
   }
   LOG_DEBUG("IEEE80211ConflictManager::isSMBusy: %s (for WifiMode = %s) ",
         BOOL_TO_STR(isSMBusy), wifiMode.c_str());
   return isSMBusy;
} // bool IEEE80211ConflictManager::isSMBusy

void IEEE80211ConflictManager::sendConflictResolverMsg(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("IEEE80211ConflictManager::sendConflictResolverMsg");
   WBL_ASSERT_AND_EXIT(!msg);

   ConflictManagerFactory* conflictmgrFactory = ConflictManagerFactory::getInstance();
   ICmdConflictResolver* conflictResolverIf =
         (conflictmgrFactory) ? (conflictmgrFactory->getConflictResolverIf()) : nullptr;
   if (conflictResolverIf)
   {
      LOG_DEBUG("sendConflictResolverMsg: SM is idle,checking for conflicts");
      conflictResolverIf->resolveWBLServiceRequest(msg);
   }
} // void IEEE80211ConflictManager::sendConflictResolverMsg

void IEEE80211ConflictManager::sendDeAuthStationMsg(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("IEEE80211ConflictManager::sendDeAuthStationMsg");
   WBL_ASSERT_AND_EXIT(!msg);
   IEEE80211ClientFactory* clientfactory = IEEE80211ClientFactory::getInstance();
   ::boost::shared_ptr < DeAuthenticateStationMsg > deAuthenticateMsg =
         static_pointer_cast < DeAuthenticateStationMsg >(msg);

   DeAuthenticateStationItem deAuthItem = deAuthenticateMsg->getDeAuthStation();
   deAuthenticateMsg->setErrorCode(WBL_ERR_NOERROR);

   IEEE80211APClientIf* apClientIf = (clientfactory) ? (clientfactory->getAPClientIf()) : (nullptr);
   if (apClientIf)
   {
      if (deAuthItem.type.compare("one") == 0 || (!deAuthItem.macAddress.empty()))
      {
         if (!apClientIf->deAuthenticateStation(deAuthItem.objPath, deAuthItem.macAddress))
         {
            deAuthenticateMsg->setErrorCode(WBL_ERR_REJECTED);
         }
      }
      else if (deAuthItem.type.compare("all") == 0)
      {
         for (auto itr = _associatedStations.begin(); itr != _associatedStations.end(); ++itr)
         {
            if (!apClientIf->deAuthenticateStation(itr->second.property.apPath, itr->second.property.macAddress))
            {
               LOG_DEBUG("IEEE80211ConflictManager::sendDeAuthStation Failed for %s", itr->second.property.apPath.c_str());
               deAuthenticateMsg->setErrorCode(WBL_ERR_REJECTED);
               break;
            }
         }
      }
   }
   sendDeAuthStationResponse(deAuthenticateMsg);
} // void IEEE80211ConflictManager::sendDeAuthStationMsg

void IEEE80211ConflictManager::sendDeAuthStationResponse(::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("IEEE80211ConflictManager::sendWifiSetupResponse");
   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   WifiSetupManager* wifiSetupManager =
         (serviceFactory) ? (serviceFactory->getWifiSetupManager()) : (nullptr);
   if (wifiSetupManager)
   {
      WifiSetupService* wifiSetupServiceIf = wifiSetupManager->getAPWifiSetup();
      wifiSetupServiceIf->onWBLServiceResponse(msg);
   }
} //void IEEE80211ConflictManager::sendDeAuthStationResponse


bool IEEE80211ConflictManager::checkLCMWifiStateAvailability(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("IEEE80211ConflictManager::checkLCMWifiStateAvailability");
   bool bRet = true;
   LCMWiFiState currState = getWifiState();

   if((LCM_WIFISTATE_OFF == currState)||(LCM_WIFISTATE_BLOCK == currState))
   {
      msg->setErrorCode(WBL_ERR_WIFISTATE_BLOCKED);
      sendMsgResponse(msg);
      bRet = false;
   }
   return bRet;
} //void IEEE80211ConflictManager::checkLCMWifiStateAvailability

void IEEE80211ConflictManager::onProcessRequest(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("IEEE80211ConflictManager::onProcessRequest");
   WBL_ASSERT_AND_EXIT(!msg);

   LOG_DEBUG("onProcessRequest: MsgID = %d", msg->getMsgID());

   switch (msg->getMsgID())
   {
      case WBL_MSG_PREPARE_SETUP_REQ:
      {
         if(!checkLCMWifiStateAvailability(msg))
            return;
         ::boost::shared_ptr < PrepareSetupMsg > prepareSetupMsg = static_pointer_cast < PrepareSetupMsg >(msg);
         WifiMode requestedWifiMode = prepareSetupMsg->getWifiMode();

         if (isSMBusy(requestedWifiMode))
         {
            LOG_DEBUG("SM is busy,post WBL Error");
            prepareSetupMsg->setErrorCode(WBL_ERR_INPROGRESS);
            sendMsgResponse(prepareSetupMsg);
         }
         else
         {
            LOG_DEBUG("SM is idle,forward to ConflictResolver");
            sendConflictResolverMsg(prepareSetupMsg);
         }
      }
      break;
      case WBL_MSG_DEACTIVATE_SETUP_REQ:
      {
         ::boost::shared_ptr < DeActivateSetupMsg > deactivateSetupMsg = static_pointer_cast < DeActivateSetupMsg >(msg);
         sendConflictResolverMsg(deactivateSetupMsg);
         updateRequestType(deactivateSetupMsg);
         SMManagerFactory* smFactory = SMManagerFactory::getInstance();
         tICmdStateMachineManagerSptr smManagerIf = (smFactory) ?
                       (smFactory->getStateMachineManagerIf()) : (nullptr);
         if (smManagerIf)
         {
            smManagerIf->processRequest(msg);
         }

      }
      break;
      case WBL_MSG_RESOLVE_CONFLICT_REQ:
      {
         ::boost::shared_ptr < ResolveConflictMsg > resolveConflictMsg = static_pointer_cast < ResolveConflictMsg >(msg);
         sendConflictResolverMsg(resolveConflictMsg);
      }
      break;
      case WBL_MSG_DEAUTHENTICATE_STATION_REQ:
      {
        sendDeAuthStationMsg(msg);
      }
      break;
      case WBL_MSG_RESTORE_SETTINGS_REQ:
      {
         SMManagerFactory* smFactory = SMManagerFactory::getInstance();
         tICmdStateMachineManagerSptr smManagerIf =
               (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
         if (smManagerIf)
         {
            smManagerIf->processRequest(msg);
         }
      }
      break;
      default:
         LOG_ERROR("Unhandled case for onProcessRequest()");
      break;
   }
} // void IEEE80211ConflictManager::onProcessRequest

void IEEE80211ConflictManager::sendMsgResponse(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("IEEE80211ConflictManager::sendMsgResponse");

   WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
   WBLBaseMsgID msgID = msg->getMsgID();
   switch (msgID)
   {
      case WBL_MSG_PREPARE_SETUP_REQ:
      case WBL_MSG_DEACTIVATE_SETUP_REQ:
      case WBL_MSG_RESOLVE_CONFLICT_REQ:
      {
         WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
         IConflictManagementService* conflictMgrServiceIf =
               (serviceFactory) ? (serviceFactory->getConflictManagementServiceIf()) : (nullptr);
         if (conflictMgrServiceIf)
         {
            conflictMgrServiceIf->onWBLServiceResponse(msg);
         }
      }
      break;
      case WBL_MSG_RESTORE_SETTINGS_REQ:
      {
         WifiSetupManager* poWifiSetUpMgr =
               (serviceFactory) ? (serviceFactory->getWifiSetupManager()) : (nullptr);
         if (poWifiSetUpMgr)
         {
            poWifiSetUpMgr->onWBLServiceResponse(msg);
         }
      }
      break;
      default:
      {
         LOG_ERROR("sendMsgResponse Invalid Msg Id");
      }
      break;
   }
} //void IEEE80211ConflictManager::sendMsgResponse

void IEEE80211ConflictManager::processIEEE80211Event(IEEE80211EventType type, const IEEE80211EventMsg* eventMsg)
{

   LOG_INFO("IEEE80211ConflictManager:processIEEE80211Event: EventType = %d ", type);

   switch (type)
   {
      case IEEE80211_EVENT_STA:
         processSTAModeEvent(eventMsg);
      break;
      case IEEE80211_EVENT_AP:
         processAPModeEvent(eventMsg);
      break;
      case IEEE80211_EVENT_P2P:
      case IEEE80211_EVENT_UNKNOWN:
      default:
         LOG_ERROR("processIEEE80211Event: Invalid event type %d", type);
      break;
   }
} // void IEEE80211ConflictManager::processIEEE80211Event

void IEEE80211ConflictManager::notifyActiveWifiSetups()
{
   LOG_INFO("IEEE80211ConflictManager:notifyActiveWifiSetups");

   WifiSetUpList wifiSetupList;
   if (getWiFiSetUps(wifiSetupList))
   {
      if(!_restoringLastMode)
      {
         WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
         if (serviceFactory)
         {
            ActiveWifiSetupsMsg msg;
            msg.setWifiSetUps(wifiSetupList);
            serviceFactory->notifyActiveWifiSetups(msg);
         }
      }
      ConflictManagerFactory* conflictmgrFactory = ConflictManagerFactory::getInstance();
      ICmdConflictResolver* conflictResolverIf =
            (conflictmgrFactory) ? (conflictmgrFactory->getConflictResolverIf()) : nullptr;
      if (conflictResolverIf)
      {
         conflictResolverIf->notifyActiveWiFiSetups(wifiSetupList);
      }

      SMManagerFactory* smFactory = SMManagerFactory::getInstance();
      tICmdStateMachineManagerSptr smManagerIf =
            (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
      if (smManagerIf)
      {
         smManagerIf->notifyActiveWiFiSetups(wifiSetupList);
      }
      ILCMClient *lcmClientIf = LCMFactory::getInstance()->getLCMClientIf();
      if(lcmClientIf)
      {
         lcmClientIf->notifyActiveWiFiSetups(wifiSetupList);
      }

      cWifiConnectionStatus::getInstance()->notifyActiveWiFiSetups(wifiSetupList);

   }
} // void IEEE80211ConflictManager::notifyActiveWifiSetups

void IEEE80211ConflictManager::processAPModeEvent(const IEEE80211EventMsg* eventMsg)
{
   LOG_INFO("IEEE80211ConflictManager::processAPModeEvent");
   WBL_ASSERT_AND_EXIT(nullptr == eventMsg);

   AccessPointItem apItem;
   AssociatedStationItem asItem;
   IEEE80211EventID eventID = eventMsg->getIEEE80211EventID();
   IEEE80211ErrorCode errCode = eventMsg->getIEEE80211ErrCode();

   LOG_INFO("processAPModeEvent: EventID = %d, %s, ErrorCode = %s",
              eventID, IEEE80211_EVENT_TO_STR(eventID), IEEE80211_ERR_TO_STR(errCode));

   bool isError = (errCode != IEEE80211_ERROR_NONE);

   WifiSetUpItem wifiSetup;
   bool isWifiSetupValid = false;

   switch (eventID)
   {
      case IEEE80211_MSG_AP_MODE_APADDED_IND:
      {
         const AccessPointAddedMsg* apAddedMsg = static_cast<const AccessPointAddedMsg*>(eventMsg);
         apAddedMsg->getAccessPointItem(apItem);
         if (!isError)
         {
            addWifiSetup(apItem);

            isWifiSetupValid = getWifiSetupItem(apItem.objPath, wifiSetup);
            if (isWifiSetupValid)
            {
               WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
               ILastIntendedModeService* limServiceIf =
                     (serviceFactory) ? (serviceFactory->getLastIntendedModeServiceIf()) : (nullptr);
               if (limServiceIf)
               {
                  limServiceIf->onSetupAdded(wifiSetup);
               }
               if (serviceFactory)
               {
                  serviceFactory->getWifiSetupManager()->updateInterfaceAdded(wifiSetup);
               }
            }
         }
      }
      break;

      case IEEE80211_MSG_AP_MODE_APREMOVED_IND:
      {
         const AccessPointRemovedMsg* apRemovedMsg = static_cast<const AccessPointRemovedMsg*>(eventMsg);
         ::std::string apObjPath = apRemovedMsg->getObjPath();

         isWifiSetupValid = getWifiSetupItem(apItem.objPath, wifiSetup);
         if (!isError)
         {
            removeWifiSetup(apObjPath);
            if (isWifiSetupValid)
            {
               WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
               ILastIntendedModeService* limServiceIf =
                     (serviceFactory) ? (serviceFactory->getLastIntendedModeServiceIf()) : (nullptr);
               if (limServiceIf)
               {
                  const std::string objPath = wifiSetup.setupObjPath;
                  limServiceIf->onSetupRemoved(objPath);
               }
            }
         }
      }
      break;

      case IEEE80211_MSG_AP_MODE_ASADDED_IND:
      {
         const AssociatedStationAddedMsg* asAddedMsg = static_cast<const AssociatedStationAddedMsg*>(eventMsg);
         asAddedMsg->getAssociatedStationItem(asItem);

         if (!isError)
         {
            addAssociatedStation(asItem);
         }
         isWifiSetupValid = getWifiSetupItem(asItem.property.apPath, wifiSetup);
      }
      break;

      case IEEE80211_MSG_AP_MODE_ASREMOVED_IND:
      {
         const AssociatedStationRemovedMsg* asRemovedMsg = static_cast<const AssociatedStationRemovedMsg*>(eventMsg);
         ::std::string asObjPath = asRemovedMsg->getObjPath();

         isWifiSetupValid = (getAssociatedStationItem(asObjPath, asItem))
               && (getWifiSetupItem(asItem.property.apPath, wifiSetup));
         if (!isError)
         {
            removeAssociatedStation(asObjPath);
         }
      }
      break;

      case IEEE80211_MSG_AP_MODE_AP_POWERED_IND:
      case IEEE80211_MSG_AP_MODE_AP_SSID_IND:
      case IEEE80211_MSG_AP_MODE_AP_PASSPHRASE_IND:
      case IEEE80211_MSG_AP_MODE_AP_CURRENTCHANNEL_IND:
      case IEEE80211_MSG_AP_MODE_AP_SECURITY_IND:
      case IEEE80211_MSG_AP_MODE_AP_VENDOR_IES_IND:
      case IEEE80211_MSG_AP_MODE_AP_MACADDRESS_IND:
      case IEEE80211_MSG_AP_MODE_AP_SUPPCHANNELS_IND:
      case IEEE80211_MSG_AP_MODE_AP_MAXSTA_IND:
      case IEEE80211_MSG_AP_MODE_AP_HIDDEN_IND:
      case IEEE80211_MSG_AP_MODE_AP_INTERFACE_IND:
      case IEEE80211_MSG_AP_MODE_AP_SETOPERATINGCHANNEL:
      case IEEE80211_MSG_AP_MODE_AP_DEAUTHENTICATESTATION:
      case IEEE80211_MSG_AP_MODE_AP_POWERSTATE_IND:
      case IEEE80211_MSG_AP_MODE_AP_POWERFAILURE_IND:
      case IEEE80211_MSG_AP_MODE_AP_TYPE_IND:
      case IEEE80211_MSG_AP_MODE_AP_IPV4_CONFIGURATION_IND:
      case IEEE80211_MSG_AP_MODE_AP_TETHERING_IND:
      case IEEE80211_MSG_AP_MODE_AP_HEALTH_IND:
      case IEEE80211_MSG_AP_MODE_AP_ACCESS_NETWORK_TYPE_IND:
      {
         const AccessPointChangedMsg* apChangedMsg = static_cast<const AccessPointChangedMsg*>(eventMsg);
         apChangedMsg->getAccessPointItem(apItem);

         if (!isError)
         {
            onAPUpdate(apItem, eventID);
         }
         isWifiSetupValid = getWifiSetupItem(apItem.objPath, wifiSetup);
      }
      break;
      case IEEE80211_MSG_AP_MODE_AP_REG_VENDOR_IND:
      case IEEE80211_MSG_AP_MODE_AP_UNREG_VENDOR_IND:
      {
         const AccessPointChangedMsg* apChangedMsg = static_cast<const AccessPointChangedMsg*>(eventMsg);
         apChangedMsg->getAccessPointItem(apItem);
         isWifiSetupValid = getWifiSetupItem(apItem.objPath, wifiSetup);
         wifiSetup.property.apSetup.vendorIEs = apItem.property.vendorIEs;
      }
      break;
      case IEEE80211_MSG_AP_MODE_CLEAR_DATA_RESP:
      {
         WifiSetupEventID smEventID = WIFI_SETUP_EVENT_AP_CLEAR_DATA;
         SMManagerFactory* smFactory = SMManagerFactory::getInstance();
         tICmdStateMachineManagerSptr smManagerIf = (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
         if ((smManagerIf) && (WIFI_SETUP_EVENT_UNKNOWN != smEventID))
         {
               bool isError = (errCode != IEEE80211_ERROR_NONE);
               smManagerIf->processEvent(smEventID, wifiSetup , isError);
         }
      }
      break;
      case IEEE80211_MSG_AP_MODE_AS_MACADDRESS_IND:
      case IEEE80211_MSG_AP_MODE_AS_APPATH_IND:
      case IEEE80211_MSG_AP_MODE_AS_INTERFACE_IND:
      case IEEE80211_MSG_AP_MODE_AS_IPADDRESS_IND:
      case IEEE80211_MSG_AP_MODE_AS_HOSTNAME_IND:
      case IEEE80211_MSG_AP_MODE_ASCHANGED_IND:
      {
         const AssociatedStationChangedMsg* asChangedMsg = static_cast<const AssociatedStationChangedMsg*>(eventMsg);
         asChangedMsg->getAssociatedStationItem(asItem);

         if (!isError)
         {
            onASUpdate(asItem, eventID);
         }
         isWifiSetupValid = getWifiSetupItem(asItem.property.apPath, wifiSetup);
      }
      break;
      case IEEE80211_MSG_AP_MODE_RESTORE_START:
      {
         WBLAPRestoreType restoreType;
         const RestoreAccessPointMsg *restoreMsg = static_cast<const RestoreAccessPointMsg*>(eventMsg);
         restoreMsg->getRestoreType(restoreType);
         if(restoreType == WBL_AP_RESTORE_CONFIGS || restoreType == WBL_AP_RESTORE_POWER)
         {
            _restoringLastMode = true;
         }
      }
      break;
      case IEEE80211_MSG_AP_MODE_RESTORE_COMPLETE:
      {
         _restoringLastMode = false;
      }
      break;
      case IEEE80211_MSG_UNKNOWN:
      default:
         LOG_ERROR("processAPModeEvent: Invalid Event ID");
      break;
   }

   //! Send update to SM
   if (isWifiSetupValid)
   {
      sendAPUpdate2SM(wifiSetup, eventID, errCode);
   }
} // void IEEE80211ConflictManager::processAPModeEvent

void IEEE80211ConflictManager::processSTAModeEvent(const IEEE80211EventMsg* eventMsg)
{
   LOG_INFO("IEEE80211ConflictManager::processSTAModeEvent");

   WBL_ASSERT_AND_EXIT(nullptr == eventMsg);

   TechnologyItem staItem;
   IEEE80211EventID eventID = eventMsg->getIEEE80211EventID();
   IEEE80211ErrorCode errCode = eventMsg->getIEEE80211ErrCode();

   LOG_INFO("processSTAModeEvent: EventID = %s, ErrorCode = %s", IEEE80211_EVENT_TO_STR(eventID),
         IEEE80211_ERR_TO_STR(errCode));

   bool isError = (errCode != IEEE80211_ERROR_NONE);

   WifiSetUpItem wifiSetup;
   bool isWifiSetupValid = false;

   switch (eventID)
   {
      case IEEE80211_MSG_STA_MODE_TECHNOLOGYADDED_IND:
      {
         const TechnologyAddedMsg* staAddedMsg = static_cast<const TechnologyAddedMsg*>(eventMsg);
         staAddedMsg->getTechnologyItem(staItem);

         if (!isError)
         {
            addWifiSetup(staItem);
            isWifiSetupValid = getWifiSetupItem(staItem.objPath, wifiSetup);
            if (isWifiSetupValid)
            {
               WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
               ILastIntendedModeService* limServiceIf =
                     (serviceFactory) ? (serviceFactory->getLastIntendedModeServiceIf()) : (nullptr);
               if (limServiceIf)
               {
                  limServiceIf->onSetupAdded(wifiSetup);
               }
               if (serviceFactory)
               {
                  serviceFactory->getWifiSetupManager()->updateInterfaceAdded(wifiSetup);
               }
            }
         }
      }
      break;

      case IEEE80211_MSG_STA_MODE_TECHNOLOGYREMOVED_IND:
      {
         const TechnologyRemovedMsg* staRemovedMsg = static_cast<const TechnologyRemovedMsg*>(eventMsg);
         ::std::string staObjPath = staRemovedMsg->getObjPath();

         isWifiSetupValid = getWifiSetupItem(staObjPath, wifiSetup);
         if (!isError)
         {
            removeWifiSetup(staObjPath);
            if (isWifiSetupValid)
            {
               WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
               ILastIntendedModeService* limServiceIf =
                     (serviceFactory) ? (serviceFactory->getLastIntendedModeServiceIf()) : (nullptr);
               if (limServiceIf)
               {
                  const ::std::string objPath = wifiSetup.setupObjPath;
                  limServiceIf->onSetupRemoved(objPath);
               }
            }
         }
      }
      break;
      case IEEE80211_MSG_STA_MODE_POWERED_IND:
      case IEEE80211_MSG_STA_MODE_CONNECTED_IND:
      case IEEE80211_MSG_STA_MODE_NAME_IND:
      case IEEE80211_MSG_STA_MODE_TYPE_IND:
      case IEEE80211_MSG_STA_MODE_CURRCHANNEL_IND:
      case IEEE80211_MSG_STA_MODE_POWERSTATE_IND:
      {
         const TechnologyChangedMsg* staChangedMsg = static_cast<const TechnologyChangedMsg*>(eventMsg);
         staChangedMsg->getTechnologyItem(staItem);

         if (!isError)
         {
            onSTAUpdate(staItem, eventID);
         }
         isWifiSetupValid = getWifiSetupItem(staItem.objPath, wifiSetup);
      }
      break;
      case IEEE80211_MSG_STA_MODE_RESTORE_START:
      {
         _restoringLastMode = true;
      }
      break;
      case IEEE80211_MSG_STA_MODE_RESTORE_COMPLETE:
      {
         _restoringLastMode = false;
      }
      break;
      case IEEE80211_MSG_UNKNOWN:
      default:
         LOG_ERROR("processSTAModeEvent: Invalid Event ID");
      break;
   }

//! Send update to SM
   if (isWifiSetupValid)
   {
      sendSTAUpdate2SM(wifiSetup, eventID, errCode);
   }
} //void IEEE80211ConflictManager::processSTAModeEvent

void IEEE80211ConflictManager::addWifiSetup(const TechnologyItem& staItem)
{
   LOG_INFO("IEEE80211ConflictManager:addWifiSetup:STA");
   //TO DO:Revisit
   std::string interface = "wlan0";

   if (!interface.empty())
   {
      if (false == isWifiSetupAvailable(staItem.objPath))
      {
         LOG_DEBUG("addWifiSetup: Adding new STA setup");

         _wifiSetups[staItem.objPath].property.mode = WIFI_MODE_STA;
         _wifiSetups[staItem.objPath].nativeObjPath = staItem.objPath;
         _wifiSetups[staItem.objPath].setupObjPath = createWifiSetupObjectPath(interface);
         _wifiSetups[staItem.objPath].property.interface = interface;
         _wifiSetups[staItem.objPath].property.isPowered = staItem.property.isPowered;
         _wifiSetups[staItem.objPath].property.powerState = staItem.property.powerState;
         _wifiSetups[staItem.objPath].property.isConnected = staItem.property.isConnected;
         if (true == staItem.property.isConnected)
         {
            _wifiSetups[staItem.objPath].property.currChannel = staItem.property.currChannel;
            _wifiSetups[staItem.objPath].property.frequency = CONVERT_FREQ_TO_FREQBAND(staItem.property.frequency);
         }
      }
      //printWifiSetup(_wifiSetups[staItem.objPath]);
      notifyActiveWifiSetups();
   }
} //void IEEE80211ConflictManager::addWifiSetup

void IEEE80211ConflictManager::onSTAUpdate(const TechnologyItem& staItem, const IEEE80211EventID eventID)
{
   LOG_INFO("IEEE80211ConflictManager::onSTAUpdate");

   ::std::string objPath = staItem.objPath;
   if (!objPath.empty())
   {
      auto setupsItr = _wifiSetups.find(objPath);
      if (setupsItr != _wifiSetups.end())
      {
         switch (eventID)
         {
            case IEEE80211_MSG_STA_MODE_POWERED_IND:
            {
               setupsItr->second.property.isPowered = staItem.property.isPowered;

               WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
               ILastIntendedModeService* limServiceIf =
                     (serviceFactory) ? (serviceFactory->getLastIntendedModeServiceIf()) : (nullptr);
               if (limServiceIf)
               {
                  limServiceIf->onSetupUpdated(setupsItr->second);
               }
            }
            break;

            case IEEE80211_MSG_STA_MODE_CONNECTED_IND:
            {
               setupsItr->second.property.isConnected = staItem.property.isConnected;
               setupsItr->second.property.connectionStatus.isConnected = staItem.property.isConnected;
               setupsItr->second.property.connectionStatus.healthIndicatorValue = (staItem.property.isConnected)?staItem.property.connHealth:0;
               if (staItem.property.isConnected)
               {
                  setupsItr->second.property.frequency = CONVERT_FREQ_TO_FREQBAND(staItem.property.frequency);
                  setupsItr->second.property.currChannel = staItem.property.currChannel;
               }
            }
            break;

            case IEEE80211_MSG_STA_MODE_POWERSTATE_IND:
               (*setupsItr).second.property.powerState = staItem.property.powerState;
            break;

            case IEEE80211_MSG_STA_MODE_NAME_IND:
               (*setupsItr).second.property.interface = staItem.property.name;
            break;

            default:
               LOG_ERROR("onSTAUpdate: Event %d not handled", eventID);
            break;
         } // switch (eventID)

         //printWifiSetup((*setupsItr).second);
         notifyActiveWifiSetups();
      }
   }
} //void IEEE80211ConflictManager::onSTAUpdate()

void IEEE80211ConflictManager::sendSTAUpdate2SM(const WifiSetUpItem& staItem, const IEEE80211EventID eventID,
      const IEEE80211ErrorCode errCode)
{
   LOG_INFO("IEEE80211ConflictManager::sendSTAUpdate2SM");

   WifiSetupEventID smEventID = WIFI_SETUP_EVENT_UNKNOWN;
   switch (eventID)
   {
      case IEEE80211_MSG_STA_MODE_TECHNOLOGYADDED_IND:
         smEventID = WIFI_SETUP_EVENT_STA_TECH_ADDED;
      break;
      case IEEE80211_MSG_STA_MODE_TECHNOLOGYREMOVED_IND:
         smEventID = WIFI_SETUP_EVENT_STA_TECH_REMOVED;
      break;
      case IEEE80211_MSG_STA_MODE_POWERED_IND:
         smEventID = WIFI_SETUP_EVENT_STA_POWERED;
      break;
      case IEEE80211_MSG_STA_MODE_CONNECTED_IND:
         smEventID = WIFI_SETUP_EVENT_STA_CONNECTED;
      break;
      default:
         LOG_ERROR("sendSTAUpdate2SM: Event %d not handled", eventID);
      break;
   }

   SMManagerFactory* smFactory = SMManagerFactory::getInstance();
   tICmdStateMachineManagerSptr smManagerIf = (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
   if ((smManagerIf) && (WIFI_SETUP_EVENT_UNKNOWN != smEventID))
   {
      bool isError = (errCode != IEEE80211_ERROR_NONE);
      smManagerIf->processEvent(smEventID, staItem, isError);
   }
} //void IEEE80211ConflictManager::sendSTAUpdate2SM()

void IEEE80211ConflictManager::onAPUpdate(const AccessPointItem& apItem, const IEEE80211EventID eventID)
{
   LOG_INFO("IEEE80211ConflictManager::onAPUpdate");

   ::std::string objPath = apItem.objPath;

   if (!objPath.empty())
   {
      auto setupsItr = _wifiSetups.find(objPath);

      if (setupsItr != _wifiSetups.end())
      {
         switch (eventID)
         {
            case IEEE80211_MSG_AP_MODE_AP_POWERED_IND:
               setupsItr->second.property.isPowered = apItem.property.isPowered;
               setupsItr->second.property.connectionStatus.isConnected = apItem.property.isPowered;
               setupsItr->second.property.connectionStatus.healthIndicatorValue = (apItem.property.isPowered)?apItem.property.connHealth:0;
            break;
            case IEEE80211_MSG_AP_MODE_AP_POWERSTATE_IND:
                  setupsItr->second.property.powerState= apItem.property.powerState;
            break;
            case IEEE80211_MSG_AP_MODE_AP_SSID_IND:
                  setupsItr->second.property.apSetup.SSID = apItem.property.ssid;
            break;
            case IEEE80211_MSG_AP_MODE_AP_PASSPHRASE_IND:
                  setupsItr->second.property.apSetup.passphrase = apItem.property.passphrase;
            break;
            case IEEE80211_MSG_AP_MODE_AP_CURRENTCHANNEL_IND:
            {
                  setupsItr->second.property.currChannel = apItem.property.currChannel;
                  auto itr = apItem.property.chanList.find(apItem.property.currChannel);
                  if (itr != apItem.property.chanList.end())
                  {
                     WBLFrequencyBand band = CONVERT_FREQ_TO_FREQBAND(itr->second);
                     setupsItr->second.property.frequency = band;
                  }
            }
            break;
            case IEEE80211_MSG_AP_MODE_AP_SECURITY_IND:
                  setupsItr->second.property.apSetup.security = apItem.property.security;
            break;
            case IEEE80211_MSG_AP_MODE_AP_VENDOR_IES_IND:
                  setupsItr->second.property.apSetup.vendorIEs = apItem.property.vendorIEs;
            break;
            case IEEE80211_MSG_AP_MODE_AP_MACADDRESS_IND:
                  setupsItr->second.property.apSetup.macAddress = apItem.property.macAddress;
            break;
            case IEEE80211_MSG_AP_MODE_AP_SUPPCHANNELS_IND:
                  setupsItr->second.property.apSetup.chanList = apItem.property.chanList;
                  //Update Supported Frequencies
                  supportedFrequencies(setupsItr->second.property.apSetup);
            break;
            case IEEE80211_MSG_AP_MODE_AP_MAXSTA_IND:
                  setupsItr->second.property.apSetup.maxStations = apItem.property.maxStations;
            break;
            case IEEE80211_MSG_AP_MODE_AP_HIDDEN_IND:
                  setupsItr->second.property.apSetup.isVisible = !(apItem.property.isHidden);
            break;
            case IEEE80211_MSG_AP_MODE_AP_INTERFACE_IND:
                  setupsItr->second.property.interface = apItem.property.interface;
            break;
            case IEEE80211_MSG_AP_MODE_AP_TYPE_IND:
                  setupsItr->second.property.apSetup.type = apItem.property.type;
            break;
            case IEEE80211_MSG_AP_MODE_AP_IPV4_CONFIGURATION_IND:
                  setupsItr->second.property.apSetup.ipv4ConfigurationList = apItem.property.ipv4ConfigurationList;
            break;
            case IEEE80211_MSG_AP_MODE_AP_TETHERING_IND:
                  setupsItr->second.property.apSetup.isTethering = apItem.property.isTethering;
            break;
            case IEEE80211_MSG_AP_MODE_AP_HEALTH_IND:
                  setupsItr->second.property.connectionStatus.isConnected = apItem.property.isPowered;
                  setupsItr->second.property.connectionStatus.healthIndicatorValue = (apItem.property.isPowered)?apItem.property.connHealth:0;
            break;
            case IEEE80211_MSG_AP_MODE_AP_ACCESS_NETWORK_TYPE_IND:
                  setupsItr->second.property.apSetup.networkType = apItem.property.wifiAccessNetworkType;
            break;
            default:
                  LOG_ERROR("onAPUpdate: Event %d not handled", eventID);
            break;
         } // switch (eventID)

         //printWifiSetup(setupsItr->second);

         if ((IEEE80211_MSG_AP_MODE_AP_POWERED_IND == eventID) || (IEEE80211_MSG_AP_MODE_AP_POWERSTATE_IND == eventID)
               || (IEEE80211_MSG_AP_MODE_AP_CURRENTCHANNEL_IND == eventID) || (IEEE80211_MSG_AP_MODE_AP_TYPE_IND == eventID))
         {
            WBLServiceFactory* serviceFactory = WBLServiceFactory::getInstance();
            ILastIntendedModeService* limServiceIf =
                  (serviceFactory) ? (serviceFactory->getLastIntendedModeServiceIf()) : (nullptr);
            if (limServiceIf)
            {
               limServiceIf->onSetupUpdated(setupsItr->second);
            }
         }
         notifyActiveWifiSetups();
      }
   }
} //void IEEE80211ConflictManager::onAPUpdate

void IEEE80211ConflictManager::sendAPUpdate2SM(const WifiSetUpItem &apItem, const IEEE80211EventID eventID,
      const IEEE80211ErrorCode errCode)
{
   LOG_INFO("IEEE80211ConflictManager::sendAPUpdate2SM");

   WifiSetupEventID smEventID = WIFI_SETUP_EVENT_UNKNOWN;
   switch (eventID)
   {
      case IEEE80211_MSG_AP_MODE_APADDED_IND:
         smEventID = WIFI_SETUP_EVENT_AP_ADDED;
      break;
      case IEEE80211_MSG_AP_MODE_APREMOVED_IND:
         smEventID = WIFI_SETUP_EVENT_AP_REMOVED;
      break;
      case IEEE80211_MSG_AP_MODE_AP_POWERED_IND:
         smEventID = WIFI_SETUP_EVENT_AP_POWERED;
      break;
      case IEEE80211_MSG_AP_MODE_AP_SSID_IND:
         smEventID = WIFI_SETUP_EVENT_AP_SSID;
      break;
      case IEEE80211_MSG_AP_MODE_AP_PASSPHRASE_IND:
         smEventID = WIFI_SETUP_EVENT_AP_PASSPHRASE;
      break;
      case IEEE80211_MSG_AP_MODE_AP_CURRENTCHANNEL_IND:
         smEventID = WIFI_SETUP_EVENT_AP_CURRENTCHANNEL;
      break;
      case IEEE80211_MSG_AP_MODE_AP_SECURITY_IND:
         smEventID = WIFI_SETUP_EVENT_AP_SECURITY;
      break;
      case IEEE80211_MSG_AP_MODE_AP_VENDOR_IES_IND:
         smEventID = WIFI_SETUP_EVENT_AP_VENDORID;
      break;
      case IEEE80211_MSG_AP_MODE_AP_SUPPCHANNELS_IND:
         smEventID = WIFI_SETUP_EVENT_AP_SUPPCHANNELS;
      break;
      case IEEE80211_MSG_AP_MODE_AP_HIDDEN_IND:
         smEventID = WIFI_SETUP_EVENT_AP_HIDDEN;
      break;
      case IEEE80211_MSG_AP_MODE_ASADDED_IND:
         smEventID = WIFI_SETUP_EVENT_AS_ADDED;
      break;
      case IEEE80211_MSG_AP_MODE_ASREMOVED_IND:
         smEventID = WIFI_SETUP_EVENT_AS_REMOVED;
      break;
      case IEEE80211_MSG_AP_MODE_AP_SETOPERATINGCHANNEL:
         smEventID = WIFI_SETUP_EVENT_AP_SETOPERATINGCHANNEL;
      break;
      case IEEE80211_MSG_AP_MODE_AP_DEAUTHENTICATESTATION:
         smEventID = WIFI_SETUP_EVENT_AP_DEAUTHSTATION;
      break;
      case IEEE80211_MSG_AP_MODE_CLEAR_DATA_RESP:
         smEventID = WIFI_SETUP_EVENT_AP_CLEAR_DATA;
      break;
      case IEEE80211_MSG_AP_MODE_AP_TETHERING_IND:
         smEventID = WIFI_SETUP_EVENT_AP_TETHERING;
      break;
      case IEEE80211_MSG_AP_MODE_AP_ACCESS_NETWORK_TYPE_IND:
         smEventID = WIFI_SETUP_EVENT_AP_NETWORK_TYPE;
      break;
      case IEEE80211_MSG_AP_MODE_AP_MAXSTA_IND:
         smEventID = WIFI_SETUP_EVENT_AP_MAXSTA;
      break;
      case IEEE80211_MSG_AP_MODE_AP_REG_VENDOR_IND:
         smEventID = WIFI_SETUP_EVENT_AP_REG_VENDOR;
      break;
      case IEEE80211_MSG_AP_MODE_AP_UNREG_VENDOR_IND:
         smEventID = WIFI_SETUP_EVENT_AP_UNREG_VENDOR;
      break;
      /*case IEEE80211_MSG_AP_MODE_AP_POWERFAILURE_IND:
         smEventID = WIFI_SETUP_EVENT_AP_POWERFAILURE;
      break;*/
      default:
         LOG_ERROR("sendAPUpdate2SM: Event %d not handled", eventID);
      break;
   } // switch (eventID)

   SMManagerFactory* smFactory = SMManagerFactory::getInstance();
   tICmdStateMachineManagerSptr smManagerIf = (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
   if ((smManagerIf) && (WIFI_SETUP_EVENT_UNKNOWN != smEventID))
   {
      bool isError = (errCode != IEEE80211_ERROR_NONE);
      smManagerIf->processEvent(smEventID, apItem , isError);
   }
} //void IEEE80211ConflictManager::sendAPUpdate2SM

void IEEE80211ConflictManager::addAssociatedStation(const AssociatedStationItem& asItem)
{
   LOG_INFO("IEEE80211ConflictManager::addAssociatedStation");
   if (false == isAssociatedStationAvailable(asItem.objPath))
   {
      _associatedStations[asItem.objPath].objPath = asItem.objPath;
      _associatedStations[asItem.objPath].property.macAddress = asItem.property.macAddress;
      _associatedStations[asItem.objPath].property.apPath = asItem.property.apPath;
      _associatedStations[asItem.objPath].property.interface = asItem.property.interface;
      _associatedStations[asItem.objPath].property.ipAddress = asItem.property.ipAddress;
      _associatedStations[asItem.objPath].property.hostName = asItem.property.hostName;

      addStationtoSetups(asItem);
   }
} //void IEEE80211ConflictManager:: addAssociatedStation

void IEEE80211ConflictManager::removeAssociatedStation(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211ConflictManager:removeAssociatedStation: ObjPath = %s", objPath.c_str());

   AssociatedStationItem asItem;
   if (getAssociatedStationItem(objPath, asItem))
   {
      removeStationFromSetUps(asItem);
      _associatedStations.erase(objPath);
   }
} //void IEEE80211ConflictManager:: removeAssociatedStation

void IEEE80211ConflictManager::onASUpdate(const AssociatedStationItem& asItem, const IEEE80211EventID eventID)
{
   LOG_INFO("IEEE80211ConflictManager::onASUpdate");

   std::string objPath = asItem.objPath;
   if (!objPath.empty())
   {
      auto asItr = _associatedStations.find(objPath);
      if (asItr != _associatedStations.end())
      {
         switch (eventID)
         {
            case IEEE80211_MSG_AP_MODE_ASCHANGED_IND:
               (*asItr).second.property = asItem.property;
            break;
            case IEEE80211_MSG_AP_MODE_AS_MACADDRESS_IND:
               (*asItr).second.property.macAddress = asItem.property.macAddress;
            break;
            case IEEE80211_MSG_AP_MODE_AS_APPATH_IND:
               (*asItr).second.property.apPath = asItem.property.apPath;
            break;
            case IEEE80211_MSG_AP_MODE_AS_INTERFACE_IND:
               (*asItr).second.property.interface = asItem.property.interface;
            break;
            case IEEE80211_MSG_AP_MODE_AS_IPADDRESS_IND:
               (*asItr).second.property.ipAddress = asItem.property.ipAddress;
            break;
            case IEEE80211_MSG_AP_MODE_AS_HOSTNAME_IND:
               (*asItr).second.property.hostName = asItem.property.hostName;
            break;
            default:
               LOG_ERROR("onASUpdate: Event %d not handled", eventID);
            break;
         } // switch (eventID)
         //addStationtoSetups(asItem);
         printAssociatedStation(asItr->second);
         notifyActiveWifiSetups();
      }
   }
} //void IEEE80211ConflictManager::onASUpdate

void IEEE80211ConflictManager::addStationtoSetups(const AssociatedStationItem& asItem)
{
   LOG_INFO("IEEE80211ConflictManager::addStationtoSetups");

   std::string apObjPath = asItem.property.apPath;
   auto setupsItr = _wifiSetups.find(apObjPath);
   if (setupsItr != _wifiSetups.end())
   {
      LOG_DEBUG("addStationtoSetups: AP found with obj path %s", apObjPath.c_str());
      (*setupsItr).second.property.apSetup.stationList.push_back(asItem);

      //printWifiSetup((*setupsItr).second);
      notifyActiveWifiSetups();
   }
} //void IEEE80211ConflictManager::addStationtoSetups

void IEEE80211ConflictManager::removeStationFromSetUps(const AssociatedStationItem& asItem)
{
   LOG_INFO("IEEE80211ConflictManager::removeStationFromSetUps");

   std::string apObjPath = asItem.property.apPath;
   auto setupsItr = _wifiSetups.find(apObjPath);
   if (setupsItr != _wifiSetups.end())
   {
      LOG_DEBUG("removeStationFromSetUps: AP found with ObjPath = %s", apObjPath.c_str());

      WifiSetUpItem& item = setupsItr->second;
      AssociatedStationList& asList = item.property.apSetup.stationList;

      for (auto itr = asList.begin(); itr != asList.end(); ++itr)
      {
         if (itr->objPath == asItem.objPath)
         {
            LOG_DEBUG("Station found with obj path %s", asItem.objPath.c_str());
            asList.erase(itr);
            //item.property.apSetup.stationList.erase(itr);
            break;
         }
      }
      notifyActiveWifiSetups();
   }
} //void IEEE80211ConflictManager::removeStationFromSetUps

void IEEE80211ConflictManager::addWifiSetup(const AccessPointItem& apItem)
{
   LOG_INFO("IEEE80211ConflictManager::addWifiSetup:AP");

   std::string interface = apItem.property.interface;
   if (!interface.empty())
   {
      //! If AP is new, add setup and configure to default values
      if (false == isWifiSetupAvailable(apItem.objPath))
      {
         LOG_DEBUG("addWifiSetup: Adding new AP setup");

         //TODO - decide AP1 or AP2 based on interface?
         _wifiSetups[apItem.objPath].property.mode = WIFI_MODE_AP1;
         _wifiSetups[apItem.objPath].nativeObjPath = apItem.objPath;
         _wifiSetups[apItem.objPath].setupObjPath = createWifiSetupObjectPath(interface);
         _wifiSetups[apItem.objPath].property.interface = interface;
         _wifiSetups[apItem.objPath].property.powerState = apItem.property.powerState;
         _wifiSetups[apItem.objPath].property.isPowered = apItem.property.isPowered;
         _wifiSetups[apItem.objPath].property.powerFailure = apItem.property.powerFailure;
         _wifiSetups[apItem.objPath].property.currChannel = apItem.property.currChannel;
         _wifiSetups[apItem.objPath].property.apSetup.SSID = apItem.property.ssid;
         _wifiSetups[apItem.objPath].property.apSetup.passphrase = apItem.property.passphrase;
         _wifiSetups[apItem.objPath].property.apSetup.security = apItem.property.security;
         _wifiSetups[apItem.objPath].property.apSetup.stationList = apItem.property.asList;
         _wifiSetups[apItem.objPath].property.apSetup.isVisible = (false == apItem.property.isHidden);
         _wifiSetups[apItem.objPath].property.apSetup.macAddress = apItem.property.macAddress;
         _wifiSetups[apItem.objPath].property.apSetup.chanList = apItem.property.chanList;
         _wifiSetups[apItem.objPath].property.apSetup.maxStations = apItem.property.maxStations;
         _wifiSetups[apItem.objPath].property.apSetup.vendorIEs = apItem.property.vendorIEs;
         _wifiSetups[apItem.objPath].property.apSetup.ipv4ConfigurationList = apItem.property.ipv4ConfigurationList;
         _wifiSetups[apItem.objPath].property.apSetup.isTethering = apItem.property.isTethering;
         _wifiSetups[apItem.objPath].property.apSetup.networkType = apItem.property.wifiAccessNetworkType;

         auto itr = apItem.property.chanList.find(apItem.property.currChannel);
         if (itr != apItem.property.chanList.end())
         {
            WBLFrequencyBand band = CONVERT_FREQ_TO_FREQBAND(itr->second);
            _wifiSetups[apItem.objPath].property.frequency = band;
         }

         //_wifiSetups[apItem.objPath].property.apSetup.type = apItem.property.type;
         updateLastRequestedAPType(apItem);
         supportedFrequencies(_wifiSetups[apItem.objPath].property.apSetup);
         printWifiSetup(_wifiSetups[apItem.objPath]);
         notifyActiveWifiSetups();

      } // if (false == isWifiSetupAvailable(apItem.objPath))
   }
} // void IEEE80211ConflictManager::addWifiSetup

void IEEE80211ConflictManager::removeWifiSetup(const ::std::string& objPath)
{
   LOG_INFO("IEEE80211ConflictManager::removeWifiSetup: ObjPath: %s", objPath.c_str());

   _wifiSetups.erase(objPath);
   notifyActiveWifiSetups();

} // void IEEE80211ConflictManager::removeWifiSetup

bool IEEE80211ConflictManager::isWifiSetupAvailable(const ::std::string& objPath)
{
   bool isSetupFound = (_wifiSetups.end() != _wifiSetups.find(objPath));
   LOG_INFO("IEEE80211ConflictManager::isWifiSetupAvailable: %s", BOOL_TO_STR(isSetupFound));
   return isSetupFound;
} // bool IEEE80211ConflictManager::isWifiSetupAvailable

bool IEEE80211ConflictManager::getWifiSetupItem(const ::std::string& objPath, WifiSetUpItem& item)
{
   bool isItemFound = false;
   auto setupsItr = _wifiSetups.find(objPath);
   if (setupsItr != _wifiSetups.end())
   {
      isItemFound = true;
      item = (*setupsItr).second;
   }
   LOG_DEBUG("IEEE80211ConflictManager::getWifiSeupItem: isItemFound = %s", BOOL_TO_STR(isItemFound));
   return isItemFound;
} // void IEEE80211ConflictManager::getWifiSetupItem

bool IEEE80211ConflictManager::getAssociatedStationItem(const ::std::string& objPath, AssociatedStationItem &item)
{
   bool isItemFound = false;
   auto setupsItr = _associatedStations.find(objPath);
   if (setupsItr != _associatedStations.end())
   {
      isItemFound = true;
      item = (*setupsItr).second;
   }
   LOG_DEBUG("IEEE80211ConflictManager::getAssociatedStationItem: isItemFound = %s", BOOL_TO_STR(isItemFound));
   return isItemFound;
} // void IEEE80211ConflictManager::getAsociatedStationItem

bool IEEE80211ConflictManager::isAssociatedStationAvailable(const ::std::string& objPath)
{
   bool isStationFound = (_associatedStations.end() != _associatedStations.find(objPath));
   LOG_DEBUG("IEEE80211ConflictManager::isAssociatedStationAvailable: %s (AS ObjPath = %s)",
         BOOL_TO_STR(isStationFound), objPath.c_str());
   return isStationFound;
} // bool IEEE80211ConflictManager::isAssociatedStationAvailable

void IEEE80211ConflictManager::printWifiSetup(const WifiSetUpItem& setup) const
{
    std::string ssid = convertByteArray2String(setup.property.apSetup.SSID);

    LOG_DEBUG("ConflictManager:printWifiSetup: wifiMode = %s ",
            WIFI_MODE_TO_STR(setup.property.mode));
    LOG_DEBUG("ConflictManager:printWifiSetup: nativeObjPath = %s ",
            setup.nativeObjPath.c_str());
    LOG_DEBUG("ConflictManager:printWifiSetup: setupObjPath = %s ",
            setup.setupObjPath.c_str());
    LOG_DEBUG("ConflictManager:printWifiSetup: interface = %s ",
            setup.property.interface.c_str());
    LOG_DEBUG("ConflictManager:printWifiSetup: currChannel = %d ",
            setup.property.currChannel);
    LOG_DEBUG("ConflictManager:printWifiSetup: frequency = %s ",
            FREQ_TO_STR(setup.property.frequency));
    LOG_DEBUG("ConflictManager:printWifiSetup: isPowered = %d ",
            setup.property.isPowered);
    LOG_DEBUG("ConflictManager:printWifiSetup: PowerState = %s ",
            POWER_STATE_TO_STR(setup.property.powerState));
    LOG_DEBUG("ConflictManager:printWifiSetup: PowerOffType = %s ",
            POWEROFFTYPE_TO_STR(setup.property.powerOffType));

    if ((WIFI_MODE_AP1 == setup.property.mode)
            || (WIFI_MODE_AP2 == setup.property.mode))
    {
        LOG_DEBUG("ConflictManager:printWifiSetup: macAddress = %s ",
                setup.property.apSetup.macAddress.c_str());
        LOG_DEBUG("ConflictManager:printWifiSetup: ssid = %s ", ssid.c_str());
        LOG_DEBUG("ConflictManager:printWifiSetup: passphrase = %s ",
                setup.property.apSetup.passphrase.c_str());
        LOG_DEBUG("ConflictManager:printWifiSetup: maxStations = %d ",
                setup.property.apSetup.maxStations);
        LOG_DEBUG("ConflictManager:printWifiSetup: security = %s ",
                SECURITY_TO_STR(setup.property.apSetup.security));
        LOG_DEBUG("ConflictManager:printWifiSetup: isVisible = %d ",
                setup.property.apSetup.isVisible);
        LOG_DEBUG("ConflictManager:printWifiSetup: apType = %s ",
                AP_TYPE_TO_STR(setup.property.apSetup.type));
        LOG_DEBUG("ConflictManager:printWifiSetup: isTethering = %s ",
                BOOL_TO_STR(setup.property.apSetup.isTethering));
        LOG_DEBUG("ConflictManager:printWifiSetup: Networktype = %d ", 
                setup.property.apSetup.networkType);
        if (!setup.property.apSetup.stationList.empty())
        {
            LOG_DEBUG("Station list not empty");
            for (auto iter = setup.property.apSetup.stationList.begin();
                    iter != setup.property.apSetup.stationList.end(); iter++)
            {
                printAssociatedStation(*iter);
            }
        }
        if (!setup.property.apSetup.ipv4ConfigurationList.empty())
        {
            for (auto iter =
                    setup.property.apSetup.ipv4ConfigurationList.begin();
                    iter != setup.property.apSetup.ipv4ConfigurationList.end();
                    iter++)
            {
                IPv4Configuration ipconfig = *iter;
                LOG_DEBUG("ConflictManager:printWifiSetup: IPV4Configuration  Network = %s ",
                        ipconfig.network.c_str());
                LOG_DEBUG("ConflictManager:printWifiSetup: IPV4Configuration Netmask = %s ",
                        ipconfig.netMask.c_str());
            }
        }
        if (!setup.property.apSetup.supportedFrequencies.empty())
        {
            for (auto iter =
                    setup.property.apSetup.supportedFrequencies.begin();
                    iter != setup.property.apSetup.supportedFrequencies.end();
                    iter++)
            {
                LOG_DEBUG("ConflictManager:printWifiSetup:SupportedFrequency:%s CPW:%s AAW:%s",
                        FREQ_TO_STR(iter->first),
                        BOOL_TO_STR(iter->second.isCPWSupport),
                        BOOL_TO_STR(iter->second.isAAWSupport));
            }
        }
    }
} // void IEEE80211APClient::printWifiSetup

void IEEE80211ConflictManager::printAssociatedStation(const AssociatedStationItem& station) const
{
   LOG_DEBUG("ConflictManager:printAssociatedStation:objPath = %s",station.objPath.c_str());
   LOG_DEBUG("ConflictManager:printAssociatedStation:macAddress = %s",station.property.macAddress.c_str());
   LOG_DEBUG("ConflictManager:printAssociatedStation:apPath = %s",station.property.apPath.c_str());
   LOG_DEBUG("ConflictManager:printAssociatedStation:interface = %s",station.property.interface.c_str());
   LOG_DEBUG("ConflictManager:printAssociatedStation:ipAddress = %s",station.property.ipAddress.c_str());
   LOG_DEBUG("ConflictManager:printAssociatedStation:hostName = %s",station.property.hostName.c_str());
   
} //void IEEE80211ConflictManager::printAssociatedStation

::std::string IEEE80211ConflictManager::createWifiSetupObjectPath(::std::string& interface) const
{
   ::std::string wifiSetupObjPath = "";
   if (!interface.empty())
   {
      //TODO revise logic - maybe give Wifi mode as a param to this function to decide "AP" or "STA"
      if(interface.compare("wlan1") == 0)
      {
         wifiSetupObjPath = "/org/bosch/wbl/" + interface + "/wifisetup/AP";
      }
      else if(interface.compare("wlan0") == 0)
      {
         wifiSetupObjPath = "/org/bosch/wbl/" + interface + "/wifisetup/STA";
      }
   }
   LOG_DEBUG("IEEE80211ConflictManager::createWifiSetupObjectPath: %s ", wifiSetupObjPath.c_str());
   return wifiSetupObjPath;
} // ::std::string IEEE80211ConflictManager::createWifiSetupObjectPath

bool IEEE80211ConflictManager::isCPChannel(const uint16& channel, const WBLFrequencyBand& frequency)
{
   bool isCarplayChannel = false;
   LOG_INFO("IEEE80211ConflictManager::isCPChannel");
   if(WBL_FREQ_BAND_5_GHZ == frequency)
   {
      auto iter = find (cpChannels_5.begin(), cpChannels_5.end(),channel);
      if(iter != cpChannels_5.end())
         isCarplayChannel = true;
   }
   else if(WBL_FREQ_BAND_2_4_GHZ == frequency)
   {
      auto iter = find (cpChannels_2_4.begin(), cpChannels_2_4.end(),channel);
           if(iter != cpChannels_2_4.end())
              isCarplayChannel = true;
   }
   return isCarplayChannel;
} //bool IEEE80211ConflictManager::isCPChannel 

bool IEEE80211ConflictManager::getWiFiSetUps(WifiSetUpList& wifiSetupList)const
{
   bool isValid = false;
   if(!_wifiSetups.empty())
   {
      isValid = true;
      for(WifiSetupsListConstItr setupsListItr = _wifiSetups.begin(); setupsListItr != _wifiSetups.end(); ++setupsListItr)
      {
         wifiSetupList.push_back(setupsListItr->second);
      }
   }
   LOG_DEBUG("getWiFiSetUps:isValid: %s",BOOL_TO_STR(isValid));
   return isValid;
} //bool IEEE80211ConflictManager::getWiFiSetUps

LCMWiFiState IEEE80211ConflictManager::getWifiState()
{
   LCMWiFiState currState = LCM_WIFISTATE_UNKNOWN;
   LCMFactory* lcmFactory = LCMFactory::getInstance();
   ILCMClient* poLCMClient = (lcmFactory) ? (lcmFactory->getLCMClientIf()) : (nullptr);
   if(poLCMClient)
   {
      currState = poLCMClient->getCurrentWiFiState();
   }
   LOG_DEBUG("getWifiState:currentState: %s",WIFISTATE_TO_STR(currState));
   return currState;
} //LCMWiFiState IEEE80211ConflictManager::getWifiState

bool IEEE80211ConflictManager::checkSupportedFrequency(APSetup &apSetup,WBLFrequencyBand frequencyBand)
{
   LOG_INFO("WifiSetupService::checkSupportedFrequency");
   bool bRet = false;
   for(auto it = apSetup.chanList.begin(); it != apSetup.chanList.end(); it++ )
   {
      if(it->second >= frequencyBand)
      {
         bRet = true;
         break;
      }
   }
   return bRet;
}//bool IEEE80211ConflictManager::checkSupportedFrequency

void IEEE80211ConflictManager::supportedFrequencies(APSetup &apSetup)
{
    LOG_INFO("IEEE80211ConflictManager::%s",__func__);


    apSetup.supportedFrequencies.clear();
    //Check for CPW support
    if (checkSupportedFrequency(apSetup, WBL_FREQ_BAND_2_4_GHZ))
    {
        SPITechnologies tech;
        //Check for AAW support
        //Find 2.4GHz CPW Channel in Supported Channel list.
        for (auto itCPWChannel = cpChannels_2_4.begin();
                itCPWChannel != cpChannels_2_4.end(); itCPWChannel++)
        {
            auto itSuppChannels = apSetup.chanList.find(*itCPWChannel);
            if (itSuppChannels != apSetup.chanList.end())
            {
                tech.isCPWSupport = TRUE;
                LOG_DEBUG("IEEE80211ConflictManager::%s CPW 2.4GHz Channel:%d",__func__,*itCPWChannel);
                break;
            }
        }
        //Check for AAW support
        tech.isAAWSupport = TRUE;
        apSetup.supportedFrequencies.insert(std::pair<WBLFrequencyBand, SPITechnologies>(WBL_FREQ_BAND_2_4_GHZ, tech));
    }
    if (checkSupportedFrequency(apSetup, WBL_FREQ_BAND_5_GHZ))
    {
        SPITechnologies tech;
        //Find 5GHz CPW Channel in Supported Channel.
        for (auto itCPWChannel = cpChannels_5.begin();
                itCPWChannel != cpChannels_5.end(); itCPWChannel++)
        {
            auto itSuppChannels = apSetup.chanList.find(*itCPWChannel);
            if (itSuppChannels != apSetup.chanList.end())
            {
                LOG_DEBUG("IEEE80211ConflictManager::%s CPW 5GHz Channel:%d",__func__,*itCPWChannel);
                tech.isCPWSupport = true;
                break;
            }
        }
        //Check for AAW support
        tech.isAAWSupport = true;
        apSetup.supportedFrequencies.insert(std::pair<WBLFrequencyBand, SPITechnologies>(WBL_FREQ_BAND_5_GHZ,tech));
    }

}//bool IEEE80211ConflictManager::supportedFrequencies

void IEEE80211ConflictManager::updateLastRequestedAPType(const AccessPointItem& apItem)
{
   LOG_INFO("IEEE80211ConflictManager::%s", __func__);
   WBLAPType apType = WBL_AP_TYPE_UNKNOWN;
   bool isRestricted = false;
   APSetup apSetup;
   DBManagerFactory* dbmgrFactory = DBManagerFactory::getInstance();
   IDBManager* dbmgrIf = (dbmgrFactory) ? (dbmgrFactory->getDBManagerIf()) : (nullptr);
   if (dbmgrIf)
   {
      if(dbmgrIf->readAPConfigsFromDB(apSetup,isRestricted))
      {
          (void)isRestricted; //Ignore restricted
          IEEE80211APClientIf* apClientIf =
                IEEE80211ClientFactory::getInstance()->getAPClientIf();
          if (apClientIf)
          {
            apType = apSetup.type;
            apClientIf->setAPType(apItem.objPath,apType);
          }
      }
   }
}//bool IEEE80211ConflictManager::updateLastRequestedAPType

void IEEE80211ConflictManager::updateRequestType(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("IEEE80211ConflictManager::%s", __func__);

   ::boost::shared_ptr<DeActivateSetupMsg> deactivateSetupMsg = static_pointer_cast<DeActivateSetupMsg>(msg);
   WifiMode mode = deactivateSetupMsg->getWifiMode();
   WBLPowerOffReason requestType = deactivateSetupMsg->getRequestType();

   for(auto it= _wifiSetups.begin();it != _wifiSetups.end(); it++)
   {
      if(it->second.property.mode == mode)
      {
         it->second.property.powerOffType = requestType;
      }
   }

} //bool IEEE80211ConflictManager::updateRequestType

} // namespace bosch
} // namespace org

/** @} */
