/**
 * @file ConflictResolver.cpp
 * @author RBEI/ECO32 Jiji Anna Jacob
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief Detects and resolves conflicts of the requested
 * setup with the current setup and forwards requests to
 * the SM for further action.
 *
 * @{
 */

#include <string>

#include "ConflictResolver.h"
#include "WBLASFComponent.h"
#include "WBLTypeProperties.h"
#include "WBLHelper.h"
#include "WBLUtils.h"

namespace org
{
namespace bosch
{

using boost::static_pointer_cast;

#define CONFLICT_TOKEN_START_RANGE 	100
#define CONFLICT_TOKEN_END_RANGE 	999

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/ConflictMgr", ConflictResolver, Debug);

ConflictResolver::ConflictResolver()
{
   LOG_INFO("ConflictResolver:Constructor");
   _conflictTimerDuration = readConflictTimerDuration();
   _poConflictItem = new ConflictDetectedItem();

} //ConflictResolver::ConflictResolver

ConflictResolver::~ConflictResolver()
{
   try
   {
      LOG_INFO("ConflictResolver:Destructor");
      delete _poConflictItem;
   }catch(...){}
} //ConflictResolver::~ConflictResolve

uint32 ConflictResolver::readConflictTimerDuration()const
{
   LOG_INFO("ConflictResolver:readConflictTimerDuration");
   uint32 timerDur = 0;

   WBLPriorityManagerFactory* prioMgrFactory = WBLPriorityManagerFactory::getInstance();
   IWBLPriorityManager* prioMgrIf = (prioMgrFactory) ? (prioMgrFactory->getWBLPriorityManagerIf()) : (nullptr);
   if (prioMgrIf)
   {
      timerDur = prioMgrIf->getConflictTimer();
   }
   return timerDur;
} //uint32 ConflictResolver::readConflictTimerDuration

void ConflictResolver::readWifiPriorityList(PriorityList& list)const
{
   LOG_INFO("ConflictResolver:readWifiPriorityList");

   WBLPriorityManagerFactory* prioMgrFactory = WBLPriorityManagerFactory::getInstance();
   IWBLPriorityManager* prioMgrIf = (prioMgrFactory) ? (prioMgrFactory->getWBLPriorityManagerIf()) : (nullptr);
   if (prioMgrIf)
   {
      prioMgrIf->getWifiPriorityList(list);
   }
} //void ConflictResolver::readWifiPriorityList

void ConflictResolver::resolveWBLServiceRequest(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("ConflictResolver:resolveWBLServiceRequest");
   WBL_ASSERT_AND_EXIT(!msg);

   WBLBaseMsgID msgID = msg->getMsgID();

   switch (msgID)
   {
      case WBL_MSG_PREPARE_SETUP_REQ:
      {
         if (true == _conflictTimer.isActive())
         {
            resolveConflictOnTimerExpiry();
         }

         ::boost::shared_ptr < PrepareSetupMsg > prepareSetupMsg = static_pointer_cast < PrepareSetupMsg >(msg);
         _prepareSetupMsg = prepareSetupMsg;
         onPrepareSetupRequest(prepareSetupMsg);
      }
      break;
      case WBL_MSG_DEACTIVATE_SETUP_REQ:
      {
         ::boost::shared_ptr < DeActivateSetupMsg > deactivateSetupMsg = static_pointer_cast < DeActivateSetupMsg >(
               msg);
         _deactSetupMsg = deactivateSetupMsg;
         onDeactivateSetupRequest(deactivateSetupMsg);
      }
      break;
      case WBL_MSG_RESOLVE_CONFLICT_REQ:
      {
         ::boost::shared_ptr < ResolveConflictMsg > resolveConflictMsg = static_pointer_cast < ResolveConflictMsg >(
               msg);
         _resolveConflictMsg = resolveConflictMsg;
         onResolveConflictRequest(resolveConflictMsg);
      }
      break;
      default:
         LOG_ERROR("Unhandled service request");
         break;
   }
} // void ConflictResolver::resolveWBLServiceRequest

void ConflictResolver::notifyActiveWiFiSetups(const WifiSetUpList& wifisetups)
{
   LOG_INFO("ConflictResolver:notifyActiveWiFiSetups");

   if(wifisetups.empty())
      return;

   _wifiSetups = wifisetups;
   if(true == _conflictTimer.isActive())
   {
      bool isResolved = isConflictResolvedOnSetUpsUpdate();
      if(isResolved)
      {
         LOG_DEBUG("Conflict Resolved on update setups");

         ConflictResolvedItem item;

         item.conflictID = _poConflictItem->conflictID;
         item.conflictType = _poConflictItem->conflictType;
         item.resType = WBL_RES_SETUP_REJECTED_PRIO;

         postConflictResolved(item);
      }
   }
} // void ConflictResolver::updateActiveWiFiSetups

bool ConflictResolver::isConflictResolvedOnSetUpsUpdate()
{
   LOG_INFO("ConflictResolver:isConflictResolvedOnSetUpsUpdate");
   bool isResolved = false;

   if(_poConflictItem)
   {
      if(_poConflictItem->conflictType == WBL_CONFLICT_REQ_STA_UNAVAILABLE)
      {
         WifiSetUpItem apItem;
         if(true == isAPSetupAvailable(apItem))
         {
            uint16 connStaCount = static_cast<uint16>(apItem.property.apSetup.stationList.size());
            uint16 reserveStations = _poConflictItem->numStaToFree;
            auto freeStations = apItem.property.apSetup.maxStations - connStaCount;

            if((reserveStations > 0)&&(freeStations >= reserveStations))
            {
               LOG_INFO("Conflict Resolved");
            }
         }
      }

      else if(_poConflictItem->conflictType == WBL_CONFLICT_SIMULTANEOUS_OP_NOTALLOWED)
      {

      }

      else if(_poConflictItem->conflictType == WBL_CONFLICT_AP_CONFIG_MISMATCH)
      {

      }
   }
   return isResolved;
} //bool ConflictResolver::isConflictResolvedOnSetUpsUpdate

void ConflictResolver::onExpired(Timer& timer, boost::shared_ptr < TimerPayload > data)
{
   LOG_INFO("ConflictResolver:onExpired");
   WBL_INTENTIONALLY_UNUSED(data);

   if (&timer == &_conflictTimer)
   {
      LOG_DEBUG("ConflictTimer  %d", timer.getAct());
      resolveConflictOnTimerExpiry();
   }

} //void ConflictResolver::onExpired

void ConflictResolver::onPrepareSetupRequest(const ::boost::shared_ptr < PrepareSetupMsg > msg)
{
   LOG_INFO("ConflictResolver:onPrepareSetupRequest");
   WBL_ASSERT_AND_EXIT(!msg);

   WifiMode requestMode = msg->getWifiMode();
   WBLActionType requestAction = msg->getWifiAction();
   _prepareSetupMsg = msg;

   if (requestMode == WIFI_MODE_STA)
   {
      processSTARequestConflicts();
   }
   else if ((requestMode == WIFI_MODE_AP1)||(requestMode == WIFI_MODE_AP2))
   {
      APConfig requestedAPConfig = msg->getAPConfigItemParams();
      processAPRequestConflicts(requestedAPConfig,requestAction);
   }
} //void ConflictResolver::onPrepareSetupRequest

void ConflictResolver::printAPConfig(const APConfig& apConfig)
{
   LOG_DEBUG("ConflictResolver:printAPConfig");
   LOG_DEBUG("printAPConfig APType = %s",AP_TYPE_TO_STR(apConfig.type));
   ::std::string ssid = convertByteArray2String(apConfig.ssid.SSID);
   LOG_DEBUG("printAPConfig SSID = %s",ssid.c_str());
   LOG_DEBUG("printAPConfig passphrase = %s",apConfig.passphrase.c_str());
   LOG_DEBUG("printAPConfig security = %s",SECURITY_TO_STR(apConfig.security));
   LOG_DEBUG("printAPConfig isVisible = %d",apConfig.isVisible);
   for(auto itr = apConfig.chanList.begin(); itr != apConfig.chanList.end(); itr++)
   {
      LOG_DEBUG("printAPConfig chanList:channel = %d",*itr);
   }
   LOG_DEBUG("printAPConfig stationsToReserve = %d",apConfig.stationsToReserve);

}//void ConflictResolver::printAPConfig

void ConflictResolver::onDeactivateSetupRequest(const ::boost::shared_ptr < DeActivateSetupMsg > msg)
{
   LOG_INFO("ConflictResolver:onDeactivateSetupRequest");
   WBL_ASSERT_AND_EXIT(!msg);

   const WifiMode requestMode = msg->getWifiMode();
   _deactSetupMsg = msg;

   if (true == _conflictTimer.isActive())
   {
      if ((_poConflictItem) && (requestMode == _poConflictItem->reqConfig.mode))
      {
         LOG_DEBUG("Deactivate received for ongoing conflict,cancel");
         _conflictTimer.stop();

         if(_poConflictItem)
         {
            _poConflictItem = nullptr;
         }
      }
      _deactSetupMsg = nullptr;
   }
} //void ConflictResolver::onDeactivateSetupRequest

void ConflictResolver::onResolveConflictRequest(const ::boost::shared_ptr < ResolveConflictMsg > msg)
{
   LOG_INFO("ConflictResolver:onResolveConflictRequest");
   WBL_ASSERT_AND_EXIT(!msg);

   if (_poConflictItem)
   {
      LOG_INFO("ConflictResolver:onResolveConflictRequest ConflictID: %d / "
            "ObtainedID: %d",_poConflictItem->conflictID,msg->getConflictId());

      if (true == _conflictTimer.isActive() && _poConflictItem->conflictID == msg->getConflictId())
      {
         _conflictTimer.stop();
         resolveConflictOnUserRequest(msg->getAction());
         msg->setErrorCode(WBL_ERR_NOERROR);
      }
      else if(_poConflictItem->conflictID != msg->getConflictId())
      {
         LOG_DEBUG("Conflict ID mismatch");
         msg->setErrorCode(WBL_ERR_INVALID_ARGS);
      }
   }
   else
   {
      msg->setErrorCode(WBL_ERR_NOTSUPPORTED);
   }
   sendWBLResponseMsg(msg);
} //void ConflictResolver::onResolveConflictRequest

void ConflictResolver::processSTARequestConflicts()
{
   LOG_INFO("ConflictResolver:processSTARequestConflicts");
   WifiSetUpItem apItem;

   if (false == isAPSetupAvailable(apItem))
   {
      LOG_DEBUG("ConflictResolver:processSTARequestConflicts AP is NOP,proceed to setup");
      sendSMRequestMsg(_prepareSetupMsg);
      _prepareSetupMsg = nullptr;
   }
   else
   {
      if ((apItem.property.frequency == WBL_FREQ_BAND_2_4_GHZ || apItem.property.frequency == WBL_FREQ_BAND_5_GHZ)
            && apItem.property.isPowered)
      {
         LOG_DEBUG("ConflictResolver:processSTARequestConflicts AP is available and operating conflict detected");

         RequestTypeItem reqConfig;
         RequestTypeItem currConfig;
         WBLConflictType conflictType;

         conflictType = WBL_CONFLICT_SIMULTANEOUS_OP_NOTALLOWED;

         reqConfig.mode = WIFI_MODE_STA;

         currConfig.mode = apItem.property.mode;
         currConfig.frequency = apItem.property.frequency;

         onConflictDetected(conflictType, reqConfig, currConfig);
      }
      else
      {
         LOG_DEBUG("ConflictResolver:processSTARequestConflicts AP is available,operating in Unknown Frequency");
         sendSMRequestMsg(_prepareSetupMsg);
         _prepareSetupMsg = nullptr;
      }
   }
} //void ConflictResolver::processSTARequestConflicts

void ConflictResolver::processAPRequestConflicts(const APConfig& requestedAPConfig ,const WBLActionType& action)
{
   LOG_INFO("ConflictResolver:processAPRequestConflicts");
   WifiSetUpItem apItem;

   if(isAPSetupAvailable(apItem))
   {
      if((action == WIFI_ACTION_ACTIVATE))
      {
         //Conflict considered same for CPW & AAW
         if ((requestedAPConfig.isTypeValid) &&
               ((WBL_AP_TYPE_CARPLAY == requestedAPConfig.type) ||
                     (WBL_AP_TYPE_AAW == requestedAPConfig.type)))
         {
            checkSPIConflicts(requestedAPConfig,apItem);
         }
         else
         {
            checkAPRequestConflicts(requestedAPConfig,apItem);
         }
      }
      else
      {
         LOG_DEBUG("AP is NOP,proceed to setup");
         sendSMRequestMsg(_prepareSetupMsg);
         _prepareSetupMsg = nullptr;
      }
   }
   else
   {
      LOG_ERROR("AP Setup Item is not available");
      _prepareSetupMsg->setErrorCode(WBL_ERR_NOTSUPPORTED);
      sendWBLResponseMsg(_prepareSetupMsg);
      _prepareSetupMsg = nullptr;
   }



} //void ConflictResolver::processAPRequestConflicts
void ConflictResolver::checkSPIConflicts(const APConfig& requestedAPConfig , const WifiSetUpItem& apItem)
{
   LOG_INFO("ConflictResolver:checkSPIConflicts");
   WBLConflictType conflictType;
   RequestTypeItem reqConfig;
   RequestTypeItem currConfig;
   if((false == checkAPPowerConflict(apItem)) && (false == checkSTAPowerConflict()))
   {
      conflictType = WBL_CONFLICT_WIFI_POWER_OFF;

   }
   else
   {
      if(checkSTAPowerConflict())
      {
         currConfig.mode = WIFI_MODE_STA;
      }
      else
      {
         if((requestedAPConfig.type == apItem.property.apSetup.type))
         {
            LOG_DEBUG("CPW/AAW AP setup already exists - PrepareSetup to choose the best available frequency");
            sendSMRequestMsg(_prepareSetupMsg);
            _prepareSetupMsg = nullptr;
            return;
         }
         currConfig.mode = WIFI_MODE_AP1;
      }
      conflictType=WBL_CONFLICT_SIMULTANEOUS_OP_NOTALLOWED;
   }
   reqConfig.mode = WIFI_MODE_AP1;
   reqConfig.apConfig = requestedAPConfig;
   onConflictDetected(conflictType, reqConfig, currConfig);
}

bool ConflictResolver::checkSTAPowerConflict()
{
   LOG_INFO("ConflictResolver:checkSTAPowerConflict");
   WifiSetUpItem staItem;
   if (true == isSTASetupAvailable(staItem) && (true == staItem.property.isPowered))
   {
      LOG_DEBUG("STA is available and powered ON");
      return true;
   }
   return false;
}//void ConflictResolver::checkSTAPowerConflict

bool ConflictResolver::checkAPPowerConflict(const WifiSetUpItem& apItem)
{
   LOG_INFO("ConflictResolver:checkAPPowerConflict");
   if ((true == apItem.property.isPowered))
   {
      LOG_DEBUG("AP is already powered ON");
      return true;
   }
   return false;
} //void ConflictResolver::checkAPPowerConflict
void ConflictResolver::checkAPRequestConflicts(const APConfig& requestedAPConfig , const WifiSetUpItem& apItem)
{
   LOG_INFO("ConflictResolver:checkAPRequestConflicts");
   WBL_INTENTIONALLY_UNUSED(requestedAPConfig);

   if(checkSTAPowerConflict())
   {
      LOG_DEBUG("STA is powered ON , Simultaneous Operation not allowed :Conflict Detected");
      RequestTypeItem reqConfig;
      RequestTypeItem currConfig;
      WBLConflictType conflictType;
      conflictType = WBL_CONFLICT_SIMULTANEOUS_OP_NOTALLOWED;
      reqConfig.mode = WIFI_MODE_AP1;
      reqConfig.apConfig =  requestedAPConfig;
      currConfig.mode = WIFI_MODE_STA;
      onConflictDetected(conflictType, reqConfig, currConfig);
   }
   else if (true == apItem.property.isPowered)
   {
      LOG_DEBUG("AP is available and powered ON");

      if (false == isAPConfigMismatch(requestedAPConfig, apItem))
      {
         LOG_DEBUG("AP config is same,setup already exists");
         sendSMRequestMsg(_prepareSetupMsg);
         _prepareSetupMsg = nullptr;
      }
      else
      {
         LOG_DEBUG("AP config is different");
         uint8 connStaCount = static_cast<uint8>(apItem.property.apSetup.stationList.size());
         if (connStaCount == 0)
         {
            LOG_INFO("No stations connected,proceed to setup");
            sendSMRequestMsg(_prepareSetupMsg);
            _prepareSetupMsg = nullptr;
         }
         else
         {
            RequestTypeItem reqConfig;
            RequestTypeItem currConfig;
            WBLConflictType conflictType;
            LOG_DEBUG("stations connected,conflict detected");
            if (connStaCount > apItem.property.apSetup.maxStations - requestedAPConfig.stationsToReserve)
            {
               conflictType = WBL_CONFLICT_REQ_STA_UNAVAILABLE;
            }
            else
            {
               conflictType = WBL_CONFLICT_AP_CONFIG_MISMATCH;
            }
            reqConfig.mode = WIFI_MODE_AP1;
            reqConfig.apConfig =  requestedAPConfig;
            currConfig.mode = apItem.property.mode;
            currConfig.frequency = apItem.property.frequency;
            onConflictDetected(conflictType, reqConfig, currConfig);
         }
      }
   }
   else if (false == apItem.property.isPowered) 
   {
      LOG_DEBUG("AP is available and powered OFF,proceed to setup");
      sendSMRequestMsg(_prepareSetupMsg);
      _prepareSetupMsg = nullptr;
   }
} //void ConflictResolver::checkAPRequestConflicts

bool ConflictResolver::isAPSetupAvailable(WifiSetUpItem& apItem) const
{
   LOG_INFO("ConflictResolver:isAPSetupAvailable");
   bool isSetupFound = false;
   for (auto setupsItr = _wifiSetups.begin(); setupsItr != _wifiSetups.end(); ++setupsItr)
   {
      if ((*setupsItr).property.mode == WIFI_MODE_AP1)
      {
         isSetupFound = true;
         apItem = *setupsItr;
      }
   }
   return isSetupFound;
} // bool ConflictResolver::isAPSetupAvailable

bool ConflictResolver::isSTASetupAvailable(WifiSetUpItem& staItem) const
{
   LOG_INFO("ConflictResolver:isSTASetupAvailable");
   bool isSetupFound = false;
   for (auto setupsItr = _wifiSetups.begin(); setupsItr != _wifiSetups.end(); ++setupsItr)
   {
      if ((*setupsItr).property.mode == WIFI_MODE_STA)
      {
         isSetupFound = true;
         staItem = *setupsItr;
      }
   }
   return isSetupFound;
} // bool ConflictResolver::isSTASetupAvailable

bool ConflictResolver::isAPConfigMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   LOG_INFO("ConflictResolver:isAPConfigMismatch");

   bool retVal = false;
   retVal =((isFrequencyMismatch(requestedAPConfig, apItem)) || (isChannelMismatch(requestedAPConfig, apItem))
         || (isSSIDMismatch(requestedAPConfig, apItem)) || (isPassphraseMismatch(requestedAPConfig, apItem))
         || (isSecurityMismatch(requestedAPConfig, apItem)) || (isVisibilityMismatch(requestedAPConfig, apItem)
         || (isAPTypeMismatch(requestedAPConfig,apItem)) || (isMaxStationMismatch(requestedAPConfig,apItem)) ));
   LOG_INFO("ConflictResolver:isAPConfigMismatch retVal:%d",retVal);
   return retVal;

} //bool ConflictResolver::isAPConfigMismatch

bool ConflictResolver::isFrequencyMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   bool result = false;

   result = ((requestedAPConfig.isFrequencyValid) && (requestedAPConfig.frequency != apItem.property.frequency));

   if(true == result)
      LOG_DEBUG("ConflictResolver:Frequency Mismatch");

   return result;
} //bool ConflictResolver::isFrequencyMismatch

bool ConflictResolver::isChannelMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   bool result = true;

   if(true == requestedAPConfig.isChanListValid)
   {
      ChannelList chanList = requestedAPConfig.chanList;
      for(auto itr = chanList.begin();itr != chanList.end();++itr)
      {
         if((*itr) == apItem.property.currChannel)
            result = false;
      }
   }
   else
   {
      result = false;
   }

   if(true == result)
        LOG_DEBUG("ConflictResolver:Channel Mismatch");

   return result;
} //bool ConflictResolver::isChannelMismatch

bool ConflictResolver::isSSIDMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   LOG_DEBUG("isSSIDMismatch");

   bool result = false;
   result = ((requestedAPConfig.isSSIDValid) && (requestedAPConfig.ssid.SSID != apItem.property.apSetup.SSID));
   /*printAPConfig(requestedAPConfig);*/
   std::string tempSSID = convertByteArray2String(requestedAPConfig.ssid.SSID);
   LOG_DEBUG("SSID  %s",tempSSID.c_str());
   LOG_DEBUG("IS ssid valid %d",requestedAPConfig.isSSIDValid);


   if(true == result)
      LOG_DEBUG("ConflictResolver:SSID mismatch");

   return result;
} //bool ConflictResolver::isSSIDMismatch

bool ConflictResolver::isPassphraseMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   bool result = false;

   result = ((requestedAPConfig.isPassphraseValid)
         && (requestedAPConfig.passphrase != apItem.property.apSetup.passphrase));
   
    if(true == result)
         LOG_DEBUG("ConflictResolver:Passphrase Mismatch");

   return result;
     
} //bool ConflictResolver::isPassphraseMismatch

bool ConflictResolver::isSecurityMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   bool result = false;

   result = ((requestedAPConfig.isSecurityValid) && (requestedAPConfig.security != apItem.property.apSetup.security));

   if(true == result)
      LOG_DEBUG("ConflictResolver:Security Mismatch");

   return result;
} //bool ConflictResolver::isSecurityMismatch

bool ConflictResolver::isVisibilityMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   bool result = false;

   result = ((requestedAPConfig.isVisibilityValid)
         && (requestedAPConfig.isVisible != apItem.property.apSetup.isVisible));

   if(true == result)
        LOG_DEBUG("ConflictResolver:Visibility Mismatch");

   return result;
} //bool ConflictResolver::isVisibilityMismatch

bool ConflictResolver::isMaxStationMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   bool result = false;

   result = ((requestedAPConfig.isStationsToReserveValid)
         && (requestedAPConfig.stationsToReserve != apItem.property.apSetup.maxStations));

   if(true == result)
        LOG_DEBUG("ConflictResolver:MaxStations Mismatch");

   return result;
} //bool ConflictResolver::isAPTypeMismatch

bool ConflictResolver::isAPTypeMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const
{
   bool result = false;

   result = ((requestedAPConfig.isTypeValid) && (requestedAPConfig.type != apItem.property.apSetup.type));

   if(true == result)
        LOG_DEBUG("ConflictResolver:Type Mismatch");

   return result;
} //bool ConflictResolver::isAPTypeMismatch
void ConflictResolver::resolveConflictOnUserRequest(const uint32 action)
{
   LOG_INFO("ConflictResolver:resolveConflictOnUserRequest");

   LOG_INFO("resolveConflictOnUserRequest User Action %d",action);

   ConflictResolvedItem item;

   if (action == WBL_CONFLICT_ACTION_PREPARE)
   {
      LOG_INFO("Conflict mode selected,proceed to setup");
      item.conflictID = _poConflictItem->conflictID;
      item.conflictType = _poConflictItem->conflictType;
      item.resType = WBL_RES_SETUP_PREPARED;
      postConflictResolved(item);
      sendSMRequestMsg(_prepareSetupMsg);
      _prepareSetupMsg = nullptr;
   }
   else
   {
      item.conflictID = _poConflictItem->conflictID;
      item.conflictType = _poConflictItem->conflictType;
      item.resType = WBL_RES_SETUP_REJECTED_USER;
      postConflictResolved(item);
      _prepareSetupMsg->setErrorCode(WBL_ERR_REJECTED);
      sendWBLResponseMsg(_prepareSetupMsg);
      _prepareSetupMsg = nullptr;
   }
} //void ConflictResolver::resolveConflictOnUserRequest

void ConflictResolver::resolveConflictOnTimerExpiry()
{
   LOG_INFO("ConflictResolver:resolveConflictOnTimerExpiry");

   ConflictResolvedItem item;
   item.conflictID = _poConflictItem->conflictID;
   item.conflictType = _poConflictItem->conflictType;

   switch(_poConflictItem->conflictType)
   {
      case WBL_CONFLICT_SIMULTANEOUS_OP_NOTALLOWED:
      case WBL_CONFLICT_REQ_STA_UNAVAILABLE:
      case WBL_CONFLICT_AP_CONFIG_MISMATCH:
      case WBL_CONFLICT_WIFI_POWER_OFF:
      {
         item.resType = WBL_RES_SETUP_REJECTED_USER;
         postConflictResolved(item);
         if(_prepareSetupMsg)
         {
            _prepareSetupMsg->setErrorCode(WBL_ERR_REJECTED);
            sendWBLResponseMsg(_prepareSetupMsg);
            _prepareSetupMsg = nullptr;
         }
      }
      break;

      default:
      {
         LOG_DEBUG("Unknown Conflict Type");
      }
      break;
   }
} //void ConflictResolver::resolveConflictOnTimerExpiry

void ConflictResolver::onConflictDetected(const WBLConflictType conflictType, const RequestTypeItem& reqConfig,
      const RequestTypeItem& currConfig,uint16 staToFree)
{
   LOG_INFO("ConflictResolver:onConflictDetected");
   if(_poConflictItem)
   {
      if(_poConflictItem->conflictType == WBL_CONFLICT_UNKNOWN)
      {
         _poConflictItem->conflictID = genConflictToken();
         _poConflictItem->conflictType = conflictType;
         _poConflictItem->reqConfig = reqConfig;
         _poConflictItem->currConfig = currConfig;
         _poConflictItem->numStaToFree = staToFree;
         LOG_DEBUG("ConflictResolver:onConflictDetected: ConflictId:%d ConflictType:%d",
               _poConflictItem->conflictID,_poConflictItem->conflictType);
         postConflictDetected();
      }
      else
      {
         LOG_ERROR("ConflictResolver:onConflictDetected: Conflict Already Exist: ConflictId:%d ConflictType:%d",
               _poConflictItem->conflictID, _poConflictItem->conflictType);
      }
   }
} //void ConflictResolver::onConflictDetected

uint32 ConflictResolver::genConflictToken()
{
   LOG_INFO("ConflictResolver::generateConflictToken");
   uint32 token = 0;
   token = cUtils::getRandomNo(CONFLICT_TOKEN_START_RANGE,CONFLICT_TOKEN_END_RANGE);
   LOG_DEBUG("Generated conflict token is %d", token);
   return token;

} //uint32 ConflictResolver::genConflictToken

void ConflictResolver::postConflictDetected()
{
   LOG_INFO("ConflictResolver::postConflictDetected");

   WBLServiceFactory* svcFactory = WBLServiceFactory::getInstance();
   IConflictManagementService* conflictMgrSvcIf =
         (svcFactory) ? (svcFactory->getConflictManagementServiceIf()) : (nullptr);
   if (conflictMgrSvcIf)
   {
      ConflictDetectedMsg msg;
      msg.setConflictDetectedItem(*_poConflictItem);
      conflictMgrSvcIf->onConflictDetected(msg);
   }
   _conflictTimerDuration = readConflictTimerDuration();
   if (false == _conflictTimer.isActive() && _conflictTimerDuration > 0)
   {
      LOG_DEBUG("Starting conflict Timer for Duration: %d Seconds",_conflictTimerDuration);
      _conflictTimer.start(*this, (_conflictTimerDuration*1000), ::asf::core::Timer::Once);
   }
   else
   {
      resolveConflictOnTimerExpiry();
   }
} //void ConflictResover::postConflictDetected

void ConflictResolver::postConflictResolved(const ConflictResolvedItem& item)
{
   LOG_INFO("ConflictResolver::postConflictResolved");

   if (true == _conflictTimer.isActive())
   {
      LOG_DEBUG("Stopping conflict Timer");
      _conflictTimer.stop();
   }
   WBLServiceFactory* svcFactory = WBLServiceFactory::getInstance();
   IConflictManagementService* conflictMgrSvcIf =
         (svcFactory) ? (svcFactory->getConflictManagementServiceIf()) : (nullptr);
   if (conflictMgrSvcIf)
   {
      ConflictResolvedMsg msg;
      msg.setConflictResolvedItem(item);
      conflictMgrSvcIf->onConflictResolved(msg);
      if (_poConflictItem->conflictType != WBL_CONFLICT_UNKNOWN)
      {
         LOG_DEBUG("Clearing current conflict ConflictID: %d ",_poConflictItem->conflictID);
         _poConflictItem->clearConflict();
      }
   }
} //void ConflictResover::postConflictResolved

void ConflictResolver::sendSMRequestMsg(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("ConflictResolver:sendSMRequestMsg");

   SMManagerFactory* smFactory = SMManagerFactory::getInstance();
   tICmdStateMachineManagerSptr SMManagerIf = (smFactory) ? (smFactory->getStateMachineManagerIf()) : (nullptr);
   if (SMManagerIf)
   {
      SMManagerIf->processRequest(msg);
   }
} //void ConflictResolver::sendSMRequestMsg

void ConflictResolver::sendWBLResponseMsg(const ::boost::shared_ptr < WBLBaseMsg > msg)
{
   LOG_INFO("ConflictResolver:sendWBLResponseMsg");

   WBLServiceFactory* svcFactory = WBLServiceFactory::getInstance();
   IConflictManagementService* conflictMgrSvcIf =
         (svcFactory) ? (svcFactory->getConflictManagementServiceIf()) : (nullptr);
   if (conflictMgrSvcIf)
   {
      conflictMgrSvcIf->onWBLServiceResponse(msg);
   }
} //void ConflictResolver::sendWBLResponseMsg
} //namespace bosch
} //namespace org

/** @} */
