/**
 * @file ConflictResolver.h
 * @author RBEI/ECO32 Jiji Anna Jacob
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#ifndef _CONFLICT_RESOLVER_H
#define _CONFLICT_RESOLVER_H

#include "asf/core/Logger.h"
#include "asf/core/Timer.h"

#include "ICmdConflictResolver.h"

#include "WBLBaseMsg.h"
#include "WBLMessages.h"
#include "WBLTypes.h"
#include "WBLDefines.h"

namespace org
{
namespace bosch
{

using namespace ::asf::core;

class ConflictResolver final: public ICmdConflictResolver, public TimerCallbackIF
{
   public:
      ConflictResolver();
      virtual ~ConflictResolver();
      ConflictResolver(const ConflictResolver& ref);
      ConflictResolver& operator=(const ConflictResolver& ref);

      virtual void resolveWBLServiceRequest(const ::boost::shared_ptr < WBLBaseMsg > msg) override;
      virtual void notifyActiveWiFiSetups(const WifiSetUpList& wifisetups) override;
      virtual void onExpired(Timer& timer, boost::shared_ptr < TimerPayload > data) override;

   private:

      void onPrepareSetupRequest(const ::boost::shared_ptr < PrepareSetupMsg > msg);
      void onDeactivateSetupRequest(const ::boost::shared_ptr < DeActivateSetupMsg > msg);
      void onResolveConflictRequest(const ::boost::shared_ptr < ResolveConflictMsg > msg);

      void processSTARequestConflicts();
      void processAPRequestConflicts(const APConfig& requestedAPConfig,const WBLActionType& action);

      void checkSPIConflicts(const APConfig& requestedAPConfig , const WifiSetUpItem& apItem);
      bool checkAPPowerConflict(const WifiSetUpItem& apItem);
      bool checkSTAPowerConflict();

      void checkAPRequestConflicts(const APConfig& requestedAPConfig , const WifiSetUpItem& apItem);

      bool isAPSetupAvailable(WifiSetUpItem& apItem) const;
      bool isSTASetupAvailable(WifiSetUpItem& staItem) const;

      bool isAPConfigMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;
      bool isFrequencyMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;
      bool isChannelMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;
      bool isSSIDMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;
      bool isPassphraseMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;
      bool isSecurityMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;
      bool isVisibilityMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;
      bool isAPTypeMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;
      bool isMaxStationMismatch(const APConfig& requestedAPConfig, const WifiSetUpItem& apItem) const;

      void onConflictDetected(const WBLConflictType conflictType, const RequestTypeItem& reqConfig,
            const RequestTypeItem& currConfig,uint16 staToFree = 0);
      void postConflictDetected();
      uint32 genConflictToken();

      void postConflictResolved(const ConflictResolvedItem& item);
      void resolveConflictOnTimerExpiry();
      void resolveConflictOnUserRequest(const uint32 action);
      bool isConflictResolvedOnSetUpsUpdate();

      uint32 readConflictTimerDuration()const;
      void readWifiPriorityList(PriorityList& list)const;

      void printAPConfig(const APConfig& apConfig);

      void sendSMRequestMsg(const ::boost::shared_ptr < WBLBaseMsg > msg);
      void sendWBLResponseMsg(const ::boost::shared_ptr < WBLBaseMsg > msg);

      ConflictDetectedItem* _poConflictItem;

      WifiSetUpList _wifiSetups;
      Timer _conflictTimer;
      uint32 _conflictTimerDuration;

      ::boost::shared_ptr < PrepareSetupMsg > _prepareSetupMsg;
      ::boost::shared_ptr < DeActivateSetupMsg > _deactSetupMsg;
      ::boost::shared_ptr < ResolveConflictMsg > _resolveConflictMsg;

      DECLARE_CLASS_LOGGER();

};

}//namespace bosch
}//namespace org

#endif //_I_CONFLICT_RESOLVER_H

   /** @} */

