/**
 * @file DataProvider.h
 * @author RBEI/ECO3 Akshay Kumar S R
 * @copyright (c) 2020 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_bl
 *
 * @brief
 *
 * @{
 */

#include <yajl/yajl_parse.h>
#include <yajl/yajl_gen.h>
#include <yajl/yajl_tree.h>
#include "DataProvider.h"

namespace org
{
namespace bosch
{

DEFINE_CLASS_LOGGER_AND_LEVEL("wifi_business_logic/Common", DataProvider, Debug);
DataProvider::DataProvider() : _WBLKDSConfiguration()
{
   LOG_DEBUG("WBL DataProvider: created");
   const ::std::string configFile(WBLCONFIGPATH);
   (void)readConfiguration(_WBLKDSConfiguration, configFile);
}

DataProvider::~DataProvider()
{
   LOG_DEBUG("WBL DataProvider destructor called");
}


void DataProvider::readConfiguration(WBLConfiguration& configuration, const ::std::string& configurationFile)
{
   LOG_DEBUG("readConfiguration: configurationFile = \"%s\"", configurationFile.c_str());
   if (configurationFile.empty())
   {
      LOG_ERROR("readConfiguration: configuration file name is empty");
      return;
   }
   FILE* pFile = fopen(configurationFile.c_str(), "rb");
   if (NULL == pFile)
   {
      LOG_ERROR("readConfiguration: open file %s failed (NULL pointer returned)", configurationFile.c_str());
      return;
   }
   fseek (pFile , 0 , SEEK_END);
   const size_t  bufferSize = ftell (pFile) +1;
   rewind (pFile);
   char* fileData = new (std::nothrow)char[bufferSize];
   if (NULL == fileData)
   {
      LOG_ERROR("readConfiguration: memory allocation failed (NULL pointer returned)");
      fclose(pFile);
      return;
   }
   size_t rd = fread((void *) fileData, 1, bufferSize , pFile);
   const bool sizeOk = (0 < rd); // check if at least 1 byte was read
   const bool eofSet = (0 != feof(pFile)); // check if end-of-file indicator is set
   const bool errorSet = (0 != ferror(pFile)); // check if error indicator is set

   if(errorSet)
   {
      const int errNumber = errno;
      LOG_ERROR("readConfiguration: read file %100s failed: ERROR=%d (%s)", configurationFile.c_str(), errNumber, strerror(errNumber));
   }

   /* file read error handling */
   if ((false == sizeOk) || (false == eofSet) || (true == errorSet))
   {
      LOG_ERROR("readConfiguration: could not read configuration file %100s", configurationFile.c_str());
   }
   else
   {
      yajl_val node;
      char errbuf[256];

      /* null plug buffers */
      errbuf[0] = 0;
      fileData[rd] = '\0';

      node = yajl_tree_parse((const char *) fileData, errbuf, sizeof(errbuf));

      /* parse error handling */
      if (node == NULL)
      {
         LOG_ERROR("ReadConfigError at Node Read");
         LOG_ERROR("readConfiguration: parsing content of configuration file %100s failed", configurationFile.c_str());

      }
      else
      {
         const char * path_CPWSupportInfoRead[] = {"KDSInfo","CPWSupport","read", (const char *) 0};
         const char * path_CPWSupportInfoDefault[] = {"KDSInfo","CPWSupport","default", (const char *) 0};
         const char * path_AAWSupportInfoRead[] = {"KDSInfo","AAWSupport","read", (const char *) 0};
         const char * path_AAWSupportInfoDefault[] = {"KDSInfo","AAWSupport","default", (const char *) 0};
         const char * path_VehicleNameRead[] = {"KDSInfo","VehicleName","read", (const char *) 0};
         const char * path_VehicleNameDefault[] = {"KDSInfo","VehicleName","default", (const char *) 0};
         const char * path_ManfNameRead[] = {"KDSInfo","ManufacturerName","read", (const char *) 0};
         const char * path_ManfNameDefault[] = {"KDSInfo","ManufacturerName","default", (const char *) 0};
         const char * path_ManfNameLength[] = {"KDSInfo","ManufacturerName","length", (const char *) 0};
         const char * path_ModelNameRead[] = {"KDSInfo","ModelName","read", (const char *) 0};
         const char * path_ModelNameDefault[] = {"KDSInfo","ModelName","default", (const char *) 0};
         const char * path_ModelNameLength[] = {"KDSInfo","ModelName","length", (const char *) 0};
         const char * path_5GHzCDSupportRead[] = {"KDSInfo","5GhzCDSupport","read", (const char *) 0};
         const char * path_5GHzCDSupportDefault[] = {"KDSInfo","5GhzCDSupport","default", (const char *) 0};
         const char * path_5GHzPDSupportRead[] = {"KDSInfo","5GhzPDSupport","read", (const char *) 0};
         const char * path_5GHzPDSupportDefault[] = {"KDSInfo","5GhzPDSupport","default", (const char *) 0};
         const char * path_BrandTypeRead[] = {"KDSInfo","VehicleBrandType","read", (const char *) 0};
         const char * path_BrandTypeDefault[] = {"KDSInfo","VehicleBrandType","default", (const char *) 0};
         const char * path_CPW2_4GhzChannel[] = {"WiFiChannels","CPW","2.4GhzChannel", (const char *) 0};
         const char * path_CPW5GhzChannel[] = {"WiFiChannels","CPW","5GhzChannel", (const char *) 0};
         const char * path_AAW5GhzChannel[] = {"WiFiChannels","AAW","5GhzChannel", (const char *) 0};
         yajl_val value = yajl_tree_get(node, path_CPWSupportInfoRead, yajl_t_number);
         if (NULL != value)
         {
            configuration._CPWSupportRead = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_CPWSupportInfoRead, yajl_t_number));
            if (NULL == yajl_tree_get(node, path_CPWSupportInfoDefault, yajl_t_number))
            {
               configuration._CPWSupportDefault = 0u;
            }
            else
            {
               configuration._CPWSupportDefault = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_CPWSupportInfoDefault, yajl_t_number));
            }
         }
         if (NULL != yajl_tree_get(node, path_AAWSupportInfoRead, yajl_t_number))
         {
            configuration._AAWSupportRead = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_AAWSupportInfoRead, yajl_t_number));
            if (NULL == yajl_tree_get(node, path_AAWSupportInfoDefault, yajl_t_number))
            {
               configuration._AAWSupportDefault = 0u;
            }
            else
            {
               configuration._AAWSupportDefault = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_AAWSupportInfoDefault, yajl_t_number));
            }
         }
         if (NULL != yajl_tree_get(node, path_VehicleNameRead, yajl_t_number))
         {
            value = yajl_tree_get(node, path_VehicleNameDefault, yajl_t_string);
            if (0 != value)
            {
               if (true == YAJL_IS_STRING(value))
               {
                  configuration._VehicleNameDefault = YAJL_GET_STRING(value);
               }
               else
               {
                  LOG_ERROR("readConfiguration: value of parameter %50s is not a string", path_VehicleNameDefault[0]);
               }
            }
            else
            {
               LOG_ERROR("readConfiguration: parameter %50s does not exist in configuration file %100s",
                     path_VehicleNameDefault[0], configurationFile.c_str());
            }

         }
         if (NULL != yajl_tree_get(node, path_ManfNameRead, yajl_t_number))
         {
            configuration._ManfNameRead = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_ManfNameRead, yajl_t_number));
            value = yajl_tree_get(node, path_ManfNameDefault, yajl_t_string);
            if (0 != value)
            {
               if (true == YAJL_IS_STRING(value))
               {
                  configuration._ManfNameDefault = YAJL_GET_STRING(value);
                  LOG_DEBUG("readConfiguration: ManufacturerNameDefault = \"%s\"", configuration._ManfNameDefault.c_str());
               }
               else
               {
                  LOG_ERROR("readConfiguration: value of parameter %50s is not a string", path_ManfNameDefault[0]);
               }
            }
            else
            {
               LOG_ERROR("readConfiguration: parameter %50s does not exist in configuration file %100s",
                     path_ManfNameDefault[0], configurationFile.c_str());
            }
            if (NULL == yajl_tree_get(node, path_ManfNameLength, yajl_t_number))
            {
               configuration._ManfNameLength = 0u;
            }
            else
            {
               configuration._ManfNameLength = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_ManfNameLength, yajl_t_number));
            }
         }
         if (NULL != yajl_tree_get(node, path_ModelNameRead, yajl_t_number))
         {
            configuration._ModelNameRead = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_ModelNameRead, yajl_t_number));
            value = yajl_tree_get(node, path_ModelNameDefault, yajl_t_string);
            if (0 != value)
            {
               if (true == YAJL_IS_STRING(value))
               {
                  configuration._ModelNameDefault = YAJL_GET_STRING(value);
               }
               else
               {
                  LOG_ERROR("readConfiguration: value of parameter %50s is not a string", path_ModelNameDefault[0]);
               }
            }
            if (NULL == yajl_tree_get(node, path_ModelNameLength, yajl_t_number))
            {
               configuration._ModelNameLength = 0u;
            }
            else
            {
               configuration._ModelNameLength = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_ModelNameLength, yajl_t_number));
            }
         }
         if (NULL != yajl_tree_get(node, path_5GHzCDSupportRead, yajl_t_number))
         {
            configuration._5GhzCDSupportRead = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_5GHzCDSupportRead, yajl_t_number));
            if (NULL == yajl_tree_get(node, path_5GHzCDSupportDefault, yajl_t_number))
            {
               configuration._5GHzCDSupportDefault = 0u;
            }
            else
            {
               configuration._5GHzCDSupportDefault = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_5GHzCDSupportDefault, yajl_t_number));
            }

         }
         if(NULL!= yajl_tree_get(node,path_5GHzPDSupportRead, yajl_t_number))
         {
            configuration._5GhzPDSupportRead = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_5GHzPDSupportRead, yajl_t_number));
            if (NULL == yajl_tree_get(node, path_5GHzPDSupportDefault, yajl_t_number))
            {
               configuration._5GHzPDSupportDefault = 0u;
            }
            else
            {
               configuration._5GHzPDSupportDefault = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_5GHzPDSupportDefault, yajl_t_number));
            }
         }
         if (NULL == yajl_tree_get(node, path_BrandTypeRead, yajl_t_number))
         {
            configuration._BrandTypeRead = 0u;
         }
         else
         {
            configuration._BrandTypeRead = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_BrandTypeRead, yajl_t_number));
         }
         if (NULL == yajl_tree_get(node, path_BrandTypeDefault, yajl_t_number))
         {
            configuration._BrandTypeDefault = 0u;
         }
         else
         {
            configuration._BrandTypeDefault = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_BrandTypeDefault, yajl_t_number));
         }
         if(NULL != yajl_tree_get(node, path_CPW2_4GhzChannel, yajl_t_number))
         {
            configuration._CPW2_4GhzChannel = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_CPW2_4GhzChannel, yajl_t_number));
         }
         if(NULL != yajl_tree_get(node, path_CPW5GhzChannel, yajl_t_number))
         {
            configuration._CPW5GHzChannel = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_CPW5GhzChannel, yajl_t_number));
         }
         if(NULL != yajl_tree_get(node, path_AAW5GhzChannel, yajl_t_number))
         {
            configuration._AAW5GHzChannel = (uint8_t)YAJL_GET_INTEGER(yajl_tree_get(node, path_AAW5GhzChannel, yajl_t_number));
         }

      }
   }
   delete[] fileData;
   fclose(pFile);
}

}
}
