/**
 * @file tech_manager.c
 * @author RBEI/ECO3 Usman Sheik
 * @copyright (c) 2016 Robert Bosch Car Multimedia GmbH
 * @addtogroup wifi_mw\aifi_ap_direct_manager
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <string.h>
#include <net/if.h>
#include <linux/version.h>
#include <gio/gio.h>
#include <glib.h>
#include <log.h>
#include <device.h>
#include <genl.h>
#include <ap.h>
#include <tech_manager.h>

static char **ap_devices = NULL;
static char **p2p_devices = NULL;

static GHashTable *interfaces;
static GList *driver_list = NULL;
static GSList *orphan_wifi_devices = NULL;

typedef
struct _wireless_interface
{
    void *data;
    wirelessdriverops *driver;
    struct net_device *device;
    enum nl80211_iftype iftype;
} wifiif;

static void
wireless_interface_cleanup (gpointer data)
{
    g_free (data);
}

static void
cleanup_orphans (gpointer data)
{
    INFO ("Cleaning up device: %s", (char *) data);
    g_free (data);
}

static const char*
wireless_iftype_to_string (const enum nl80211_iftype type)
{
    switch (type) {
    default:
    case NUM_NL80211_IFTYPES:
    case NL80211_IFTYPE_UNSPECIFIED:
        return ENUMTOSTR (NL80211_IFTYPE_UNSPECIFIED);
    case NL80211_IFTYPE_ADHOC:
        return ENUMTOSTR (NL80211_IFTYPE_ADHOC);
    case NL80211_IFTYPE_STATION:
        return ENUMTOSTR (NL80211_IFTYPE_STATION);
    case NL80211_IFTYPE_AP:
        return ENUMTOSTR (NL80211_IFTYPE_AP);
    case NL80211_IFTYPE_AP_VLAN:
        return ENUMTOSTR (NL80211_IFTYPE_AP_VLAN);
    case NL80211_IFTYPE_WDS:
        return ENUMTOSTR (NL80211_IFTYPE_WDS);
    case NL80211_IFTYPE_MONITOR:
        return ENUMTOSTR (NL80211_IFTYPE_MONITOR);
    case NL80211_IFTYPE_MESH_POINT:
        return ENUMTOSTR (NL80211_IFTYPE_MESH_POINT);
    case NL80211_IFTYPE_P2P_CLIENT:
        return ENUMTOSTR (NL80211_IFTYPE_P2P_CLIENT);
    case NL80211_IFTYPE_P2P_GO:
        return ENUMTOSTR (NL80211_IFTYPE_P2P_GO);
    case NL80211_IFTYPE_P2P_DEVICE:
        return ENUMTOSTR (NL80211_IFTYPE_P2P_DEVICE);
#if LINUX_VERSION_CODE >= KERNEL_VERSION(4,0,0)
    case NL80211_IFTYPE_OCB:
        return ENUMTOSTR (NL80211_IFTYPE_OCB);
#endif
    }

    return ENUMTOSTR (NL80211_IFTYPE_UNSPECIFIED);
}

static int
wireless_tech_is_blacklisted (const char *ifname)
{
    char **interfaces;

    return_val_if_fail (ifname, -EINVAL);

    if (!ap_devices)
        goto p2p;

    interfaces = ap_devices;
    for ( ; *interfaces; interfaces++)
        if (!g_strcmp0 (ifname, *interfaces))
            return 1;

p2p:
    if (!p2p_devices)
        return 0;

    interfaces = p2p_devices;
    for ( ; *interfaces; interfaces++)
        if (!g_strcmp0 (ifname, *interfaces))
            return 1;

    return 0;
}

static wirelessdriverops*
wireless_tech_find_driver (const enum nl80211_iftype type)
{
    GList *temp;
    wirelessdriverops *driver;

    temp = driver_list;
    for ( ; temp; temp = temp->next) {
        driver = temp->data;
        if (driver && driver->iftype == type)
            return driver;
    }

    return NULL;
}

static void
wireless_tech_probe_driver (wirelessdriverops *driver)
{
    unsigned int index;
    wifiif *interface;
    GHashTableIter iter;
    gpointer key, value;
    const char *name = NULL, *address = NULL;

    return_if_fail (driver);

    DEBUG ("driver: %p name: %s", driver, driver->name);

    g_hash_table_iter_init (&iter, interfaces);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        interface = value;
        if (!interface || interface->driver)
            continue;

        if (driver->iftype != interface->iftype)
            continue;

        name = device_get_name (interface->device);
        index = device_get_index (interface->device);
        address = device_get_address (interface->device);

        if (driver->add) {
            interface->data = driver->add (index, name, address);
            if (!interface->data)
                ERROR ("Failed to create the %s device", driver->name);
        }

        interface->driver = driver;
    }
}

static void
wireless_tech_remove_data (wifiif *iface)
{
    unsigned int index;
    int ret;
    const char *name = NULL,
            *address = NULL;

    return_if_fail (iface);

    name = device_get_name (iface->device);
    index = device_get_index (iface->device);
    address = device_get_address (iface->device);

    if (iface->driver->remove) {
        ret = iface->driver->remove (index, name, address);
        if (ret < 0)
            ERROR ("Failed to remove the data for %s [error : %s]",
                   name, strerror (-ret));
    }

    iface->driver = NULL;
}

static void
wireless_tech_remove_driver (wirelessdriverops *driver)
{
    wifiif *interface;
    GHashTableIter iter;
    gpointer key, value;

    return_if_fail (driver);

    DEBUG ("driver: %p name: %s", driver, driver->name);
    g_hash_table_iter_init (&iter, interfaces);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        interface = value;
        if (interface && interface->driver == driver)
            wireless_tech_remove_data (interface);
    }
}

static int
handle_device_added (unsigned int index,
                     const char *name,
                     const char *address)
{
    int ret;
    wifiif *iface;
    wirelessdriverops *driver;

    return_val_if_fail (name && address, -EINVAL);

    DEBUG ("index: %d name: %s address: %s", index,
           name, address);

    ret = wireless_tech_is_blacklisted (name);
    if (ret != 1) {
        ERROR ("Device \"%s\" is blacklisted", name);
        return 0;
    }

    /* intended purpose of the device? */
    ret = genl_get_interface_type (name);
    if (ret < 0) {
        ERROR ("Failed to get the device type information %s", strerror (-ret));
        orphan_wifi_devices = g_slist_append (orphan_wifi_devices, g_strdup (name));
        return ret;
    }

    DEBUG ("Device \"%s\" will be managed as : %s", name,
           wireless_iftype_to_string ((enum nl80211_iftype) ret));

    iface = g_hash_table_lookup (interfaces, GINT_TO_POINTER (index));
    return_val_if_fail (!iface, -EALREADY);
    iface = g_try_malloc0 (sizeof (*iface));
    return_val_if_fail (iface, -ENOMEM);

    iface->device = device_get (index);
    iface->iftype = (enum nl80211_iftype) ret;

    /* Do we have the respective driver? */
    driver = wireless_tech_find_driver (iface->iftype);
    if (driver) {
        INFO ("Driver found for the device : %s [%s]", name, driver->name);
        if (driver->add) {
            iface->data = (void *) driver->add (index, name, address);
            if (iface->data) {
                if (driver->start) {
                    ret = driver->start (iface->data);
                    if (ret < 0)
                        ERROR ("Failed to start the %s", driver->name);
                }
            } else {
                ERROR ("Failed to create the %s device", driver->name);
            }
        }
        iface->driver = driver;
    }

    g_hash_table_replace (interfaces, GINT_TO_POINTER (index), iface);
    return 0;
}

static int
handle_device_removed (unsigned int index,
                       const char *name,
                       const char *address)
{
    int ret = 0;
    wifiif *wif;
    gboolean removed;

    return_val_if_fail (name && address, -EINVAL);

    DEBUG ("index %d name %s address %s", index,
           name, address);

    wif = g_hash_table_lookup (interfaces, GINT_TO_POINTER (index));
    return_val_if_fail (wif, -ENOENT);
    if (wif->driver && wif->driver->remove) {

        if (wif->driver->stop) {
            ret = wif->driver->stop (wif->data);
            if (ret <0)
                ERROR ("Failed to stop : %s error : %s", wif->driver->name, strerror (-ret));
        }

        ret = wif->driver->remove (index, name, address);
        if (ret < 0)
            ERROR ("Failed to remove the data for %s [error : %s]", name, strerror (-ret));
    }

    removed = g_hash_table_remove (interfaces, GINT_TO_POINTER (index));
    if (removed)
        INFO ("Successfully removed the wireless interface entry : %s", name);

    return ret;
}

static deviceops wifi_devices = {
    .client = "wireless",
    .device_added = handle_device_added,
    .device_removed = handle_device_removed,
};

static void
wireless_tech_remove_orphan (const char *name)
{
    char *orphan;
    GSList *temp;
    int found = 0;

    return_if_fail (name);

    temp = orphan_wifi_devices;
    for ( ; temp; temp = temp->next) {
        orphan = temp->data;
        if (orphan && !g_strcmp0 (orphan, name)) {
            found = 1;
            break;
        }
    }

    if (!found)
        return;

    orphan_wifi_devices = g_slist_remove (orphan_wifi_devices, name);
    g_free (orphan);
}

int
wireless_tech_handle_orphans (const char *name)
{
    GSList *temp;
    char *orphan = NULL;
    struct net_device *orphandev;
    const char *address;
    unsigned int index;
    int found = 0, ret;

    return_val_if_fail (name, -EINVAL);

    temp = orphan_wifi_devices;
    for ( ; temp; temp = temp->next) {
        orphan = temp->data;
        if (orphan && !g_strcmp0 (orphan, name)) {
            found = 1;
            break;
        }
    }

    if (!found)
        return 0;

    index = if_nametoindex (name);
    if (!index) {
        ERROR ("Failed to get the index for interface : %s", name);
        return -ENXIO;
    }

    orphandev = device_get (index);
    if (!orphandev) {
        ERROR ("Device for %s is not maintained", name);
        return -ENOENT;
    }

    address = device_get_address (orphandev);
    return_val_if_fail (address, -ENODATA);
    ret = handle_device_added (index, name, address);

    if (!ret || ret == -EALREADY)
        wireless_tech_remove_orphan (name);
    if (ret < 0) {
        ERROR ("Failed to handle the device : %s [%s]",
               name, strerror (-ret));
    }

    return ret;
}

int
wireless_tech_driver_register (wirelessdriverops *driver)
{
    int found = 0;
    GList *temp;
    wirelessdriverops *ops;

    return_val_if_fail (driver, -EINVAL);

    DEBUG ("driver %p name %s type %s", driver, driver->name,
           wireless_iftype_to_string (driver->iftype));

    temp = driver_list;
    for ( ; temp; temp = temp->next) {
        ops = temp->data;
        if (ops == driver) {
            found = 1;
            break;
        }
    }

    if (!found) {
        driver_list = g_list_append (driver_list, driver);
        wireless_tech_probe_driver (driver);
        return 0;
    }

    return -EEXIST;
}

int
wireless_tech_driver_unregister (wirelessdriverops *driver)
{
    int found = 0;
    GList *temp;
    wirelessdriverops *ops;

    return_val_if_fail (driver, -EINVAL);

    DEBUG ("driver %p name %s type %s", driver, driver->name,
           wireless_iftype_to_string (driver->iftype));

    temp = driver_list;
    for ( ; temp; temp = temp->next) {
        ops = temp->data;
        if (ops == driver) {
            found = 1;
            break;
        }
    }

    if (found) {
        driver_list = g_list_remove (driver_list, driver);
        wireless_tech_remove_driver (driver);
        return 0;
    }

    return -ENOENT;
}


int
wireless_tech_init (char *apdevices, char *p2pdevices)
{
    int ret;

    if (apdevices)
        ap_devices = g_strsplit (apdevices, ",", -1);

    if (p2pdevices)
        p2p_devices = g_strsplit (p2pdevices, ",", -1);

    ret = device_notifier_register (&wifi_devices);
    if (ret < 0) {
        ERROR ("Failed to register the device ops");
        return ret;
    }

    interfaces = g_hash_table_new_full (g_direct_hash, g_direct_equal,
                                        NULL, wireless_interface_cleanup);

    return ret;
}

int
wireless_tech_deinit ()
{
    int ret;

    ret = device_notifier_unregister (&wifi_devices);
    if (ret < 0)
        ERROR ("Failed to unregister the device ops");

    g_hash_table_destroy (interfaces);
    g_list_free (driver_list);

    g_slist_free_full (orphan_wifi_devices, cleanup_orphans);

    g_strfreev (ap_devices);
    g_strfreev (p2p_devices);

    return ret;
}

/** @} */
