/**
 * @file objectmanager_if.c
 * @author RBEI/ECO3-Karthikeyan Madeswaran
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <errno.h>
#include <string.h>
#include <glib.h>
#include <log.h>
#include "inc/declarations.h"
#include <objectmanager_if.h>
#include <org-bosch-wapdman-objmanager-generated.h>

typedef
struct __wapdman_object {
    char *path;
    GSList *interfaces;
} wapdmanobject;

typedef
struct
{
    gchar *path;
    GDBusConnection *bus;
    GHashTable *objects;
    OrgFreedesktopDBusObjectManager *service;
} ObjectManager;

static ObjectManager *object_manager = NULL;

static void
objman_interface_cleanup (gpointer data)
{
    InterfaceTable *iface = data;

    return_if_fail (iface);
    g_free (iface->name);
    g_free (iface);
}

static void
objman_object_cleanup (gpointer data)
{
    wapdmanobject *object = data;

    return_if_fail (object);
    g_slist_free_full (object->interfaces, objman_interface_cleanup);
    g_free (object->path);
    g_free (object);
}

static gboolean
objman_get_managed_objects (OrgFreedesktopDBusObjectManager *objmanager,
                            GDBusMethodInvocation *invocation,
                            gpointer data)
{
    int ret;
    GSList *interfaces;
    GVariant *variant = NULL;
    GHashTableIter iter;
    gpointer key, value;
    InterfaceTable *iface;
    wapdmanobject *object;
    const PropertyTable *property;
    ObjectManager *manager = data;
    GVariantBuilder *ifacebuilder, *propbuilder,
            *objbuilder;

    return_val_if_fail (objmanager && invocation && manager, FALSE);

    DEBUG ("Manager: %s objmanager: %p invocation: %p",
           manager->path ? manager->path : "unknown",
           objmanager, invocation);

    objbuilder = g_variant_builder_new (G_VARIANT_TYPE ("a{oa{sa{sv}}}"));

    g_hash_table_iter_init (&iter, manager->objects);
    while (g_hash_table_iter_next (&iter, &key, &value)) {

        object = value;
        continue_if_fail (object);

        DEBUG ("Registered object: %p "
               "Object path : %s",
               object,
               object->path ? object->path : "unknown");

        interfaces = object->interfaces;
        ifacebuilder = g_variant_builder_new (G_VARIANT_TYPE ("a{sa{sv}}"));
        for ( ; interfaces; interfaces = interfaces->next) {

            iface = interfaces->data;
            continue_if_fail (iface);

            DEBUG ("Available interface: %p "
                   "Interface name: \"%s\"",
                   iface,
                   iface->name ? iface->name : "unknown");

            property = iface->properties;
            propbuilder = g_variant_builder_new (G_VARIANT_TYPE ("a{sv}"));
            for ( ; property && property->name; property++, variant = NULL) {

                DEBUG ("Avaialable Property: %s", property->name);

                continue_if_fail (property->get && property->type);
                ret = property->get (property, &variant, iface->userdata);
                if (ret < 0) {
                    ERROR ("Failed to get the \"%s\" "
                           "value: %s/%d",
                           property->name ? property->name : "unknown",
                           strerror (-ret), -ret);
                    continue;
                }

                g_variant_builder_add (propbuilder, "{sv}", property->name, variant);
            }

            g_variant_builder_add (ifacebuilder, "{sa{sv}}", iface->name, propbuilder);
            g_variant_builder_unref (propbuilder);
        }

        g_variant_builder_add (objbuilder, "{oa{sa{sv}}}", object->path, ifacebuilder);
        g_variant_builder_unref (ifacebuilder);
	}

    org_freedesktop_dbus_object_manager_complete_get_managed_objects
            (objmanager, invocation, g_variant_builder_end (objbuilder));

    g_variant_builder_unref (objbuilder);
    return TRUE;
}

static int
objman_get_object (const char *path,
                   ObjectManager *manager,
                   wapdmanobject **object)
{
    wapdmanobject *obj;

    return_val_if_fail (path && object, -EINVAL);
    return_val_if_fail (!*object, -EEXIST);
    return_val_if_fail (manager, -ENOTCONN);

    obj = g_hash_table_lookup (manager->objects, path);
    if (!obj) {
        obj = g_try_malloc0 (sizeof (*obj));
        return_val_if_fail (obj, -ENOMEM);
        obj->path = g_strdup (path);
        g_hash_table_replace (manager->objects, obj->path, obj);
    }

    *object = obj;
    return 0;
}

static InterfaceTable*
objman_get_interface (const char *name,
                      wapdmanobject *object)
{
    GSList *temp;
    InterfaceTable *iface = NULL;

    return_val_if_fail (name && object, NULL);
    temp = object->interfaces;
    for ( ; temp; temp = temp->next) {
        iface = temp->data;
        if (iface && !g_strcmp0 (iface->name, name))
            return iface;
    }

    return NULL;
}

static int
objman_emit_interfacesadded (ObjectManager *manager,
                             const char *path,
                             InterfaceTable *iface)
{
    int ret;
    GVariant *value = NULL;
    const PropertyTable *temp;
    GVariantBuilder builder, ifbuilder;

    return_val_if_fail (manager && path && iface, -EINVAL);

    DEBUG ("Object Path: %s, "
           "interface name: %s", path, iface->name);

    g_variant_builder_init (&builder, G_VARIANT_TYPE ("a{sv}"));
    g_variant_builder_init (&ifbuilder, G_VARIANT_TYPE ("a{sa{sv}}"));

    temp = iface->properties;
    for ( ; temp && temp->name; temp++, value = NULL) {
        continue_if_fail (temp->get);
        ret = temp->get (temp, &value, iface->userdata);
        if (ret < 0) {
            ERROR ("Failed to get the \"%s\" value: %s/%d",
                   temp->name, strerror (-ret), -ret);
            continue;
        }
        g_variant_builder_add (&builder, "{sv}", temp->name, value);
    }

    g_variant_builder_add (&ifbuilder, "{sa{sv}}", iface->name, &builder);

    org_freedesktop_dbus_object_manager_emit_interfaces_added
            (manager->service, path, g_variant_builder_end (&ifbuilder));

    return 0;
}

int
objman_register_interfaces (const char *path,
                            const char *name,
                            const PropertyTable *properties,
                            void *userdata)
{
    int ret;
    wapdmanobject *object = NULL;
    ObjectManager *manager = object_manager;
    InterfaceTable *iface;

    return_val_if_fail (path && name, -EINVAL);
    return_val_if_fail (manager, -ENOTCONN);

    ret = objman_get_object (path, manager, &object);
    if (ret < 0) {
        ERROR ("Failed to register interface [\"%s\"] "
               "with the objectmanager for path : %s "
               "[error : %s/%d]", name, path, strerror (-ret), -ret);
        return ret;
    }

    iface = objman_get_interface (name, object);
    return_val_if_fail (!iface, -EALREADY);

    iface = g_try_malloc0 (sizeof (*iface));
    return_val_if_fail (iface, -ENOMEM);

    iface->name = g_strdup (name);
    iface->properties = properties;
    iface->userdata = userdata;

    object->interfaces = g_slist_append (object->interfaces, iface);
    ret = objman_emit_interfacesadded (manager, path, iface);
    if (ret < 0) {
        ERROR ("Failed to emit the interfacesadded signal for "
               "interface \"%s\" path \"%s\" error [%s/%d]", name,
               path, strerror (-ret), -ret);
    }

    return ret;
}

static int
objman_emit_interfacesremoved (ObjectManager *manager,
                               const char *path,
                               InterfaceTable *iface)
{
    const char *interfacesremoved [2];

    return_val_if_fail (manager && path && iface, -EINVAL);

    DEBUG ("Object Path: %s, interface name: %s", path, iface->name);

    interfacesremoved [0] = iface->name;
    interfacesremoved [1] = NULL;
    org_freedesktop_dbus_object_manager_emit_interfaces_removed
            (manager->service, path, interfacesremoved);

    return 0;
}

int
objman_unregister_interfaces (const char *path,
                              const char *name)
{
    int ret;
    InterfaceTable *iface;
    wapdmanobject *object = NULL;
    ObjectManager *manager = object_manager;

    return_val_if_fail (path && name, -EINVAL);
    return_val_if_fail (manager, -ENOTCONN);

    ret = objman_get_object (path, manager, &object);
    if (ret < 0) {
        ERROR ("Interface \"%s\" was not added, error: %s/%d",
               name, strerror (-ret), -ret);
        return ret;
    }

    iface = objman_get_interface (name, object);
    return_val_if_fail (iface, -ENOENT);
    ret = objman_emit_interfacesremoved (manager, path, iface);
    if (ret < 0)
        return ret;

    object->interfaces = g_slist_remove (object->interfaces, iface);
    objman_interface_cleanup ((gpointer) iface);
    if (g_slist_length (object->interfaces) == 0)
        g_hash_table_remove (manager->objects, path);
    return 0;
}

static int
objman_service_init (GDBusConnection *bus,
                     const char *path,
                     ObjectManager **objmanager)
{
    int ret = -EIO;
    ObjectManager *manager;
    gboolean expt = FALSE;
    GError *error = NULL;

    return_val_if_fail (bus, -ENOLINK);
    return_val_if_fail (path && objmanager, -EINVAL);
    return_val_if_fail (!*objmanager, -EEXIST);

    manager = g_try_malloc0 (sizeof (*manager));
    return_val_if_fail (manager, -ENOMEM);

    DEBUG ("ObjectManager object path: %s manager : %p", path, manager);
    manager->objects = g_hash_table_new_full (g_str_hash, g_str_equal,
                                              NULL, objman_object_cleanup);
    manager->service =
            org_freedesktop_dbus_object_manager_skeleton_new ();

    g_signal_connect (manager->service, "handle_get_managed_objects",
                      G_CALLBACK (objman_get_managed_objects),
                      manager);

    expt = g_dbus_interface_skeleton_export (G_DBUS_INTERFACE_SKELETON (manager->service),
                                             bus, path, &error);
    if (!expt) {
        ERROR ("Failed to export the object manager interfaces : %s",
               error->message);
        g_error_free (error);
        goto failed;
    }

    manager->bus = bus;
    manager->path = g_strdup (path);

    *objmanager = manager;
    return 0;

failed:
    g_object_unref (manager->service);
    g_free (manager);
    return ret;
}

static int
objman_service_deinit (ObjectManager **manager)
{
    return_val_if_fail (manager, -EINVAL);
    return_val_if_fail (*manager, -EALREADY);

    g_object_unref ((*manager)->service);
    g_free ((*manager)->path);
    g_hash_table_destroy ((*manager)->objects);
    g_free (*manager);
    *manager = NULL;

    return 0;
}

int
objman_init (GDBusConnection *conn)
{
    int ret;
    char *path = WAPDMAN_OBJMAN_OBJECT_PATH;

    return_val_if_fail (conn, -ENOLINK);

    DEBUG ("Exporting object manager on DBus connection: %p",
           conn);

    ret = objman_service_init (conn, path, &object_manager);
    if (ret < 0)
        ERROR ("Failed to intialize the objectmanager service");

    return ret;
}

int
objman_deinit ()
{
    int ret;

    DEBUG ("");

    ret = objman_service_deinit (&object_manager);
    if (ret < 0)
        ERROR ("Failed to cleanup the objectmanager service");

    return ret;
}

/** @} */
