/**
 * @file manager_if.c
 * @author RBEI/ECO3-Karthikeyan Madeswaran
 * @copyright (c) 2015 Robert Bosch Car Multimedia GmbH
 * @addtogroup
 *
 * @brief
 *
 * @{
 */

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <ap.h>
#include <log.h>
#include <manager_if.h>
#include "inc/declarations.h"
#include <objectmanager_if.h>
#include <org-bosch-wapdman-manager-generated.h>

typedef
struct {
    char *path;
    Manager *service;
	GDBusConnection *bus;
} ManagerObject;

static ManagerObject *manager = NULL;

static const PropertyTable properties [] = {
    { NULL }
};

static InterfaceTable manager_interfaces [] = {
    { WAPDMAN_MANAGER_INTERFACE_NAME, properties, NULL},
    { NULL }
};

static int
manager_register_interfaces (ManagerObject *manager)
{
    int ret = 0;
    const InterfaceTable *interface;

    return_val_if_fail (manager, -EINVAL);

    interface = manager_interfaces;
    for ( ; interface && interface->name; interface++) {
        ret = objman_register_interfaces (manager->path, interface->name,
                                          interface->properties, manager);
        if (ret < 0) {
            ERROR ("Failed to register the interface [\"%s\"]"
                   "for the object : %s error: %s/%d", interface->name,
                   manager->path, strerror (-ret), -ret);
        }
    }

    return ret;
}

static int
manager_unregister_interfaces (const ManagerObject *manager)
{
    int ret = 0;
    const InterfaceTable *interface;

    return_val_if_fail (manager, -EINVAL);

    interface = manager_interfaces;
    for ( ; interface && interface->name; interface++) {
        ret = objman_unregister_interfaces (manager->path, interface->name);
        if (ret < 0) {
            ERROR ("Failed to register the interface [\"%s\"]"
                   "for the object : %s error: %s/%d", interface->name,
                   manager->path, strerror (-ret), -ret);
        }
    }

    return ret;
}

static gboolean
manager_find_access_point (Manager *object,
                           GDBusMethodInvocation *invocation,
                           const gchar *pattern,
                           gpointer data)
{
//    char *path = NULL;
    ManagerObject *man = data;

    return_val_if_fail (object && man && pattern && invocation, FALSE);

    DEBUG ("manager : %p invocation : %p pattern : %s",
           man, invocation, pattern);

    /* get the access point object from the interface */

    return FALSE;
}

static gboolean
manager_get_access_points (Manager *object,
                           GDBusMethodInvocation *invocation,
                           gpointer data)
{
    int ret;
    GVariant *variant = NULL;
    ManagerObject *man = data;

    return_val_if_fail (object && man && invocation, FALSE);

    DEBUG ("object %p invocation %p data %p", object,
           invocation, data);

    ret = access_point_get_all (&variant);
    if (ret < 0) {
        ERROR ("Failed to get the available access point details: %s/%d",
               strerror (-ret), -ret);
        return FALSE;
    }

    manager_complete_get_access_points (object, invocation, variant);
    return TRUE;
}

static int
manager_service_init (GDBusConnection *bus,
                      const char *path,
                      ManagerObject **man)
{
    int ret = -EIO;
    ManagerObject *manager;
    gboolean expt = FALSE;
    GError *error = NULL;

    return_val_if_fail (bus, -ENOLINK);
    return_val_if_fail (path && man, -EINVAL);
    return_val_if_fail (!*man, -EEXIST);

    manager = g_try_malloc0 (sizeof (*manager));
    return_val_if_fail (manager, -ENOMEM);

    DEBUG ("Manager object path: %s manager: %p", path, manager);

    manager->service = manager_skeleton_new ();

    g_signal_connect (manager->service, "handle-find-access-point",
                      G_CALLBACK (manager_find_access_point),
                      manager);

    g_signal_connect (manager->service, "handle-get-access-points",
                      G_CALLBACK (manager_get_access_points),
                      manager);

    expt = g_dbus_interface_skeleton_export (G_DBUS_INTERFACE_SKELETON (manager->service),
                                             bus, path, &error);
    if (!expt) {
        ERROR ("Failed to export the manager interfaces : %s", error->message);
        g_error_free (error);
        goto failed;
    }

    manager->bus = bus;
    manager->path = g_strdup (path);

    *man = manager;
    return 0;

failed:
    g_object_unref (manager->service);
    g_free (manager);
    return ret;
}

static int
manager_service_deinit (ManagerObject **man)
{
    int ret;

    return_val_if_fail (man, -EINVAL);
    return_val_if_fail (*man, -EALREADY);

    ret = manager_unregister_interfaces (*man);
    if (ret < 0) {
        ERROR ("Failed to unregister the manager interfaces [%s] : "
               "%s/%d", (*man)->path, strerror (-ret), -ret);
    }

    g_object_unref ((*man)->service);
    g_free ((*man)->path);
    g_free (*man);
    *man = NULL;

    return 0;
}

int
manager_accesspoint_added (const char *path)

{
    ManagerObject *man = manager;

    return_val_if_fail (path, -EINVAL);
    return_val_if_fail (man, -ENOTCONN);

    manager_emit_access_point_added (man->service, path);
    return 0;
}

int
manager_accesspoint_removed (const char *path)

{
    ManagerObject *man = manager;

    return_val_if_fail (path, -EINVAL);
    return_val_if_fail (man, -ENOTCONN);

    manager_emit_access_point_removed (man->service, path);
    return 0;
}

int
manager_init (GDBusConnection *conn)
{
    int ret;
    const char *path = WAPDMAN_MANAGER_OBJECT_PATH;

    return_val_if_fail (conn, -ENOLINK);

    DEBUG ("Exporting Manager instance on DBus connection: %p", conn);

    ret = manager_service_init (conn, path, &manager);
    if (ret < 0) {
        ERROR ("Failed to intialize the manager service : %s/%d",
               strerror (-ret), -ret);
        return ret;
    }

    ret = manager_register_interfaces (manager);
    if (ret < 0)
        ERROR ("Failed to register the manager interfaces : %s/%d",
               strerror (-ret), -ret);

    return ret;
}

int
manager_deinit ()
{
    int ret;

    DEBUG ("");

    ret = manager_service_deinit (&manager);
    if (ret < 0)
        ERROR ("Failed to cleanup the manager service");

    return ret;
}

/** @} */
